/**
* @file ATHAiboControl.cpp
*
* Implementation of the ATHAiboControl application
*
* @author Martin Ltzsch
*/

#include "stdafx.h"
#include "ATHAiboControl.h"
#include "ATHAiboControlDlg.h"
#include "Platform/GTAssert.h"
#include "Tools/Debugging/QueueFillRequest.h"
#include "Tools/Debugging/DebugKeyTable.h"
#include "Representations/JoystickData.h"
#include "Representations/Perception/CameraParameters.h"
#include "Tools/Debugging/DebugKeyTable.h"
#include "Tools/Debugging/QueueFillRequest.h"
#include "Tools/Module/SolutionRequest.h"

#define HAVE_BOOLEAN
#include "Representations/Perception/JPEGImage.h"

BEGIN_MESSAGE_MAP(CATHAiboControlApp, CWinApp)
//{{AFX_MSG_MAP(CATHAiboControlApp)
//}}AFX_MSG
ON_COMMAND(ID_HELP, CWinApp::OnHelp)
END_MESSAGE_MAP()


CATHAiboControlApp::CATHAiboControlApp()
: joystickIsConnected(false), wlanIsConnected(false),
dlg1(0), dlg2(0), configuration(ers7AllImages)
{
}

CATHAiboControlApp theApp;

CATHAiboControlApp* getATHAiboControlApp()
{
  return &theApp;
}

void CATHAiboControlApp::onChangeConfiguration()
{
  DebugKeyTable debugKeyTable;
  switch(configuration)
  {
  case ers7AllImages:
  case ers210AllImages:
    debugKeyTable.set(DebugKeyTable::sendJPEGImage,DebugKey::always);
    break;
  case ers210SomeImages:
  case ers7SomeImages:
    debugKeyTable.set(DebugKeyTable::sendJPEGImage,DebugKey::every_n_ms,500);
    break;
  }
  toRobot.out.bin << debugKeyTable;
  toRobot.out.finishMessage(idDebugKeyTable);

  SolutionRequest setting;
  setting.solutions[SolutionRequest::selfLocator] = SolutionRequest::disabled;
  setting.solutions[SolutionRequest::imageProcessor] = SolutionRequest::disabled;
  setting.solutions[SolutionRequest::behaviorControl] = SolutionRequest::ath2004BehaviorControlAgentJoystickControlled;
  setting.solutions[SolutionRequest::collisionDetector] = SolutionRequest::disabled;
  switch(configuration)
  {
  case ers7AllImages:
  case ers7SomeImages:
    setting.solutions[SolutionRequest::walkingEngineNormal] = SolutionRequest::invKinERS7WalkingEngine;
    break;
  case ers210AllImages:
  case ers210SomeImages:
    setting.solutions[SolutionRequest::walkingEngineNormal] = SolutionRequest::invKinGT2003WalkingEngine;
    break;
  }
  setting.solutions[SolutionRequest::ballLocator] = SolutionRequest::disabled;
  setting.solutions[SolutionRequest::teamBallLocator] = SolutionRequest::disabled;
  setting.solutions[SolutionRequest::playersLocator] = SolutionRequest::disabled;
  setting.solutions[SolutionRequest::headControl] = SolutionRequest::gt2003HeadControl;
  setting.solutions[SolutionRequest::sensorDataProcessor] = SolutionRequest::defaultSensorDataProcessor;
  setting.solutions[SolutionRequest::robotStateDetector] = SolutionRequest::defaultRobotStateDetector;
  setting.solutions[SolutionRequest::sensorBehaviorControl] = SolutionRequest::disabled;
  setting.solutions[SolutionRequest::motionControl] = SolutionRequest::defaultMotionControl;
  setting.solutions[SolutionRequest::getupEngine] = SolutionRequest::defaultGetupEngine;
  setting.solutions[SolutionRequest::soundControl] = SolutionRequest::defaultSoundControl;
  setting.solutions[SolutionRequest::sensorActorLoop] = SolutionRequest::disabled;
  toRobot.out.bin << setting;
  toRobot.out.finishMessage(idSolutionRequest);


  QueueFillRequest defaultQueueFillRequest;
  defaultQueueFillRequest.mode = QueueFillRequest::overwriteOlder;
  toRobot.out.bin << defaultQueueFillRequest;
  toRobot.out.finishMessage(idQueueFillRequest);

/*  CameraParameters defaultCameraParameters;
  defaultCameraParameters.theGain = CameraParameters::gain_high;
  defaultCameraParameters.theWhiteBalance = CameraParameters::wb_fl_mode;
  defaultCameraParameters.theShutterSpeed = CameraParameters::shutter_slow;
  toRobot.out.bin << defaultCameraParameters;
  toRobot.out.finishMessage(idCameraParameters);*/
}

BOOL CATHAiboControlApp::InitInstance()
{
  if (!AfxSocketInit())
  {
    AfxMessageBox(IDP_SOCKETS_INIT_FAILED);
    return FALSE;
  }
  
  SetRegistryKey(_T("www.robocup.de/aiboteamhumboldt"));
  
  ip = GetProfileInt("","ip",0);
  configuration = (Configuration)GetProfileInt("","configuration",0);

  // Mit der nchsten Zeile kann man das Aussehen des 2. Dialoges testen,
  // ohne dass das WLan verbunden sein muss.
  // dlg2 = new CATHAiboControlDlg2(); dlg2->DoModal(); delete dlg2; dlg2 = 0;

  while(1)
  {
    onChangeConfiguration();
    dlg1 = new CATHAiboControlDlg1();
    if (dlg1->DoModal() == IDCANCEL) return FALSE;
    delete dlg1;
    dlg1 = 0;
    
    if (wlanIsConnected && joystickIsConnected)
    {
      dlg2 = new CATHAiboControlDlg2();
      dlg2->DoModal();
      delete dlg2;
      dlg2 = 0;
      wlanThread.wlanDisconnect();
    }
  }
  return FALSE;
}

bool CATHAiboControlApp::handleMessage(InMessage& message)
{
  switch(message.getMessageID())
  {
  case idText:
    {
      ASSERT(message.getMessageSize());
      char* buf = (char*)malloc(message.getMessageSize()*2+1);
      message.text.readAll(buf);
      PRINT(buf);
      delete[](buf);
      return true;
    }
  case idJPEGImage:
    {
      JPEGImage jpegImage;
      message.bin >> jpegImage;
      jpegImage.toImage(image);
      updateDialogs();
      return true;
    }
  case idImage:
    {
      message.bin >> image;
      updateDialogs();
    }
  default:
    return false;
  }
}

BOOL CATHAiboControlApp::OnIdle(LONG lCount) 
{
  this->OnIdle();

  return CWinApp::OnIdle(lCount);
}

void CATHAiboControlApp::updateDialogs()
{
  if (dlg1 != 0) dlg1->update();
  if (dlg2 != 0) dlg2->update();
}

void CATHAiboControlApp::OnIdle() 
{
  JOYINFOEX ji;
  MMRESULT joyresult;

  joyresult=joyGetPosEx(JOYSTICKID1,&ji);

  if (joyresult == 0)
  {
    if (!joystickIsConnected) 
    {
      joystickIsConnected = true;
      updateDialogs();
    }
    sendJoystickData(ji);
  }
  else
  {
    if (joystickIsConnected)
    {
      joystickIsConnected = false;
      updateDialogs();
    }
  }

  {  
    SYNC_WITH(wlanThread);
    
    if (wlanThread.isRunning())
    {
      if (wlanThread.wlanIsConnected())
      {
        if (!wlanIsConnected)
        {
          wlanIsConnected = true;
          updateDialogs();
        }
        toRobot.moveAllMessages(wlanThread.queueToRobot);
      }
      else
      {
        if (wlanIsConnected)
        {
          wlanIsConnected = false;
          updateDialogs();
        }
      }
    }
  }
  {
    SYNC; // now no other threads can access the handled queues
    
    fromRobot.handleAllMessages(*this);
    fromRobot.clear();
  }

  if (wlanIsConnected && joystickIsConnected)
  {
    if (dlg1!=0) dlg1->EndModalLoop(0);
  }

}

void CATHAiboControlApp::sendJoystickData(JOYINFOEX& ji)
{
  JoystickData d;

  d.x = -1.0*((double)ji.dwYpos-32767.0)/32768.0;
  d.y = -1.0*((double)ji.dwXpos-32767.0)/32768.0;
  d.z = 1.0*((double)ji.dwRpos-32767.0)/32768.0;
  d.accel = -1.0*((double)ji.dwZpos-32767.0)/32768.0;
  d.button = ji.dwButtons;
  d.coolie = (ji.dwPOV==65535?-1:ji.dwPOV/4500) + 1;

  toRobot.out.bin << d;
  toRobot.out.finishMessage(idJoystickData);
  toRobot.removeRepetitions();
}

int CATHAiboControlApp::ExitInstance() 
{
  wlanThread.wlanDisconnect();
  WriteProfileInt("","ip",ip);
  WriteProfileInt("","configuration",configuration);

  return CWinApp::ExitInstance();
}

CATHAiboControlWLanThread::CATHAiboControlWLanThread()
{
  tcpConnection = 0;
  wlanNeedsToBeConnected = false;
  start(this,communicate);
}

void CATHAiboControlWLanThread::communicate()
{
  while(isRunning())
  {
    if(wlanNeedsToBeConnected)
    {
      if(tcpConnection)
        delete tcpConnection;
      char strRemoteIP[32];
      sprintf(strRemoteIP, "%i.%i.%i.%i", remoteIP >> 24,
        remoteIP >> 16 & 255,
        remoteIP >> 8 & 255,
        remoteIP & 255);
      tcpConnection = new TcpConnection(strRemoteIP, remotePort);
      
      if(wlanIsConnected())
      {
        char s[200];
        sprintf(s,"CATHAiboControlWLanThread : connected to %s:%i.",strRemoteIP,remotePort);
        PRINT(s);
      }
      else
      {
        char s[200];
        sprintf(s,"CATHAiboControlWLanThread : could not connect to %s:%i.",strRemoteIP,remotePort);
        PRINT(s);
      }
      wlanNeedsToBeConnected = false;
    }
    
    if(wlanIsConnected())
    {
      char* sendData,
        * receivedData;
      int sendSize = 0,
        receivedSize = 0;
      
      sendData = 0;
      
      // If there is something to send, prepare a package
      {
        SYNC;
        if(!queueToRobot.isEmpty())
        {
          OutBinarySize size;
          size << queueToRobot;
          sendSize = size.getSize();
          sendData = new char[sendSize];
          OutBinaryMemory stream(sendData);
          stream << queueToRobot;
        }
      }
      
      // exchange data with the router
      if(tcpConnection->sendAndReceive(sendData,sendSize,receivedData,receivedSize))
        queueToRobot.clear();
      
      // If a package was prepared, remove it
      if(sendSize)
        delete [] sendData;
      
      // If a package was received from the router program, add it to receiver queue
      if(receivedSize > 0)
      {
        InBinaryMemory stream(receivedData,receivedSize);
        stream >> queueFromRobot;
        delete [] receivedData;
        
        // move all messages to the queue from physical robots
        
        SYNC_WITH(*getATHAiboControlApp());
        queueFromRobot.moveAllMessages(getATHAiboControlApp()->fromRobot);
        
      }
    }
    Sleep(10);
  }
}

bool CATHAiboControlWLanThread::wlanIsConnected()
{
  if(tcpConnection && !tcpConnection->isConnected())
  {
    delete tcpConnection;
    tcpConnection = 0;
  }
  return tcpConnection != 0;
}

/** This function establishes the connection over the WLan interface and must be called before using it, of course **/
void CATHAiboControlWLanThread::wlanConnect(const unsigned long theRemoteIP, int theRemotePort)
{
  SYNC;
  remoteIP = theRemoteIP;
  remotePort = theRemotePort;
  wlanNeedsToBeConnected = true;
  char strRemoteIP[32];
  sprintf(strRemoteIP, "%i.%i.%i.%i", remoteIP >> 24,
    remoteIP >> 16 & 255,
    remoteIP >> 8 & 255,
    remoteIP & 255);
  char s[200];
  sprintf(s,"CATHAiboControlWLanThread : trying to connect to %s:%i.", strRemoteIP, remotePort);
  PRINT(s);
}

/** This function terminates an WLan connection **/
void CATHAiboControlWLanThread::wlanDisconnect()
{
  {
    SYNC;
    if(wlanIsConnected())
    {
    /** send empty DebugKeyTable and immediateReadWrite to prevent useless full queues on robot
    * @todo this might be not usefull when in QueueFillRequest::overwriteOlder mode
      */
      QueueFillRequest queueFillRequest;
      queueToRobot.out.bin << queueFillRequest;
      queueToRobot.out.finishMessage(idQueueFillRequest);
      DebugKeyTable debugKeyTable;
      queueToRobot.out.bin << debugKeyTable;
      queueToRobot.out.finishMessage(idDebugKeyTable);
    }  
  }
  Sleep(800);
  SYNC;
  if(wlanIsConnected())
  {
    delete tcpConnection;
    tcpConnection = 0;
    PRINT("CATHAiboControlWLanThread : disconnected");
  }
}

CATHAiboControlWLanThread::~CATHAiboControlWLanThread()
{
}


/*
* Change Log:
* 
* $Log: ATHAiboControl.cpp,v $
* Revision 1.5  2004/01/26 23:03:16  loetzsch
* ATHAiboControl disables Vision and Selflocalization, as they are too slow
* on non-SuperCore ERS210
*
* Revision 1.4  2004/01/26 17:47:42  loetzsch
* bug fix
*
* Revision 1.3  2004/01/26 13:35:02  loetzsch
* improved
*
* Revision 1.2  2004/01/24 20:18:40  loetzsch
* improved ATH AiboControl
*
* Revision 1.1  2004/01/24 14:55:28  loetzsch
* created ATH AiboControl
*
*/

