/**
* @file ATHAiboControl.h
*
* Definition of the ATHAiboControl application
*
* @author Martin Ltzsch
*/

#ifndef __ATHAiboControl_h_
#define __ATHAiboControl_h_

#include "StdAfx.h"
#include "resource.h"		
#include "Tools/MessageQueue/MessageQueue.h"
#include "Platform/Win32/Thread.h"
#include "Platform/Win32Linux/TCPConnection.h"
#include "Representations/Perception/Image.h"
#include <mmsystem.h>

/// class prototypes to avoid inclusion of ATHAiboControlDlg.h 
class CATHAiboControlDlg1; 
class CATHAiboControlDlg2; 

/**
* @class CATHAiboControlWLanThread
*
* A thread that manages the communication between the queues and the Wireless Lan
*
* @author Martin Ltzsch
*/
class CATHAiboControlWLanThread : public Thread<CATHAiboControlWLanThread>
{
public:
  DECLARE_SYNC; /**< Make this object synchronizable. */
  
  /** Constructor */
  CATHAiboControlWLanThread();
  
  /** Destructor */
  ~CATHAiboControlWLanThread();
  
  /** A queue for messages to be sent via the WLAN to the robot */
  MessageQueue queueToRobot;
  
  /** A queue for messages to be received from the WLAN */
  MessageQueue queueFromRobot;
  
  /** Returns whether WLAN connection is established */
  bool wlanIsConnected();
  
  /** This function establishes the connection over the WLan interface and must be called before using it, of course **/
  void wlanConnect(const unsigned long remoteIP, int remotePort);
  
  /** This function terminates an WLan connection **/
  void wlanDisconnect();
  
private:  
  /** The thread main function */
  void communicate();
  
  /** a TCP/IP client class handling the WLan interface **/
  TcpConnection* tcpConnection;
  unsigned long remoteIP;
  int remotePort;
  bool wlanNeedsToBeConnected;
};


/** 
* @class CATHAiboControlApp
* 
* The application class of ATHAiboControl
*/
class CATHAiboControlApp : public CWinApp, public MessageHandler
{
public:
  DECLARE_SYNC; /**< Make this object synchronizable. */

  /** Constructor */
	CATHAiboControlApp();

  /** The first configuration dialog */
  CATHAiboControlDlg1* dlg1;

  /** The second main dialog */
  CATHAiboControlDlg2* dlg2;

  /** updates the visible of the both dialogs */
  void updateDialogs();

  /** A message queue for messages to the robot */
  MessageQueue toRobot;

  /** A message queue for messages from the robot */
  MessageQueue fromRobot;

  /** The ip address of the robot */
  unsigned long ip;

  /** A WLAN thread */
  CATHAiboControlWLanThread wlanThread;

  /** Idle Processing */
  void OnIdle();

  /** Determines if a joystick is connected */
  bool joystickIsConnected;

  /** Determines if the WLAN connection is established */
  bool wlanIsConnected;

  /** the last received image */
  Image image;

  /** sends the joystick data to the robot */
  void sendJoystickData(JOYINFOEX& ji);

  /** different configurations */
  enum Configuration { ers7AllImages, ers7SomeImages, ers210AllImages, ers210SomeImages };

  /** The current configuration */
  enum Configuration configuration;

  /** is called when configuration changed */
  void onChangeConfiguration();

  //{{AFX_VIRTUAL(CATHAiboControlApp)
	public:
	virtual BOOL InitInstance();
	virtual int ExitInstance();
	virtual BOOL OnIdle(LONG lCount);
	//}}AFX_VIRTUAL


	//{{AFX_MSG(CATHAiboControlApp)
	//}}AFX_MSG
	DECLARE_MESSAGE_MAP()

  /** 
  * Called from a MessageQueue to distribute messages.
  * Use message.getMessageID to decide if the message is relavant for 
  * the MesssageHandler derivate.
  * Use message.bin, message.text or message.config as In streams to get the data from.
  * @param message The message that can be read.
  * @return true if the message was read (handled).
  */
  virtual bool handleMessage(InMessage& message);

};

/** Returns a pointer to the CATHAiboControlApp */
CATHAiboControlApp* getATHAiboControlApp();

#endif // __ATHAiboControl_h_

/** 
* Change Log:
*
* $Log: ATHAiboControl.h,v $
* Revision 1.2  2004/01/26 13:35:02  loetzsch
* improved
*
* Revision 1.1  2004/01/24 14:55:28  loetzsch
* created ATH AiboControl
*
*/
