/**
* @file ATH2004BallLocator.cpp
* 
* This file contains a hypothesis based ball-localization class.
*
* @author <a href="mailto:goehring@informatik.hu-berlin.de">Daniel Goehring</a>
* @author <a href="mailto:jhoffman@informatik.hu-berlin.de">Jan Hoffmann</a>
*/


/* 
to do's:

  + wie bewerten der BallHypothesen mit Abstands- und Winkeldifferenzen
  + consts in header
  + NORMALIZE CHANGES BY Frame!!!
  
    
      done's:
      + implement BallHypothesis class
      + implement BallHypothesis class methods
      + dynamisiere Hypothesengroesse in Header und Constructor
      + Hypothesen und Mutterklasse generiert
      + Hypothesen zufaellig erzeugt
      + Akkumulierte Bewertungen berechnet, Normalisierung der Werte
      + Hypothesen mit unterschiedlichen Farben
      + Odometrie auf die Hypothesen draufgerechnet
      + Ballbewegung im naechsten Takt propagiert
      + Propagierung hochbewerteter Hypothesen in naechsten Takt
      + Propagierung eines geringen neuen Anteils zufaellig
      + Verrauschung der Daten durch addNoise - Methode
      + 4 BallZustaende: Bounced, Grabbed, Kicked, Idle
      + 
*/

#include "ATH2004BallLocator.h"
#include "Tools/Debugging/Debugging.h"

#include "Tools/Actorics/RobotDimensions.h"
#include "Tools/FieldDimensions.h"
#include "Platform/SystemCall.h"
#include "Tools/Debugging/DebugDrawings.h"
#include <math.h>
#include "Tools/Math/Matrix2x2.h"
#include "Platform/GTAssert.h"



SingleBallHypothesis::SingleBallHypothesis()
{
  id  = random(10000);
  dob = 0;
  SingleBallHypothesis::reset();
}

void SingleBallHypothesis::set(double distance, double angle, double speed, double direction)
{
  this->distance  = distance;
  this->angle     = angle;
  this->speed     = speed;
  this->direction = direction;
}

void SingleBallHypothesis::reset()
{
  distance    = random(5000);
  angle       = random() * pi2;
  speed       = random(1500); 
  direction   = random() * pi2;
  evaluation  = 0.00001;
  accumulation= 0.00001;
  MUTATE = 0.1;
  g_Roll = 160;
  t_Call = 0.32;
}



void SingleBallHypothesis::propagatePosition()
{
  double x,y;
  
  //recalculate local ball distance + angle, polar - cartesian ... addition - polar
  
  //called every 32 ms can be considered as constant???
  
  x = cos(angle)*distance + cos(direction)*speed*t_Call; /////!!!!!0.032s
  y = sin(angle)*distance + sin(direction)*speed*t_Call;
  
  
  distance  = sqrt(x*x+y*y);
  
  if (x != 0)
  {
    angle   = atan(y/x);
    
    
    if (x < 0)
    {
      angle += pi;
    }
    
    
    if (speed > 0)
    {
      speed = speed - t_Call * g_Roll;  // time of execution mults drag coefficient
    }
    
    if (speed < 0)
    {
      speed = 0;
    }
    
    if (distance > 5000) 
    {
      reset();
    }
  }
  else
  {
    reset();
  }
  //recalculate velocity speed 
}


void SingleBallHypothesis::setDistance(double distance)
{
  this->distance = distance;
}

void SingleBallHypothesis::setAngle(double angle)
{
  this->angle = angle;
}

void SingleBallHypothesis::setSpeed(double speed)
{
  this->speed = speed;
}

void SingleBallHypothesis::setDirection(double direction)
{
  this->direction = direction;
}

void SingleBallHypothesis::compensateOdometry(Pose2D odometry)
{
  double x,y;
  
  //recalculate local ball distance + angle, polar - cartesian ... addition - polar
  x = cos(angle)*distance;
  y = sin(angle)*distance;
  x-= odometry.translation.x;
  y-= odometry.translation.y;
  
  distance  = sqrt(x*x+y*y);
  
  if (x != 0) 
  {
    angle     = atan(y/x);
    
    if (x < 0)
    {
      angle += pi;
    }
    
    angle -= odometry.rotation;
    
    //recalculate velocity direction 
    direction -= odometry.rotation;
  }
  else
  {reset();}
}

double SingleBallHypothesis::getDistance()
{
  return distance;
}

double SingleBallHypothesis::getAngle()
{
  return angle;
}

double SingleBallHypothesis::getSpeed()
{
  return speed;
}

double SingleBallHypothesis::getDirection()
{
  return direction;
}

BallHypothesis::BallHypothesis()
{
  reset();
}

void BallHypothesis::reset()
{
  for (int i = 0; i < maxNumOfHypos; i++)
  {
    hypos[i].reset();
  }
}


/** This method compensates the BallPositions by Odometry data
* for all Hypothesis
*/

void BallHypothesis::compensateOdometry(Pose2D odometry)
{
  for (int i = 0; i < maxNumOfHypos; i++)
  {
    hypos[i].compensateOdometry(odometry);
  }
}


ATH2004BallLocator::ATH2004BallLocator(const BallLocatorInterfaces& interfaces)
: BallLocator(interfaces)
{
  lastTimeBallSeen = 0;
  //KALMAN  
 /* 
  sx = sy = 1000;
  sx_ = sy_ = 0;
  px = py = 250000;
  px_ = py_ = 0;
  rx = ry = 500;
  kx = ky = 0;
  qx = qy = 100;
  */

  oldSeenPosX = 0;
  oldSeenPosY = 0;
  oldModPosX = 0;
  oldModPosY = 0;
  oldModSpeedX = 0;
  oldModSpeedY = 0;

  Sx.x = 1000;
  Sy.x = 1000;
  Sx.y = 0;
  Sy.y = 0;



  Px.c[0].x = 250000;
  Py.c[0].x = 250000;
  Px.c[1].x = 250000;
  Py.c[1].x = 250000;
  Px.c[0].y = 250000;
  Py.c[0].y = 250000;
  Px.c[1].y = 250000;
  Py.c[1].y = 250000;


  
  
  
  //KALMAN
}


BallHypothesis ballHypothesis;

void ATH2004BallLocator::execute()
{
  Vector2<double> ballPositionOnField;
  Vector2<double> ballVelocityOnField;
  
  
  double timeDiff = (double )(timeOfImageProcessing - lastTimeBallSeen)/1000; // 1/msec -> 1/sec
  lastTimeBallSeen = timeOfImageProcessing;
  
  if(!ballPercept.ballWasSeen)
  {
    consecutiveFramesBallSeen = 0;
    consecutiveFramesBallNotSeen++;
  }
  else // ball was seen
  { Vector2<double> ballOffset;
    ballPercept.getOffset(ballOffset);

    // if there is a gap of n frames during which the ball was not seen
    // reset the "timeWhenLastSeenConsecutively"
    if (consecutiveFramesBallNotSeen > 5)
    {
      ballPosition.seen.timeWhenFirstSeenConsecutively = timeOfImageProcessing;
    }
    
    // if ball has not been seen for a while but 
    // is seen now, directly (re-)set to percept...
    if (consecutiveFramesBallNotSeen > 10) 
    {
      Sx.x = ballOffset.x;
      Sy.x = ballOffset.y;
      Sx.y = 0;
      Sy.y = 0;
      oldSeenPosX = oldSeenPosY = oldModPosX = oldModPosY = oldModSpeedX = oldModSpeedY = 0;
    }
    // ball was seen recently
    else  
    {
      // calc ball speed:
      if ((consecutiveFramesBallSeen > 1) && (timeDiff > 0))
      {
      }
    }
    
    
    consecutiveFramesBallSeen++;
    consecutiveFramesBallNotSeen = 0;
    
    ballPosition.seen.timeWhenLastSeen = timeOfImageProcessing;
    
    
    //KALMAN BEGINS
   /*
    
    zx = ballOffset.x;
      sx_ = sx;         //old estimate
      px_ = px + qx;
      kx  = px_ / (px_ + rx);
      px  = (kx - 1) * (kx - 1) * px_ + kx * kx * rx;
      sx  = sx_ + kx * (zx - sx_);
      ballPosition.seen.x = sx;
    
      zy = ballOffset.y;
      sy_ = sy;
      py_ = py + qy;
      ky  = py_ / (py_ + ry);
      py  = (ky - 1) * (ky - 1) * py_ + ky * ky * ry;
      sy  = sy_ + ky * (zy - sy_);
      ballPosition.seen.y = sy;
    */
    
    
    Rx.c[0].x = 300;
    Rx.c[0].y = Rx.c[1].x = 200;
    Rx.c[1].y = 500;
    
    Ry.c[0].x = 500;
    Ry.c[0].y = Rx.c[1].x = 75;
    Ry.c[1].y = 150;


   // Qx.c[0].x = Qx.c[0].y = Qx.c[1].x = Qx.c[1].y = 100;
   // Qy.c[0].x = Qy.c[0].y = Qy.c[1].x = Qy.c[1].y = 100;   

    Qx.c[0].x = 500;
    Qy.c[0].x = 500;

    

// init ends



    
    Zx.x = ballOffset.x;
    //Zx.y = (ballOffset.x - oldSeenPosX) / 30;
    
    
    Sx_ = Sx;
    Sx_.x += oldModSpeedX/30;
    Px_ = Px + Qx;
    
    Kx = Px_ * (Px_ + Rx).invert();
    
    Sx = Sx_ + Kx * (Zx - Sx_);
    Px = (I - Kx) * Px_;
    
    // Y 
    
    //FrameDiffs einfuehren
    
     
    
    Zy.x = ballOffset.y;
    //Zy.y = (ballOffset.y - oldSeenPosY) / 30;
    
    //Propagate Ball Speed without Odometry
    
    Sy_ = Sy;
    Sy_.x += oldModSpeedY/30;
    Py_ = Py + Qy;
    
    Ky = Py_ * (Py_ + Ry).invert();
    
    Sy = Sy_ + Ky * (Zy - Sy_);
    Py = (I - Ky) * Py_;
    
    
    oldModPosX   = ballPosition.seen.x = Sx.x;
    oldModSpeedX = ballPosition.seen.speed.x = Sx.y;
    
    oldModPosY   = ballPosition.seen.y = Sy.x;
    oldModSpeedY = ballPosition.seen.speed.y = Sy.y;
    
    oldSeenPosX = ballOffset.x;
    oldSeenPosY = ballOffset.y;
 
    
    ballPosition.propagated.x  = ballPosition.seen.x;
    ballPosition.propagated.y  = ballPosition.seen.y;
    
    
    
    //KALMAN ENDS
    
    /*
    
      ballPosition.seen = 
      Geometry::relative2FieldCoord(robotPose, ballOffset.x, ballOffset.y); 
      
        ballPosition.propagated.x  = ballPosition.seen.x;
        ballPosition.propagated.y  = ballPosition.seen.y;
        
          
            if (SystemCall::getTimeSince(ballPosition.seen.timeWhenFirstSeenConsecutively) > 350)
            {
            ballPosition.seen.timeUntilSeenConsecutively = timeOfImageProcessing; 
            }
            }
            
              
                
                  
                    //compensate BallHypothesis by odometry
                    ballHypothesis.compensateOdometry(odometryData - lastRobotOdometry);
                    
                      // paint all new BallHypothesis out
                      for (int i = 0; i < BallHypothesis::maxNumOfHypos; i++) 
                      {
                      //transform local positions to field positions using self localization data
                      ballPositionOnField = 
                      Geometry::relative2FieldCoord(robotPose, cos(ballHypothesis.hypos[i].getAngle())
                      *ballHypothesis.hypos[i].getDistance(), sin(ballHypothesis.hypos[i].getAngle())
                      *ballHypothesis.hypos[i].getDistance());
                      
                        ballVelocityOnField =
                        Geometry::relative2FieldCoord(robotPose, cos(ballHypothesis.hypos[i].getAngle())
                        *ballHypothesis.hypos[i].getDistance() + cos(ballHypothesis.hypos[i].getDirection())
                        * 50, sin(ballHypothesis.hypos[i].getAngle())
                        *ballHypothesis.hypos[i].getDistance() + sin(ballHypothesis.hypos[i].getDirection())
                        * 50);
                        
                          //if ((i % 4) == 0)
                          //Drawings::black
                          ARROW(ballLocatorField, ballPositionOnField.x, ballPositionOnField.y, ballVelocityOnField.x, ballVelocityOnField.y, 2, 2,  ballHypothesis.hypos[i].getID()%12);
                          }
                          
                            Vector2<double> ballOffset;
                            ballPercept.getOffset(ballOffset);
                            ballPositionOnField = Geometry::relative2FieldCoord(robotPose, ballOffset.x, ballOffset.y);
                            CIRCLE(ballLocatorField, ballPositionOnField.x, ballPositionOnField.y, 50, 2, 0, Drawings::blue);
                            
                              
                                lastRobotOdometry = odometryData;
                                
                                  
                                    
    DEBUG_DRAWING_FINISHED(ballLocatorField);  */
  }  
}


void ATH2004BallLocator::compensateOdometry()
{
  lastRobotOdometry = odometryData;
}        



bool ATH2004BallLocator::handleMessage(InMessage& message)
{
  return false;
}

/*
* Change log :
* 
* $Log: ATH2004BallLocator.cpp,v $
* Revision 1.38  2004/05/22 16:28:45  goehring
* no message
*
* Revision 1.37  2004/05/18 21:17:59  goehring
* Kalman component improved
*
* Revision 1.36  2004/05/17 21:14:57  goehring
* ATH-Kalman implementation
*
* Revision 1.35  2004/04/21 20:11:40  goehring
* BallLocators added
*
* Revision 1.20  2003/12/28 17:52:33  goehring
* BallStates added, BallPositionPropagation by speed implemented
*
* Revision 1.19  2003/12/28 13:54:36  goehring
* Particle Propagation and Mutation implemented
*
* Revision 1.18  2003/12/28 12:22:10  goehring
* Probability Accumulation added
*
* Revision 1.17  2003/12/28 11:44:55  goehring
* Evaluation and Normalization beta added
*
* Revision 1.16  2003/12/20 14:50:10  goehring
* BallHypotheses odometry compensated, set-Value methods implemented
*
* Revision 1.15  2003/12/19 16:03:59  goehring
* BallHypothesises now relative to Robot, no odometry data implemented
*
* Revision 1.14  2003/12/19 11:04:48  goehring
* Hypothesis particles randomly generated and displayed on field -
* not yet transformed relatively to the robot's position.
*
* Revision 1.13  2003/12/18 15:34:15  jhoffman
* changed wording: rc configuration -> robotcontrol process layout
*
* Revision 1.12  2003/12/18 14:58:05  jhoffman
* fixed font size for splash screen, moved copyright notice into picture
*
* Revision 1.11  2003/12/17 15:35:06  goehring
* Ballhypothesis
*
* Revision 1.10  2003/12/16 16:55:05  goehring
* Ballhypthesis moved to ATH2004BallLocator
*
* Revision 1.9  2003/12/15 17:27:46  goehring
* Ballhypotheses extended
*
* Revision 1.8  2003/12/09 16:33:54  jhoffman
* added class ballhypothesis
*
* Revision 1.7  2003/12/04 13:28:40  goehring
* Concept change
*
* Revision 1.2  2003/12/02 15:12:16  goehring
* BallPosition algorithm added
*
* Revision 1.1  2003/12/02 13:51:01  goehring
* no message
*
*
*/
