/**
* @file KalmanBallLocator.h
* Contains a BallLocator implementation using Kalman filtering
*
* @author <a href="mailto:stefanuhrig@gmx.net">Stefan Uhrig</a>
*/
//------------------------------------------------------------------------------
#ifndef KALMANBALLLOCATOR_H_INCLUDED
#define KALMANBALLLOCATOR_H_INCLUDED
//------------------------------------------------------------------------------
#include "BallLocator.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "KalmanProcessModels/BaseModel.h"
#include "BallSideDetector.h"

// Replacing of new with DEBUG_NEW prevents including of STL headers
// --> undefine it
// NEVER replace such operators in headers because it influences all the
// following included headers. Also don't use using namespace ...
// Use such things only in the .cpp files where things are implemented
#ifdef new
  #ifdef DEBUG_NEW
    #undef new
  #endif
#endif
#include <vector>
//------------------------------------------------------------------------------
/**
* @class KalmanBallLocator
* A BallLocator using Kalman filtering. This class is the main class managing
* different Kalman process models
*/
class KalmanBallLocator : public BallLocator
{
public:
  KalmanBallLocator(BallLocatorInterfaces& interfaces);
  virtual ~KalmanBallLocator();

public:
  /**
  * Executes the KalmanBallLocator module
  */
  virtual void execute();
  /**
  * Handles idKalmanData debug messages
  * @param message The message
  * @return true if message was handled
  */
  virtual bool handleMessage(InMessage& message);

private:
  /**
  * Called if ball was seen. Performs update step.
  */
  void handleSeenBall();
  /**
  * Called if ball was not seen. Performs predict step.
  */
  void handleUnseenBall();
  /**
  * Checks if the ball rolls by right or left of the robot
  */
  void setBallState();
  /**
  * setUnknownResult is called if an exception occurs.
  * However, this should not happen, but if, this prevents the robot from crashing.
  */
  void setUnknownResult();
  /**
  * True if passed position is nonsense.
  * @param x x-position in millimeter
  * @param y y-position in millimeter
  * @return true if position is nonsense
  */
  bool isNonSensePos(double x, double y);
  /**
  * Adds a process model to the list of managed process models.
  * @param pModel Pointer to process model derived from KalmanProcessModelBase
  */
  void addModel(KalmanProcessModelBase* pModel);
  /**
  * Frees the managed process models
  */
  void destroyModels();
  /**
  * Sends the process model states as debug messages
  */
  void sendProcessModelStates();
  /**
  * Draws the global ball positions and data model ball positions on the field
  */
  void drawBallPosition();

private:
  //! Vector containing all instantiated process models
  std::vector<KalmanProcessModelBase*> kalmanModels;
  //! Vector containing the result of calls to the update function of the
  //! process models
  std::vector<KalmanUpdateResult> kalmanUpdateResults;
  //! Vector containing the result of calls to the predict function of the
  //! process models
  std::vector<KalmanPredictResult> kalmanPredictResults;

  double x_rel;  ///< x-position of ball relative to robot
  double y_rel;  ///< y-position relative to robot
  double x_abs;  ///< x-position of ball on field
  double y_abs;  ///< y-position of ball on field
  double vx_rel; ///< speed of ball in x-direction relative to robot
  double vy_rel; ///< speed of ball in y-direction relative to robot
  double vx_abs; ///< speed of ball in x-direction on field
  double vy_abs; ///< speed of ball in y-direction on field
  bool ballWasSeen; ///< true if ball was seen
  double last_seen_x_rel; ///< x-position of last seen ball relative to robot
  double last_seen_y_rel; ///< y-position of last seen ball relative to robot


  //! Last odometry data of robot
  OdometryData lastOdometry;

  //! If true, process model states are sent as debug messages
  bool bSendProcessModelStates;

  //! used for detecting whether the ball is fast and close
  BallSideDetector ballSideDetector;

  //! 1.2*length of field diagonal
  static double dFieldDiagonalLength;
};
//------------------------------------------------------------------------------
#endif
//------------------------------------------------------------------------------

/*
 * Change log :
 * $Log: KalmanBallLocator.h,v $
 * Revision 1.5  2004/04/20 14:12:16  uhrig
 *  - odometry processing corrected
 *  - fixed position model disabled
 *  - covariance matrices tuned
 *
 * Revision 1.4  2004/04/07 12:28:56  risler
 * ddd checkin after go04 - first part
 *
 * Revision 1.3  2004/03/31 00:09:46  risler
 * set last seen position when ball is not seen
 *
 * Revision 1.2  2004/03/30 15:48:06  risler
 * various bugfixes from stefan
 *
 * Revision 1.1.1.1  2004/03/29 08:28:51  Administrator
 * initial transfer from tamara
 *
 * Revision 1.3  2004/03/15 12:28:52  risler
 * change log added
 *
 *
 */
