/**
* @file ConstantSpeedModel.cpp
* Contains a KalmanBallLocator process model for balls with constant speed
*
* @author <a href="mailto:stefanuhrig@gmx.net">Stefan Uhrig</a>
*/
//------------------------------------------------------------------------------
#include "Tools/Math/MVTools.h"
#include "ConstantSpeedModel.h"
#include "Tools/Debugging/Debugging.h"
//------------------------------------------------------------------------------
// Discrete process update matrix (dt must be set during update)
double KalmanConstantSpeedModel::defaultA[16] =
{
  1.0, 0.0, 0.0, 0.0,
  0.0, 1.0, 0.0, 0.0,
  0.0, 0.0, 1.0, 0.0,
  0.0, 0.0, 0.0, 1.0
};
// Initial internal covariance matrix of filter
double KalmanConstantSpeedModel::defaultP[16] = 
{
  1.0, 0.0, 0.0, 0.0,
  0.0, 1.0, 0.0, 0.0,
  0.0, 0.0, 1.0, 0.0,
  0.0, 0.0, 0.0, 1.0
};
// Process covariance matrix
// Entries must be multiplied with powers of dt during update
// Assume variances of 0.004 m^2 and 0.004 m^2 per s^2
double KalmanConstantSpeedModel::defaultQ[16] =
{
  0.004, 0.000, 0.004, 0.000,
  0.000, 0.004, 0.000, 0.004,
  0.004, 0.000, 0.004, 0.000,
  0.000, 0.004, 0.000, 0.004
};
// Measurement covariance matrix
// Assume variances of 0.0001 m^2 and 0.0001 m^2 per s^2
double KalmanConstantSpeedModel::defaultR[16] =
{
  0.01*0.01, 0.00*0.00, 0.00*0.00, 0.00*0.00,
  0.00*0.00, 0.01*0.01, 0.00*0.00, 0.00*0.00,
  0.00*0.00, 0.00*0.00, 0.01*0.01, 0.00*0.00,
  0.00*0.00, 0.00*0.00, 0.00*0.00, 0.01*0.01
};
//------------------------------------------------------------------------------
KalmanConstantSpeedModel::KalmanConstantSpeedModel() : KalmanProcessModelBase()
{
  A = defaultA;
  globQ = defaultQ;
  globR = defaultR;
  reset();
}
//------------------------------------------------------------------------------
KalmanConstantSpeedModel::~KalmanConstantSpeedModel()
{
}
//------------------------------------------------------------------------------
void KalmanConstantSpeedModel::reset()
{
  initState = NotInited;
  P = defaultP;
  lastLiklihood = 0.0;
  lastTime = 0.0;
  double zeros[4] = {0.0, 0.0, 0.0, 0.0};
  x_act = zeros;
}
//------------------------------------------------------------------------------
KalmanUpdateResult
  KalmanConstantSpeedModel::update(double time, double x, double y)
{
  KalmanUpdateResult res;

  // Initialise filter if necessary
  if (initState == NotInited)
  {
    // Initialise position
    x_act[0] = z[0] = x;
    x_act[1] = z[1] = y;

    initState = PositionInited;
    lastTime = time;

    res.liklihood = 0.0;
    return res;
  }

  // Determine and check time difference 
  double dt = time - lastTime;
  if (MVTools::isNearZero(dt))
  {
    // Send last state if time difference is near zero
    res.state.x = x_act[0];
    res.state.y = x_act[1];
    res.state.vx = x_act[2];
    res.state.vy = x_act[3];
    res.liklihood = lastLiklihood;
    return res;
  }

  if (initState == PositionInited)
  {
    // Initialise speed
    x_act[2] = z[2] = (x - x_act[0]) / dt;
    x_act[3] = z[3] = (y - x_act[1]) / dt;
    x_act[0] = z[0] = x;
    x_act[1] = z[1] = y;

    initState = Ready;
    lastTime = time;

    res.liklihood = 0.0;
    return res;
  }

  
  try
  {
    // Adapt process covariance matrix
    double dt2 = dt*dt;
    double dt3 = dt2*dt;
    double dt4 = dt2*dt2;
    Q = globQ;
    Q[0][0] *= dt4;
    Q[0][2] *= dt3;
    Q[1][1] *= dt4;
    Q[1][3] *= dt3;
    Q[2][0] *= dt3;
    Q[2][2] *= dt2;
    Q[3][1] *= dt3;
    Q[3][3] *= dt2;
  
    // Perform time update
    try
    {
      A[0][2] = dt;
      A[1][3] = dt;
      x_minus = A*x_act;                              // (2.1)
      P_minus = A*P*transpose(A) + Q;                 // (2.2)
  
      // Update measured state
      z[2] = (x - z[0])/dt;
      z[3] = (y - z[1])/dt;
      z[0] = x;
      z[1] = y;
    }
    catch (MVException m)
    {
      // We shouldn't end up here      
      OutputException("time-update", m);
      reset();
      res.liklihood = 0.0;
      return res;
    }

    // Calculate weight (see equation 2.6)
    R = globR;
    C = P_minus + R;
    double determinant;
    try
    {
      C_inv = invert(C);
      determinant = det(C);
      if (determinant < 0.0)
        throw MVException();
      if (MVTools::isNearZero(determinant))
        throw MVException();
    }
    catch (MVException m)
    {
      // We shouldn't end up here      
      OutputException("invert_c", m);
      reset();
      res.liklihood = 0.0;
      return res;
    }
    
    double exponent = -0.5*(z - x_minus)*(C_inv*(z - x_minus));
    MVException m;
    if (exponent > MVTools::getMaxExpDouble())
    {
      // Sometimes exponent ist too high to represent the result with
      // a double. Set it to the maximum value
      exponent = MVTools::getMaxExpDouble();
    }
    double denominator = 4*pi*pi*sqrt(determinant);
    res.liklihood = exp(exponent)/(denominator);

    // Measurement update
    try
    {
      K = P_minus * C_inv;                            // (2.3)
      x_act = x_minus + K*(z - x_minus);              // (2.4)
      P = (P.eye() - K)*P_minus;                      // (2.5)
    }
    catch (MVException m)
    {
      // We should not end up here
      OutputException("measurement-update", m);
      reset();
      res.liklihood = 0.0;
      return res;
    }

    // Set state und save information of this update step
    lastLiklihood = res.liklihood;
    res.state.x = x_act[0];
    res.state.y = x_act[1];
    res.state.vx = x_act[2];
    res.state.vy = x_act[3];
    lastTime = time;
  }
  catch (...)
  {
    // We should not end up here
    OUTPUT(idText, text,
           "UnknownException in KalmanConstantSpeedModel::update()");
    reset();
    res.liklihood = 0.0;
    return res;
  }

  return res;
}
//------------------------------------------------------------------------------
KalmanPredictResult KalmanConstantSpeedModel::predict(double time)
{
  KalmanPredictResult res;

  // If model was not initialised yet we can't predict a state
  if (initState != Ready)
    return res;

  // Predict a state using the process model
  try
  {
    x_predict = A*x_act;
  }
  catch (MVException m)
  {
    // We should not end up here
    OutputException("prediction", m);
    reset();
    res.liklihood = 0.0;
    return res;
  }
  res.state.x = x_predict[0];
  res.state.y = x_predict[1];
  res.state.vx = x_predict[2];
  res.state.vy = x_predict[3];
  res.liklihood = lastLiklihood;

  return res;
}
//------------------------------------------------------------------------------
void KalmanConstantSpeedModel::adapt(const OdometryData& lastOdometry,
                                     const OdometryData& actualOdometry)
{
  // Adapt state and measured state to the new position of the robot
  Vector2<double> tempBallPosition = 
    (lastOdometry + Pose2D(x_act[0]*1000.0, x_act[1]*1000.0) -
     actualOdometry).translation;
  x_act[0] = tempBallPosition.x / 1000.0;
  x_act[1] = tempBallPosition.y / 1000.0;
    
  tempBallPosition = 
    (lastOdometry + Pose2D(z[0]*1000.0, z[1]*1000.0) -
     actualOdometry).translation;
  z[0] = tempBallPosition.x / 1000.0;
  z[1] = tempBallPosition.y / 1000.0;

  double vx, vy;
  double speed_rot = lastOdometry.getAngle() - actualOdometry.getAngle();
  vx = x_act[2];
  vy = x_act[3];
  x_act[2] = vx*cos(speed_rot) - vy*sin(speed_rot);
  x_act[3] = vx*sin(speed_rot) + vy*cos(speed_rot);
  
  vx = z[2];
  vy = z[3];
  z[2] = vx*cos(speed_rot) - vy*sin(speed_rot);
  z[3] = vx*sin(speed_rot) + vy*cos(speed_rot);
}
//------------------------------------------------------------------------------

/*
 * Change log :
 * $Log: ConstantSpeedModel.cpp,v $
 * Revision 1.4  2004/04/20 14:12:16  uhrig
 *  - odometry processing corrected
 *  - fixed position model disabled
 *  - covariance matrices tuned
 *
 * Revision 1.3  2004/04/07 12:28:56  risler
 * ddd checkin after go04 - first part
 *
 * Revision 1.2  2004/03/30 15:47:30  risler
 * covariance matrix fixed should be symmetric
 *
 * Revision 1.1.1.1  2004/03/29 08:28:51  Administrator
 * initial transfer from tamara
 *
 * Revision 1.2  2004/03/15 12:28:52  risler
 * change log added
 *
 *
 */
