/**
* @file MSH2004BallLocator.cpp
* 
* This file contains the default class for ball-localization.
*
* @author <a href="mailto:grypho@tempus-vivit.net">Carsten Schumann</a>
*/



#include "MSH2004BallLocator.h"
#include "Tools/Debugging/Debugging.h"

#include "Tools/FieldDimensions.h"
#include "Platform/SystemCall.h"
#include "Tools/Debugging/DebugDrawings.h"
#include <math.h>
#include "Platform/GTAssert.h"
#include "Tools/Math/Geometry.h"


static int const sensingZone1Border = 500;
static int const sensingZone2Border = 1000;
static int const sensingZone3Border = 1250;
static unsigned long const speedTrigger = 1000;


MSH2004BallLocator::MSH2004BallLocator(const BallLocatorInterfaces& interfaces)
: BallLocator(interfaces)
{
  lastTimeBallSeen = 0;
  lastSeenBallPositionX = 0;
  lastSeenBallPositionY = 0;
  lastPositionChangeX = 0;  
  lastPositionChangeY = 0;  
  predictedBallPositionX = 0;
  predictedBallPositionY = 0;
  this->currentFrame = 0;

  this->ringbufferIndex=-1;
  for (int i=0; i< MSH2004BallLocatorRingBufferSize; i++){
	this->ringbuffer[i]=NULL;
  }

  // initialising private variables
  freshSensed  = true;
  ballSensedPos.x = 0;
  ballSensedPos.y = 0;
  ballSensedTime = 0;
  ballSensedRelPos.x = 0;
  ballSensedRelPos.y = 0;

}

void MSH2004BallLocator::execute()
{
  this->currentFrame++;
  

//  OUTPUT(idText,text,"ODO: R"<<odometryData.rotation<<" X"<<odometryData.translation.x<<" Y"<<odometryData.translation.y);
//  OUTPUT(idText,text,"SL: R"<<robotPose.rotation<<" X"<<robotPose.translation.x<<" Y"<<robotPose.translation.y);
//  OUTPUT(idText,text,"SHIFTOFF= "<<odometryData.rotation-robotPose.rotation);



//  double timeDiff = (double )(timeOfImageProcessing - lastTimeBallSeen)/1000; // 1/msec -> 1/sec
//  lastTimeBallSeen = timeOfImageProcessing;

  
  if(ballPercept.ballWasSeen)
  {
  	ballPosition.validity=1.0;  
    ballPosition.seen.timeWhenLastSeen=SystemCall::getCurrentSystemTime();
//    OUTPUT(idText,text,"CONTACT.");
    this->addNewBallPercept(ballPercept, robotPose, odometryData);

    if (consecutiveFramesBallNotSeen > 5)
      ballPosition.seen.timeWhenFirstSeenConsecutively = timeOfImageProcessing;

    consecutiveFramesBallSeen++;
    consecutiveFramesBallNotSeen = 0;

    ballPosition.seen.timeWhenLastSeen = timeOfImageProcessing;

    if (SystemCall::getTimeSince(ballPosition.seen.timeWhenFirstSeenConsecutively) > 350)
    {
      ballPosition.seen.timeUntilSeenConsecutively = timeOfImageProcessing; 
    }

  } else {
    consecutiveFramesBallSeen = 0;
    consecutiveFramesBallNotSeen++;
    ballPosition.validity*=0.85;
  }
  Vector2<double> currentBallPosition;
  Vector2<double> ballMotion;
  if (this->ringbufferIndex>=0){
	  ballMotion=this->determineBallMotionVector(ballPosition.motionValidity);
	  currentBallPosition=this->extrapolateBallPosition(this->ringbufferIndex,ballMotion);

	  ballPosition.seen=this->ringbuffer[getRingbufferIndex(this->ringbufferIndex)]->getBallpositionOnField();
	  ballPosition.seen.speed=ballMotion;
//  Debug-Ausgabe

	  int i;
	  Vector2<double> ballOnFieldLast;
	  Vector2<double> ballOnField;
	  RobotPose pose;
	  RobotPose poseLast;
	  RobotPose correctedPoseLast;
	  RobotPose correctedPose;
	  for (i=0; i< MSH2004BallLocatorRingBufferSize && this->ringbufferIndex>=i ; i++){
		int currentIndex=getRingbufferIndex(this->ringbufferIndex-i);
		Drawings::Color ballColor=Drawings::gray;
		Drawings::Color poseColor=Drawings::gray;
		Drawings::Color correctedPoseColor=Drawings::gray;
		Drawings::Color ballMovement=Drawings::gray;
    
		if (this->ringbuffer[currentIndex]->entryRecent(this->currentFrame)){
		  ballColor=Drawings::yellow;
		  poseColor=Drawings::black;
		  correctedPoseColor=Drawings::blue;
		  ballMovement=Drawings::red;
		}


		if (i>0){
		  poseLast=pose;
		  pose=this->ringbuffer[currentIndex]->getPose();
		  LINE(ballLocatorField,(int)pose.translation.x,(int)pose.translation.y,(int)poseLast.translation.x,(int)poseLast.translation.y,1,0,poseColor);
		} else {
		  pose=this->ringbuffer[currentIndex]->getPose();
		}      
		CIRCLE(ballLocatorField,(int)pose.translation.x,(int)pose.translation.y,41-4*i,1,Drawings::ps_solid,poseColor);

    
    
		if (i>0){
		  correctedPoseLast=correctedPose;
		  correctedPose=this->ringbuffer[currentIndex]->getCorrectedPose();
		  LINE(ballLocatorField,(int)correctedPose.translation.x,(int)correctedPose.translation.y,(int)correctedPoseLast.translation.x,(int)correctedPoseLast.translation.y,1,0,correctedPoseColor);
		} else {
		  correctedPose=this->ringbuffer[currentIndex]->getCorrectedPose();
		}
		CIRCLE(ballLocatorField,(int)correctedPose.translation.x,(int)correctedPose.translation.y,41-4*i,1,Drawings::ps_solid,correctedPoseColor);


    
		int linewidth;
		Drawings::PenStyle penstyle;
		if (this->ringbuffer[currentIndex]->getMovementValidity()>0.5){
			linewidth=3;
			penstyle=Drawings::ps_solid;		
		} else {
			linewidth=1;
			penstyle=Drawings::ps_dot;		
		}
		Vector2<double> currentBallMovement;
		if (i>0){
		  ballOnFieldLast=ballOnField;
		  ballOnField=this->ringbuffer[currentIndex]->getBallpositionOnField();
		  currentBallMovement=this->ringbuffer[currentIndex]->getBallMovement();
		  LINE(ballLocatorField,(int)ballOnField.x,(int)ballOnField.y,(int)ballOnField.x+(currentBallMovement.x*1000),(int)ballOnField.y+(currentBallMovement.y*1000),linewidth,penstyle,ballMovement);
		  LINE(ballLocatorField,(int)ballOnField.x,(int)ballOnField.y,(int)ballOnFieldLast.x,(int)ballOnFieldLast.y,1,0,ballColor);
	//      if (this->entryRecent(currentIndex)){
	//        OUTPUT(idText,text,"Showing motion between "<<currentIndex<<" and "<<(currentIndex-1)%10<<" as active ");
	//      }
		} else {
		  ballOnField=this->ringbuffer[currentIndex]->getBallpositionOnField();
		  currentBallMovement=this->ringbuffer[currentIndex]->getBallMovementPerMS();
		  LINE(ballLocatorField,(int)ballOnField.x,(int)ballOnField.y,(int)ballOnField.x+(currentBallMovement.x*1000),(int)ballOnField.y+(currentBallMovement.y*1000),linewidth,penstyle,ballMovement);

		}

		CIRCLE(ballLocatorField,(int)ballOnField.x,(int)ballOnField.y,41-4*i,1,Drawings::ps_solid,ballColor);
	  }

	  ballOnField=this->ringbuffer[getRingbufferIndex(this->ringbufferIndex)]->getBallpositionOnField();
	  CIRCLE(ballLocatorField,(int)currentBallPosition.x,(int)currentBallPosition.y,42,1,Drawings::ps_solid,Drawings::red);
	  LINE(ballLocatorField,(int)ballOnField.x,(int)ballOnField.y,(int)ballOnField.x+(ballMotion.x*1000),(int)ballOnField.y+(ballMotion.y*1000),3,0,Drawings::blue);
  } else {
	if (ballPosition.validity>0.1) {
		  ballPosition.validity/=1.5;
	}
    ballPosition.seen.x=0;
    ballPosition.seen.y=0;
    ballPosition.seen.speed.x=0;
    ballPosition.seen.speed.y=0;
    ballPosition.motionValidity=0;
  }
  DEBUG_DRAWING_FINISHED(ballLocatorField); 



  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ballPosition.seen);
  double ballAngle    = toDegrees(Geometry::angleTo(robotPose.getPose(),
                                                    ballPosition.seen));
  Vector2<double> ballRelPos;
  ballRelPos.x = ballDistance * 
    cos(Geometry::angleTo(robotPose.getPose(),ballPosition.seen));
  ballRelPos.y = ballDistance * 
    sin(Geometry::angleTo(robotPose.getPose(),ballPosition.seen));

  if ((ballDistance > sensingZone1Border) && (ballDistance < sensingZone2Border))
  { 
    ballSensedPos = ballPosition.seen;
    ballSensedRelPos = ballRelPos;
    if ((ballAngle > -90) && (ballAngle <= 90))
      ballSensedTime = SystemCall::getCurrentSystemTime();
  }

  if (ballDistance > sensingZone3Border) {
    freshSensed  = true;
    ballPosition.ballState = BallModel::none;
  }

  if (    (ballAngle > -90)
       && (ballAngle <  90)
       && (ballDistance  <  sensingZone1Border)
       && (freshSensed == true))
  {
    freshSensed = false;
    if (    (SystemCall::getCurrentSystemTime() - ballSensedTime <  speedTrigger)
         && (ballSensedRelPos.x - ballRelPos.x  >= 1))
    {
      double side;
      side = ballSensedRelPos.y + (ballSensedRelPos.x / (ballRelPos.x - ballSensedRelPos.x))*(ballRelPos.y - ballSensedRelPos.y);
      if (side < 0) // ball passed on right side
        ballPosition.ballState = BallModel::ballRollsByRight;
      else
        ballPosition.ballState = BallModel::ballRollsByLeft;
    }
  }


}


Vector2<double> MSH2004BallLocator::extrapolateBallPosition(int lastInBufferID,Vector2<double> ballMotion){
	if (lastInBufferID>=0){
	  MSH2004BallLocatorElement* lastInBuffer=this->ringbuffer[getRingbufferIndex(lastInBufferID)];
	
	  long timeDifference=this->ringbuffer[getRingbufferIndex(lastInBufferID)]->getAge(this->currentFrame);
	  if (timeDifference<10){ // < 10 ms
		  return lastInBuffer->getBallpositionOnField();
	  } else {
  		Vector2<double> newBallPosition;
	  	return lastInBuffer->getBallpositionOnField()+(ballMotion*timeDifference);
	  } 
	} else {
		Vector2<double> ret;
		ret.x=0;
		ret.y=0;
		return ret;
	}
	

}



bool MSH2004BallLocator::handleMessage(InMessage& message)
{
  return false;
}


Matrix2x2<double> MSH2004BallLocator::getShiftoffMatrix(){
  //only the angle matters, not the displacement vector!!
  //calculate matrix.
  double integratedShiftoff=0;
  double sum=0;
  for (int i=0; i< MSH2004BallLocatorRingBufferSize && this->ringbufferIndex>=i ; i++){
    integratedShiftoff+=this->ringbuffer[getRingbufferIndex(this->ringbufferIndex-i)]->getRotationShiftoff()*(MSH2004BallLocatorRingBufferSize-i+1);
    sum+=MSH2004BallLocatorRingBufferSize-i+1;
  }
  integratedShiftoff/=sum;

  Matrix2x2<double> shiftoff;
  shiftoff.c[0].x=-cos(integratedShiftoff);
  shiftoff.c[1].x=-sin(integratedShiftoff);
  shiftoff.c[0].y=sin(integratedShiftoff);
  shiftoff.c[1].y=shiftoff.c[0].x;
  
//  OUTPUT(idText,text,"RingbufferShiftoff="<<integratedShiftoff);

  return shiftoff;

}

void MSH2004BallLocator::addNewBallPercept(const BallPercept& ballPercept, const RobotPose& robotPose, const OdometryData& odometryPose){
  this->ringbufferIndex++;

  int currentPercept=getRingbufferIndex(this->ringbufferIndex);
  int lastPercept=getRingbufferIndex(this->ringbufferIndex-1);

  for (int i=0; i<MSH2004BallLocatorRingBufferSize; i++){
  	if (ringbuffer[i]!=NULL){
	    ringbuffer[i]->clearTempData();
  	}
  }

  if (this->ringbuffer[currentPercept] != NULL) {
    delete this->ringbuffer[currentPercept];
  }
  this->ringbuffer[currentPercept]=new MSH2004BallLocatorElement(ballPercept,robotPose,odometryData,this->currentFrame);	


  //if more than one percept is in buffer (percept0 + percept1)
  if (this->ringbufferIndex>=1){
    Pose2D localizationMovement=this->ringbuffer[currentPercept]->getPose()-this->ringbuffer[lastPercept]->getPose();
    Pose2D odometryMovement=this->ringbuffer[currentPercept]->getOdometry()-this->ringbuffer[lastPercept]->getOdometry();
    Pose2D movementOffset=localizationMovement-odometryMovement;
    this->recalculatePosesInBuffer();
    this->recalculateMotionVectorsInBuffer();
  }
}

void MSH2004BallLocator::recalculatePosesInBuffer(){

  // get affine figure to transform odometry coordinates into field coordinates
  Matrix2x2<double> shiftoffMatrix= this->getShiftoffMatrix();

  // because we assume that the current location (by selflocator) is right, we can calculate the ball position now.
  this->ringbuffer[getRingbufferIndex(this->ringbufferIndex)]->recalculateBallOnField();


  for (int i=1; i< MSH2004BallLocatorRingBufferSize && this->ringbufferIndex>=i ; i++){
	MSH2004BallLocatorElement* currentElement=this->ringbuffer[getRingbufferIndex(this->ringbufferIndex-i)];

	//calculate a new position for element i based on position of element i-1 (i-1 is in the future)
	currentElement->recalculatePose(
		this->ringbuffer[getRingbufferIndex(this->ringbufferIndex-i+1)],
		shiftoffMatrix
	);
	//now element i has a valid position assuming, element 0 has a valid position.
    //for all elements in ringbuffer do ball position calculation based on ball percept
    //at time i and new calculated robot position at time i.
	currentElement->recalculateBallOnField();
  }

}

void MSH2004BallLocator::recalculateMotionVectorsInBuffer(){
  int i;
  for (i=1; i< MSH2004BallLocatorRingBufferSize && this->ringbufferIndex>=i ; i++){
    int currentIndex=getRingbufferIndex(this->ringbufferIndex-i);
    int futuralIndex=getRingbufferIndex(this->ringbufferIndex-i+1);
	this->ringbuffer[currentIndex]->recalculateBallMovement(this->ringbuffer[futuralIndex] , currentFrame);
  }
  for (i=1; i< MSH2004BallLocatorRingBufferSize && this->ringbufferIndex>=i ; i++){
	MSH2004BallLocatorElement* currentElement= this->ringbuffer[getRingbufferIndex(this->ringbufferIndex-i)];
	for (int j=1; j< MSH2004BallLocatorRingBufferSize-i && this->ringbufferIndex>=i+j ; j++){
	  MSH2004BallLocatorElement* comparatorElement= this->ringbuffer[getRingbufferIndex(this->ringbufferIndex-i-j)];
	  if (currentElement->movementInvertedTo(comparatorElement) && comparatorElement->entryRecent(this->currentFrame)){
	    currentElement->setMovementValidity(0.0);
		comparatorElement->setMovementValidity(0.0);
	  }
	}
  }
}


int MSH2004BallLocator::getRingbufferIndex(int index){
  return index & MSH2004BallLocatorRingBufferMask;
  


}

Vector2<double> MSH2004BallLocator::determineBallMotionVector(double& motionValidity){
  Vector2<double> motionVector;
  motionVector.x=0;
  motionVector.y=0;
  double overallWeight=0;
  double validitySum=0;
  int count=0;
//OUTPUT(idText,text,"-----CurrentFrame="<<this->currentFrame);
  for (int i=1; i< MSH2004BallLocatorRingBufferSize && 
	            this->ringbufferIndex>=i && 
				this->ringbuffer[getRingbufferIndex(this->ringbufferIndex-i)]->entryRecent(this->currentFrame); 
       i++){
    count++;
	  MSH2004BallLocatorElement* currentElement=this->ringbuffer[getRingbufferIndex(this->ringbufferIndex-i)];
    validitySum+=currentElement->getMovementValidity();
    const Vector2<double>& ballMovement=currentElement->getBallMovementPerMS();
    double currentWeight= (MSH2004BallLocatorRingBufferSize-i)*currentElement->getMovementValidity();
//  OUTPUT(idText,text,"BallMovement: X="<<ballMovement.x<<" Y="<<ballMovement.y<<" weight="<<currentWeight);
    motionVector+= ballMovement*currentWeight;
    overallWeight+=currentWeight;
  }

  //normalize motion vector
//division durch 0 wenn nur 1 percept in queue
	motionVector/=overallWeight;
//  OUTPUT(idText,text,"CurrentMovement: "<<motionVector.abs()<<" at Percept "<<this->ringbufferIndex);
//  OUTPUT(idText,text,"OverallMovement: X="<<motionVector.x<<" Y="<<motionVector.y);

  if (count>0)  {
    motionValidity=validitySum/count;
  } else {
    motionValidity=0.0;
  }

  return motionVector;

}

MSH2004BallLocator::~MSH2004BallLocator(){
  for (int i=0; i<MSH2004BallLocatorRingBufferSize; i++){
	if (this->ringbuffer[i]!=NULL){
      delete(this->ringbuffer[i]);
	  this->ringbuffer[i]=NULL;
	}
  }
}
	

/*
* Change log :
* 
* $Log: MSH2004BallLocator.cpp,v $
* Revision 1.16  2004/05/12 19:33:12  kerdels
* merged the behavior changes during australian, american and japan open
*
* Revision 1.15  2004/04/08 15:33:00  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.14  2004/03/08 00:58:36  roefer
* Interfaces should be const
*
* Revision 1.13  2004/01/28 15:39:59  schumann
* added time since last seen for ball percept
*
* Revision 1.12  2004/01/27 14:26:46  lohmann
* changes by Schumann
*
* Revision 1.11  2004/01/26 12:40:52  schumann
* comments added
*
* Revision 1.10  2004/01/22 20:08:00  schumann
* added validity for movement
* added elemination of jumping ball
* improved calculation of ball movement
*
* Revision 1.9  2004/01/21 11:48:52  schumann
* Added motion extrapolation and cleaned up balllocator
*
* Revision 1.8  2004/01/15 14:05:17  schumann
* Motion is calculated now. Still no output to following modules
*
* Revision 1.7  2003/12/31 23:50:35  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.6  2003/12/29 15:51:13  roefer
* Again, for the Nth time, warnings removed. Will that ever stop?
*
* Revision 1.5  2003/12/23 13:40:41  roefer
* Warnings removed
*
* Revision 1.4  2003/12/19 12:53:27  schumann
* some minor additions
*
* Revision 1.3  2003/12/16 19:43:06  roefer
* Warnings removed
*
* Revision 1.2  2003/12/10 11:09:40  schumann
* added loclization ringbuffer with odometrical compensation of localization gaps
*
* Revision 1.1  2003/12/03 13:29:27  schumann
* added MSH2004 Ball Locator
*
*
*
*/
