/**
 * @file MSH2004BallLocator.h
 * 
 * This file contains default class for ball-localization.
 *
 * @author <a href="mailto:grypho@tempus-vivit.net">Carsten Schumann</a>
 */

#ifndef __MSH2004BallLocator_h_
#define __MSH2004BallLocator_h_

#include "BallLocator.h"
#include "Tools/Debugging/GenericDebugData.h"
#include "Tools/Math/Matrix2x2.h"
#include "MSH2004BallLocatorElement.h"

/**
* @class MSH2004BallLocator
*
* A solution of the BallLocator module.
*
*
* @author <a href="mailto:grypho@tempus-vivit.net">Carsten Schumann</a>
*/

//rotation - percept

typedef struct 
{
  RobotPose pose;
  RobotPose correctedPose;
  BallPercept ballPercept;
  Vector2<double> ballOnField;
  Vector2<double> ballMovement;
  long systemTime;
  long systemFrame;
  bool jumpDetected;
  OdometryData odometry;



} MSH2004BallLocatorRingBufferElement;

const int MSH2004BallLocatorRingBufferSize=8; //must be 2^n (2,4,8,16,...)
const int MSH2004BallLocatorRingBufferMask=7;

class MSH2004BallLocator : public BallLocator
{
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the BallLocator module.
  */
  MSH2004BallLocator(const BallLocatorInterfaces& interfaces);
  ~MSH2004BallLocator();
  /** Executes the module */
  virtual void execute();

private:
    MSH2004BallLocatorElement* ringbuffer[MSH2004BallLocatorRingBufferSize];
    double odometryRotationOffsetRingbuffer[MSH2004BallLocatorRingBufferSize];
    int ringbufferIndex;
	
	long currentFrame;
    Vector2<double> lastBallSeen;
    Pose2D lastBallPercept,
           lastRobotOdometry; 
    long consecutiveFramesBallSeen;
    long consecutiveFramesBallNotSeen;
    unsigned long lastTimeBallSeen;

    Vector2<double> relative2FieldCoord(RobotPose rp, double x, double y);
    void compensateOdometry();

    bool handleMessage(InMessage& message);

    double lastSeenBallPositionX;
    double lastSeenBallPositionY;
    double lastPositionChange;
    double lastPositionChangeX;
    double lastPositionChangeY;
    double predictedBallPositionX;
    double predictedBallPositionY;

    
    
    Vector2<double> getMovementVectorPerSecond(const MSH2004BallLocatorRingBufferElement event1, const MSH2004BallLocatorRingBufferElement event2);
    void addNewBallPercept(const BallPercept& ballPercept, const RobotPose& robotPose, const OdometryData& odometryMovementSinceLastPercept);
    void recalculatePosesInBuffer();
    void recalculateMotionVectorsInBuffer();

    Vector2<double> determineBallMotionVector(double& motionValidity);
	Vector2<double> extrapolateBallPosition(int lastInBufferID,Vector2<double> ballMotion);
    Matrix2x2<double> getShiftoffMatrix();
    int getRingbufferIndex(int index);

  // Variables used to determine if the ball just passed by on the
  // left or right side of the robot
  bool freshSensed;
  Vector2<double> ballSensedPos;
  unsigned long   ballSensedTime;
  Vector2<double> ballSensedRelPos;

};

#endif// __MSH2004BallLocator_h_

/*
 * Change log : 
 * $Log: MSH2004BallLocator.h,v $
 * Revision 1.8  2004/05/12 19:33:12  kerdels
 * merged the behavior changes during australian, american and japan open
 *
 * Revision 1.7  2004/03/08 00:58:36  roefer
 * Interfaces should be const
 *
 * Revision 1.6  2004/01/22 20:08:00  schumann
 * added validity for movement
 * added elemination of jumping ball
 * improved calculation of ball movement
 *
 * Revision 1.5  2004/01/21 11:48:52  schumann
 * Added motion extrapolation and cleaned up balllocator
 *
 * Revision 1.4  2004/01/15 14:05:17  schumann
 * Motion is calculated now. Still no output to following modules
 *
 * Revision 1.3  2003/12/19 12:53:27  schumann
 * some minor additions
 *
 * Revision 1.2  2003/12/10 11:09:40  schumann
 * added loclization ringbuffer with odometrical compensation of localization gaps
 *
 * Revision 1.1  2003/12/03 13:29:27  schumann
 * added MSH2004 Ball Locator
 *
 */
