/**
* @file PIDSmoothedBallLocator.cpp
* 
* This file contains the default class for ball-localization.
*
* @author <a href="mailto:jhoffman@informatik.hu-berlin.de">Jan Hoffmann</a>
*/

#include "PIDSmoothedBallLocator.h"
#include "Tools/FieldDimensions.h"
#include "Platform/SystemCall.h"
#include "Tools/Debugging/DebugDrawings.h"
#include <math.h>
#include "Tools/Math/Matrix2x2.h"
#include "Platform/GTAssert.h"


PIDSmoothedBallLocator::PIDSmoothedBallLocator(const BallLocatorInterfaces& interfaces)
: BallLocator(interfaces),
ballPosX (0, 0.9, 0, 0, -xPosOpponentGoal, xPosOpponentGoal, 500000),
ballPosY (0, 0.9, 0, 0, -yPosLeftFlags, yPosLeftFlags, 500000),
ballSpeedX(0, 0.3, 0, 0, -500, 500, 100000),
ballSpeedY(0, 0.3, 0, 0, -500, 500, 100000),
ballAngle(0, 0.9, 0, 0, -pi, pi, 500000),
ballDistance(0, 0.5, 0, 0, 0, 10000),
consecutiveFramesBallSeen(0),
consecutiveFramesBallNotSeen(0),
ballSideDetector(ballPercept, robotPose, calibrationRequest, ballPosition)
{
  lastTimeBallSeen = 0;
}

/*
smoothing in the PID smoothed value should not be step based
but time based!
*/
void PIDSmoothedBallLocator::execute()
{
  ballSideDetector.execute();

  //determine a ball in front of a close goal.
  ballPosition.seen.ballInFrontOfOpponentGoal = false;

  Player::teamColor ownColor = getPlayer().getTeamColor();
  colorClass opponentGoalColor = ownColor == Player::red ? skyblue : yellow;
  
  if(landmarksPercept.numberOfGoals > 0)
  {
    double angleToBall = ballPercept.getAngleSizeBased();
    double ballDistance = ballPercept.getDistanceSizeBased();
    if(
      landmarksPercept.goals[0].color == opponentGoalColor &&
      ballPercept.ballWasSeen &&
      landmarksPercept.goals[0].x.min < angleToBall &&
      angleToBall < landmarksPercept.goals[0].x.max &&
      ballDistance < 1100
      )
    {
      ballPosition.seen.ballInFrontOfOpponentGoal = true;
    }
  }

  double timeDiff = (double )(timeOfImageProcessing - lastTimeBallSeen)/1000; // 1/msec -> 1/sec
  lastTimeBallSeen = timeOfImageProcessing;
	Vector2<double> ballOnField;

  compensateOdometry();

  // calculate error
  double const ballMeasurementError = 159.1 * tan(openingAngleHeight_ERS210/2)/176*3;
  ballPosition.seen.distanceError = 
    ballMeasurementError *
    ballPercept.getDistance() * ballPercept.getDistance() /
    (ballRadius * 159.1 /* focal length ers219 */);
  ballPosition.seen.angleError = 0.05;

  // ball not seen
  if(!ballPercept.ballWasSeen)
  {
    consecutiveFramesBallSeen = 0;
    consecutiveFramesBallNotSeen++;

	  Vector2<double> propSpeed = ballPosition.propagated.getPropagatedSpeed(timeOfImageProcessing);
	  Vector2<double> propBallPos = ballPosition.propagated.getPropagatedPosition(timeOfImageProcessing);
		    
	  ballSpeedX.resetTo(propSpeed.x);
    ballSpeedY.resetTo(propSpeed.y);

    ballPosX.resetTo(propBallPos.x); 
    ballPosY.resetTo(propBallPos.y); 
		ballAngle.resetTo(propBallPos.angle());
		ballDistance.resetTo(propBallPos.abs());

    ballOnField = Geometry::relative2FieldCoord(robotPose, propBallPos.x, propBallPos.y);
    CIRCLE(ballLocatorField, ballOnField.x, ballOnField.y, 90, 3, 0, Drawings::blue);

    // move seen to compensate for jumping localization:
    ballOnField = Geometry::relative2FieldCoord(robotPose, lastBallSeen.x, lastBallSeen.y);
    ballPosition.seen.x = ballOnField.x;
    ballPosition.seen.y = ballOnField.y;
  }
  else // ball was seen
  {
    // if there is a gap of n frames during which the ball was not seen
    // reset the "timeWhenLastSeenConsecutively"
    if (consecutiveFramesBallNotSeen > 5)
      ballPosition.seen.timeWhenFirstSeenConsecutively = timeOfImageProcessing;

    Vector2<double> ballOffset;
    ballPercept.getOffset(ballOffset);

    // calc ball position:
    if (consecutiveFramesBallNotSeen > 10) // if ball has not been seen for a while but is seen now, directly set to percept...
    {
      ballPosX.resetTo(ballOffset.x); 
      ballPosY.resetTo(ballOffset.y);
			ballAngle.resetTo(ballPercept.getAngle());
			ballDistance.resetTo(ballPercept.getDistance());

      ballSpeedX.reset();
      ballSpeedY.reset();
			
			//ballOnField = Geometry::relative2FieldCoord(robotPose, ballPosX.getVal(), ballPosY.getVal());
			//debug 

      //clipBallOutOfRobot(ballDistance,ballAngle);
			ballOnField = Geometry::relative2FieldCoord(robotPose, ballDistance.getVal()*cos(ballAngle.getVal()), ballDistance.getVal()*sin(ballAngle.getVal()));
			CIRCLE(ballLocatorField, ballOnField.x, ballOnField.y, 200, 3, 0, Drawings::red);  
    }
    else  // if it has been seen recently, perform smoothing...
    {
      // calc ball speed:
      if ((consecutiveFramesBallSeen > 1) && timeDiff > 0)
      {
        double vx = (ballOffset.x - ballPosX.getVal())/timeDiff;
        double vy = (ballOffset.y - ballPosY.getVal())/timeDiff;

        double deltaDistance  = (ballPercept.getDistanceBearingBased() + 
                                ballPercept.getDistanceSizeBased())/2
                                 - ballDistance.getVal();
        double deltaAngle     = ballPercept.getAngle()  - ballAngle.getVal();

        // compare if the deltas are greater than 
        // the standard deviation (error) of the 
        // ball measurement:
        if ((fabs(deltaDistance) < ballPosition.seen.distanceError) && 
          (fabs(deltaAngle) < ballPosition.seen.angleError))
        {
          //OUTPUT(idText, text, "---");
          ballSpeedX.resetTo(0.0);
          ballSpeedY.resetTo(0.0);
        }
        else if (consecutiveFramesBallSeen > 1)
        {
          ballSpeedX.approximateVal(vx);
          ballSpeedY.approximateVal(vy);
        }
        else
        {
          ballSpeedX.resetTo(vx);
          ballSpeedY.resetTo(vy);
        }
		    
				
				ballPosition.propagated.timeOfObservation = timeOfImageProcessing;
				ballPosition.propagated.observedSpeed.x = ballSpeedX.getVal();
				ballPosition.propagated.observedSpeed.y = ballSpeedY.getVal();
      }
        
      // calc position:
      ballPosX.approximateVal(ballOffset.x); 
      ballPosY.approximateVal(ballOffset.y);  

//      ballAngle.approximateVal(ballPercept.getAngleBearingBased()); 
//      ballDistance.approximateVal(ballPercept.getDistanceBearingBased()); 
//      ballDistance.approximateVal(ballPercept.getDistanceSizeBased()); 

      // old version
      ballAngle.approximateVal(ballPercept.getAngle()); 
      ballDistance.approximateVal(ballPercept.getDistance());  
      
      //clipBallOutOfRobot(ballDistance,ballAngle);
      lastBallSeen.x = ballDistance.getVal()*cos(ballAngle.getVal());//ballPosX.getVal();
      lastBallSeen.y = ballDistance.getVal()*sin(ballAngle.getVal());//ballPosY.getVal();

			/* 2do: make this angle/dist */
			ballPosition.propagated.positionWhenLastObserved.x = ballPosX.getVal();
			ballPosition.propagated.positionWhenLastObserved.y = ballPosY.getVal();
      
      Vector2<double> ballOnField = Geometry::relative2FieldCoord(robotPose, ballDistance.getVal()*cos(ballAngle.getVal()), ballDistance.getVal()*sin(ballAngle.getVal()));
      ballPosition.seen.x = ballOnField.x;
      ballPosition.seen.y = ballOnField.y;
      ballPosition.propagated.x  = ballPosition.seen.x;
      ballPosition.propagated.y  = ballPosition.seen.y;
    }
    consecutiveFramesBallSeen++;
    consecutiveFramesBallNotSeen = 0;

    ballPosition.seen.timeWhenLastSeen = timeOfImageProcessing;



    if (SystemCall::getTimeSince(ballPosition.seen.timeWhenFirstSeenConsecutively) > 350)
    {
      ballPosition.seen.timeUntilSeenConsecutively = timeOfImageProcessing; 
    }
  }
  
  ballPosition.seen.speed = Vector2<double>(ballSpeedX.getVal(), ballSpeedY.getVal()); 
  Vector2<double> speedOnField(ballPosition.seen.x + ballSpeedX.getVal(), ballPosition.seen.y + ballSpeedY.getVal());
  //OUTPUT(idText, text, " " << ballPosition.seen.speed.x << ", " << ballPosition.seen.speed.y);
  Vector2<double> robotToBall(ballPosition.seen.x, ballPosition.seen.y);
  robotToBall -= robotPose.translation;
  robotToBall.normalize();
  robotToBall *= ballPosition.seen.distanceError;

//  ballPosition.seen.speed.x = ballSpeedX.getVal()*cos(robotPose.rotation) + ballSpeedY.getVal()*sin(robotPose.rotation); 
//  ballPosition.seen.speed.y = -ballSpeedX.getVal()*sin(robotPose.rotation) + ballSpeedY.getVal()*cos(robotPose.rotation);

  CIRCLE(ballLocatorField, ballPosition.seen.x, ballPosition.seen.y, 50, 3, 0, Drawings::orange);

  LINE(ballLocatorField, 
    ballPosition.seen.x - robotToBall.x, ballPosition.seen.y - robotToBall.y, 
    ballPosition.seen.x + robotToBall.x, ballPosition.seen.y + robotToBall.y, 
    2, 0, Drawings::orange);

  robotToBall.normalize();
  robotToBall *= (ballPercept.getDistance() * ballPosition.seen.angleError);

  LINE(ballLocatorField, 
    ballPosition.seen.x + robotToBall.y, ballPosition.seen.y - robotToBall.x, 
    ballPosition.seen.x - robotToBall.y, ballPosition.seen.y + robotToBall.x, 
    2, 0, Drawings::orange);

  ARROW(ballLocatorField, ballPosition.seen.x, ballPosition.seen.y, speedOnField.x, speedOnField.y, 1, 3, Drawings::red);
  /*
  // ball intercept point

  // achtung: division by zero if b1=b2=0!!!


  double theta, 
    b1 = ballPosX.getVal() - robotPose.translation.x,
    b2 = ballPosY.getVal() - robotPose.translation.y, 
    v1 = ballSpeedX.getVal(), 
    v2 = ballSpeedY.getVal(),
    maxSpeed = 150;

  theta = asin((b2*b2*v1 - b1*b2*v2 + b1*sqrt(b2*b2*(maxSpeed*maxSpeed - v1*v1) + 2*b1*b2*v1*v2 + b1*b1*(maxSpeed*maxSpeed - v2*v2)))/
          (maxSpeed*(b1*b1 + b2*b2)));

  LINE(ballLocatorField, robotPose.translation.x, robotPose.translation.y, 
    robotPose.translation.x + sin(theta)*500, robotPose.translation.y + cos(theta)*500, 
    3, 0, Drawings::blue); 

  theta -= pi;
  theta *= -1;

  LINE(ballLocatorField, robotPose.translation.x, robotPose.translation.y, 
    robotPose.translation.x + sin(theta)*500, robotPose.translation.y + cos(theta)*500, 
    3, 0, Drawings::blue); 
  */
  DEBUG_DRAWING_FINISHED(ballLocatorField);  
}

void PIDSmoothedBallLocator::clipBallOutOfRobot(PIDsmoothedValue& ballDistance, PIDsmoothedValue& ballAngle)
{
  double x=ballDistance.getVal()*cos(ballAngle.getVal());
  double y=ballDistance.getVal()*sin(ballAngle.getVal());
  bool changed=false;

  if ((x<90)&&(x>-130)&&(fabs(y)<70))
  {
    x=90;
    changed=true;
  }
  else if ((x<=-130)&&(x>-230)&&(fabs(y)<70))
  {
    x=-230;
    changed=true;
  }
  else if ((x<90)&&(x>-230))
  {
    if ((y>-115)&&(y<0))
    {
      y=-115;
      changed=true;
    }
    else if ((y<115)&&(y>=0))
    {
      y=115;
      changed=true;
    }
  }
  if (changed)
  {
    double dist=sqrt(x*x+y*y);
    double arc=atan2(y,x);
    ballDistance.resetTo(dist);
    ballAngle.resetTo(arc);
  }
}

void PIDSmoothedBallLocator::compensateOdometry()
{
  lastBallSeen = (lastRobotOdometry + Pose2D(lastBallSeen) - odometryData).translation;
  Vector2<double> b = (lastRobotOdometry + Pose2D(Vector2<double>(ballPosX.getVal(), ballPosY.getVal()))
                       - odometryData).translation;
  ballPosX.setTo(b.x);
  ballPosY.setTo(b.y);
  lastRobotOdometry = odometryData;
}        


bool PIDSmoothedBallLocator::handleMessage(InMessage& message)
{
  bool handled = false;
  
  switch(message.getMessageID())
  {
  case idGenericDebugData:
    {
      GenericDebugData d;
      message.bin >> d;
      if(d.id == GenericDebugData::ballLocatorPIDs)
      {
        OUTPUT(idText,text,"generic debug message handled by module PIDSmoothedBallLocator");
        
        ballPosX.setWeightP(d.data[0]);
        ballPosX.setWeightI(d.data[1]);
        ballPosX.setWeightD(d.data[2]);
        
        ballPosY.setWeightP(d.data[0]);
        ballPosY.setWeightI(d.data[1]);
        ballPosY.setWeightD(d.data[2]);
        
        ballSpeedX.setWeightP(d.data[3]);
        ballSpeedX.setWeightI(d.data[4]);
        ballSpeedX.setWeightD(d.data[5]);
        
        ballSpeedY.setWeightP(d.data[3]);
        ballSpeedY.setWeightI(d.data[4]);
        ballSpeedY.setWeightD(d.data[5]);
      }
      handled = true;
      break;
    }
  }
  return handled;
}

/*
* Change log :
* 
* $Log: PIDSmoothedBallLocator.cpp,v $
* Revision 1.14  2004/04/11 18:47:30  roefer
* Warning removed
*
* Revision 1.13  2004/04/05 17:56:45  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.6  2004/04/03 23:14:16  dueffert
* clipping corrected but not used any more
*
* Revision 1.5  2004/04/03 11:18:25  dueffert
* clipping improved
*
* Revision 1.4  2004/04/02 23:19:26  dueffert
* clipping added
*
* Revision 1.3  2004/04/02 15:32:23  juengel
* Improved calculation of "ball is in front of goal".
*
* Revision 1.2  2004/04/02 11:38:41  juengel
* calculation of ballModel.ballInFrontOfGoal added
*
* Revision 1.1.1.1  2004/03/31 11:16:37  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.12  2004/03/25 21:22:09  juengel
* Restored old ball position calculation.
*
* Revision 1.11  2004/03/25 09:10:40  jhoffman
* speed calculation improved
*
* Revision 1.10  2004/03/21 19:09:22  juengel
* Moved call of BallSideDetector.
*
* Revision 1.9  2004/03/21 12:39:32  juengel
* Added CalibrationRequest to BallLocatorInterfaces.
*
* Revision 1.8  2004/03/08 00:58:36  roefer
* Interfaces should be const
*
* Revision 1.7  2004/01/20 17:57:04  heinze
* worked at ball-side-detector
*
* Revision 1.6  2004/01/20 16:45:52  loetzsch
* The BallLocator has access to the complete BallPosition again
*
* Revision 1.5  2004/01/13 18:35:24  heinze
* added ball-side-detector
*
* Revision 1.4  2003/12/31 23:50:35  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.3  2003/11/15 17:09:02  juengel
* changed BallPercept
*
* Revision 1.2  2003/10/21 09:11:24  neubach
* corrected clipping of small speeds
*
* Revision 1.1  2003/10/06 13:33:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/26 11:38:51  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.2  2003/09/01 10:20:11  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.27  2003/06/27 13:23:32  jhoffman
* changed debug drawing for ball speed from line to arrow
*
* Revision 1.26  2003/06/22 14:54:34  risler
* bugfix: speed should be absolute
*
* Revision 1.25  2003/06/22 10:51:45  dueffert
* nonsense ball speed corrected
*
* Revision 1.24  2003/06/21 14:20:22  jhoffman
* changed setting of time from "getcurrentsystemtime" to "timeOfImageProcessing"
*
* Revision 1.23  2003/06/20 16:40:25  jhoffman
* bug removed
*
* Revision 1.22  2003/06/20 16:37:51  jhoffman
* smoothing lessened
*
* Revision 1.21  2003/06/15 14:26:44  jhoffman
* + moved "relative2FieldCoord" to Geometry
* + added member function to ballposition to calculate the propagated position and speed for a given time
* + propagated speed and time calculation using exponential decay instead of using an iterative calculation
* + in motion you can now use propageted ball pos at 125 Hz rather then the framerate determined by cognition
*
* Revision 1.20  2003/06/03 08:57:12  roefer
* Modelling of speed for unseen ball improved
*
* Revision 1.19  2003/05/01 17:09:06  loetzsch
* Redesign of ball modeling:
* - Modularized class BallPosition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*
* Revision 1.18  2003/04/14 16:06:51  loetzsch
* ATH after GermanOpen CVS merge
* added timeUntilSeenConsecutively
*
* Revision 1.2  2003/04/11 21:50:02  Jan Hoffmann
* ballPosition.timeUntilSeenConsecutively is calculated.
*
* Revision 1.1.1.1  2003/04/09 14:22:20  loetzsch
* started Aibo Team Humboldt's GermanOpen CVS
*
* Revision 1.17  2003/04/03 21:07:21  jhoffman
* added clipping of small speeds and "consecutive time ball seen"
*
* Revision 1.16  2003/04/02 15:35:54  jhoffman
* parameter adjustment
*
* Revision 1.15  2003/04/02 11:01:53  jhoffman
* added odometry correction
*
* Revision 1.14  2003/04/02 08:58:50  jhoffman
* no message
*
* Revision 1.13  2003/04/02 08:56:47  jhoffman
* ball locator stores and smoothes relative ball positions and adds robot pose
* each frame to generate the position on the field
*
* Revision 1.12  2003/03/31 13:44:43  jhoffman
* error fixed
*
* Revision 1.11  2003/03/31 13:41:55  jhoffman
* added propagated speed
*
* Revision 1.10  2003/03/30 21:10:48  jhoffman
* simplified code
*
* Revision 1.9  2003/03/25 19:24:16  jhoffman
* fixed div/0
*
* Revision 1.8  2003/03/25 13:55:53  dueffert
* warning removed
*
* Revision 1.7  2003/03/24 13:33:17  jhoffman
* - added damping for speed when ball is not seen
* - "ballseen" is only updated when a ball is actually seen
* - "ballknown" is updated always, if ball is not seen, ball speed is added
*
* Revision 1.6  2003/03/19 16:41:19  jhoffman
* fixed debug drawing error
*
* Revision 1.5  2003/03/19 15:37:47  jhoffman
* now calculates proper speed
*
* Revision 1.4  2003/03/06 12:04:38  dueffert
* missing braces added
*
* Revision 1.3  2003/03/05 18:08:27  jhoffman
* added boundaries for ball pos
*
* Revision 1.2  2003/03/05 14:58:16  jhoffman
* ball locator takes into account ball speed when ball is not seen for
* a brief moment (e.g. when it is temporarily behind another robot); this
* makes ball following better
*
* Revision 1.1  2003/03/05 10:34:05  jhoffman
* no message
*
*
*/
