/**
* @file VoteBasedBallLocator.cpp
* 
* This file contains a class for ball-localization.
*
* @author <a href="mailto:jhoffman@informatik.hu-berlin.de">Jan Hoffmann</a>
*/

#include "VoteBasedBallLocator.h"
#include "Tools/FieldDimensions.h"
#include "Platform/SystemCall.h"
#include "Tools/Debugging/DebugDrawings.h"
#include <math.h>
#include "Tools/Math/Matrix2x2.h"
#include "Platform/GTAssert.h"


VoteBasedBallLocator::VoteBasedBallLocator(const BallLocatorInterfaces& interfaces)
: BallLocator(interfaces),
consecutiveFramesBallSeen(0),
consecutiveFramesBallNotSeen(0)
{
  lastTimeBallSeen = 0;
}

void VoteBasedBallLocator::execute()
{
  double timeDiff = (double )(timeOfImageProcessing - lastTimeBallSeen)/1000; // 1/msec -> 1/sec
  lastTimeBallSeen = timeOfImageProcessing;
	Vector2<double> ballOnField, anotherPointOnTheField;
	int i;

  compensateOdometry();

  // ball not seen
  if(!ballPercept.ballWasSeen)
  {
    consecutiveFramesBallSeen = 0;
    consecutiveFramesBallNotSeen++;

		Vector2<double> propSpeed = ballPosition.propagated.getPropagatedSpeed(timeOfImageProcessing);
		Vector2<double> propBallPos = ballPosition.propagated.getPropagatedPosition(timeOfImageProcessing);
		/*    
		ballSpeedX.resetTo(propSpeed.x);
    ballSpeedY.resetTo(propSpeed.y);

    ballPosX.resetTo(propBallPos.x); 
    ballPosY.resetTo(propBallPos.y); 
		ballAngle.resetTo(propBallPos.angle());
		ballDistance.resetTo(propBallPos.abs());
		*/

    ballOnField = Geometry::relative2FieldCoord(robotPose, propBallPos.x, propBallPos.y);
    CIRCLE(ballLocatorField, ballOnField.x, ballOnField.y, 90, 3, 0, Drawings::blue);

    // move seen to compensate for jumping localization:
    //ballOnField = Geometry::relative2FieldCoord(robotPose, lastBallSeen.x, lastBallSeen.y);
    ballPosition.seen.x = ballOnField.x;
    ballPosition.seen.y = ballOnField.y;
  }
  else // ball was seen
  {
    // if there is a gap of n frames during which the ball was not seen
    // reset the "timeWhenLastSeenConsecutively"
    if (consecutiveFramesBallNotSeen > 5)
      ballPosition.seen.timeWhenFirstSeenConsecutively = timeOfImageProcessing;

    // calc ball position:
    //if (consecutiveFramesBallNotSeen > 10) // if ball has not been seen for a while but is seen now, directly set to percept...
    //{
    //}
    else  // if it has been seen recently, perform smoothing...
    {
      // calc ball speed:
      if ((consecutiveFramesBallSeen > 1) && timeDiff > 0)
      {
				/*
        double vx = (ballPercept.balls[0].offset.x - ballPosX.getVal()) / timeDiff;
        double vy = (ballPercept.balls[0].offset.y - ballPosY.getVal()) / timeDiff;
        
        // clip small speeds
        if (fabs(vx) < 10)
          ballSpeedX.resetTo(0.0);
        else if (consecutiveFramesBallSeen > 1)
          ballSpeedX.approximateVal(vx);
        else
          ballSpeedX.resetTo(vx);

        if (fabs(vy) < 10)
          ballSpeedY.resetTo(0.0);
        else if (consecutiveFramesBallSeen > 1)
          ballSpeedY.approximateVal(vy);
        else
          ballSpeedX.resetTo(vy);
					
				ballPosition.propagated.timeOfObservation = timeOfImageProcessing;
				ballPosition.propagated.observedSpeed.x = ballSpeedX.getVal();
				ballPosition.propagated.observedSpeed.y = ballSpeedY.getVal();
				*/
      }
        
      Vector2<double> ballOffset;
      ballPercept.getOffset(ballOffset);

      measurements.add(ballOffset);
      angleMeasurements.add(ballPercept.getAngle()); 
      distMeasurements.add(ballPercept.getDistance()); 
			timeStamps.add(timeOfImageProcessing/1000.0);

      ballPosition.seen.x = ballOnField.x;
      ballPosition.seen.y = ballOnField.y;
      ballPosition.propagated.x  = ballPosition.seen.x;
      ballPosition.propagated.y  = ballPosition.seen.y;
    }
    consecutiveFramesBallSeen++;
    consecutiveFramesBallNotSeen = 0;
		ballPosition.seen.timeWhenLastSeen = timeOfImageProcessing;
	  if (SystemCall::getTimeSince(ballPosition.seen.timeWhenFirstSeenConsecutively) > 350)
      ballPosition.seen.timeUntilSeenConsecutively = timeOfImageProcessing; 
  }
			

	// Voting for best hypothesis:

	int p1, p2, v;
	double bestVote = 0.0, currentVote; 
	Vector2<double> bestPos, bestSpeed;
	Vector2<double> hPos, hSpeed;

	for (p1 = 0; p1 < 10; p1++)
	{
		for(p2 = p1 + 1; p2 < 10; p2++) //start with the same to include zero speeds!
		{
			// [1] create hypothesis
		  //     from two points
			//		 offset = angleMeasurement[p1];

			//double td = timeStamps[p1] - timeStamps[p2];
		 	hSpeed = (measurements[p1] - measurements[p2])/(timeStamps[p1] - timeStamps[p2]);
			hPos = measurements[p1];

			//ballOnField = Geometry::relative2FieldCoord(robotPose, measurements[p1].x, measurements[p1].y);			
			//ARROW(ballLocator, ballOnField.x, ballOnField.y, hSpeed.x, hSpeed.y, 2, 2, Drawings::blue);

			// [2] perform voting
			currentVote = (10.0-p1)/10.0;
			for(v = 0; v < 10; v++)
			{
				// only have positions vote
				// that were not used for
				// generating the hypothesis:
				if((v != p1) && (v != p2))//&& (hSpeed.y < 0.0))
				{
					double timeSinceLastMeasurement = (timeStamps[p1] - timeStamps[v]);

					double weight = 1/((timeStamps[0] - timeStamps[v])*3 + 1);//(10.0-v)/10.0;

					Vector2<double> hypo = hPos + hSpeed*timeSinceLastMeasurement;

					double vt = weight*vote(hypo, angleMeasurements[v], distMeasurements[v]);
					currentVote += vt;
						
					
					if (p2 % 3 == 0)
					{
					if (currentVote > 0.0)
					{
						// debug output
						if((hSpeed*timeSinceLastMeasurement).abs() < 1000)
							ballOnField = Geometry::relative2FieldCoord(robotPose, hypo.x, hypo.y);
						else 
							ballOnField = Geometry::relative2FieldCoord(robotPose, hPos.x, hPos.y);
						//CIRCLE(ballLocator, ballOnField.x, ballOnField.y, 40*vt+5, 1, 0, Drawings::green);
					}
					else 
					{
						// debug output
						if((hSpeed*timeSinceLastMeasurement).abs() < 1000)
							ballOnField = Geometry::relative2FieldCoord(robotPose, hypo.x, hypo.y);
						else 
							ballOnField = Geometry::relative2FieldCoord(robotPose, hPos.x, hPos.y);
						//CIRCLE(ballLocator, ballOnField.x, ballOnField.y, 3, 1, 0, Drawings::red);
					}
					}
					// measure distance at time stored in timestamp
					// weigh(t) using exponential decay with time 
				}
			}
			// [3] Save hypothesis and vote into a buffer
			if (currentVote > bestVote)
			{
				bestVote = currentVote;
				double timeDiffUntilNow = timeStamps[0] - timeStamps[p1];
				bestPos = hPos + hSpeed*timeDiffUntilNow;// this must be the currentTime...
				bestSpeed = hSpeed;
				ballOnField = Geometry::relative2FieldCoord(robotPose, bestPos.x, bestPos.y);
				anotherPointOnTheField = Geometry::relative2FieldCoord(robotPose, bestPos.x + hSpeed.x*1, bestPos.y + hSpeed.y*1);
				//CIRCLE(ballLocator, ballOnField.x, ballOnField.y, 20, 2, 0, Drawings::blue);
				//ARROW(ballLocator, ballOnField.x, ballOnField.y, anotherPointOnTheField.x, anotherPointOnTheField.y,  2, 2, Drawings::blue);
			}					
		}
	}

	// [4] scan buffer for best hypothesis, if more
	// than one with the same votes exist, perform 
	// averaging
	ballOnField = Geometry::relative2FieldCoord(robotPose, bestPos.x, bestPos.y);
	anotherPointOnTheField = Geometry::relative2FieldCoord(robotPose, bestPos.x + bestSpeed.x*5, bestPos.y + bestSpeed.y*5);
	ARROW(ballLocatorField, ballOnField.x, ballOnField.y, anotherPointOnTheField.x, anotherPointOnTheField.y,  2, 2, Drawings::blue);
	CIRCLE(ballLocatorField, ballOnField.x, ballOnField.y, 20, 2, 0, Drawings::blue);

	// debug output: show last 
	ballOnField = Geometry::relative2FieldCoord(robotPose, measurements[0].x, measurements[0].y);			
	CIRCLE(ballLocatorField, ballOnField.x, ballOnField.y, 50, 1, 0, Drawings::orange);  
	for (i = 1; i < 10; i++)
	{
		ballOnField = Geometry::relative2FieldCoord(robotPose, measurements[i].x, measurements[i].y);			
		//CIRCLE(ballLocator, ballOnField.x, ballOnField.y, 50, 1, 0, Drawings::orange);  
	}

  DEBUG_DRAWING_FINISHED(ballLocatorField);  

	ballOnField = Geometry::relative2FieldCoord(robotPose, bestPos.x, bestPos.y);
	ballPosition.seen.x = ballOnField.x;
	ballPosition.seen.y = ballOnField.y;

	anotherPointOnTheField = Geometry::relative2FieldCoord(robotPose, bestSpeed.x, bestSpeed.y);
  ballPosition.seen.speed = anotherPointOnTheField;	
}




void VoteBasedBallLocator::compensateOdometry()
{
	int i;
	
	for (i = 0; i < 10; i++)
	{
		measurements.updateEntry((lastRobotOdometry + Pose2D(measurements[i]) - odometryData).translation, i);
		angleMeasurements.updateEntry(measurements[i].angle(), i);
		distMeasurements.updateEntry(measurements[i].abs(), i);
	}

  lastRobotOdometry = odometryData;
}        


bool VoteBasedBallLocator::handleMessage(InMessage& message)
{
  bool handled = false;
  
  switch(message.getMessageID())
  {
  case idGenericDebugData:
    {
      GenericDebugData d;
      message.bin >> d;
      if(d.id == GenericDebugData::ballLocatorPIDs)
      {
        OUTPUT(idText,text,"generic debug message handled by module VoteBasedBallLocator");
        
        //ballPosX.setWeightP(d.data[0]);
      }
      handled = true;
      break;
    }
  }
  return handled;
}

	
double VoteBasedBallLocator::vote(Vector2<double> hypothesis, double angleMeasurement, double distMeasurement)
{
	double dAngle = fabs(hypothesis.angle() - angleMeasurement);
	double hDist = hypothesis.abs();
	double dDistance = fabs(hDist - distMeasurement);

	// if angle error is (absolutely) small 
	// AND distance error is (relatively) small
	// vote for the hypothesis
	if ((dAngle < .1) && (dDistance < hDist*.2))
		return 1.0;
	else
		return 0.0;
}


/*
* Change log :
* 
* $Log: VoteBasedBallLocator.cpp,v $
* Revision 1.6  2004/03/08 00:58:37  roefer
* Interfaces should be const
*
* Revision 1.5  2004/01/20 16:45:52  loetzsch
* The BallLocator has access to the complete BallPosition again
*
* Revision 1.4  2003/12/31 23:50:35  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.3  2003/11/15 17:09:02  juengel
* changed BallPercept
*
* Revision 1.2  2003/10/28 13:27:49  dueffert
* warnings removed
*
* Revision 1.1  2003/10/22 10:22:57  goehring
* VoteBasedBallLocator
*
*
*/
