/** 
* @file ATH2004ContinuousRules/AvoidObstaclesAheadTurning.h
*
* @author Matthias Jngel
*/

#ifndef __ATH2004AvoidObstaclesAheadTurning_h_
#define __ATH2004AvoidObstaclesAheadTurning_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/Debugging/DebugDrawings.h"

namespace ATH2004ContinuousRules
{
  
/**@class AvoidObstaclesAheadTurning
* rule which avoids collision with the obstacles
  */
  class AvoidObstaclesAheadTurning: public ContinuousRule{
  public:
    
  /** constructor
    */
    AvoidObstaclesAheadTurning(const BehaviorControlInterfaces& interfaces,
      double maxForce = 1.5, 
      double maxForceLength = 100.0, 
      double influenceLength = 450.0)
      :ContinuousRule(interfaces, "AvoidObstaclesAheadTurning"),
      maxForce(maxForce),
      maxForceLength(maxForceLength),
      influenceLength(influenceLength)
    {};
    
    /**
    * Returns whether this rule generates absolute or robot relative coordinates.
    * This rule is robot relative.
    */
    virtual bool isRelative() {return true;}
    
    /** executes the basic behavior rule
    * @param robotPose the current robots pose at which the rule is to be evaluated
    * @param walk the direction and speed of the suggested motion coded as an vector (output)
    * @param ra the rotation angle, the direction the robot should be directed (output)
    * @param rweight the rotation weight the weight of the suggested rotation (output)
    */
    virtual void execute(const RobotPose& robotPose,
      Vector2<double>& walk,
      double& ra, double& rweight)
    {
      ra = 0;
      rweight = 0.0;
      walk.x = 0;
      walk.y = 0;
      if(obstaclesModel.corridorInFront < influenceLength) 
      {
        if(obstaclesModel.getAngleOfNextFreeSector(pi/4, 0, (int)influenceLength) < 0)
          ra = -pi_2;
        else
          ra = pi_2;
        
        walk.x = -getRepellingForce(obstaclesModel.corridorInFront, maxForce, maxForceLength, influenceLength);
        rweight = 1.0;
      }
    };
    
  protected:
    /** the maximum length of the result vector*/
    double maxForce;
    /** the distance below which the result vector length is maxForce (in mm)*/
    double maxForceLength;
    /** the distance above which the result vector length is 0 (in mm)*/
    double influenceLength;
  };
  
}


#endif 


/*
* Change log:
*
* $Log: AvoidObstaclesAheadTurning.h,v $
* Revision 1.3  2004/03/08 00:58:40  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/28 15:04:09  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/
