/** 
* @file ATH2004ContinuousRules/AvoidOpponents.h
*
* @author Max Risler
*/

#ifndef __ATH2004AvoidOpponents_h_
#define __ATH2004AvoidOpponents_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"

namespace ATH2004ContinuousRules
{

/**@class AvoidOpponents
 * rule which avoids collision with the opponents
 */
class AvoidOpponents: public ContinuousRule{
public:

  /** constructor
   */
  AvoidOpponents(const BehaviorControlInterfaces& interfaces,
                 double maxForce = 3.0, 
                 double maxForceLength = 150.0, 
                 double influenceLength = 250.0)
      :ContinuousRule(interfaces, "AvoidOpponents"),
       maxForce(maxForce),
       maxForceLength(maxForceLength),
       influenceLength(influenceLength)
  {};


 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    Vector2<double> fromOpponent;
    double distance;

    walk.x=walk.y=0;

    Geometry::Line robot(robotPose.translation,
                         robotPose*Vector2<double>(-bodyLength,0)-robotPose.translation);

    for (int i=0;i<playerPoseCollection.numberOfOpponentPlayers;i++)
    {
      distance = Geometry::getDistanceToEdge(robot,playerPoseCollection.getOpponentPlayerPose(i).getPose().translation);

      fromOpponent = robotPose.translation - playerPoseCollection.getOpponentPlayerPose(i).getPose().translation;
      fromOpponent.normalize();

      walk += fromOpponent * 
              getRepellingForce(distance, maxForce, maxForceLength, influenceLength) * 
              playerPoseCollection.getOpponentPlayerPose(i).getValidity();
    }

    ra=0;
    rweight=0;
  };

protected:
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is maxForce (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;
};

}


#endif 


/*
* Change log:
*
* $Log: AvoidOpponents.h,v $
* Revision 1.3  2004/03/08 00:58:41  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/28 15:04:09  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/
