/** 
* @file ATH2004SimpleBasicBehaviors.h
*
* Declaration of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#ifndef __ATH2004SimpleBasicBehaviors_h_
#define __ATH2004SimpleBasicBehaviors_h_

#include "../../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2BasicBehavior.h"
#include "Tools/Actorics/WalkAccelerationRestrictor.h"
#include "Tools/Evolution/Population.h"
#include "Tools/Math/PIDsmoothedValue.h"
#include "Modules/WalkingEngine/InvKinWalkingEngine.h"
#include "Modules/WalkingEngine/InvKinWalkingParameterSets.h"



/**
* A Basic Behavior for going to the ball
* @author Matthias Jngel
* @author Martin Ltzsch
*/
class ATH2004BasicBehaviorGoToBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorGoToBall(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball.distance",distanceAtEnd);
    registerParameter("go-to-ball.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball.distance" */
  double distanceAtEnd;
  
  /** parameter "go-to-ball.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};


/**
* A Basic Behavior for going to the ball
*/
class ATH2004BasicBehaviorDribbleAtBorder : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorDribbleAtBorder(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("dribble-at-border", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("dribble-at-border.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball.distance" */
  double maxSpeed;
  
};

/**
* A Basic Behavior for dribbling in the center of the field
*/
class ATH2004BasicBehaviorDribbleInCenterOfField : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorDribbleInCenterOfField(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("dribble-in-center-of-field", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("dribble-in-center-of-field.speed", speed);
	registerParameter("dribble-in-center-of-field.modi", modi);
	direction = 361;
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "dribble-in-center-of-field.speed" */
  double speed;
  /** parameter "dribble-in-center-of-field.modi" */
  double modi;
  // dribble direction (angle)
  double direction;
  // dribble side
  int side;
  // rotation
  double rotationM;
  
};

/**
* A Basic Behavior for going behind the ball 
*/
class ATH2004BasicBehaviorGetBehindBall2 : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorGetBehindBall2(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("get-behind-ball2", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball.distance" */
  double speed;
  int side;
};

/**
* A Basic Behavior for going to the ball and reaching it with certain distance (x and y relative to robot) and direction
* @author Uwe Dffert
* @author Matthias Jngel
* @author Martin Ltzsch
*/
class ATH2004BasicBehaviorGoToBallWithDirection : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorGoToBallWithDirection(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball-with-direction", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball-with-direction.distance.x",distanceX);
    registerParameter("go-to-ball-with-direction.distance.y",distanceY);
    registerParameter("go-to-ball-with-direction.max-speed", maxSpeed);
    registerParameter("go-to-ball-with-direction.direction", direction);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball-with-direction.distance.x" */
  double distanceX;
  /** parameter "go-to-ball-with-direction.distance.y" */
  double distanceY;
  /** parameter "go-to-ball-with-direction.max-speed" */
  double maxSpeed;
  /** parameter "go-to-ball-with-direction.direction" */
  double direction;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};


/**
* A Basic Behavior for avoiding obstacles
* @author Matthias Jngel
*/
class ATH2004BasicBehaviorAvoidObstacles : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorAvoidObstacles(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("avoid-obstacles", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("avoid-obstacles.turn-distance",turnDistance);
    registerParameter("avoid-obstacles.stop-distance",stopDistance);
    registerParameter("avoid-obstacles.walk-speed", walkSpeed);
    registerParameter("avoid-obstacles.turn-speed", turnSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "avoid-obstacles.turn-distance" */
  double turnDistance;
  
  /** parameter "avoid-obstacles.stop-distance" */
  double stopDistance;
  
  /** parameter "avoid-obstacles.walk-speed" */
  double walkSpeed;
  
  /** parameter "avoid-obstacles.turn-speed" */
  double turnSpeed;
};

/**
* A Basic Behavior for going to a point on the field
* @author Uwe Dffert
* @author Martin Ltzsch
*/
class ATH2004BasicBehaviorGoToPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorGoToPoint(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point.x", x);
    registerParameter("go-to-point.y", y);
    registerParameter("go-to-point.destination-angle", destinationAngle);
    registerParameter("go-to-point.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point.x" */
  double x;
  
  /** parameter "go-to-point.y" */
  double y;
  
  /** parameter "go-to-point.destination-angle" */
  double destinationAngle;
  
  /** parameter "go-to-point.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;

  friend class ATH2004BasicBehaviorGetBehindBall;
};

/**
* A Basic Behavior for going to a point on the field without slowing down near the destination
* @author Martin Ltzsch
*/
class ATH2004BasicBehaviorGoToPointFast : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorGoToPointFast(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-fast", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point-fast.x", x);
    registerParameter("go-to-point-fast.y", y);
    registerParameter("go-to-point-fast.destination-angle", destinationAngle);
    registerParameter("go-to-point-fast.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point-fast.x" */
  double x;
  
  /** parameter "go-to-point-fast.y" */
  double y;
  
  /** parameter "go-to-point-fast.destination-angle" */
  double destinationAngle;
  
  /** parameter "go-to-point-fast.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for precisely going to a point on the field
* @author Uwe Dffert
* @author Martin Ltzsch
* @author Thomas Rfer
*/
class ATH2004BasicBehaviorGoToPointPrecisely : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorGoToPointPrecisely(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-precisely", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point-precisely.x", x);
    registerParameter("go-to-point-precisely.y", y);
    registerParameter("go-to-point-precisely.destination-angle", destinationAngle);
    registerParameter("go-to-point-precisely.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point-precisely.x" */
  double x;
  
  /** parameter "go-to-point-precisely.y" */
  double y;
  
  /** parameter "go-to-point-precisely.destination-angle" */
  double destinationAngle;
  
  /** parameter "go-to-point-precisely.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to a point on the field and avoiding obstacles on the way there.
* @author Jan Hoffmann
*/
class ATH2004BasicBehaviorGoToPointAndAvoidObstacles : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorGoToPointAndAvoidObstacles(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-and-avoid-obstacles", errorHandler),
    BehaviorControlInterfaces(interfaces),
	  speedPhi(0, .1, 0, 0, -pi_4, pi_4, pi/10),
	  speedX(0, .6, 0, 0, -80, 200, 100),
	  speedY(0, .1, 0, 0, -100, 100, 100)
  {
    registerParameter("go-to-point-and-avoid-obstacles.x", x);
    registerParameter("go-to-point-and-avoid-obstacles.y", y);
    registerParameter("go-to-point-and-avoid-obstacles.max-speed", maxSpeed);
    registerParameter("go-to-point-and-avoid-obstacles.far-away-avoidance", farAwayAvoidance);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-point-and-avoid-obstacles.x" */
  double x;
  
  /** parameter "go-to-point-and-avoid-obstacles.y" */
  double y;
  
  /** parameter "go-to-point-and-avoid-obstacles.max-speed" */
  double maxSpeed;

  /** parameter "go-to-point-and-avoid-obstacles.far-away-avoidance" */
  double farAwayAvoidance;
        
  /** smoothes the rotation */  
  PIDsmoothedValue speedPhi;

  /** smoothes the speed in forward direction */  
  PIDsmoothedValue speedX;

  /** smoothes the speed in sideways direction */  
  PIDsmoothedValue speedY;
};


/**
* A Basic Behavior for getting behind the ball
* @author Max Risler
* @author Martin Ltzsch
*/
class ATH2004BasicBehaviorGetBehindBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004BasicBehaviorGetBehindBall(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("get-behind-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("get-behind-ball.distance",distance);
    registerParameter("get-behind-ball.left-right", leftRight);
    registerParameter("get-behind-ball.max-speed", maxSpeed);
    registerParameter("get-behind-ball.angle", angle);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "get-behind-ball.distance" */
  double distance;
  
  /** parameter "get-behind-ball.left-right" */
  double leftRight;
  
  /** parameter "get-behind-ball.max-speed" */
  double maxSpeed;
  
  /** parameter "get-behind-ball.angle" */
  double angle;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
  
  /** Walks to a point */
  void gotoPoint(double x, double y, double maxSpeed);
};

/**
* @class SimpleBasicBehaviors
*
* Creates and registers simple basic behaviors
*/
class ATH2004SimpleBasicBehaviors : public BehaviorControlInterfaces
{
public:
  /**
  * Constructor
  */
  ATH2004SimpleBasicBehaviors(const BehaviorControlInterfaces& interfaces,
                       Xabsl2ErrorHandler& errorHandler)
    : BehaviorControlInterfaces(interfaces),
      errorHandler(errorHandler),
      basicBehaviorAvoidObstacles(interfaces,errorHandler),
      basicBehaviorGetBehindBall(interfaces,errorHandler),
      basicBehaviorGoToBall(interfaces,errorHandler),
      basicBehaviorGoToBallWithDirection(interfaces,errorHandler),
      basicBehaviorGoToPoint(interfaces,errorHandler),
      basicBehaviorGoToPointFast(interfaces,errorHandler),
      basicBehaviorGoToPointPrecisely(interfaces,errorHandler),
      basicBehaviorGoToPointAndAvoidObstacles(interfaces,errorHandler),
	  basicBehaviorDribbleAtBorder(interfaces,errorHandler),
	  basicBehaviorDribbleInCenterOfField(interfaces,errorHandler),
	  basicBehaviorGetBehindBall2(interfaces,errorHandler)
  {}

  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine);

private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;

  //!@name Basic Behaviors
  //!@{
  ATH2004BasicBehaviorAvoidObstacles  basicBehaviorAvoidObstacles;  
  ATH2004BasicBehaviorGetBehindBall  basicBehaviorGetBehindBall;
  ATH2004BasicBehaviorGoToBall  basicBehaviorGoToBall;
  ATH2004BasicBehaviorGoToBallWithDirection  basicBehaviorGoToBallWithDirection;
  ATH2004BasicBehaviorGoToPoint  basicBehaviorGoToPoint;  
  ATH2004BasicBehaviorGoToPointFast  basicBehaviorGoToPointFast;  
  ATH2004BasicBehaviorGoToPointPrecisely  basicBehaviorGoToPointPrecisely;  
  ATH2004BasicBehaviorGoToPointAndAvoidObstacles  basicBehaviorGoToPointAndAvoidObstacles;
  ATH2004BasicBehaviorDribbleAtBorder  basicBehaviorDribbleAtBorder;
  ATH2004BasicBehaviorDribbleInCenterOfField  basicBehaviorDribbleInCenterOfField;
  ATH2004BasicBehaviorGetBehindBall2  basicBehaviorGetBehindBall2;

  //!@}
};

#endif // __ATH2004SimpleBasicBehaviors_h_

/*
* Change Log
* 
* $Log: ATH2004SimpleBasicBehaviors.h,v $
* Revision 1.12  2004/03/17 09:30:32  jhoffman
* small changes to go-to-point-and-avoid-obstacles
*
* Revision 1.11  2004/03/16 14:00:17  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.10  2004/03/08 00:58:39  roefer
* Interfaces should be const
*
* Revision 1.2  2004/03/15 17:11:39  hoffmann
* - added ATH2004HeadControl
* - added ATH2004LEDControl
* - headmotiontester shows "tilt2"
* - motion process updates odometry while no new robotPose is received, added to motion request
* - some ui adjustments
* - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
*
* Revision 1.9  2004/03/02 10:10:11  mellmann
* 3 dribble-modes in DribbleInCenterOfField
*
* Revision 1.8  2004/01/08 13:51:48  loetzsch
* added go-to-point-fast
*
* Revision 1.7  2003/12/11 17:10:22  mellmann
* new Variable "side" in dribbleInCenterOfField
*
* Revision 1.6  2003/12/04 17:22:54  mellmann
* Added "GetBehindBall2" and "DribbleInCenterOfField".
*
* Revision 1.5  2003/11/25 19:53:36  heinze
* added basic behavior dribble-at-border
*
* Revision 1.4  2003/10/28 14:04:50  dueffert
* unused evolution removed
*
* Revision 1.3  2003/10/28 13:57:45  dueffert
* unused evolution removed
*
* Revision 1.2  2003/10/28 13:25:52  dueffert
* spelling improved
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

