/**
* @file ATH2004BehaviorControl.cpp
* 
* Implementation of class ATH2004BehaviorControl.
*
* @author Martin Ltzsch
*/

#include "ATH2004BehaviorControl.h"
#include "Tools/Player.h"

ATH2004BehaviorControl::ATH2004BehaviorControl(const BehaviorControlInterfaces& interfaces)
: Xabsl2BehaviorControl(interfaces,SolutionRequest::ath2004),
ballSymbols(interfaces),
configurationSymbols(interfaces,configuration),
headAndTailSymbols(interfaces),
joystickSymbols(interfaces),
ledAndSoundSymbols(interfaces),
mathFunctions(interfaces),
motionRequestSymbols(interfaces),
roboCupGameManagerSymbols(interfaces),
robotPoseSymbols(interfaces),
robotStateSymbols(interfaces),
specialVisionSymbols(interfaces),
strategySymbols(interfaces),
challengeSymbols(interfaces),
simpleBasicBehaviors(interfaces,errorHandler),
continuousBasicBehaviors(interfaces,configuration,errorHandler),
commonBasicBehaviors(interfaces,errorHandler),
lastSound(SoundRequest::none)
{
  Xabsl2FileInputSource file("Xabsl2/ath04-ic.dat");
  init(file);
}

ATH2004BehaviorControl::~ATH2004BehaviorControl()
{
}

void ATH2004BehaviorControl::registerSymbolsAndBasicBehaviors()
{
  simpleBasicBehaviors.registerBasicBehaviors(*pEngine);
  continuousBasicBehaviors.registerBasicBehaviors(*pEngine);
  commonBasicBehaviors.registerBasicBehaviors(*pEngine);
  
  ballSymbols.registerSymbols(*pEngine);
  configurationSymbols.registerSymbols(*pEngine);
  headAndTailSymbols.registerSymbols(*pEngine);
  joystickSymbols.registerSymbols(*pEngine);
  ledAndSoundSymbols.registerSymbols(*pEngine);
  mathFunctions.registerSymbols(*pEngine);
  motionRequestSymbols.registerSymbols(*pEngine);
  roboCupGameManagerSymbols.registerSymbols(*pEngine);
  robotPoseSymbols.registerSymbols(*pEngine);
  robotStateSymbols.registerSymbols(*pEngine);
  specialVisionSymbols.registerSymbols(*pEngine);
  strategySymbols.registerSymbols(*pEngine);
  challengeSymbols.registerSymbols(*pEngine);
}

void ATH2004BehaviorControl::execute()
{
  ballSymbols.update();
  configurationSymbols.update();
  roboCupGameManagerSymbols.update();
  robotPoseSymbols.update();
  strategySymbols.update();
  
  continuousBasicBehaviors.update();
  
  // set the outgoint behavior team message to none
  outgoingBehaviorTeamMessage.message = BehaviorTeamMessage::none;
  
  // set the upper leds depending on the dynamic role.
  switch (strategySymbols.role)
  {
  case BehaviorTeamMessage::striker:
    ledRequest.redTopLEDs = LEDRequest::bothOn;
    break;
  case BehaviorTeamMessage::offensiveSupporter:
    ledRequest.redTopLEDs = LEDRequest::bothFastBlink;
    break;
  case BehaviorTeamMessage::defensiveSupporter:
  default:
    ledRequest.redTopLEDs = LEDRequest::bothOff;
  }
  
  // Set the tail request depending on whether the ball was seen
  if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 120)
  {
    // ball seen now
    motionRequest.tailRequest = MotionRequest::wagHorizontalFast;
  }
  else if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 2000)
  {
    // ball seen
    motionRequest.tailRequest = MotionRequest::wagHorizontal;
  }
  else if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) 
    > BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
    && SystemCall::getTimeSince(ballPosition.communicated.timeWhenLastObserved) < 3000)
  {
    // ball known
    motionRequest.tailRequest = MotionRequest::wagVertical;
  }
  else
  {
    // ball not known
    motionRequest.tailRequest = MotionRequest::noTailWag;
  }
  
  // set head control mode lookToStars to detect missing settings
  headControlMode.headControlMode = HeadControlMode::lookToStars;
    
  // execute the engine
  executeEngine();

  if((robotState.getState() == RobotState::crashed)||
     (robotState.getState() == RobotState::rollLeft)||
     (robotState.getState() == RobotState::rollRight))
  {
    if ((motionRequest.motionType == MotionRequest::walk &&
      motionRequest.walkType != MotionRequest::upsideDown)||
      (motionRequest.motionType == MotionRequest::stand))
    {
      motionRequest.motionType = MotionRequest::getup;
      return;
    }
  }

  SoundRequest::SoundID currentSound = soundRequest.soundID;
  
  if (headControlMode.headControlMode == HeadControlMode::lookToStars)
  {
    soundRequest.soundID = SoundRequest::rob101;
  }
  else if (
    strategySymbols.role == BehaviorTeamMessage::striker &&
    currentSound == lastSound)
  {
    if (strategySymbols.robotIsStuck)
    {
      soundRequest.soundID = SoundRequest::bing03;
    }
    else if (strategySymbols.obstaclesAreClose)
    {
      soundRequest.soundID = SoundRequest::bing01;
    }
    else if (robotState.getConsecutiveCollisionTimeAggregate() > 2000)
    {
      soundRequest.soundID = SoundRequest::bing05;
    }
  }

  lastSound = currentSound;

}

bool ATH2004BehaviorControl::handleMessage(InMessage& message)
{
  return (configuration.handleMessage(message) ||
    Xabsl2BehaviorControl::handleMessage(message));
}


/*
* Change log :
* 
* $Log: ATH2004BehaviorControl.cpp,v $
* Revision 1.7  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
* Revision 1.6  2004/05/04 10:48:57  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.5  2004/03/16 14:00:17  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.4  2004/03/08 00:58:43  roefer
* Interfaces should be const
*
* Revision 1.2  2004/03/11 11:45:16  loetzsch
* removed shoot logger from ATH2004BehaviorControl
*
*
* Revision 1.3  2004/02/03 13:20:47  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.2  2004/01/28 21:16:56  loetzsch
* added ATH2004ShootLogger
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

