/** 
* @file ATH2004StrategySymbols.cpp
*
* Implementation of class ATH2004StrategySymbols.
*
* @author Martin Ltzsch
*/

#include "ATH2004StrategySymbols.h"
#include "Tools/FieldDimensions.h"

ATH2004StrategySymbols::ATH2004StrategySymbols(const BehaviorControlInterfaces& interfaces)
: BehaviorControlInterfaces(interfaces),
timeWhenBallWasStartedToCatch(0), timeUntilBallWasCaught(0), 
goodShot(-1)
{
  role = BehaviorTeamMessage::goalie;
  estimatedTimeToReachBall = 0.0;
  nextShot=44;
}


void ATH2004StrategySymbols::registerSymbols(Xabsl2Engine& engine)
{
  // "robot-number"
  engine.registerDecimalInputSymbol("robot-number", this,
    (double(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getRobotNumber);

  // "role"
  engine.registerEnumeratedInputSymbol("role", (int *)&role);
  engine.registerEnumeratedInputSymbolEnumElement("role", "goalie", BehaviorTeamMessage::goalie);
  engine.registerEnumeratedInputSymbolEnumElement("role", "striker", BehaviorTeamMessage::striker);
  engine.registerEnumeratedInputSymbolEnumElement("role", "defensive-supporter", BehaviorTeamMessage::defensiveSupporter);
  engine.registerEnumeratedInputSymbolEnumElement("role", "offensive-supporter", BehaviorTeamMessage::offensiveSupporter);

  // "estimated-time-to-reach-ball"
  engine.registerDecimalInputSymbol("estimated-time-to-reach-ball", (double *)&estimatedTimeToReachBall);

  // "sent-game-state"
  engine.registerEnumeratedOutputSymbol("sent-game-state",(int*)&outgoingBehaviorTeamMessage.gameState);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.sleep",BehaviorTeamMessage::initial);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.initial",BehaviorTeamMessage::initial);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.ready",BehaviorTeamMessage::ready);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.playing",BehaviorTeamMessage::playing);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.penalized",BehaviorTeamMessage::penalized);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.finished",BehaviorTeamMessage::finished);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.set",BehaviorTeamMessage::set);

  // "another-player-is-in-ready-state"
  engine.registerBooleanInputSymbol("another-player-is-in-ready-state",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getAnotherPlayerIsInReadyState);

  // "team-message"
  engine.registerEnumeratedOutputSymbol("team-message",(int*)&outgoingBehaviorTeamMessage.message);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.none", BehaviorTeamMessage::none);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.just-performed-a-kick",BehaviorTeamMessage::justPerformedAKick);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.performing-a-kick",BehaviorTeamMessage::performingAKick);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.preparing-a-kick",BehaviorTeamMessage::preparingAKick);

  // "another-teammate-is-preparing-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-is-preparing-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getAnotherTeammateIsPreparingAKick);

  // "another-teammate-is-performing-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-is-performing-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getAnotherTeammateIsPerformingAKick);

  // "another-teammate-just-performed-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-just-performed-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getAnotherTeammateJustPerformedAKick);

  // "the-striker-is-playing-near-the-opponent-goal"
  engine.registerBooleanInputSymbol("the-striker-is-playing-near-the-opponent-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getTheStrikerIsPlayingNearTheOpponentGoal);

  // "the-striker-is-not-playing-near-the-opponent-goal"
  engine.registerBooleanInputSymbol("the-striker-is-not-playing-near-the-opponent-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getTheStrikerIsNotPlayingNearTheOpponentGoal);

  // "the-striker-is-playing-near-the-own-goal"
  engine.registerBooleanInputSymbol("the-striker-is-playing-near-the-own-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getTheStrikerIsPlayingNearTheOwnGoal);

  // "the-striker-is-not-playing-near-the-own-goal"
  engine.registerBooleanInputSymbol("the-striker-is-not-playing-near-the-own-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getTheStrikerIsNotPlayingNearTheOwnGoal);

  // "catch-ball"
  engine.registerBooleanInputSymbol("catch-ball", this,
    (bool (Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getCaught);

  // "catch-ball-time"
  engine.registerDecimalInputSymbol("catch-ball-time",this,
    (double (Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getCatchTime);

  // "robot-is-stuck"
  engine.registerBooleanInputSymbol("robot-is-stuck", &robotIsStuck);

  // "obstacles-are-close"
  engine.registerBooleanInputSymbol("obstacles-are-close", &obstaclesAreClose);

  engine.registerDecimalInputSymbol("search-ball.x", this,
    (double (Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getSearchBallX);

  // "get-good-shot"
  engine.registerDecimalInputFunction("get-good-shot",this,
    (double (Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getGoodShot);
  engine.registerDecimalInputFunctionParameter("get-good-shot","get-good-shot.angle",getGoodShotAngle);

  // "get-good-x"
  engine.registerDecimalInputFunction("get-good-x", this,
    (double(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getGoodX);

  // "get-good-y"
  engine.registerDecimalInputFunction("get-good-y", this,
    (double(Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getGoodY);

  // "last-good-shoot"
  engine.registerDecimalInputSymbol("last-good-shot",&goodShot);

  // "goalie-go-to-ball"
  engine.registerBooleanInputSymbol("goalie-go-to-ball",  this,
    (bool (Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::goalieGoToBall);

  // "get-angle-to-teammate
  engine.registerDecimalInputFunction("get-angle-to-teammate",this,
    (double (Xabsl2FunctionProvider::*)())&ATH2004StrategySymbols::getAngleToTeammate);
  engine.registerDecimalInputFunctionParameter("get-angle-to-teammate","get-angle-to-teammate.index",ATH2004StrategySymbols::angleToTeammateIndex);

  engine.registerEnumeratedInputSymbol("goalie-clear-ball-behavior", (int *)&goalieClearBallBehavior);
  engine.registerEnumeratedInputSymbolEnumElement("goalie-clear-ball-behavior", "active", ATH2004StrategySymbols::ACTIVE);
  engine.registerEnumeratedInputSymbolEnumElement("goalie-clear-ball-behavior", "passive", ATH2004StrategySymbols::PASSIVE);


}

void ATH2004StrategySymbols::update()
{
  estimateTimeToReachBall();
  computeRole();

  outgoingBehaviorTeamMessage.estimatedTimeToReachBall = estimatedTimeToReachBall;
  outgoingBehaviorTeamMessage.dynamicRole = role;

  if (((executedMotionRequest.motionType == MotionRequest::specialAction
    && 
    (executedMotionRequest.specialActionType == MotionRequest::catchBall2
    || executedMotionRequest.specialActionType == MotionRequest::lookForLandmarksWithBallCaught)
    )
    || 
    (executedMotionRequest.motionType == MotionRequest::walk 
    && executedMotionRequest.walkType == MotionRequest::turnWithBall)
    )
    || headControlMode.headControlMode == HeadControlMode::catchBall)
  {
    if (SystemCall::getTimeSince(timeUntilBallWasCaught) > 1000)
    {
      timeWhenBallWasStartedToCatch = SystemCall::getCurrentSystemTime();
    }

    timeUntilBallWasCaught = SystemCall::getCurrentSystemTime();
  }

  // robot is stuck
  if(obstaclesModel.getPercentageOfLowDistanceObstaclesInRange(0, pi_2, 300) > 0.2)
  {
    robotIsStuck = true;
    obstaclesAreClose = true;
  }
  else
  {
    robotIsStuck = false;
    if(obstaclesModel.getPercentageOfLowDistanceObstaclesInRange(0, pi_2, 500) > 0.15)
      obstaclesAreClose = true;
    else
      obstaclesAreClose = false;
  }
}

void ATH2004StrategySymbols::computeRole()
{
  int i;
  if (getPlayer().getPlayerNumber() == Player::one)
  {
    // no role changes for the goalie
    role = BehaviorTeamMessage::goalie;
    return;
  }

  // fall-back if no wlan
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
    && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
    && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
    case Player::two:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x < 75)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::defensiveSupporter;
      break;
    case Player::three:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x > 500)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::offensiveSupporter;
      break;
    case Player::four:
    default:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x > -75)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::offensiveSupporter;
      break;
    }
    return;
  }

  // with wlan

  // estimate the closest other teammate to the ball
  double minTeammateTime = 100000.0;
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() ) // only new messages
    {
      if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::goalie &&
        (
        teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing ||
        teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::ready ||
        teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::set
        )
        )
        // the robot is not interested in 
        // (1) how fast the goalie can approach the ball 
        // (2) how fast a penalized field player can approach the ball
      {
        double time = 
          teamMessageCollection[i].behaviorTeamMessage.estimatedTimeToReachBall;

        // bonus for current striker
        if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker ) 
          time -= 500.0;

        if (time < minTeammateTime ) 
        {
          minTeammateTime = time;
        }
      }
    }
  }

  // bonus for current striker
  if (role == BehaviorTeamMessage::striker)
    minTeammateTime += 1000.0;

  // assign striker-role if nearest to the ball
  if ( estimatedTimeToReachBall < minTeammateTime )
  {
    role = BehaviorTeamMessage::striker;
    return;
  }

  // assign supporting roles
  double maxTeammateX = (double)xPosOwnGroundline;
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() )
    {
      if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::striker 
        && teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::goalie 
        && 
        (
        teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing ||
        teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::ready ||
        teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::set 
        )
        )
      {
        double teammateX = 
          teamMessageCollection[i].robotPose.translation.x;

        // bonus for current offensive supporter
        if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::offensiveSupporter ) 
          teammateX += 300.0;

        if (teammateX > maxTeammateX ) 
        {
          maxTeammateX = teammateX;
        }
      }
    }
  }

  // bonus for current offensive supporter
  if ( role == BehaviorTeamMessage::offensiveSupporter ) 
    maxTeammateX -= 300.0;

  if ( robotPose.translation.x >= maxTeammateX ) 
    role = BehaviorTeamMessage::offensiveSupporter;
  else 
    role = BehaviorTeamMessage::defensiveSupporter;
}

void ATH2004StrategySymbols::estimateTimeToReachBall()
{
  // stay striker if ball is caught
  if (getCaught())
  {
    estimatedTimeToReachBall = 0.0;
    return;
  }

  // account for distance to ball
  estimatedTimeToReachBall = Geometry::distanceTo(robotPose.translation, ballPosition.seen) / 0.2;

  // account if the robot is between the ball and the opponent goal

  // the position of the robot
  Vector2<double> robotPosition = robotPose.translation;
  double angleToLeftOpponentGoalPost = Geometry::angleTo(robotPosition, Vector2<double>(xPosOpponentGroundline,yPosLeftGoal));
  double angleToRightOpponentGoalPost = Geometry::angleTo(robotPosition, Vector2<double>(xPosOpponentGroundline,yPosRightGoal));
  if(angleToLeftOpponentGoalPost < angleToRightOpponentGoalPost)
  {
    angleToLeftOpponentGoalPost += pi2;
  }
  double angleToOpponentGoal= (angleToLeftOpponentGoalPost + angleToRightOpponentGoalPost) / 2.0;
  double absoluteAngleToBall = Geometry::angleTo(robotPosition,ballPosition.seen);
  double angleBetweenBallAndOpponentGoal = normalize(angleToOpponentGoal - absoluteAngleToBall);

  estimatedTimeToReachBall += 400.0 * fabs(angleBetweenBallAndOpponentGoal);

  // longer if ball not seen
  estimatedTimeToReachBall += 2.0 * SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);

  // test for obstacles
  /*
  if ( isOpponentBetweenRobotAndBall(robotPose) || 
  isOwnBetweenRobotAndBall (robotPose) )
  time += 5000.0;
  */

}

double ATH2004StrategySymbols::getAngleToTeammate()
{

  //OUTPUT(idText, text,"::nrofownplayer:"<<playerPoseCollection.numberOfOwnPlayers);


  if(1 <= playerPoseCollection.numberOfOwnPlayers)
  {

    //OUTPUT(idText, text,"::angle:"<<toDegrees(Geometry::angleTo(robotPose, playerPoseCollection.getOwnPlayerPose((int)angleToTeammateIndex).getPose().translation)));

    return toDegrees(Geometry::angleTo(robotPose, playerPoseCollection.getOwnPlayerPose((int)angleToTeammateIndex).getPose().translation));
  }
  else
  {
    return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>(xPosOpponentGroundline,yPosCenterGoal)));
  }
}

bool ATH2004StrategySymbols::getAnotherPlayerIsInReadyState()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.gameState 
        == BehaviorTeamMessage::ready)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004StrategySymbols::getAnotherTeammateIsPreparingAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::preparingAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004StrategySymbols::getAnotherTeammateIsPerformingAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::performingAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004StrategySymbols::getAnotherTeammateJustPerformedAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::justPerformedAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004StrategySymbols::getTheStrikerIsPlayingNearTheOpponentGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x > 1400)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004StrategySymbols::getTheStrikerIsNotPlayingNearTheOpponentGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x < 1200)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004StrategySymbols::getTheStrikerIsPlayingNearTheOwnGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x < -1200)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004StrategySymbols::getTheStrikerIsNotPlayingNearTheOwnGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x > -1000)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004StrategySymbols::getCaught()
{
  return (SystemCall::getTimeSince(timeUntilBallWasCaught) < 500);  
}

double ATH2004StrategySymbols::getCatchTime()
{
  return (SystemCall::getTimeSince(timeUntilBallWasCaught) < 500?
    timeUntilBallWasCaught - timeWhenBallWasStartedToCatch : 0);
}

double ATH2004StrategySymbols::getSearchBallX()
{
  // fall-back if no wlan
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
    && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
    && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
    case Player::two:
      return -1000;
    case Player::three:
      return 0;
    case Player::four:
    default:
      return 1000;
    }
  }
  else
  {
    switch (role)
    {
    case BehaviorTeamMessage::defensiveSupporter:
      return -1000;
    case BehaviorTeamMessage::offensiveSupporter:
      return 1000;
    case BehaviorTeamMessage::striker:
    default:
      return 0;
    }
  }
}

double ATH2004StrategySymbols::getGoodShot()
{
  // if (ballPosition.seen.timeWhenLastSeen > 100) return -1;
  // own left corner
  if ((ballPosition.seen.x < -1900) && (ballPosition.seen.y > -1100) && (ballPosition.seen.y < -1000)) return MotionRequest::athErs7HeadKickRight;

  // own left corner2
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.x > -1900) && (ballPosition.seen.y > -1050)) return MotionRequest::kickWithRightToLeft;

  // own left side
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.y > -1050) && (ballPosition.seen.y < 0)) return MotionRequest::kickWithLeftToRight;

  // own right corner
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.y < -1100) && (ballPosition.seen.y > -1000)) return MotionRequest::athErs7HeadKickLeft;

  // own right corner2
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.x > -1900) && (ballPosition.seen.y > -1050))  return MotionRequest::kickWithLeftToRight;

  // own right side
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.y < 1050) && (ballPosition.seen.y > 0)) return MotionRequest::kickWithRightToLeft;

  // opponent left corner
  if ((ballPosition.seen.x > 1800) && (ballPosition.seen.y > -1100) && (ballPosition.seen.y < -500)) return MotionRequest::openArm;

  // opponent left corner2
  if ((ballPosition.seen.x > 1600) && (ballPosition.seen.y < -1000)) return MotionRequest::kickWithRightToLeft;

  // opponent right corner
  if ((ballPosition.seen.x > 1800) && (ballPosition.seen.y > 500) && (ballPosition.seen.y < 1100)) return MotionRequest::openArm;

  // opponent right corner2
  if ((ballPosition.seen.x > 1600) && (ballPosition.seen.y > 1000)) return MotionRequest::kickWithLeftToRight;

  // left border 
  if (ballPosition.seen.y < -1050) return MotionRequest::lanceKickSoftRight;

  // right border
  if (ballPosition.seen.y > 1050) return MotionRequest::lanceKickSoftLeft;

  // near opponent goal
  if (ballPosition.seen.x > 1200) return MotionRequest::athErs7ForwardLeapingKick;

  // in center of field
  return MotionRequest::athErs7ExactForwardKick;

}

double ATH2004StrategySymbols::getGoodX()
{
  int distance = 200;
  // own left corner and own right corner
  if (((ballPosition.seen.x < -1900) && (ballPosition.seen.y > -1100) && (ballPosition.seen.y < -1000))||((ballPosition.seen.x < -1800) && (ballPosition.seen.y < -1100) && (ballPosition.seen.y > -1000)))
    return (distance*(ballPosition.seen.x-xPosOpponentGoal)/sqrt(sqr(ballPosition.seen.x-xPosOpponentGoal)+sqr(ballPosition.seen.y-yPosCenterGoal))+ballPosition.seen.x);

  // own left corner2
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.x > -1900) && (ballPosition.seen.y > -1050)) return MotionRequest::kickWithRightToLeft;

  // own left side
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.y > -1050) && (ballPosition.seen.y < 0)) return MotionRequest::kickWithLeftToRight;

  // own right corner2
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.x > -1900) && (ballPosition.seen.y > -1050))  return MotionRequest::kickWithLeftToRight;

  // own right side
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.y < 1050) && (ballPosition.seen.y > 0)) return MotionRequest::kickWithRightToLeft;

  // opponent left corner
  if ((ballPosition.seen.x > 1800) && (ballPosition.seen.y > -1100) && (ballPosition.seen.y < -500)) return MotionRequest::openArm;

  // opponent left corner2
  if ((ballPosition.seen.x > 1600) && (ballPosition.seen.y < -1000)) return MotionRequest::kickWithRightToLeft;

  // opponent right corner
  if ((ballPosition.seen.x > 1800) && (ballPosition.seen.y > 500) && (ballPosition.seen.y < 1100)) return MotionRequest::openArm;

  // opponent right corner2
  if ((ballPosition.seen.x > 1600) && (ballPosition.seen.y > 1000)) return MotionRequest::kickWithLeftToRight;

  // left border 
  if (ballPosition.seen.y < -1050) return MotionRequest::lanceKickSoftRight;

  // right border
  if (ballPosition.seen.y > 1050) return MotionRequest::lanceKickSoftLeft;

  // near opponent goal
  if (ballPosition.seen.x > 1200) return MotionRequest::athErs7ForwardLeapingKick;

  // in center of field
  return MotionRequest::athErs7ExactForwardKick;
  //return 0;
}

double ATH2004StrategySymbols::getGoodY()
{
  int distance = 200;
  // own left corner or own right corner
  if (((ballPosition.seen.x < -1900) && (ballPosition.seen.y > -1100) && (ballPosition.seen.y < -1000))||((ballPosition.seen.x < -1800) && (ballPosition.seen.y < -1100) && (ballPosition.seen.y > -1000)))
    return (distance*(ballPosition.seen.y-yPosCenterGoal)/sqrt(sqr(ballPosition.seen.x-xPosOpponentGoal)+sqr(ballPosition.seen.y-yPosCenterGoal))+ballPosition.seen.y);

  // own left corner2
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.x > -1900) && (ballPosition.seen.y > -1050)) return MotionRequest::kickWithRightToLeft;

  // own left side
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.y > -1050) && (ballPosition.seen.y < 0)) return MotionRequest::kickWithLeftToRight;

  // own right corner2
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.x > -1900) && (ballPosition.seen.y > -1050))  return MotionRequest::kickWithLeftToRight;

  // own right side
  if ((ballPosition.seen.x < -1800) && (ballPosition.seen.y < 1050) && (ballPosition.seen.y > 0)) return MotionRequest::kickWithRightToLeft;

  // opponent left corner
  if ((ballPosition.seen.x > 1800) && (ballPosition.seen.y > -1100) && (ballPosition.seen.y < -500)) return MotionRequest::openArm;

  // opponent left corner2
  if ((ballPosition.seen.x > 1600) && (ballPosition.seen.y < -1000)) return MotionRequest::kickWithRightToLeft;

  // opponent right corner
  if ((ballPosition.seen.x > 1800) && (ballPosition.seen.y > 500) && (ballPosition.seen.y < 1100)) return MotionRequest::openArm;

  // opponent right corner2
  if ((ballPosition.seen.x > 1600) && (ballPosition.seen.y > 1000)) return MotionRequest::kickWithLeftToRight;

  // left border 
  if (ballPosition.seen.y < -1050) return MotionRequest::lanceKickSoftRight;

  // right border
  if (ballPosition.seen.y > 1050) return MotionRequest::lanceKickSoftLeft;

  // near opponent goal
  if (ballPosition.seen.x > 1200) return MotionRequest::athErs7ForwardLeapingKick;

  // in center of field
  return MotionRequest::athErs7ExactForwardKick;
}

bool ATH2004StrategySymbols::getDecisionGoalieGoToBall(bool lastDecision)
{
  const double maxBalldistToGoalie = 900;			// 90cm
  const double maxBalldistToPenalty = 400;		// 40cm
  const double minBalldistToPenalty = 100;		// 10cm
  const double maxBallspeedDirGoal = 500;			// 50cm/s

  double balldistToGoalie;
  double ballSpeedToGoal;			// >0 : ball move to Goal <0 : ball don't move to goal


  balldistToGoalie = Geometry::distanceTo(robotPose,ballPosition.seen);

  /*first version , it's not good to use it*/
  ballSpeedToGoal = ballPosition.seen.speed.abs();   

  if ( lastDecision == true )
  {

    /* if ball close to the penaltyarea,GO!!!*/
    if ( FieldDimensions::distanceToOwnPenaltyArea(ballPosition.seen) < minBalldistToPenalty ) 
      return ( true );

    //keep the ball
    if ( ( balldistToGoalie < 100 ) && ( FieldDimensions::distanceToOwnPenaltyArea(ballPosition.seen) < maxBalldistToPenalty ) ) 
      return ( true );

    //back to Goal distance to ball is bigger than distance to obstacles
    if ( ( balldistToGoalie > 100 ) && ( obstaclesAreClose == true ) ) 
      return ( false );

  }
  else
  {

    /* if ball close to the penaltyarea,GO!!!*/
    if ( FieldDimensions::distanceToOwnPenaltyArea(ballPosition.seen) < minBalldistToPenalty ) 
      return ( true );

    if ( FieldDimensions::distanceToOwnPenaltyArea(ballPosition.seen) > maxBalldistToPenalty ) 
      return ( false );

    /*should remove, dist is not good (validity)*/
    if ( balldistToGoalie > maxBalldistToGoalie ) return ( false );

    //bad	/*enemy*/
    if ( ( obstaclesAreClose == true ) && ( balldistToGoalie > 100 ) ) return ( false );

    //bad	/*ballSpeed*/
    if ( ballSpeedToGoal > maxBallspeedDirGoal ) return ( false );

  }

  return false;
}

bool ATH2004StrategySymbols::goalieGoToBall()
{

  //const double maxBalldistToGoalie = 900;			// 90cm
  //const double maxBalldistToPenalty = 400;		// 40cm
  //const double minBalldistToPenalty = 100;		// 10cm
  //const double maxBallspeedDirGoal = 500;			// 50cm/s

  int useKindOfShoot = 0;

  static bool decision;			//have to know whats the last decision
  double balldistToGoalie;
  double ballSpeedToGoal;			// >0 : ball move to Goal <0 : ball don't move to goal
  int ballArea;
  int maxBalldistToGoalieWithArea = 0;


  balldistToGoalie = Geometry::distanceTo(robotPose,ballPosition.seen);

  /*first version , it's not good to use it*/
  ballSpeedToGoal = ballPosition.seen.speed.abs();   


  decision = getDecisionGoalieGoToBall(decision);

  /*field*/
  /*get Sektor of Ball*/
  /*behind penaltyAreaLine*/
  /* ballArea: 0=penaltyArea 1=Leftcorner 2=Rightcorner 4=front 5=leftfront 6=rightfront*/

  if ( ballPosition.seen.x < xPosOwnPenaltyArea ) ballArea = 0;
  else ballArea = 4;

  if ( ballPosition.seen.y < yPosLeftPenaltyArea ) ballArea+= 1;
  if ( ballPosition.seen.y > yPosRightPenaltyArea ) ballArea+= 2;

  //some bad
  if ( ballArea == 0 ) maxBalldistToGoalieWithArea = 9999; // if ball is in penaltyarea
  if ( ballArea == 4 ) maxBalldistToGoalieWithArea = 500;  // if ball is in front of penaltyarea
  if ( ( ballArea == 5 ) || ( ballArea == 6 ) ) maxBalldistToGoalieWithArea = 500; // left and right front 
  if ( ( ballArea == 1 ) || ( ballArea == 2 ) ) maxBalldistToGoalieWithArea = yPosLeftSideline; // left and right corner

  //some bad : balldistToGoal is maybe better
  if ( balldistToGoalie < maxBalldistToGoalieWithArea ) return ( decision = true ); 

  if ( ( ballArea <= 4 ) || ( ballArea == 0 ) ) useKindOfShoot = 0; //forward
  if ( ballArea == 1 ) useKindOfShoot = 2; //right
  if ( ballArea == 2 ) useKindOfShoot = 1; //left


  if ( useKindOfShoot == 0 ) goalieClearBallBehavior = ACTIVE;
  if ( useKindOfShoot == 2 ) goalieClearBallBehavior = ACTIVE;
  if ( useKindOfShoot == 1 ) goalieClearBallBehavior = ACTIVE;

  if ( ballSpeedToGoal > 300 ) goalieClearBallBehavior = PASSIVE;



  return decision;
}

/*
* Change Log
* 
* $Log: ATH2004StrategySymbols.cpp,v $
* Revision 1.26  2004/04/08 10:08:41  risler
* fixed and unified team color symbols
*
* Revision 1.25  2004/04/05 17:56:45  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.3  2004/04/03 00:56:31  juengel
* Role change in set and ready.
*
* Revision 1.2  2004/04/01 20:49:42  loetzsch
* renamed athErs7UnswBash to athErs7ExactForwardKick
*
* Revision 1.1.1.1  2004/03/31 11:16:38  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.24  2004/03/25 17:40:15  loetzsch
* adaptations to changes in the game controller and the led request
*
* Revision 1.23  2004/03/16 18:37:23  brueckne
* Added ATH2004StrategySymbols::getGoodX/Y
*
* Revision 1.22  2004/03/08 00:58:44  roefer
* Interfaces should be const
*
* Revision 1.21  2004/03/04 23:01:55  roefer
* Warnings and errors removed
*
* Revision 1.20  2004/03/04 17:25:00  sombrutz
* added symbol goalie-clear-ball-behavior
*
* Revision 1.19  2004/03/04 15:08:16  sombrutz
* added some comments
* added getDecisionGoalieGoToBall(bool lastDecision)
*
* Revision 1.18  2004/03/03 15:10:43  loetzsch
* added some comments in computeRole()
*
* Revision 1.17  2004/03/02 00:26:20  roefer
* Warnings removed
*
* Revision 1.16  2004/02/27 20:33:19  brueckne
* improved penalty-shooter-hertha
* chooses nearly everywhere the right shot
*
* Revision 1.15  2004/02/27 18:21:02  brueckne
* continued with penalty-shooter hertha
*
* Revision 1.14  2004/02/27 15:29:04  juengel
* Added getTeamColor().
*
* Revision 1.13  2004/02/27 10:43:28  spranger
* added get-Angle-to-Teammate function and parameter
*
* Revision 1.12  2004/02/26 17:17:29  loetzsch
* bug fixes and renaming of "shoot" to "shot"
*
* Revision 1.11  2004/02/26 14:41:47  brueckne
* Continued penality shooter hertha
*
* Revision 1.10  2004/02/15 23:32:55  sombrutz
* new stuff for goalie-go-to-ball
*
* Revision 1.9  2004/02/13 17:34:49  sombrutz
* add code for goalieGoToBall
*
* Revision 1.8  2004/02/13 11:32:31  sombrutz
* add GoalieGoToBall ( used by the new goalie)
*
* Revision 1.7  2004/01/22 12:43:13  dueffert
* flush removed because it doesnt work properly on robots
*
* Revision 1.6  2004/01/20 17:53:01  brueckne
* try to save datas in shoots.log
*
* Revision 1.5  2004/01/13 18:06:30  brueckne
* One step closer to getGoodShoot
*
* Revision 1.4  2004/01/08 17:32:38  brueckne
* Still working on getGoodShoot
*
* Revision 1.3  2003/12/11 17:08:17  brueckne
* Continued penalty-shooter-hertha
*
* Revision 1.2  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/26 22:49:35  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

