/** 
* @file ATH2004ERS7ContinuousRules/AvoidOwnPenaltyArea.h
*
* 
* @author Martin Kallnik
*/

#ifndef __ATH2004ERS7AvoidOwnPenaltyArea_h_
#define __ATH2004ERS7AvoidOwnPenaltyArea_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace ATH2004ERS7ContinuousRules
{

/**@class AvoidOwnPenaltyArea
 * rule which avoids the own penalty area
 */
class AvoidOwnPenaltyArea: public ContinuousRule{
private:
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is maxForce (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;

public:

  /** constructor
   */
  AvoidOwnPenaltyArea(BehaviorControlInterfaces& interfaces,
                 double maxForce = 3.0, 
                 double maxForceLength = 100.0, 
                 double influenceLength = 300.0)
      :ContinuousRule(interfaces, "AvoidOwnPenaltyArea"),
       maxForce(maxForce),
       maxForceLength(maxForceLength),
       influenceLength(influenceLength)
  {};
 
 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
	 * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    if (getPlayer().getPlayerNumber() != Player::one)
    {
    
      double distance = min (
        FieldDimensions::distanceToOwnPenaltyArea ( robotPose.translation ),
        FieldDimensions::distanceToOwnPenaltyArea ( robotPose*Vector2<double>(-bodyLength,0) ) );
    
      walk = robotPose.translation;
      walk.x -= xPosOwnGroundline;
      walk.normalize();
    
      walk *= getRepellingForce(distance, maxForce, maxForceLength, influenceLength);
    } else {
      // no avoid own penalty area for goalie
      walk.x = walk.y = 0;
    }
    
    ra=0;
    rweight=0;
  };
};

}


#endif 


/*
* Change log:
*
* $Log: AvoidOwnPenaltyArea.h,v $
* Revision 1.1  2004/03/16 14:00:18  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.1  2004/03/06 12:52:12  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
* created local cvs for Gnne
*
* Revision 1.3  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.2  2003/10/28 15:04:09  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004ERS7BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004ERS7 three times (BB2004, DDD2004, MSH2004)
*
*/
