/** 
* @file ATH2004ERS7SimpleBasicBehaviors.cpp
*
* Implementation of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#include "Tools/Math/PIDsmoothedValue.h"
#include "ATH2004ERS7SimpleBasicBehaviors.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Matrix2x2.h"
#include "Tools/Math/Common.h"

#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Debugging/Debugging.h"

//needed for dribbleInCenterOfField and getBehindBall2
#include "Tools/FieldDimensions.h"

void ATH2004ERS7BasicBehaviorGoToBallOld::execute()
{
  //accelerationRestrictor.saveLastWalkParameters();
  
  if (maxSpeed == 0) maxSpeed = 350;
  
  double maxSpeedX = maxSpeed;
  double maxSpeedY = maxSpeed;
  
  double maxTurnSpeed = fromDegrees(60);
  
  double timeToReachX, timeToReachY, timeToTurnToAngle, estimatedTimeToReachDestination;
  
  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ball);
  double ballX = ballDistance * cos(Geometry::angleTo(robotPose.getPose(),ball)) + 50;
  double ballY = ballDistance * sin(Geometry::angleTo(robotPose.getPose(),ball));
  
  if(ballDistance == 0)
    ballDistance = 1;
  double ballAngle = Geometry::angleTo(robotPose.getPose(),ball);
  
  double diffX = distanceAtEnd / ballDistance * ballX;
  double diffY = distanceAtEnd / ballDistance * ballY;
  
  Vector2<double> destination(ballX - diffX, ballY - diffY);
  double distanceToDestination;
  distanceToDestination = fabs(ballDistance - distanceAtEnd);
  
  timeToReachX = fabs(destination.x) / maxSpeedX;
  timeToReachY = fabs(destination.y) / maxSpeedY;
  timeToTurnToAngle = ballAngle / maxTurnSpeed;
  estimatedTimeToReachDestination = 
    (timeToReachX < timeToReachY) ? timeToReachY : timeToReachX;
  estimatedTimeToReachDestination = 
    (estimatedTimeToReachDestination < timeToTurnToAngle) ? timeToTurnToAngle : estimatedTimeToReachDestination;
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  motionRequest.walkParams.translation.x = 
    destination.x / estimatedTimeToReachDestination;
  motionRequest.walkParams.translation.y = 
    0.5 * (destination.y / estimatedTimeToReachDestination);
  
  
  motionRequest.walkParams.rotation = ballAngle * 0.6;
  
  if (motionRequest.walkParams.rotation > maxTurnSpeed)
    motionRequest.walkParams.rotation = maxTurnSpeed;
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    motionRequest.walkParams.rotation = -maxTurnSpeed;
  
  
  if ((fabs(toDegrees(ballAngle))<10)&&(distanceToDestination<20))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
  else
  {
    //    accelerationRestrictor.restrictAccelerations(250,250,180);
  }
}

void ATH2004ERS7BasicBehaviorGoToBall::execute()
{
  //accelerationRestrictor.saveLastWalkParameters();
  
  double targetAngle;
 
  if (maxSpeed == 0) maxSpeed = 350;
  if (targetAngleToBall == 0) 
    targetAngle = 0;
  else 
    targetAngle = targetAngleToBall;


  double maxTurnSpeed = fromDegrees(180);
  
  Vector2<double> ballInWorldCoord = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  double angleToBall = Geometry::angleTo(robotPose.getPose(),ballInWorldCoord);
  double distanceToBall = Geometry::distanceTo(robotPose.getPose(),ballInWorldCoord);

  // if distance is between 400 and 600,
  // perform linear interpolation
  double factor = 1 - (distanceToBall - 400) / 200;
  if (factor > 1) factor = 1;
  if (factor < 0) factor = 0;

  //targetAngle *= factor;
  //angleToBall += targetAngle;  

  // destination = ball position in robot 
  // coordintes plus 200mm in x direction
  Vector2<double> destination(
      distanceToBall * cos(angleToBall) + 400, 
      distanceToBall * sin(angleToBall) + targetAngleToBall);


  if (distanceToBall < 50)
  {
    destination.x = 1; 
    destination.y = 0;
  }

  // adjust forward speed:
  // if a lot of turning is necessary, don't walk so fast!
  factor = (pi-fabs(angleToBall))/pi;
  if (factor > 1) factor = 1;
  if (factor < 0) factor = 0;

  destination.normalize();
  destination *= (maxSpeed*factor);

  /*
  factor = .2 + distanceToBall / 500;
  if (factor > 1) factor = 1;
  if (factor < 0) factor = 0;

  destination *= factor;
*/

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;  
  motionRequest.walkParams.translation.x = destination.x;
  motionRequest.walkParams.translation.y = destination.y;

  factor = 1.5;
  /*
  factor = .2 + 1.5 * distanceToBall / 800;
  if (factor > 1.5) factor = 1.5;
  if (factor < 0) factor = 0;
*/
  motionRequest.walkParams.rotation = angleToBall * factor; /* this should also be factored!!!! */
  
  // clip rotation speed
  motionRequest.walkParams.rotation = min(motionRequest.walkParams.rotation, maxTurnSpeed);
  motionRequest.walkParams.rotation = max(motionRequest.walkParams.rotation, -maxTurnSpeed);  
}

void ATH2004ERS7BasicBehaviorGoaliePosition::execute()
{
  if (maxSpeed == 0)
    maxSpeed = 150;
  //double maxV = maxSpeed;

  double radius = 630;
  Vector2<double> pointToGuard = Vector2<double>(xPosOwnGroundline-370, 0);
  Vector2<double> ballInWorldCoord = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
//  double angleToBall = Geometry::angleTo(robotPose.getPose(),ballInWorldCoord);
  
  ARROW(ballLocatorField, pointToGuard.x, pointToGuard.y, ballInWorldCoord.x, ballInWorldCoord.y, 1, 3, Drawings::red);
  Vector2<double> guardPosition = ballInWorldCoord - pointToGuard;

  if (guardPosition.angle() > .75)
  {
    guardPosition.x = 1;
    guardPosition.y = .8;
  } 
  else if (guardPosition.angle() < -.75)
  {
    guardPosition.x = 1;
    guardPosition.y = -.8;
  } 

  guardPosition.normalize();
  guardPosition *= radius;
  guardPosition += pointToGuard;

  CIRCLE(ballLocatorField, guardPosition.x, guardPosition.y, 10, 3, 0, Drawings::red);

  Vector2<double> fromRobotOutward = robotPose.translation - pointToGuard;
  ARROW(ballLocatorField, robotPose.translation.x, robotPose.translation.y, 
    robotPose.translation.x + fromRobotOutward.y, robotPose.translation.y - fromRobotOutward.x, 1, 3, Drawings::red);

  Vector2<double> direction = guardPosition - robotPose.translation;
  //Vector2<double> onCircle(-fromRobotOutward.y, fromRobotOutward.x);
  //onCircle.normalize();
  //double leftOrRight = onCircle * direction;
  //direction = onCircle *leftOrRight;

  Vector2<double> c1(cos(robotPose.rotation), -sin(robotPose.rotation)), 
    c2(sin(robotPose.rotation), cos(robotPose.rotation));
  Matrix2x2<double> rotate(c1, c2);

  direction = rotate*direction;
  if (direction.abs() > maxSpeed)
  {
    direction.normalize();
    direction *= maxSpeed;
  }
  
  double rotation = (fromRobotOutward.angle()-robotPose.rotation);//= .25*angleToBall;

  if (rotation > 0.0)
	{
		if (robotPose.getAngle() + rotation > pi/4)
			rotation = pi/4 - robotPose.getAngle();
	}
	else
	{
		if (robotPose.getAngle() + rotation < -pi/4)
			rotation = -pi/4 - robotPose.getAngle();
	}

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;  
  motionRequest.walkParams.translation.x = direction.x;
  motionRequest.walkParams.translation.y = direction.y;
  motionRequest.walkParams.rotation = rotation; 
  DEBUG_DRAWING_FINISHED(ballLocatorField);  
}





void ATH2004ERS7BasicBehaviorTurnAroundPoint::execute()
{
  double maxTurnSpeed = fromDegrees(180);
  if (forwardComponent == 0) forwardComponent = 200.0;
  
  Vector2<double> destinationInWorldCoord(x,y);
  double angleToDestination = Geometry::angleTo(robotPose.getPose(),destinationInWorldCoord);
  double distance = Geometry::distanceTo(robotPose.getPose(),destinationInWorldCoord);

  // destination = ball position in robot 
  // coordintes plus 100mm in x direction
  Vector2<double> destination(
      distance * cos(angleToDestination), 
      distance * sin(angleToDestination));
  
  double factor = (distance - radius) / 800; 
  if (factor > 1) factor = 1;
  if (factor < 0) factor = 0;

  destination *= factor;
  destination.y += (leftRight > 0 ? 1 : -1)*(200 - forwardComponent/5)*(1-factor);

  if (destination.x < forwardComponent)   
    destination.x = forwardComponent;

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;  
  motionRequest.walkParams.translation.x = destination.x;
  motionRequest.walkParams.translation.y = destination.y;
  motionRequest.walkParams.rotation = angleToDestination * 2;
  
  // clip rotation speed
  motionRequest.walkParams.rotation = min(motionRequest.walkParams.rotation, maxTurnSpeed);
  motionRequest.walkParams.rotation = max(motionRequest.walkParams.rotation, -maxTurnSpeed); 
}

void ATH2004ERS7BasicBehaviorGoForwardToPoint::execute()
{
  accelerationRestrictor.saveLastWalkParameters();
  
  Vector2<double> destination(x,y);
  
  if (maxSpeed == 0) maxSpeed = 350;
  
  double maxTurnSpeed = fromDegrees(120);
  
  double angleToDestination = Geometry::angleTo(robotPose,destination);
  
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
  motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  
  motionRequest.walkParams.rotation = angleToDestination*2;
  if (motionRequest.walkParams.rotation > maxTurnSpeed)
  {
    motionRequest.walkParams.rotation = maxTurnSpeed;
  }
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
  {
    motionRequest.walkParams.rotation = -maxTurnSpeed;
  }

  accelerationRestrictor.restrictAccelerations(300,300,200);
  
}



void ATH2004ERS7BasicBehaviorGoToPointAndAvoidObstacles::execute()
{
  double 
    obstacleAvoidanceDistance,
    distanceToDestination = Geometry::distanceTo(robotPose.getPose(),Vector2<double>(x,y)),
    angleToDestination = Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)),
    widthOfCorridor,
    targetSpeed, 
    targetAngle, 
    freeSpaceInFront,
    minSpeed,
    distanceBelowWhichObstaclesAreAvoided;
  int mode = 0;

  // reset PIDs if last execution is long ago
  if (SystemCall::getTimeSince(lastTimeExecuted) > 500)
  {
    speedX.resetTo(100);
    speedY.resetTo(0);
    turnSpeed.resetTo(0); 
  }
  lastTimeExecuted = SystemCall::getCurrentSystemTime();
   
  if (maxSpeed == 0) maxSpeed = 350;

  // avoidance level:
  // 0 = rugby (very little avoidance)
  // 1 = football (medium...)
  // 2 = standard, stay clear of obstacles

  switch((int)avoidanceLevel)
  {
  case 0: widthOfCorridor = 150; break;
  case 1: widthOfCorridor = 200; break;
  case 2: 
  default: widthOfCorridor = 250; break;
  }

  distanceBelowWhichObstaclesAreAvoided = 500;
  minSpeed = -150;

  // to do: make the following some function of 
  // the distance to the destination to make sure
  // the destination is reached!!
  obstacleAvoidanceDistance = distanceToDestination;
  // perform clipping
  obstacleAvoidanceDistance = min(obstacleAvoidanceDistance, distanceBelowWhichObstaclesAreAvoided);

  // derive the forward speed of the robot
  // from the free range in front of the robot
  targetSpeed = freeSpaceInFront = obstaclesModel.getDistanceInCorridor(0.0, widthOfCorridor);
  targetSpeed -= 300;
  // perform clipping
  targetSpeed = max(minSpeed, targetSpeed);
  targetSpeed = min(maxSpeed, targetSpeed);

  // default: head to where there's more free space
  double leftForward = obstaclesModel.getTotalFreeSpaceInSector(.7, 1.39, obstacleAvoidanceDistance);
  double rightForward = obstaclesModel.getTotalFreeSpaceInSector(-.7, 1.39, obstacleAvoidanceDistance);
  targetAngle =	2*(leftForward - rightForward) / (leftForward + rightForward);
  if(fabs(targetAngle) < .5)
    targetAngle = sgn(targetAngle)*.5;

  //if (targetAngle > .8) targetAngle = .8;
  //if (targetAngle > -.8) targetAngle = -.8;

  // if there's space in front, turn towards goal
  // also turn for goal if corridor in the direction of the goal is free
  if (freeSpaceInFront > obstacleAvoidanceDistance)
  {
    // determine the next free sector in the desired direction
    double nextFree = obstaclesModel.getAngleOfNextFreeSector(pi/4, angleToDestination, (int )obstacleAvoidanceDistance);

    // check if there is enough space in the direction of "nextFree"
    if (obstaclesModel.getDistanceInCorridor(nextFree, widthOfCorridor) > obstacleAvoidanceDistance)
    {
  	  if (fabs(nextFree) < .8)
	    {
	      targetAngle = nextFree;
		    mode = 1;
	    }
    } 
  }

  if (obstaclesModel.getDistanceInCorridor(angleToDestination, widthOfCorridor) > obstacleAvoidanceDistance)
  {
    targetAngle = angleToDestination;
    mode = 2;
  }

  //OUTPUT(idText, text, "mode " << mode << ", a: " << targetAngle << ", v: " << targetSpeed);
  
  if(obstaclesModel.getDistanceInCorridor((double)sgn(targetAngle) * pi/2, distanceBelowWhichObstaclesAreAvoided/2) < 100)
  {
	  targetSpeed = maxSpeed;
    targetAngle = 0;  
    //OUTPUT(idText, text, "turning overruled");
  }

  // adjust forward speed:
  // if a lot of turning is necessary, don't walk so fast!
  double factor = .5+(pi-fabs(targetAngle))/pi;
  if (factor > 1) factor = 1;
  if (factor < 0) factor = 0;

  targetSpeed *= factor;

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = speedX.approximateVal(targetSpeed);
  motionRequest.walkParams.translation.y = speedY.approximateVal(0);
  motionRequest.walkParams.rotation = turnSpeed.approximateVal(targetAngle);
  //OUTPUT(idText, text, "turning speed " << turnSpeed.getVal());
}



void ATH2004ERS7BasicBehaviorGoToPoint::execute()
{
  accelerationRestrictor.saveLastWalkParameters();
  
  Vector2<double> destination(x,y);
  const Vector2<double>& self = robotPose.translation;
  
  if (maxSpeed==0) maxSpeed = 350;
  
  double maxTurnSpeed = fromDegrees(40);
  
  double distanceToDestination = Geometry::distanceTo(self,destination);
  
  double angleDifference = normalize(fromDegrees(destinationAngle) - robotPose.rotation);
  
  double angleToDestination = Geometry::angleTo(robotPose,destination);
  
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  //this time does not necessarily include time for turning!:
  double estimatedTimeToReachDestination;
  if (distanceToDestination > 200)
  {
    estimatedTimeToReachDestination = (distanceToDestination+200)/ maxSpeed;
    
    motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
    motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  }
  else
  {
    estimatedTimeToReachDestination = 2*distanceToDestination / maxSpeed;
    if (distanceToDestination > 30)
    {
      motionRequest.walkParams.translation.x = 
        cos(angleToDestination) * maxSpeed*distanceToDestination/200;
      motionRequest.walkParams.translation.y = 
        sin(angleToDestination) * maxSpeed*distanceToDestination/200;
    }
    else
    {
      motionRequest.walkParams.translation.x = 0;
      motionRequest.walkParams.translation.y = 0;
    }
  }
  if (estimatedTimeToReachDestination==0)
  {
    estimatedTimeToReachDestination = 0.001;
  }
  
  if (fabs(toDegrees(angleDifference)) > 20)
  {
    motionRequest.walkParams.rotation = 
      angleDifference / estimatedTimeToReachDestination;
    if (motionRequest.walkParams.rotation > maxTurnSpeed)
    {
      motionRequest.walkParams.rotation = maxTurnSpeed;
    }
    if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    {
      motionRequest.walkParams.rotation = -maxTurnSpeed;
    }
  }
  else
  {
    motionRequest.walkParams.rotation = 2*angleDifference;
  }
  
  
  
  if ((fabs(toDegrees(angleDifference))<10)&&(distanceToDestination<80))
  {
    motionRequest.walkParams.translation.x = 0;
    motionRequest.walkParams.translation.y = 0;
    motionRequest.walkParams.rotation = 0;
  }
  
  accelerationRestrictor.restrictAccelerations(150,150,100);
  
}

void ATH2004ERS7BasicBehaviorDribbleAtBorder::execute()
{
  //Ballposition
  Vector2<double> ball = ballPosition.seen;
  
  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ball);
  // unused double ballX = ballDistance * cos(Geometry::angleTo(robotPose.getPose(),ball));
  double ballY = ballDistance * sin(Geometry::angleTo(robotPose.getPose(),ball));
  
  /* Some requests */
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = 200;
  
  /* setting the angle of the robot depending on its position to the opponents ground line */
  double destinationAngle;
  destinationAngle = (robotPose.translation.x - xPosHalfWayLine) * 90 / (xPosOpponentGroundline - xPosHalfWayLine);
  
  if (robotPose.translation.y < 0)
  {
    /* robot is on the left side of field and uses the left foot for dribbling*/  
    motionRequest.walkParams.translation.y = ballY + 70;
  }
  else
  {
    /* robot is on the left side of field  and uses the right foot for dribbling*/  
    motionRequest.walkParams.translation.y = ballY - 70;
  }
  {
    if (motionRequest.walkParams.translation.y > 100) 
    {
      motionRequest.walkParams.translation.y = 100;
      
      if (robotPose.translation.x > xPosOpponentPenaltyArea)
      {
        motionRequest.walkParams.rotation = - robotPose.getAngle() + fromDegrees(destinationAngle);
      }
      else
      {
        motionRequest.walkParams.rotation = 0;
      }
      
    }
    if (motionRequest.walkParams.translation.y < -100)
    {
      motionRequest.walkParams.translation.y = -100;
      if (robotPose.translation.x > xPosOpponentPenaltyArea)
      {
        motionRequest.walkParams.rotation = - robotPose.getAngle() + fromDegrees(- destinationAngle);
      }
      else
      {
        motionRequest.walkParams.rotation = 0;
      }
    }
    
    //motionRequest.walkParams.translation.y = 0;
    motionRequest.motionType = MotionRequest::walk;
  }
}

void ATH2004ERS7BasicBehaviorDribbleInCenterOfField::execute()
{
  if(!((int)speed))
    speed = 300;
  //Ballposition
  Vector2<double> ball = ballPosition.seen;

  Vector2<double> goalPos(2000,0);
  

  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ball);
  double ballX = ballDistance * cos(Geometry::angleTo(robotPose.getPose(),ball));
  double ballY = ballDistance * sin(Geometry::angleTo(robotPose.getPose(),ball));
  double sideOffset = 76;

  if(!side)
    side = 1;
  
  if((ball.y < -500))
    side = 1;
  
  if((ball.y > 500))
    side = -1;

  if(modi != 2)
	direction = 361;
  
  if(robotPose.translation.x > 1800)
	  speed = 100;
	else
		if(robotPose.translation.x > 1600)
			speed = 200;
			else
				speed = 300;

  double targetX = ballX - 80;
  double targetY = ballY + sideOffset * side;
  
  double scale = 1;
  double threshold = 80;
  
  if(fabs(targetY) > threshold)
    scale = 0;
  else
    scale = (1-fabs(targetY/threshold))*threshold;
  
  
  double targetDistance = sqrt(sqr(targetX)+sqr(targetY));
  
  double unscale = (sqr(targetX)*(sqr(targetDistance)+scale))/(sqr(targetX)+scale)-sqr(targetDistance);
  
  double speedX = speed * (sqr(targetX/(targetDistance+1+unscale)))*sgn(targetX);
  double speedY = speed * (sqr(targetY/(targetDistance+1+scale)))*sgn(targetY);
  
  // ---------rotation calculation-------
  switch((int)modi){
	// modi 1: dribble parallel to x-axis in direction of opponent goal
	case 1: rotationM = - robotPose.getAngle();
			break;
	// modi 2: fix the init-direction of robot as the dribble-direction
	case 2: if(direction == 361) // 361 is the init-value if direction
				direction = robotPose.getAngle();

			rotationM = direction - robotPose.getAngle();
			break;

	// modi 3: dribble in the direction of opponent goal
	case 3: if(sqrt(sqr(targetY)+sqr(targetX)) < 250)
				rotationM = ((goalPos-ball).angle()-robotPose.getAngle());
			else
				if(sqrt(sqr(targetY)+sqr(targetX)) > 350)
					rotationM = - robotPose.getAngle();
				else
					if(!rotationM)
						rotationM = - robotPose.getAngle();
			break;
	//dribble without direction correction
	default: rotationM = 0;
			 break;  // :)
  }//end switch
  // ------------------------------------

  if(fabs(targetY) < 60)
    speedX = speed - speedY;
  

  motionRequest.walkParams.translation.x = speedX;
  motionRequest.walkParams.translation.y = speedY;
  motionRequest.walkParams.rotation = rotationM;
  
  motionRequest.motionType = MotionRequest::walk;
}//end DribbleInCenterOfField


 /*
 * Change Log
 * 
 * $Log: ATH2004ERS7SimpleBasicBehaviors.cpp,v $
 * Revision 1.18  2004/05/04 10:48:58  loetzsch
 * replaced all enums
 * xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
 * by
 * behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
 * (this mechanism was neither fully implemented nor used)
 *
 * Revision 1.17  2004/04/05 17:56:46  loetzsch
 * merged the local German Open CVS of the aibo team humboldt with the tamara CVS
 *
 * Revision 1.4  2004/04/03 22:28:27  hoffmann
 * improved goalie positioning simple behavior
 *
 * Revision 1.3  2004/04/03 19:41:40  hoffmann
 * added parameter to turn to point
 *
 * Revision 1.2  2004/04/02 23:12:14  hoffmann
 * added goalie-position basic behavior
 *
 * Revision 1.1.1.1  2004/03/31 11:16:39  loetzsch
 * created ATH repository for german open 2004
 *
 * Revision 1.16  2004/03/29 19:13:35  jhoffman
 * turn for ball turns differently as a function of distance
 *
 * Revision 1.15  2004/03/29 18:09:16  jhoffman
 * turn for ball turns slightly slower
 *
 * Revision 1.14  2004/03/29 13:15:19  jhoffman
 * - new go-to-ball
 * - old go-to-ball renamed to go-to-ball-old
 * - new turn around point
 *
 * Revision 1.13  2004/03/28 14:08:05  jhoffman
 * added go-to-ball-test
 *
 * Revision 1.12  2004/03/27 21:09:26  loetzsch
 * improved the goalie
 *
 * Revision 1.11  2004/03/26 15:24:20  loetzsch
 * improved border behaviors
 *
 * Revision 1.10  2004/03/25 09:25:01  jhoffman
 * parameter adjustments to obstacle avoidance
 *
 * Revision 1.9  2004/03/24 18:41:29  juengel
 * Slowed down rotation and side speed of goToBall.
 *
 * Revision 1.8  2004/03/23 16:14:20  juengel
 * Removed OUTPUTS.
 *
 * Revision 1.7  2004/03/23 14:41:56  juengel
 * Added dribble behaviors.
 *
 * Revision 1.6  2004/03/21 19:12:43  roefer
 * Warnings removed
 *
 * Revision 1.5  2004/03/19 19:01:06  jhoffman
 * added obstacle avoidance parameter "avoidance level" (0 - min, 1 - mid, 2 - max avoidance)
 *
 * Revision 1.4  2004/03/18 13:33:06  loetzsch
 * removed warnings
 *
 * Revision 1.3  2004/03/17 09:32:39  jhoffman
 * small changes to go-to-point-and-avoid-obstacles
 *
 * Revision 1.2  2004/03/16 14:18:18  loetzsch
 * tuned parameters of turn-around-point
 *
 * Revision 1.1  2004/03/16 14:00:18  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.6  2004/03/16 09:02:13  juengel
 * bug fixed
 *
 * Revision 1.5  2004/03/16 08:32:41  loetzsch
 * increased speed
 * improved turn-around-point
 *
 *
*/

