/**
* @file ATH2004ERS7BehaviorControl.cpp
* 
* Implementation of class ATH2004ERS7BehaviorControl.
*
* @author Martin Ltzsch
*/

#include "ATH2004ERS7BehaviorControl.h"
#include "Tools/Player.h"

ATH2004ERS7BehaviorControl::ATH2004ERS7BehaviorControl(BehaviorControlInterfaces& interfaces)
: Xabsl2BehaviorControl(interfaces,SolutionRequest::ath2004ers7),
calibrationSymbols(interfaces),
ballSymbols(interfaces),
configurationSymbols(interfaces),
headAndTailSymbols(interfaces),
joystickSymbols(interfaces),
kickSelectionSymbols(interfaces),
ledAndSoundSymbols(interfaces),
mathFunctions(interfaces),
motionRequestSymbols(interfaces),
roboCupGameManagerSymbols(interfaces),
robotPoseSymbols(interfaces),
robotStateSymbols(interfaces),
specialVisionSymbols(interfaces),
strategySymbols(interfaces),
challengeSymbols(interfaces),
simpleBasicBehaviors(interfaces,errorHandler),
continuousBasicBehaviors(interfaces,errorHandler),
commonBasicBehaviors(interfaces,errorHandler),
shootLogger(interfaces),
profiler()
{
  Xabsl2FileInputSource file("Xabsl2/ath7-ic.dat");
  init(file);

  if (!errorHandler.errorsOccurred)
  {
    profiler.init(*pEngine);
  }
}

ATH2004ERS7BehaviorControl::~ATH2004ERS7BehaviorControl()
{
}

void ATH2004ERS7BehaviorControl::registerSymbolsAndBasicBehaviors()
{
  simpleBasicBehaviors.registerBasicBehaviors(*pEngine);
  continuousBasicBehaviors.registerBasicBehaviors(*pEngine);
  commonBasicBehaviors.registerBasicBehaviors(*pEngine);
  calibrationSymbols.registerSymbols(*pEngine);
  ballSymbols.registerSymbols(*pEngine);
  configurationSymbols.registerSymbols(*pEngine);
  headAndTailSymbols.registerSymbols(*pEngine);
  joystickSymbols.registerSymbols(*pEngine);
  ledAndSoundSymbols.registerSymbols(*pEngine);
  kickSelectionSymbols.registerSymbols(*pEngine);
  mathFunctions.registerSymbols(*pEngine);
  motionRequestSymbols.registerSymbols(*pEngine);
  roboCupGameManagerSymbols.registerSymbols(*pEngine);
  robotPoseSymbols.registerSymbols(*pEngine);
  robotStateSymbols.registerSymbols(*pEngine);
  specialVisionSymbols.registerSymbols(*pEngine);
  strategySymbols.registerSymbols(*pEngine);
  challengeSymbols.registerSymbols(*pEngine);
}

void ATH2004ERS7BehaviorControl::execute()
{
  for(int i = 0; i < 14; i++) ledRequest.faceLED[i] = LEDRequest::oooo;
  ledRequest.backFrontWhiteLED = LEDRequest::oooo;
  ledRequest.backMiddleWhiteLED = LEDRequest::oooo;
  ledRequest.backRearWhiteLED = LEDRequest::oooo;
  ledRequest.backMiddleOrangeLED = LEDRequest::oooo;

  ledRequest.backFrontBlueLED = 
    getPlayer().getTeamColor() == Player::blue ? LEDRequest::llll : LEDRequest::oooo;
  ledRequest.backRearRedLED = 
    getPlayer().getTeamColor() == Player::red ? LEDRequest::llll : LEDRequest::oooo;

  ledRequest.headWhiteLED = LEDRequest::oooo;
  ledRequest.headOrangeLED = LEDRequest::oooo;

  ballSymbols.update();
  configurationSymbols.update();
  roboCupGameManagerSymbols.update();
  robotPoseSymbols.update();
  strategySymbols.update();
  
  continuousBasicBehaviors.update();
  
  // set the outgoint behavior team message to none
  outgoingBehaviorTeamMessage.message = BehaviorTeamMessage::none;
  
  // set the lilac face leds depending on the dynamic role.
  switch (strategySymbols.role)
  {
  case BehaviorTeamMessage::striker:
    ledRequest.faceLED[13] = LEDRequest::llll;
    break;
  case BehaviorTeamMessage::offensiveSupporter:
    ledRequest.faceLED[13] = LEDRequest::olol;
    break;
  case BehaviorTeamMessage::defensiveSupporter:
    ledRequest.faceLED[13] = LEDRequest::ooll;
    break;
  default:
    ledRequest.faceLED[13] = LEDRequest::oooo;
  }

  // visualize best-angle-to-opponent-goal(-no-obstacles) with the top white leds.
  double bestAngle;
  if(getPlayer().getPlayerNumber() == Player::one)
  {
    bestAngle = robotPoseSymbols.getGoalieGoalKickAngle();
    if (ballPosition.ballState == BallModel::ballRollsByLeft)
      ledRequest.faceLED[1] = LEDRequest::llll;	
    if (ballPosition.ballState == BallModel::ballRollsByRight)
      ledRequest.faceLED[0] = LEDRequest::llll;	
    if (ballPosition.ballState == BallModel::ballRollsFast)
    {
      ledRequest.faceLED[2] = LEDRequest::llll;	
      ledRequest.faceLED[3] = LEDRequest::llll;	
    }
  }
  else
    bestAngle = robotPoseSymbols.getBestAngleToOpponentGoal();

  if     (bestAngle >  75) ledRequest.faceLED[ 7] = LEDRequest::lolo; //?_____   
  else if(bestAngle >  45) ledRequest.faceLED[ 7] = LEDRequest::llll; // x____
  else if(bestAngle >  15) ledRequest.faceLED[ 9] = LEDRequest::llll; // _x___      
  else if(bestAngle > -15) ledRequest.faceLED[10] = LEDRequest::llll; // __x__
  else if(bestAngle > -45) ledRequest.faceLED[ 8] = LEDRequest::llll; // ___x_
  else if(bestAngle > -75) ledRequest.faceLED[ 6] = LEDRequest::llll; // ____x
  else                     ledRequest.faceLED[ 6] = LEDRequest::lolo; // _____?
  // Set the tail request depending on whether the ball was seen
  if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 80)
  {
    // ball seen now
    ledRequest.headOrangeLED = LEDRequest::llll;
    motionRequest.tailRequest = MotionRequest::noTailWag;
  }
  else if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 2000)
  {
    // ball seen
    ledRequest.headWhiteLED = LEDRequest::llll;
    motionRequest.tailRequest = MotionRequest::noTailWag;
  }
  else if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) 
    > BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
    && SystemCall::getTimeSince(ballPosition.communicated.timeWhenLastObserved) < 3000)
  {
    // ball known
    motionRequest.tailRequest = MotionRequest::wagHorizontalFast;
  }
  else
  {
    // ball not known
    motionRequest.tailRequest = MotionRequest::noTailWag;
  }

  if (strategySymbols.robotIsStuck)
  {
    ledRequest.faceLED[4] = LEDRequest::llll;
    ledRequest.faceLED[5] = LEDRequest::llll;
  }
  else if (strategySymbols.obstaclesAreClose)
  {
    ledRequest.faceLED[11] = LEDRequest::llll;
  }
  
  // set head control mode lookToStars to detect missing settings
  headControlMode.headControlMode = HeadControlMode::lookToStars;

  soundRequest.soundID = SoundRequest::none;
    
  // execute the engine
  ////////////////////////
  executeEngine();
  ////////////////////////

  if((robotState.getState() == RobotState::crashed)||
     (robotState.getState() == RobotState::rollLeft)||
     (robotState.getState() == RobotState::rollRight))
  {
    if ((motionRequest.motionType == MotionRequest::walk &&
      motionRequest.walkType != MotionRequest::upsideDown)||
      (motionRequest.motionType == MotionRequest::stand))
    {
      motionRequest.motionType = MotionRequest::getup;
      return;
    }
  }

  // do profiling
  if(strategySymbols.doProfile == ATH2004ERS7StrategySymbols::doProfiling)
    profiler.doProfiling(*pEngine, robotPose.frameNumber);
  if(strategySymbols.writeProfile == ATH2004ERS7StrategySymbols::writeProfiles){
    profiler.recordCollectedLogs();
    strategySymbols.writeProfile = ATH2004ERS7StrategySymbols::dontWriteProfiles;
  }
  else if(strategySymbols.writeProfile == ATH2004ERS7StrategySymbols::writeCompleteProfiles){
    profiler.recordCompleteLog();
    strategySymbols.writeProfile = ATH2004ERS7StrategySymbols::dontWriteProfiles;
  }

    

  // if the head control mode was not set, play a warning sound
  if (headControlMode.headControlMode == HeadControlMode::lookToStars)
  {
    soundRequest.soundID = SoundRequest::rob101;
  }

  if(sensorDataBuffer.frame[0].data[SensorData::chin] == 1)
    soundRequest.soundID = SoundRequest::bark2;

  // execute the shoot logger, but not during a game
  // shootLogger.execute();
}

bool ATH2004ERS7BehaviorControl::handleMessage(InMessage& message)
{
  return 
    (Xabsl2BehaviorControl::handleMessage(message) ||
    kickSelectionSymbols.handleMessage(message)
    );
}


/*
* Change log :
* 
* $Log: ATH2004ERS7BehaviorControl.cpp,v $
* Revision 1.27  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
* Revision 1.26  2004/05/04 17:39:41  loetzsch
* bug fix
*
* Revision 1.25  2004/05/04 16:25:18  juengel
* sound output improved ???
*
* Revision 1.24  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.23  2004/04/30 09:26:44  spranger
* redone profiler-interface
*
* Revision 1.22  2004/04/22 11:34:52  spranger
* added more logging mechanisms
*
* Revision 1.21  2004/04/20 18:17:59  spranger
* added profiler
*
* Revision 1.20  2004/04/13 12:01:41  spranger
* added rudimentary profiling
*
* Revision 1.19  2004/03/29 15:03:28  juengel
* robotIsStuck and obstaclesAreClose is shown with face LEDs
*
* Revision 1.18  2004/03/28 18:05:31  juengel
* Best angle to opponent goal is visualized.
*
* Revision 1.17  2004/03/28 17:53:31  juengel
* Ball rolls fast is shown.
*
* Revision 1.16  2004/03/28 16:04:56  juengel
* bug fixed
*
* Revision 1.15  2004/03/28 15:53:47  jhoffman
* goalie displays left or right ball with its leds
*
* Revision 1.14  2004/03/27 21:40:22  juengel
* goalie-goal-kick-angle is visualized
*
* Revision 1.13  2004/03/26 15:31:13  juengel
* Changed meaning of head.
*
* Revision 1.12  2004/03/26 13:56:10  juengel
* Top white face leds indicate best angle to opponent goal (new version).
*
* Revision 1.11  2004/03/26 07:04:34  juengel
* Top white face leds indicate best angle to opponent goal.
*
* Revision 1.10  2004/03/26 06:42:25  loetzsch
* several improvements in the high level options
*
* Revision 1.9  2004/03/25 21:21:03  juengel
* Head LEDs show ballSeen etc.
*
* Revision 1.8  2004/03/25 17:40:15  loetzsch
* adaptations to changes in the game controller and the led request
*
* Revision 1.7  2004/03/23 17:03:21  loetzsch
* shoot logger is not invoked during a test game
*
* Revision 1.6  2004/03/21 19:17:40  juengel
* - barking when chin pressed
* - back*WhiteLeds are initialized
* - removed "obstacle sounds"
*
* Revision 1.5  2004/03/21 12:42:20  juengel
* - Moved calibration symbols to common symbols.
* - face LEDs are initialized with oooo
* - lilac face LEDs show player role
*
* Revision 1.4  2004/03/20 00:20:18  kerdels
* removed some errors
*
* Revision 1.2  2004/03/17 19:57:40  juengel
* handleMessage() for kickSelectionSymbols added.
*
* Revision 1.1  2004/03/16 14:00:17  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.2  2004/03/09 13:44:49  loetzsch
* added class ATH2004ERS7KickSelectionSymbols
*
* Revision 1.1  2004/03/06 12:52:11  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
* created local cvs for Gnne
*
* Revision 1.3  2004/02/03 13:20:47  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.2  2004/01/28 21:16:56  loetzsch
* added ATH2004ERS7ShootLogger
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004ERS7BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004ERS7 three times (BB2004, DDD2004, MSH2004)
*
*/

