/**
* @file ATH2004ERS7ShootLogger.h
* 
* Definition of class ATH2004ERS7ShootLogger.
*
* @author Martin Ltzsch
*/

#ifndef __ATH2004ERS7ShootLogger_h_
#define __ATH2004ERS7ShootLogger_h_

#include "../BehaviorControl.h"

/**
* @class ATH2004ERS7ShootLogger
*
* Runs in parallel to the ATH Xabsl engine and logs the preconditions
* and results of each kick.
*
* @author Martin Ltzsch
*/ 
class ATH2004ERS7ShootLogger : public BehaviorControlInterfaces
{
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004ERS7ShootLogger(BehaviorControlInterfaces& interfaces);

  /** destructor */
  ~ATH2004ERS7ShootLogger();

  /** Executes the logger */
  void execute();

private:
  /** Different states of the shoot logger */
  enum State { initial, kickRequested, kickStarted, kickFinished, 
    waitingForBall, wait };

  /** The state of the logger */
  State state;

  /** Changes the state of the logger */
  void changeState(State newState);

  /** The time of the last state change */
  unsigned long timeOfLastStateChange;

  /** 
  * Writes a collected record either to "kicks.log" or, if requested by 
  * a debug key, to the debug message queue.
  */
  void record();

  /** Writes a recorded log entry to a stream */
  void writeRecordToStream(Out& out);

  /** The relative x position of the ball when the kick was requested */
  double ballXR;

  /** The relative y position of the ball when the kick was requested */
  double ballYR;

  /** The time since the ball was seen last when the kick was requested */
  unsigned long ballTR;

  /** The requested kick */
  MotionRequest::SpecialActionID kick;

  /** The relative x position of the ball when the kick was started */
  double ballXS;

  /** The relative y position of the ball when the kick was started */
  double ballYS;

  /** The time since the ball was seen last when the kick was started */
  unsigned long ballTS;

  /** The odometry data when the kick was started */
  OdometryData odometryDataWhenKickWasStarted;

  /** The relative x position of the ball after the kick */
  double ballXA;

  /** The relative y position of the ball after the kick */
  double ballYA;

  /** The time since the kick was finished when the record was written */
  unsigned long tA;

  /** Returns the relative x position of a ball position */
  double getBallX(Pose2D from);

  /** Returns the relative y position of a ball position */
  double getBallY(Pose2D from);
};

#endif// __ATH2004ERS7ShootLogger_h_

/*
 * Change log :
 * 
 * $Log: ATH2004ERS7ShootLogger.h,v $
 * Revision 1.3  2004/03/20 00:20:18  kerdels
 * removed some errors
 *
 * Revision 1.1  2004/03/16 14:00:17  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.1  2004/03/06 12:52:11  loetzsch
 * cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
 *
 * Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
 * created local cvs for Gnne
 *
 * Revision 1.4  2004/02/04 14:09:56  loetzsch
 * improved
 *
 * Revision 1.3  2004/02/03 23:16:46  loetzsch
 * added debug key sendShootRecords
 *
 * Revision 1.2  2004/01/30 13:25:28  loetzsch
 * using odometry data in shoot logger,
 * first draft of visualization
 *
 * Revision 1.1  2004/01/28 21:16:56  loetzsch
 * added ATH2004ShootLogger
 *
 */
