/** 
* @file BB2004SimpleBasicBehaviors.cpp
*
* Implementation of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Tim Laue
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#include "BB2004SimpleBasicBehaviors.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"
#include "Tools/Location.h"
#include "Tools/Debugging/Debugging.h"
#include <cassert>


void BB2004SimpleBasicBehaviors::registerBasicBehaviors(Xabsl2Engine& engine)
{
  engine.registerBasicBehavior(basicBehaviorGoToBallPotentialField);
  engine.registerBasicBehavior(basicBehaviorGoToPosePotentialField);
  engine.registerBasicBehavior(basicBehaviorRun);
  engine.registerBasicBehavior(basicBehaviorWalkOmni);
  engine.registerBasicBehavior(basicBehaviorRandomWalk);
  engine.registerBasicBehavior(basicBehaviorDribble);
  engine.registerBasicBehavior(basicBehaviorKick);
}


bool BB2004BasicBehaviorTools::ballAtRobot()
{
  return (sensorDataBuffer.lastFrame().data[SensorData::bodyPsd]>110000);
}


void BB2004BasicBehaviorTools::approachCloseBall()
{
  Vector2<double> relVec(Geometry::vectorTo(robotPose, ballPosition.seen));
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.rotation = atan2(relVec.y, relVec.x);
  if(SystemCall::getTimeSince(headModeChanged) > 600)
  {
    if(searchBall)
    {
      searchBall = false;
      headControlMode.headControlMode = HeadControlMode::searchAuto;
    }
    else
    {
      searchBall = true;
      headControlMode.headControlMode = HeadControlMode::searchForBall;
    }
  }
  if(SystemCall::getTimeSince(lastTimeApproached) > 1000)
  {
    headControlMode.headControlMode = HeadControlMode::searchForBall;
    headModeChanged = SystemCall::getCurrentSystemTime();
    searchBall = true;
  }
  if(fabs(motionRequest.walkParams.rotation) > 0.15)
  {
    if(Geometry::distanceTo(robotPose, ballPosition.seen) > 200)
    {
      motionRequest.walkParams.translation = relVec;
      motionRequest.walkParams.translation.normalize(); 
      motionRequest.walkParams.translation *= 120;
    }
    else
    {
      motionRequest.walkParams.translation.x = 0.0;
      motionRequest.walkParams.translation.y = 0.0;
    }
  /*  if(toDegrees(fabs(motionRequest.walkParams.rotation)) < 50)
    {
      motionRequest.walkParams.rotation *= 2.0;
    }*/
  }
  else
  {
    motionRequest.walkParams.translation = relVec;
    motionRequest.walkParams.translation.normalize(); 
    motionRequest.walkParams.translation *= 140;
  }
  lastTimeApproached = SystemCall::getCurrentSystemTime();
}


double BB2004BasicBehaviorTools::rotationToOpponentGoal()
{
/*  if(obstaclesModel.angleToFreePartOfGoalWasDetermined[ObstaclesModel::opponentGoal])
  {
    return obstaclesModel.angleToFreePartOfGoal[ObstaclesModel::opponentGoal];
  }
  else
  {*/
    return (Geometry::angleTo(robotPose, Vector2<double>(xPosOpponentGroundline,0)));
  //}
}


void BB2004BasicBehaviorGoToBallPotentialField::execute()
{
  PotentialfieldResult result;
  potentialfields.execute(robotPose, ballPosition, playerPoseCollection, 
                         obstaclesModel, teamMessageCollection, result);

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = 365 * result.motion.pos.x;
  motionRequest.walkParams.translation.y = 250 * result.motion.pos.y;
  motionRequest.walkParams.rotation = result.motion.rotation;
}


BB2004BasicBehaviorGoToPosePotentialField::BB2004BasicBehaviorGoToPosePotentialField
                                            (const BehaviorControlInterfaces& interfaces,
                                            Xabsl2ErrorHandler& errorHandler):
                                            Xabsl2BasicBehavior("go-to-pose-potential-field", errorHandler),
                                            BehaviorControlInterfaces(interfaces),
                                            potentialfields("Bremen/gotopose.pfc"),
                                            forwardSpeed(150)
{
  registerParameter("go-to-pose-potential-field.x", x);
  registerParameter("go-to-pose-potential-field.y", y);
  registerParameter("go-to-pose-potential-field.destination-angle", destinationAngle);
  Location currentLocation(getLocation());
  InTextFile parametersFile(currentLocation.getFilename("runparam.cfg"));
  /*if (parametersFile.exists())
  {
    parametersFile >> forwardSpeed;
  }*/
}


void BB2004BasicBehaviorGoToPosePotentialField::execute()
{
  PotentialfieldResult result;
  Pose2D destination;
  destination.translation.x = x;
  destination.translation.y = y;
  destination.rotation = fromDegrees(destinationAngle);
  potentialfields.execute(robotPose, ballPosition, playerPoseCollection, 
                         obstaclesModel, teamMessageCollection, destination, true, result);

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  Vector2<double> relVec(-robotPose.translation + destination.translation);
  double distanceToPoint(relVec.abs());
  if(result.action == "do-turn")
  {
    double rotationDifference = destination.rotation - robotPose.rotation;
    while(rotationDifference>pi) rotationDifference -= 2*pi;
    while(rotationDifference<-pi) rotationDifference += 2*pi;
    if(fabs(toDegrees(rotationDifference))<10)
    {
      if(distanceToPoint < 50.0)
      {
        motionRequest.motionType = MotionRequest::stand;
      }
      else
      {
        double ca(cos(-robotPose.rotation));
        double sa(sin(-robotPose.rotation));
        double newX(relVec.x*ca - relVec.y*sa);
        relVec.y = relVec.y*ca + relVec.x*sa;
        relVec.x = newX;
        motionRequest.walkParams.translation = relVec;
        motionRequest.walkParams.translation.normalize();
        motionRequest.walkParams.translation *= 60.0;
        motionRequest.walkParams.rotation = 0;
      }
    }
    else
    {
      motionRequest.walkParams.translation.x = 0;
      motionRequest.walkParams.translation.y = 0;
      motionRequest.walkParams.rotation = rotationDifference;
      if(fabs(toDegrees(rotationDifference)) < 30.0)
      {
        motionRequest.walkParams.rotation /= fabs(rotationDifference);
        motionRequest.walkParams.rotation *= fromDegrees(30.0);
      }
    }
  }
  else
  {
    motionRequest.walkParams.translation.x = forwardSpeed * result.motion.pos.x;
    motionRequest.walkParams.translation.y = forwardSpeed * result.motion.pos.y;
    motionRequest.walkParams.rotation = result.motion.rotation;
    if(distanceToPoint < 150)
    {
      motionRequest.walkParams.translation.x /= 3.0;
      motionRequest.walkParams.translation.y /= 3.0; 
    }
    if(fabs(toDegrees(motionRequest.walkParams.rotation)) > 60)
    {
      motionRequest.walkParams.translation.x = 0;
      motionRequest.walkParams.translation.y = 0; 
    }
  }
}


BB2004BasicBehaviorRandomWalk::BB2004BasicBehaviorRandomWalk
                                            (const BehaviorControlInterfaces& interfaces,
                                            Xabsl2ErrorHandler& errorHandler):
                                            Xabsl2BasicBehavior("random-walk", errorHandler),
                                            BehaviorControlInterfaces(interfaces),
                                            potentialfields("Bremen/randwalk.pfc")
{}


void BB2004BasicBehaviorRandomWalk::execute()
{
  PotentialfieldResult result;
  potentialfields.execute(robotPose, ballPosition, playerPoseCollection, 
                          obstaclesModel, teamMessageCollection, result);
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = 320 * result.motion.pos.x;
  if(motionRequest.walkParams.translation.x < 0.0)
  {
    motionRequest.walkParams.translation.x *= -1.0;
  }
  motionRequest.walkParams.translation.y = 320 * result.motion.pos.y;
  motionRequest.walkParams.rotation = result.motion.rotation;
  motionRequest.walkParams.rotation /= 4.0;
  headControlMode.headControlMode = HeadControlMode::lookStraightAhead;
}


BB2004BasicBehaviorRun::BB2004BasicBehaviorRun(const BehaviorControlInterfaces& interfaces,
                                               Xabsl2ErrorHandler& errorHandler)
                                             : Xabsl2BasicBehavior("run", errorHandler),
                                               BehaviorControlInterfaces(interfaces),
                                               forwardSpeed(280), runState(turnState)
{
  registerParameter("run.speed-x", speedX);
  registerParameter("run.speed-y", speedY);
  registerParameter("run.speed-max", maxSpeed);
  registerParameter("run.rotation-speed", rotationSpeed);
  registerParameter("run.use-rotation", useRotation);
  Location currentLocation(getLocation());
  InTextFile parametersFile(currentLocation.getFilename("runparam.cfg"));
  if (parametersFile.exists())
  {
    parametersFile >> forwardSpeed;
  }
}
  

void BB2004BasicBehaviorRun::execute()
{
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  if(runState == turnState)
  {
    motionRequest.walkParams.translation.x = 0.0;
    motionRequest.walkParams.translation.y = 0.0;
    if(rotationSpeed != 0.0)
    {
      motionRequest.walkParams.rotation = rotationSpeed / fabs(rotationSpeed);
      if(fabs(rotationSpeed) > 140)
      {
        motionRequest.walkParams.rotation *= pi;
      }
      else if(fabs(rotationSpeed) < 30)
      {
        motionRequest.walkParams.rotation *= fromDegrees(30);
      }
      else
      {
        motionRequest.walkParams.rotation *= fabs(fromDegrees(rotationSpeed));
      }
    }
    if(fabs(rotationSpeed) < 15)
    {
      runState = dashState;
    }
  }
  else // runState == dashState
  {
    motionRequest.walkParams.translation.x = forwardSpeed;
    motionRequest.walkParams.translation.y = 0.0;//speedY*forwardSpeed;
    motionRequest.walkParams.rotation = fromDegrees(rotationSpeed);
    if(fabs(motionRequest.walkParams.rotation) > 0.5)
    {
      if(fabs(motionRequest.walkParams.rotation) < 0.9)
      {
        motionRequest.walkParams.rotation /= fabs(motionRequest.walkParams.rotation);
        motionRequest.walkParams.rotation *= 0.5;
      }
      else
      {
        runState = turnState;
      }
    }
    else if(fabs(motionRequest.walkParams.rotation) > 0.2)
    {
      motionRequest.walkParams.rotation /= fabs(motionRequest.walkParams.rotation);
      motionRequest.walkParams.rotation *= 0.5;
    }
  }
}


BB2004BasicBehaviorWalkOmni::BB2004BasicBehaviorWalkOmni(const BehaviorControlInterfaces& interfaces,
                                                         Xabsl2ErrorHandler& errorHandler)
                                                       : Xabsl2BasicBehavior("walk-omni", errorHandler),
                                                         BehaviorControlInterfaces(interfaces),
                                                         forwardSpeed(250)
{
  registerParameter("walk-omni.speed-x", speedX);
  registerParameter("walk-omni.speed-y", speedY);
  registerParameter("walk-omni.speed-max", maxSpeed);
  registerParameter("walk-omni.rotation-speed", rotationSpeed);
  registerParameter("walk-omni.use-rotation", useRotation);
  maxSpeed = 1000;
  /*Location currentLocation(getLocation());
  InTextFile parametersFile(currentLocation.getFilename("runparam.cfg"));
  if (parametersFile.exists())
  {
    parametersFile >> forwardSpeed;
  }*/
}
  

void BB2004BasicBehaviorWalkOmni::execute()
{
  double currentSpeed(maxSpeed < forwardSpeed ? maxSpeed : forwardSpeed);
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = speedX*currentSpeed;
  motionRequest.walkParams.translation.y = speedY*currentSpeed;
  motionRequest.walkParams.rotation = fromDegrees(rotationSpeed);
}


void BB2004BasicBehaviorDribble::execute()
{
	if(tools.ballAtRobot())
  {
    headControlMode.headControlMode = HeadControlMode::BBautoCatchBall;
  }
  else
  {
    headControlMode.headControlMode = HeadControlMode::searchForBall;
  }
  if(SystemCall::getTimeSince(timeStamp) > 300)
  {
    dribbleState = walk;
    timeOfStateChange = SystemCall::getCurrentSystemTime();
  }
  switch(dribbleState)
  {
  case walk:
    {
      soundRequest.soundID = SoundRequest::bark2;
      tools.approachCloseBall();   
      if(tools.ballAtRobot())
      {
        dribbleState = hold;
        lastState = walk;
        timeOfStateChange = SystemCall::getCurrentSystemTime();
      }
      break;
    }
  case hold:
    {
      /*motionRequest.motionType = MotionRequest::specialAction;*/
      motionRequest.motionType = MotionRequest::stand;
      if(!tools.ballAtRobot())
      {
        dribbleState = walk;
        lastState = hold;
        timeOfStateChange = SystemCall::getCurrentSystemTime();
      }
      else if(SystemCall::getTimeSince(timeOfStateChange) > 400)
      {
        dribbleState = moveBall;
        lastState = hold;
        timeOfStateChange = SystemCall::getCurrentSystemTime();
      }
      break;
    }
  case moveBall:
    {
      doDribbling();
    }
  }
  timeStamp = SystemCall::getCurrentSystemTime();
}


void BB2004BasicBehaviorDribble::doDribbling()
{
  double rot(tools.rotationToOpponentGoal());
  double rotBall(Geometry::angleTo(robotPose, ballPosition.seen));
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  if(SystemCall::getTimeSince(moveTimeStamp) > 1000)
  {
    moveState = turn;
    moveTimeStamp = SystemCall::getCurrentSystemTime();
    timeOfMoveStateChange = SystemCall::getCurrentSystemTime();
  }
  switch(moveState)
  {
  case turn:
    {
      if(toDegrees(fabs(rot)) < 15.0 || SystemCall::getTimeSince(timeOfMoveStateChange) > 1300) 
      {
        moveState = dash;
        timeOfMoveStateChange = SystemCall::getCurrentSystemTime();
      }
      else
      {
        motionRequest.walkParams.translation.x = 0.0;
        motionRequest.walkParams.translation.y = 0.0;
        motionRequest.walkParams.rotation = rot;
        if(toDegrees(fabs(rot)) < 40)
        {
          motionRequest.walkParams.rotation /= fabs(rot);
          motionRequest.walkParams.rotation *= fromDegrees(40);
        }
        if(toDegrees(fabs(rot)) > 160)
        {
          motionRequest.walkParams.rotation /= fabs(rot);
          motionRequest.walkParams.rotation *= fromDegrees(160);
        }
        break;
      }
    }
  case dash:
    {
      motionRequest.walkParams.translation.x = 350;
      motionRequest.walkParams.translation.y = 0;
      motionRequest.walkParams.rotation = rotBall;
      if(SystemCall::getTimeSince(timeOfMoveStateChange) < 500)
      {
        headControlMode.headControlMode = HeadControlMode::searchForBall;
      }
      else
      {
        headControlMode.headControlMode = HeadControlMode::searchAuto;
      }
      if(toDegrees(fabs(rot)) > 30 && SystemCall::getTimeSince(timeOfMoveStateChange) > 1000)
      {
        moveState = turn;
        timeOfMoveStateChange = SystemCall::getCurrentSystemTime();
      }
    }
  }
}


void BB2004BasicBehaviorKick::execute()
{
  if(tools.ballAtRobot())
  {
    headControlMode.headControlMode = HeadControlMode::BBautoCatchBall;
  }
  else
  {
    headControlMode.headControlMode = HeadControlMode::searchForBall;
  }
  if(SystemCall::getTimeSince(timeStamp) > 300)
  {
    kickState = walk;
    timeOfStateChange = SystemCall::getCurrentSystemTime();
  }
  switch(kickState)
  {
  case walk:
    {
      soundRequest.soundID = SoundRequest::bark1;
      tools.approachCloseBall();   
      if(tools.ballAtRobot())
      {
        kickState = hold;
        lastState = walk;
        timeOfStateChange = SystemCall::getCurrentSystemTime();
      }
      break;
    }
  case hold:
    {
      /*motionRequest.motionType = MotionRequest::specialAction;*/
      motionRequest.motionType = MotionRequest::stand;
      if(!tools.ballAtRobot())
      {
        kickState = walk;
        lastState = hold;
        timeOfStateChange = SystemCall::getCurrentSystemTime();
      }
      else if(SystemCall::getTimeSince(timeOfStateChange) > 400)
      {
        kickState = kick;
        lastState = hold;
        timeOfStateChange = SystemCall::getCurrentSystemTime();
      }
      break;
    }
  case kick:
    {
      doKick();
    }
  }
  timeStamp = SystemCall::getCurrentSystemTime();
}


void BB2004BasicBehaviorKick::doKick()
{
  if(kickID >= 0.0 && kickID <= 2.0)
  {
    motionRequest.motionType = MotionRequest::specialAction;
    if(kickID == 0.0)
      motionRequest.specialActionType = MotionRequest::BB2004ChestFallKick;
    else if(kickID == 1.0)
      motionRequest.specialActionType = MotionRequest::BB2004ArmKickLeft;
    else if(kickID == 2.0)
      motionRequest.specialActionType = MotionRequest::BB2004ArmKickRight;
    //if(SystemCall::getTimeSince(timeOfStateChange) > 500) 
    if(executedMotionRequest.specialActionType == MotionRequest::BB2004ChestFallKick ||
       executedMotionRequest.specialActionType == MotionRequest::BB2004ArmKickLeft ||
       executedMotionRequest.specialActionType == MotionRequest::BB2004ArmKickRight)
    {
      kickState = walk;
      lastState = kick;
      timeOfStateChange = SystemCall::getCurrentSystemTime();
    }
  }
  else
  {
    assert(false);
  }
}


/*
* Change Log
* 
* $Log: BB2004SimpleBasicBehaviors.cpp,v $
* Revision 1.15  2004/05/21 14:43:34  tim
* changed parameters for random walk
*
* Revision 1.14  2004/04/09 14:06:44  tim
* integrated changes from GO2004
*
* Revision 1.29  2004/04/03 05:21:08  tim
* improved behavior
*
* Revision 1.28  2004/04/02 21:11:18  tim
* changed parameters
*
* Revision 1.27  2004/04/02 19:18:57  tim
* changed parameters
*
* Revision 1.26  2004/04/02 01:11:02  tim
* Changed behavior
*
* Revision 1.25  2004/04/01 23:36:27  tim
* Changed behavior
*
* Revision 1.24  2004/04/01 22:16:11  tim
* Changed behavior
*
* Revision 1.23  2004/04/01 16:43:41  tim
* Changed parameters
*
* Revision 1.22  2004/04/01 16:05:43  tim
* Changed go to pose parameters
*
* Revision 1.21  2004/04/01 13:15:13  tim
* "improved" behavior
*
* Revision 1.20  2004/03/31 21:55:53  tim
* changed behavior
*
* Revision 1.19  2004/03/31 21:22:35  tim
* changed behavior
*
* Revision 1.18  2004/03/31 20:13:06  tim
* changed behavior
*
* Revision 1.17  2004/03/30 22:45:13  tim
* changed behavior
*
* Revision 1.16  2004/03/30 21:00:05  fritsche
* very simple dribble
*
* Revision 1.15  2004/03/30 14:50:55  tim
* "improved" behavior ;-)
*
* Revision 1.14  2004/03/29 19:55:52  tim
* changed parameters
*
* Revision 1.13  2004/03/29 11:52:39  tim
* several small changes
*
* Revision 1.12  2004/03/27 09:34:04  tim
* changed BB2004 behavior
*
* Revision 1.11  2004/03/24 14:05:49  tim
* several small changes
*
* Revision 1.10  2004/03/22 10:23:03  tim
* added random walk
* changed parameters
*
* Revision 1.9  2004/03/19 15:38:00  tim
* changed BB2004BehaviorControl
*
* Revision 1.8  2004/03/15 12:50:30  tim
* Adaptions to new GameController
*
* Revision 1.7  2004/02/03 13:20:47  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.6  2004/01/23 14:56:48  tim
* Added files for BremenByters potential field behavior
*
* Revision 1.5  2004/01/20 15:43:55  tim
* Added potential fields to Bremen Byters behavior
*
* Revision 1.4  2003/12/09 16:29:06  jhoffman
* removed OUTPUTs
*
* Revision 1.3  2003/10/28 13:57:45  dueffert
* unused evolution removed
*
* Revision 1.2  2003/10/28 13:25:52  dueffert
* spelling improved
*
* Revision 1.1  2003/10/26 22:49:36  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

