/** 
* @file BB2004SimpleBasicBehaviors.h
*
* Declaration of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#ifndef __BB2004SimpleBasicBehaviors_h_
#define __BB2004SimpleBasicBehaviors_h_

#include "../../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2BasicBehavior.h"
#include "Modules/WalkingEngine/InvKinWalkingEngine.h"
#include "Modules/WalkingEngine/InvKinWalkingParameterSets.h"
#include "../BBPotentialfields.h"


/**
* A utility class implementing some useful functions for BB2004BasicBehaviors
*/
class BB2004BasicBehaviorTools : public BehaviorControlInterfaces
{
public:
  /** Empty constructor*/
  BB2004BasicBehaviorTools(const BehaviorControlInterfaces& interfaces): BehaviorControlInterfaces(interfaces)
  { lastTimeApproached = 0; headModeChanged = 0; searchBall = false;}

  /** Detects, if the ball is at the body PSD sensor*/
  bool ballAtRobot();

  /** Approaches a ball being near to the robot*/
  void approachCloseBall();

  /** The angle to the opponent goal*/
  double rotationToOpponentGoal();

private:
  unsigned long lastTimeApproached, headModeChanged;
  bool searchBall;
};


/**
* A Basic Behavior for going to the ball using a potential field
* @author Tim Laue
*/
class BB2004BasicBehaviorGoToBallPotentialField : public Xabsl2BasicBehavior, 
                                                  public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  BB2004BasicBehaviorGoToBallPotentialField(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball-potential-field", errorHandler),
    BehaviorControlInterfaces(interfaces),
    potentialfields("Bremen/gtbllast.pfc")
  {}
  
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** An instance of the potential field architecture*/
  BBPotentialfields potentialfields;
};


/**
* A Basic Behavior for moving the robot to a given pose
* @author Tim Laue
*/
class BB2004BasicBehaviorGoToPosePotentialField : public Xabsl2BasicBehavior, 
                                                  public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  BB2004BasicBehaviorGoToPosePotentialField(const BehaviorControlInterfaces& interfaces,
                                            Xabsl2ErrorHandler& errorHandler);
    
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** An instance of the potential field architecture*/
  BBPotentialfields potentialfields;
  /** Parameter go-to-pose-potential-field.x*/
  double x;
  /** Parameter go-to-pose-potential-field.y*/
  double y;
  /** Parameter go-to-pose-potential-field.destination-angle*/
  double destinationAngle;
  /** The maximum forward speed*/
  double forwardSpeed;
};


/**
* A Basic Behavior for running
* @author Tim Laue
*/
class BB2004BasicBehaviorRun : public Xabsl2BasicBehavior, 
                               public BehaviorControlInterfaces
{
public:
 /* Constructor.
  * @param errorHandler Is invoked when errors occur
  * @param interfaces The paramters of the BehaviorControl module.
  */
  BB2004BasicBehaviorRun(const BehaviorControlInterfaces& interfaces,
                         Xabsl2ErrorHandler& errorHandler);
  
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** Parameter "run.speed-x" */
  double speedX;
  /** Parameter "run.speed-y" */
  double speedY;
  /** Parameter "run.speed-max" */
  double maxSpeed;
  /** Parameter "run.rotation-speed" */
  double rotationSpeed;
  /** Parameter "run.use-rotation */
  double useRotation;
  /** The standard forward speed */
  double forwardSpeed;
  /** The state of the basic behavior*/
  enum RunState {turnState, dashState} runState;
};


/**
* A Basic Behavior for omnidirectional walking
* @author Tim Laue
*/
class BB2004BasicBehaviorWalkOmni : public Xabsl2BasicBehavior, 
                                    public BehaviorControlInterfaces
{
public:
 /* Constructor.
  * @param errorHandler Is invoked when errors occur
  * @param interfaces The paramters of the BehaviorControl module.
  */
  BB2004BasicBehaviorWalkOmni(const BehaviorControlInterfaces& interfaces,
                              Xabsl2ErrorHandler& errorHandler);
  
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** Parameter "run.speed-x" */
  double speedX;
  /** Parameter "run.speed-y" */
  double speedY;
  /** Parameter "run.speed-max" */
  double maxSpeed;
  /** Parameter "run.rotation-speed" */
  double rotationSpeed;
  /** Parameter "run.use-rotation */
  double useRotation;
  /** The standard forward speed */
  double forwardSpeed;
};


/**
* A Basic Behavior for randomly walking
* @author Tim Laue
*/
class BB2004BasicBehaviorRandomWalk : public Xabsl2BasicBehavior, 
                                      public BehaviorControlInterfaces
{
public:
 /* Constructor.
  * @param errorHandler Is invoked when errors occur
  * @param interfaces The paramters of the BehaviorControl module.
  */
  BB2004BasicBehaviorRandomWalk(const BehaviorControlInterfaces& interfaces,
                                Xabsl2ErrorHandler& errorHandler);
  
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** An instance of the potential field architecture*/
  BBPotentialfields potentialfields;
};


/**
* A Basic Behavior for dribbling
* @author Tim Laue
*/
class BB2004BasicBehaviorDribble : public Xabsl2BasicBehavior, 
                                   public BehaviorControlInterfaces
{
public:
 /* Constructor.
  * @param errorHandler Is invoked when errors occur
  * @param interfaces The paramters of the BehaviorControl module.
  */
  BB2004BasicBehaviorDribble(const BehaviorControlInterfaces& interfaces,
                             Xabsl2ErrorHandler& errorHandler)
                           : Xabsl2BasicBehavior("dribble", errorHandler),
                             BehaviorControlInterfaces(interfaces),
                             tools(interfaces)
  {
    registerParameter("dribble.x", translation.x);
    registerParameter("dribble.y", translation.y);
    dribbleState = walk;
    moveState = dash;
    lastState = walk;
    timeStamp = 0;
    timeOfMoveStateChange = 0;
    moveTimeStamp = 0;
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** Parameters "dribble.x" and "dribble.y"*/
  Vector2<double> translation;
  /** A time stamp*/
  unsigned long timeStamp, timeOfStateChange, timeOfMoveStateChange, moveTimeStamp;
  /** The state of the dribble behavior*/
  enum DribbleState {walk, hold, moveBall} dribbleState, lastState;
  enum MoveState {turn, dash} moveState;
  /** Some utility functions*/
  BB2004BasicBehaviorTools tools;

  /** The dribble function*/
  void doDribbling();
};


/**
* A Basic Behavior for kicking the ball.
* @author Tim Laue
*/
class BB2004BasicBehaviorKick : public Xabsl2BasicBehavior, 
                                public BehaviorControlInterfaces
{
public:
 /* Constructor.
  * @param errorHandler Is invoked when errors occur
  * @param interfaces The paramters of the BehaviorControl module.
  */
  BB2004BasicBehaviorKick(const BehaviorControlInterfaces& interfaces,
                          Xabsl2ErrorHandler& errorHandler)
                        : Xabsl2BasicBehavior("kick", errorHandler),
                          BehaviorControlInterfaces(interfaces),
                          tools(interfaces)
  {
    registerParameter("kick.id", kickID);
    registerParameter("kick.rotation", rotation);
    registerParameter("kick.x", translation.x);
    registerParameter("kick.y", translation.y);
    kickState = walk;
    lastState = walk;
    timeStamp = 0;
    timeOfStateChange;
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** Parameter "kick.id" */
  double kickID;
  /** Parameter "kick.rotation*/
  double rotation;
  /** Parameters "kick.x" and "kick.y"*/
  Vector2<double> translation;
  /** A time stamp*/
  unsigned long timeStamp, timeOfStateChange;
  /** The state of the kick behavior*/
  enum KickState {walk, hold, kick} kickState, lastState;
  /** Some utility functions*/
  BB2004BasicBehaviorTools tools;
  /** Executes the kick*/
  void doKick();
};



/**
* @class BB2004SimpleBasicBehaviors
*
* Creates and registers simple basic behaviors
*/
class BB2004SimpleBasicBehaviors : public BehaviorControlInterfaces
{
public:
  /**
  * Constructor
  */
  BB2004SimpleBasicBehaviors(const BehaviorControlInterfaces& interfaces,
                       Xabsl2ErrorHandler& errorHandler)
    : BehaviorControlInterfaces(interfaces),
      errorHandler(errorHandler),
      basicBehaviorGoToBallPotentialField(interfaces,errorHandler),
      basicBehaviorGoToPosePotentialField(interfaces,errorHandler),
      basicBehaviorRun(interfaces,errorHandler),
      basicBehaviorWalkOmni(interfaces,errorHandler),
      basicBehaviorRandomWalk(interfaces,errorHandler),
      basicBehaviorDribble(interfaces,errorHandler),
      basicBehaviorKick(interfaces,errorHandler)
  {}

  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine);

private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;

  //!@name Basic Behaviors
  //!@{
  BB2004BasicBehaviorGoToBallPotentialField basicBehaviorGoToBallPotentialField;  
  BB2004BasicBehaviorGoToPosePotentialField basicBehaviorGoToPosePotentialField;  
  BB2004BasicBehaviorRun basicBehaviorRun;
  BB2004BasicBehaviorWalkOmni basicBehaviorWalkOmni;
  BB2004BasicBehaviorRandomWalk basicBehaviorRandomWalk;
  BB2004BasicBehaviorDribble basicBehaviorDribble;
  BB2004BasicBehaviorKick basicBehaviorKick;
  //!@}
};

#endif // __BB2004SimpleBasicBehaviors_h_

/*
* Change Log
* 
* $Log: BB2004SimpleBasicBehaviors.h,v $
* Revision 1.17  2004/04/09 14:06:44  tim
* integrated changes from GO2004
*
* Revision 1.18  2004/04/03 05:21:08  tim
* improved behavior
*
* Revision 1.17  2004/03/31 20:13:06  tim
* changed behavior
*
* Revision 1.16  2004/03/27 09:34:04  tim
* changed BB2004 behavior
*
* Revision 1.15  2004/03/24 14:05:49  tim
* several small changes
*
* Revision 1.14  2004/03/22 10:23:04  tim
* added random walk
* changed parameters
*
* Revision 1.13  2004/03/19 15:38:00  tim
* changed BB2004BehaviorControl
*
* Revision 1.12  2004/03/15 12:50:30  tim
* Adaptions to new GameController
*
* Revision 1.11  2004/03/08 00:58:46  roefer
* Interfaces should be const
*
* Revision 1.10  2004/01/23 16:24:24  tim
* no message
*
* Revision 1.9  2004/01/23 15:22:34  tim
* bugs fixed
*
* Revision 1.8  2004/01/23 15:03:53  tim
* no message
*
* Revision 1.7  2004/01/23 14:56:48  tim
* Added files for BremenByters potential field behavior
*
* Revision 1.6  2004/01/20 15:43:55  tim
* Added potential fields to Bremen Byters behavior
*
* Revision 1.5  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.4  2003/10/28 14:04:50  dueffert
* unused evolution removed
*
* Revision 1.3  2003/10/28 13:57:45  dueffert
* unused evolution removed
*
* Revision 1.2  2003/10/28 13:25:52  dueffert
* spelling improved
*
* Revision 1.1  2003/10/26 22:49:36  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

