/** 
* @file BB2004StrategySymbols.cpp
*
* Implementation of class BB2004StrategySymbols.
*
* @author Martin Ltzsch
* @author Tim Laue
*/

#include "BB2004StrategySymbols.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Location.h"


BB2004StrategySymbols::BB2004StrategySymbols(const BehaviorControlInterfaces& interfaces)
: BehaviorControlInterfaces(interfaces),
timeWhenBallWasStartedToCatch(0), timeUntilBallWasCaught(0)
{
  role = BehaviorTeamMessage::goalie;
  baseStrategy = standard;
  estimatedTimeToReachBall = 0.0;
  timeOfLastTaskSelection = 0;
  int strategyInt;
  InTextFile parametersFile(getLocation().getFilename("basestrt.cfg"));
  if (parametersFile.exists())
  {
    parametersFile >> strategyInt;
    baseStrategy = (BaseStrategy)strategyInt;
  }
}


void BB2004StrategySymbols::registerSymbols(Xabsl2Engine& engine)
{
  // "robot-number"
  engine.registerDecimalInputSymbol("robot-number", this,
    (double(Xabsl2FunctionProvider::*)())&BB2004StrategySymbols::getRobotNumber);
  
  // "role"
  engine.registerEnumeratedInputSymbol("role", (int *)&role);
  engine.registerEnumeratedInputSymbolEnumElement("role", "goalie", BehaviorTeamMessage::goalie);
  engine.registerEnumeratedInputSymbolEnumElement("role", "striker", BehaviorTeamMessage::striker);
  engine.registerEnumeratedInputSymbolEnumElement("role", "defensive-supporter", BehaviorTeamMessage::defensiveSupporter);
  engine.registerEnumeratedInputSymbolEnumElement("role", "offensive-supporter", BehaviorTeamMessage::offensiveSupporter);
  
  // "task"
  engine.registerEnumeratedInputSymbol("task", (int *)&task);
  engine.registerEnumeratedInputSymbolEnumElement("task", "play-goalie", playGoalie);
  engine.registerEnumeratedInputSymbolEnumElement("task", "standard-defend", standardDefend);
  engine.registerEnumeratedInputSymbolEnumElement("task", "standard-support", standardSupport);
  engine.registerEnumeratedInputSymbolEnumElement("task", "play-ball", playBall);
  engine.registerEnumeratedInputSymbolEnumElement("task", "search-ball", searchBall);
  
  // "estimated-time-to-reach-ball"
  engine.registerDecimalInputSymbol("estimated-time-to-reach-ball", (double *)&estimatedTimeToReachBall);
  
  // "sent-game-state"
  engine.registerEnumeratedOutputSymbol("sent-game-state",(int*)&outgoingBehaviorTeamMessage.gameState);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.sleep",BehaviorTeamMessage::initial);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.initial",BehaviorTeamMessage::initial);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.ready",BehaviorTeamMessage::ready);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.playing",BehaviorTeamMessage::playing);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.penalized",BehaviorTeamMessage::penalized);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.finished",BehaviorTeamMessage::finished);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.set",BehaviorTeamMessage::set);
  
  // "another-player-is-in-ready-state"
  engine.registerBooleanInputSymbol("another-player-is-in-ready-state",
    this,(bool(Xabsl2FunctionProvider::*)())&BB2004StrategySymbols::getAnotherPlayerIsInReadyState);
  
  // "team-message"
  engine.registerEnumeratedOutputSymbol("team-message",(int*)&outgoingBehaviorTeamMessage.message);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.none", BehaviorTeamMessage::none);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.just-performed-a-kick",BehaviorTeamMessage::justPerformedAKick);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.performing-a-kick",BehaviorTeamMessage::performingAKick);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.preparing-a-kick",BehaviorTeamMessage::preparingAKick);
  
  // "another-teammate-is-preparing-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-is-preparing-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&BB2004StrategySymbols::getAnotherTeammateIsPreparingAKick);
  
  // "another-teammate-is-performing-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-is-performing-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&BB2004StrategySymbols::getAnotherTeammateIsPerformingAKick);
  
  // "another-teammate-just-performed-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-just-performed-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&BB2004StrategySymbols::getAnotherTeammateJustPerformedAKick);
  
  // "catch-ball"
  engine.registerBooleanInputSymbol("catch-ball", this,
    (bool (Xabsl2FunctionProvider::*)())&BB2004StrategySymbols::getCaught);
  
  // "catch-ball-time"
  engine.registerDecimalInputSymbol("catch-ball-time",this,
    (double (Xabsl2FunctionProvider::*)())&BB2004StrategySymbols::getCatchTime);
  
  // "robot-is-stuck"
  engine.registerBooleanInputSymbol("robot-is-stuck", &robotIsStuck);
  
  // "obstacles-are-close"
  engine.registerBooleanInputSymbol("obstacles-are-close", &obstaclesAreClose);
  
  engine.registerDecimalInputSymbol("search-ball.x", this,
    (double (Xabsl2FunctionProvider::*)())&BB2004StrategySymbols::getSearchBallX);
}

void BB2004StrategySymbols::update()
{
  estimateTimeToReachBall();
  computeRole();
  computeTask();
  
  outgoingBehaviorTeamMessage.estimatedTimeToReachBall = estimatedTimeToReachBall;
  outgoingBehaviorTeamMessage.dynamicRole = role;
  
  if (((executedMotionRequest.motionType == MotionRequest::specialAction
    && 
    (executedMotionRequest.specialActionType == MotionRequest::catchBall2
    || executedMotionRequest.specialActionType == MotionRequest::lookForLandmarksWithBallCaught)
    )
    || 
    (executedMotionRequest.motionType == MotionRequest::walk 
    && executedMotionRequest.walkType == MotionRequest::turnWithBall)
    )
    || headControlMode.headControlMode == HeadControlMode::catchBall)
  {
    if (SystemCall::getTimeSince(timeUntilBallWasCaught) > 1000)
    {
      timeWhenBallWasStartedToCatch = SystemCall::getCurrentSystemTime();
    }
    
    timeUntilBallWasCaught = SystemCall::getCurrentSystemTime();
  }
  
  // robot is stuck
  if(obstaclesModel.getPercentageOfLowDistanceObstaclesInRange(0, pi_2, 300) > 0.2)
  {
    robotIsStuck = true;
    obstaclesAreClose = true;
  }
  else
  {
    robotIsStuck = false;
    if(obstaclesModel.getPercentageOfLowDistanceObstaclesInRange(0, pi_2, 500) > 0.15)
      obstaclesAreClose = true;
    else
      obstaclesAreClose = false;
  }
}

void BB2004StrategySymbols::computeRole()
{
  if(SystemCall::getTimeSince(timeOfLastTaskSelection) < 2500)
  {
    return;
  }
  else
  {
    timeOfLastTaskSelection = SystemCall::getCurrentSystemTime();
  }
  int i;
  if (getPlayer().getPlayerNumber() == Player::one)
  {
    // no role changes for the goalie
    role = BehaviorTeamMessage::goalie;
    return;
  }
  
  // fall-back if no wlan
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
    && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
    && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
    case Player::two:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x < 75)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::defensiveSupporter;
      break;
    case Player::three:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x > 500)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::offensiveSupporter;
      break;
    case Player::four:
    default:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x > -75)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::offensiveSupporter;
      break;
    }
    return;
  }
  
  // with wlan
  
  // estimate the closest other teammate to the ball
  double minTeammateTime = 100000.0;
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() )
    {
      if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::goalie 
        && teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing) 
      {
        double time = 
          teamMessageCollection[i].behaviorTeamMessage.estimatedTimeToReachBall;
        
        // bonus for current striker
        if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker ) 
          time -= 500.0;
        
        if (time < minTeammateTime ) 
        {
          minTeammateTime = time;
        }
      }
    }
  }
  
  // bonus for current striker
  if (role == BehaviorTeamMessage::striker)
    minTeammateTime += 1000.0;
  
  // assign striker-role if nearest to the ball
  if ( estimatedTimeToReachBall < minTeammateTime )
  {
    role = BehaviorTeamMessage::striker;
    return;
  }
  
  // assign supporting roles
  double maxTeammateX = (double)xPosOwnGroundline;
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() )
    {
      if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::striker 
        && teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::goalie 
        && teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing)
      {
        double teammateX = 
          teamMessageCollection[i].robotPose.translation.x;
        
        // bonus for current offensive supporter
        if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::offensiveSupporter ) 
          teammateX += 300.0;
        
        if (teammateX > maxTeammateX ) 
        {
          maxTeammateX = teammateX;
        }
      }
    }
  }
  
  // bonus for current offensive supporter
  if ( role == BehaviorTeamMessage::offensiveSupporter ) 
    maxTeammateX -= 300.0;
  
  if ( robotPose.translation.x >= maxTeammateX ) 
    role = BehaviorTeamMessage::offensiveSupporter;
  else 
    role = BehaviorTeamMessage::defensiveSupporter;
}


void BB2004StrategySymbols::computeTask()
{
  // A goalie is a goalie is a goalie... :
  if(getPlayer().getPlayerNumber() == Player::one)
  {
    task = playGoalie;
    return;
  }
  // If the ball position is unknown -> start search :
  if(ballPosition.getTimeSinceLastKnown(2000) > 3000)
  {
    task = searchBall;
    return;
  }
  // Fall-back if no WLAN (quite dumb solution):
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
    && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
    && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
    case Player::two: task = standardDefend;
    case Player::three: task = standardSupport;
    default: task = playBall;
    }
    return;
  }
  // Determine the player going to the ball:
  double minTeammateTime = 1000000.0;
  int i;
  int indexOfTeammateNearestToTheBall(0);
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() && teamMessageCollection[i].playerNumberOfSender != Player::one
      && teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing) 
    {
      double time(teamMessageCollection[i].behaviorTeamMessage.estimatedTimeToReachBall);
      if (time < minTeammateTime ) 
      {
        minTeammateTime = time;
        indexOfTeammateNearestToTheBall = i;
      }
    }
  }
  // bonus if currently playing the ball
  if (task == playBall)
  {
    minTeammateTime += 1000.0;
  }
  // assign task if nearest to the ball
  if (estimatedTimeToReachBall < minTeammateTime)
  {
    task = playBall;
    return;
  }
  
  // Now we know, that the robot will not be the goalie and will also not
  // play the the ball. A supporting role has to be determined. There may
  // also be a second supporting robot left, to whom such a role will be
  // assigned, too:
  int numberOfSupportingTeammates(0);
  int indexOfOtherSupportingTeammate(0);
  for(i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ )
  {
    if (i != indexOfTeammateNearestToTheBall && 
      teamMessageCollection[i].isActual() && teamMessageCollection[i].playerNumberOfSender != Player::one
      && teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing) 
    {
      numberOfSupportingTeammates = 1;
      indexOfOtherSupportingTeammate = i;
    }
  }
  // Simple solution:
  if(numberOfSupportingTeammates)
  {
    if(robotPose.translation.x < teamMessageCollection[indexOfOtherSupportingTeammate].robotPose.translation.x)
    {
      task = standardDefend;
    }
    else
    {
      task = standardSupport;
    }
  }
  else
  {
    task = standardDefend;
  }
}


void BB2004StrategySymbols::estimateTimeToReachBall()
{
  // stay striker if ball is caught
  if (task == playBall && Geometry::distanceTo(robotPose, ballPosition.seen) < 400 && ballPosition.getTimeSinceLastKnown(10000) < 400)
  {
    estimatedTimeToReachBall = 0.0;
    return;
  }
  
  // account for distance to ball
  estimatedTimeToReachBall = Geometry::distanceTo(robotPose.translation, ballPosition.seen) / 0.2;
  
  // account if the robot is between the ball and the opponent goal
  
  // the position of the robot
  Vector2<double> robotPosition = robotPose.translation;
  double angleToLeftOpponentGoalPost = Geometry::angleTo(robotPosition, Vector2<double>(xPosOpponentGroundline,yPosLeftGoal));
  double angleToRightOpponentGoalPost = Geometry::angleTo(robotPosition, Vector2<double>(xPosOpponentGroundline,yPosRightGoal));
  if(angleToLeftOpponentGoalPost < angleToRightOpponentGoalPost)
  {
    angleToLeftOpponentGoalPost += pi2;
  }
  double angleToOpponentGoal= (angleToLeftOpponentGoalPost + angleToRightOpponentGoalPost) / 2.0;
  double absoluteAngleToBall = Geometry::angleTo(robotPosition,ballPosition.seen);
  double angleBetweenBallAndOpponentGoal = normalize(angleToOpponentGoal - absoluteAngleToBall);
  
  estimatedTimeToReachBall += 400.0 * fabs(angleBetweenBallAndOpponentGoal);
  
  // longer if ball not seen
  estimatedTimeToReachBall += 2.0 * SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);
  if(SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) > 2000)
  {
    estimatedTimeToReachBall += 10000;
  }
}


bool BB2004StrategySymbols::getAnotherPlayerIsInReadyState()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.gameState 
        == BehaviorTeamMessage::ready)
      {
        return true;
      }
    }
  }
  return false;
}

bool BB2004StrategySymbols::getAnotherTeammateIsPreparingAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::preparingAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool BB2004StrategySymbols::getAnotherTeammateIsPerformingAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::performingAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool BB2004StrategySymbols::getAnotherTeammateJustPerformedAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::justPerformedAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool BB2004StrategySymbols::getCaught()
{
  return (SystemCall::getTimeSince(timeUntilBallWasCaught) < 500);  
}

double BB2004StrategySymbols::getCatchTime()
{
  return (SystemCall::getTimeSince(timeUntilBallWasCaught) < 500?
    timeUntilBallWasCaught - timeWhenBallWasStartedToCatch : 0);
}

double BB2004StrategySymbols::getSearchBallX()
{
  // fall-back if no wlan
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
    && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
    && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
    case Player::two:
      return -1000;
    case Player::three:
      return 0;
    case Player::four:
    default:
      return 1000;
    }
  }
  else
  {
    switch (role)
    {
    case BehaviorTeamMessage::defensiveSupporter:
      return -1000;
    case BehaviorTeamMessage::offensiveSupporter:
      return 1000;
    case BehaviorTeamMessage::striker:
    default:
      return 0;
    }
  }
}

/*
* Change Log
* 
* $Log: BB2004StrategySymbols.cpp,v $
* Revision 1.11  2004/04/09 14:06:44  tim
* integrated changes from GO2004
*
* Revision 1.11  2004/04/02 19:18:57  tim
* changed parameters
*
* Revision 1.10  2004/03/29 11:52:39  tim
* several small changes
*
* Revision 1.9  2004/03/27 09:34:04  tim
* changed BB2004 behavior
*
* Revision 1.8  2004/03/25 17:40:15  loetzsch
* adaptations to changes in the game controller and the led request
*
* Revision 1.7  2004/03/25 12:05:55  tim
* added new tasks
*
* Revision 1.6  2004/03/24 14:05:50  tim
* several small changes
*
* Revision 1.5  2004/03/19 15:38:00  tim
* changed BB2004BehaviorControl
*
* Revision 1.4  2004/03/15 12:50:31  tim
* Adaptions to new GameController
*
* Revision 1.3  2004/03/08 00:58:51  roefer
* Interfaces should be const
*
* Revision 1.2  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/26 22:49:37  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

