/**
* @file BehaviorControl.h
* 
* Implementation of class BehaviorControl
*/

#ifndef __BehaviorControl_h_
#define __BehaviorControl_h_

#include "Tools/Module/Module.h"

#include "Representations/Cognition/RobotPose.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Cognition/PlayerPoseCollection.h"
#include "Representations/Cognition/ObstaclesModel.h"
#include "Representations/Cognition/RobotState.h"
#include "Representations/Cognition/SampleSet.h"
#include "Representations/Motion/MotionRequest.h"
#include "Representations/Motion/LEDRequest.h"
#include "Representations/Motion/HeadControlMode.h"
#include "Representations/Motion/OdometryData.h"
#include "Representations/Sound/SoundRequest.h"
#include "Representations/Perception/SpecialPercept.h"
#include "Representations/Perception/SpecialVisionRequest.h"
#include "Representations/Perception/CalibrationRequest.h"
#include "Representations/Perception/SensorDataBuffer.h"
#include "Representations/RoboCup/GameControlData.h"
#include "Representations/WLan/TeamMessageCollection.h"
#include "Representations/JoystickData.h"
#include "Tools/MessageQueue/InMessage.h"
#include "Modules/WalkingEngine/InvKinWalkingParameters.h"
#include "Modules/WalkingEngine/UDParameterSet.h"
#include "Representations/Perception/PSDPercept.h"

#ifdef _WIN32
#pragma warning(disable:4786) 
// the constructor with all it's long parameter type names results in a too
// long debug identifier
#endif


/**
* @class BehaviorControlInterfaces
* 
* The interfaces of the BehaviorControl module.
*/
class BehaviorControlInterfaces
{
public:
  /** Constructor.*/
  BehaviorControlInterfaces(
    const RobotPose& robotPose,
    const BallModel& ballPosition,
    const PlayerPoseCollection& playerPoseCollection,
    const ObstaclesModel& obstaclesModel,
    const RobotState& robotState,
    const MotionRequest& executedMotionRequest,
    const SpecialPercept& specialPercept,
    const TeamMessageCollection& teamMessageCollection,
    const JoystickData& joystickData,
    const OdometryData& odometryData,
    const SampleSetProxy<PoseSample>& selfLocatorSamples,
    const SensorDataBuffer& sensorDataBuffer,
    const PSDPercept& psdPercept,
    GameControlData& gameControlData,
    MotionRequest& motionRequest,
    LEDRequest& ledRequest,
    HeadControlMode& headControlMode,
    SoundRequest& soundRequest,
    InvKinWalkingParameters& invKinWalkingParameters,
    UDParameters& udParameters,
    unsigned long& walkParameterTimeStamp,
    SpecialVisionRequest& specialVisionRequest,
    CalibrationRequest& calibrationRequest,
    BehaviorTeamMessage& outgoingBehaviorTeamMessage)

    : robotPose(robotPose),
    ballPosition(ballPosition),
    playerPoseCollection(playerPoseCollection),
    obstaclesModel(obstaclesModel),
    robotState(robotState),
    executedMotionRequest(executedMotionRequest),
    specialPercept(specialPercept),
    teamMessageCollection(teamMessageCollection),
    joystickData(joystickData),
    odometryData(odometryData),
    selfLocatorSamples(selfLocatorSamples),
    sensorDataBuffer(sensorDataBuffer),
    psdPercept(psdPercept),
    gameControlData(gameControlData),
    motionRequest(motionRequest),
    ledRequest(ledRequest),
    headControlMode(headControlMode),
    soundRequest(soundRequest),
    invKinWalkingParameters(invKinWalkingParameters),
    udParameters(udParameters),
    walkParameterTimeStamp(walkParameterTimeStamp),
    specialVisionRequest(specialVisionRequest),
    calibrationRequest(calibrationRequest),
    outgoingBehaviorTeamMessage(outgoingBehaviorTeamMessage)
  {}
  
protected:
  /** The robots position and orientation */
  const RobotPose& robotPose;
  
  /** The position and the speed of the ball */
  const BallModel& ballPosition;
  
  /** The positions of the own and opponent players */
  const PlayerPoseCollection& playerPoseCollection;

  /** The positions of the obstacles */
  const ObstaclesModel& obstaclesModel;
  
  /** Pressed switches and fall downs */
  const RobotState& robotState;

  /** The motion request that is currently executed by the MotionControl */
  const MotionRequest& executedMotionRequest;
  
  /** Special percepts for challenges */
  const SpecialPercept& specialPercept;
  
  /** Messages from teammates */
  const TeamMessageCollection& teamMessageCollection;
  
  /** Data produced by an extern sensor called joystick */
  const JoystickData& joystickData;
  
  /** Odomety Data */
  const OdometryData& odometryData;

  /**  The sample set proxy linked to the sample set of the active locator */
  const SampleSetProxy<PoseSample>& selfLocatorSamples;

  /** The sensor data */
  const SensorDataBuffer& sensorDataBuffer;

  /** The PSDPercept*/
  const PSDPercept& psdPercept;

  /** Messages that are sent by the referee */ 
  GameControlData& gameControlData;
  
  /** The motion request to be set by the behavior */
  MotionRequest& motionRequest;
  
  /** The led request to be set by the behavior */
  LEDRequest& ledRequest;
  
  /** The head control mode to be set by the behavior */
  HeadControlMode& headControlMode;
  
  /** The acoustic message to be set by the behavior */
  SoundRequest& soundRequest;

  /** walking parameter sets to be used by the WalkingEngine, eg calculated by evolution behavior */
  InvKinWalkingParameters& invKinWalkingParameters;
  UDParameters& udParameters;
  unsigned long& walkParameterTimeStamp;
  
  /** A request for special vision modules for challenges */
  SpecialVisionRequest& specialVisionRequest;
  
  /** A request for calibration */
  CalibrationRequest& calibrationRequest;
  
  /** Messages to teammates */
  BehaviorTeamMessage& outgoingBehaviorTeamMessage;
};

/**
* @class BehaviorControl 
*
* The Base class for behavior controls
*/
class BehaviorControl : public Module, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param interfaces The paramters of the BehaviorControl module.
  */
  BehaviorControl(const BehaviorControlInterfaces& interfaces)
    : BehaviorControlInterfaces(interfaces)
  {}
  
  /** Destructor */
  virtual ~BehaviorControl() {}
};

#endif //__BehaviorControl_h_

/*
* Change log :
* 
* $Log: BehaviorControl.h,v $
* Revision 1.18  2004/04/16 14:56:36  dueffert
* cleanup for Martins data flow graphics
*
* Revision 1.17  2004/04/14 06:37:15  loetzsch
* removed the hack "kickInProgress" from the BehaviorControlInterfaces
*
* Revision 1.16  2004/04/08 15:33:00  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.15  2004/03/29 15:30:48  dueffert
* evolution mode idea added
*
* Revision 1.14  2004/03/19 16:36:33  dueffert
* reference bug fixed
*
* Revision 1.13  2004/03/17 16:31:20  kerdels
* added boolean input symbol "robot-pose.something-in-front-of-chest" utilizing the chest distance sensor
*
* Revision 1.12  2004/03/16 16:00:09  tim
* Added SensorDataBuffer to BehaviorControlInterfaces
*
* Revision 1.11  2004/03/15 12:50:31  tim
* Adaptions to new GameController
*
* Revision 1.10  2004/03/15 12:37:15  dueffert
* measurement of free choosen MotionRequest allowed now
*
* Revision 1.9  2004/03/08 00:58:52  roefer
* Interfaces should be const
*
* Revision 1.8  2004/02/23 16:48:27  dueffert
* several improvements for measurement of walking
*
* Revision 1.7  2004/02/18 14:49:19  dueffert
* behavior control can now change walking parameters
*
* Revision 1.6  2004/02/16 17:58:06  dueffert
* packageCognitionMotion extended with invkin parameters
*
* Revision 1.5  2004/02/03 13:19:49  spranger
* renamed all references to  class BallPosition to BallModel
*
* Revision 1.4  2003/10/12 11:46:30  juengel
* Added CalibrationRequest.
*
* Revision 1.3  2003/09/26 21:23:20  loetzsch
* renamed class JointState to CollisionPercept
*
* Revision 1.2  2003/09/26 15:27:48  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.11  2003/06/10 15:05:02  goehring
* CollisionDetectorInterface extended by JointState
*
* Revision 1.10  2003/05/12 14:08:39  brunn
* renamed selfLocationSampleSetProxy to selfLocatorSamples
* "Ha, glad am I that no one knew that Rumpelstiltskin I am styled"
*
* Revision 1.9  2003/05/12 12:28:08  brunn
* renamed sampleSetProxy to selfLocationSampleSetProxy
* added selfLocationSampleSetProxy to BehaviourControl-Interfaces
*
* Revision 1.8  2003/04/03 17:35:27  dueffert
* free part of goal kick added
*
* Revision 1.7  2003/03/10 13:55:22  juengel
* Added ObstaclesModel to the interfaces.
*
* Revision 1.6  2003/03/06 11:06:38  dueffert
* warning removed
*
* Revision 1.5  2003/02/27 12:02:30  dueffert
* JoystickData added
*
* Revision 1.4  2003/02/07 16:40:50  dueffert
* Bombay stuff added after cleanup
*
* Revision 1.3  2002/09/17 23:55:20  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.2  2002/09/12 12:24:10  juengel
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:12  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.6  2002/07/23 13:33:40  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.5  2002/06/20 00:40:19  Thomas Rfer
* WLan crash removed
*
* Revision 1.4  2002/06/10 23:43:27  loetzsch
* fixed VC Compiler warnings
*
* Revision 1.3  2002/06/09 15:24:51  loetzsch
* Added TeamMessageCollection and BehaviorTeamMessage to the execute of BehaviorControl
*
* Revision 1.2  2002/05/10 17:29:01  juengel
* Added SpecialVision and SpecialPercept.
*
* Revision 1.7  2002/05/05 18:52:01  loetzsch
* added
* - GameControlData,
* - Receivers for GameControlData
* - access by behavior to GameControlData
*
* Revision 1.6  2002/05/04 12:43:37  loetzsch
* The currently executed MotionRequest is now sent from the MotionControl
* to the BehaviorControl via the OdometryData structure
*
* Revision 1.5  2002/04/28 19:15:14  giese
* SoundPlay added...
*
* Revision 1.4  2002/03/25 09:22:12  rentmeister
* Behavior Dialog
*
* Revision 1.3  2001/12/21 14:09:39  roefer
* Added several destructors
*
* Revision 1.2  2001/12/10 17:47:06  risler
* change log added
*
*/
