/** 
* @file DDD2004ContinuousRules/AvoidBorderTurning.h
*
* @author Matthias Jngel
*/

#ifndef __DDD2004AvoidBorderTurning_h_
#define __DDD2004AvoidBorderTurning_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Debugging/DebugDrawings.h"

namespace DDD2004ContinuousRules
{
  
/**@class AvoidBorderTurning
* rule which avoids collision with field borders
  */
  class AvoidBorderTurning: public ContinuousRule{
  public:
    
  /** constructor
    */
    AvoidBorderTurning(const BehaviorControlInterfaces& interfaces)
      :ContinuousRule(interfaces, "AvoidBorderTurning")
    {
    };
    
    /**
    * Returns whether this rule generates absolute or robot relative coordinates.
    * This rule is robot relative.
    */
    virtual bool isRelative() {return true;}
    
    /** executes the basic behavior rule
    * @param robotPose the current robots pose at which the rule is to be evaluated
    * @param walk the direction and speed of the suggested motion coded as an vector (output)
    * @param ra the rotation angle, the direction the robot should be directed (output)
    * @param rweight the rotation weight the weight of the suggested rotation (output)
    */
    virtual void execute(const RobotPose& robotPose,
      Vector2<double>& walk,
      double& ra,
      double& rweight)
    {
      ra=0;
      rweight=0;
      
      Vector2<int> point1((int)robotPose.translation.x, (int)robotPose.translation.y);
      Vector2<double> point2double;
      point2double = (robotPose + Vector2<double>(500, 0)).translation;
      Vector2<int> point2((int)point2double.x, (int)point2double.y);
      
/*      
      LARGE_DOT(lineFollowerField, point1.x, point1.y, Drawings::red, Drawings::red);
      LARGE_DOT(lineFollowerField, point2.x, point2.y, Drawings::green, Drawings::green);
      LINE(lineFollowerField, point1.x, point1.y, point2.x, point2.y, 6, Drawings::ps_solid, Drawings::gray);
*/      
      if(FieldDimensions::clipLineWithFieldAndGoalAreas(point1, point2) )
      {
/*
        LINE(lineFollowerField, point1.x, point1.y, point2.x, point2.y, 6, Drawings::ps_solid, Drawings::white);
        LARGE_DOT(lineFollowerField, point1.x, point1.y, Drawings::red, Drawings::white);
        LARGE_DOT(lineFollowerField, point2.x, point2.y, Drawings::green, Drawings::white);
*/        
        Vector2<double> vectorToBorder;
        FieldDimensions::vectorToBorderIncludingGoals(robotPose.translation, vectorToBorder);
/*        LINE(lineFollowerField, 
          point1.x, point1.y, 
          point1.x + (int)vectorToBorder.x, 
          point1.y + (int)vectorToBorder.y,
          3, Drawings::ps_solid, Drawings::blue);
  */      
        rweight = 1.0;
        if(normalize((point2 - point1).angle() - vectorToBorder.angle()) > 0)
          ra = pi_4;
        else
          ra = -pi_4;
      }
//      DEBUG_DRAWING_FINISHED(lineFollowerField);
    }
  };
  
}

#endif 


/*
* Change log:
*
* $Log: AvoidBorderTurning.h,v $
* Revision 1.3  2004/03/08 00:59:01  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/28 15:04:09  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:38  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/
