/** 
* @file DDD2004SimpleBasicBehaviors.cpp
*
* Implementation of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#include "Tools/Math/PIDsmoothedValue.h"
#include "DDD2004SimpleBasicBehaviors.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"

#include "Tools/Debugging/Debugging.h"
#include "Platform/SystemCall.h"

void DDD2004SimpleBasicBehaviors::registerBasicBehaviors(Xabsl2Engine& engine)
{
  engine.registerBasicBehavior(basicBehaviorAvoidObstacles);
  engine.registerBasicBehavior(basicBehaviorGetBehindBall);
  engine.registerBasicBehavior(basicBehaviorGoToBall);
  engine.registerBasicBehavior(basicBehaviorGoToBallWithDirection);
  engine.registerBasicBehavior(basicBehaviorGoToPoint);
  engine.registerBasicBehavior(basicBehaviorGoToPointPrecisely);
  engine.registerBasicBehavior(basicBehaviorGoToPointAndAvoidObstacles);
}

void DDD2004BasicBehaviorGoToBallWithDirection::execute()
{
  //accelerationRestrictor.saveLastWalkParameters();
  
  if (maxSpeed == 0)
  {
    maxSpeed = 200;
  }
  
  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  
  Vector2<double> pointToReach(ball.x-cos(direction)*distanceX-sin(direction)*distanceY,
    ball.y-sin(direction)*distanceX-cos(direction)*distanceY);
  

  double destinationDistance = Geometry::distanceTo(robotPose.getPose(),pointToReach);
  double destinationX = destinationDistance * cos(Geometry::angleTo(robotPose.getPose(),pointToReach));
  double destinationY = destinationDistance * sin(Geometry::angleTo(robotPose.getPose(),pointToReach));

/*
  Vector2<double> pointToReachMirrored(ball.x-cos(direction)*distanceX+sin(direction)*distanceY,
    ball.y-sin(direction)*distanceX+cos(direction)*distanceY);

  double destinationDistanceMirrored = Geometry::distanceTo(robotPose.getPose(),pointToReach);
	double destinationYMirrored = destinationDistanceMirrored * sin(Geometry::angleTo(robotPose.getPose(),pointToReachMirrored));
 
	if (fabs(destinationYMirrored) < fabs(destinationY))
		{
		destinationY = destinationYMirrored;
		}
*/

  if(destinationDistance < 1)
  {
    destinationDistance = 1;
  }
  
  double minimalTime=destinationDistance/maxSpeed;
  
  double angleDiff=direction-robotPose.rotation;
  if (angleDiff<-pi)
  {
    angleDiff += pi2;
  }
  else if (angleDiff >=pi)
  {
    angleDiff -= pi2;
  }
  
  double maxTurnSpeed = fromDegrees(60);
  double turnTime = angleDiff/maxTurnSpeed;
  if (turnTime>minimalTime)
  {
    minimalTime=turnTime;
  }
  minimalTime += 0.2; //get slower near to destination
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  motionRequest.walkParams.translation.x = destinationX / minimalTime;
  motionRequest.walkParams.translation.y = destinationY / minimalTime;
  
  motionRequest.walkParams.rotation = angleDiff;
  
  if (motionRequest.walkParams.rotation > maxTurnSpeed)
    motionRequest.walkParams.rotation = maxTurnSpeed;
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    motionRequest.walkParams.rotation = -maxTurnSpeed;
  
  if ((fabs(toDegrees(angleDiff))<10)&&(destinationDistance<10))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

void DDD2004BasicBehaviorGoToBall::execute()
{
  //accelerationRestrictor.saveLastWalkParameters();

  if (maxSpeed == 0) maxSpeed = walkMaxForwardSpeed;

  double maxTurnSpeed = fromDegrees(60);

/*
  old version
  double maxSpeedX = maxSpeed;
  double maxSpeedY = maxSpeed;
  double timeToReachX, timeToReachY, timeToTurnToAngle, estimatedTimeToReachDestination;

  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ball);
  double ballX = ballDistance * cos(Geometry::angleTo(robotPose.getPose(),ball)) + 50;
  double ballY = ballDistance * sin(Geometry::angleTo(robotPose.getPose(),ball));

  if(ballDistance == 0)
    ballDistance = 1;
  double ballAngle = Geometry::angleTo(robotPose.getPose(),ball);

  double diffX = distanceAtEnd / ballDistance * ballX;
  double diffY = distanceAtEnd / ballDistance * ballY;

  Vector2<double> destination(ballX - diffX, ballY - diffY);
  double distanceToDestination;
  distanceToDestination = fabs(ballDistance - distanceAtEnd);

  timeToReachX = fabs(destination.x) / maxSpeedX;
  timeToReachY = fabs(destination.y) / maxSpeedY;
  timeToTurnToAngle = ballAngle / maxTurnSpeed;
  estimatedTimeToReachDestination = 
    (timeToReachX < timeToReachY) ? timeToReachY : timeToReachX;
  estimatedTimeToReachDestination = 
    (estimatedTimeToReachDestination < timeToTurnToAngle) ? timeToTurnToAngle : estimatedTimeToReachDestination;

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;

  motionRequest.walkParams.translation.x = 
    destination.x / estimatedTimeToReachDestination;
  motionRequest.walkParams.translation.y = 
    destination.y / estimatedTimeToReachDestination;


  motionRequest.walkParams.rotation = ballAngle;

  if (motionRequest.walkParams.rotation > maxTurnSpeed)
    motionRequest.walkParams.rotation = maxTurnSpeed;
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    motionRequest.walkParams.rotation = -maxTurnSpeed;


  if ((fabs(toDegrees(ballAngle))<10)&&(distanceToDestination<20))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
  else
  {
//    accelerationRestrictor.restrictAccelerations(250,250,180);
  }
*/

  static bool usingDash = false;
  static long lastChangeTime = 0;

  double timeToReachX, timeToReachY, estimatedTimeToReachDestination;
  double angle = Geometry::angleTo(robotPose.getPose(),ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted));

  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ball);
  double ballX = ballDistance * cos(Geometry::angleTo(robotPose.getPose(),ball)) + 50;
  double ballY = ballDistance * sin(Geometry::angleTo(robotPose.getPose(),ball));

  if(ballDistance == 0)
    ballDistance = 1;
  double ballAngle = Geometry::angleTo(robotPose.getPose(),ball);

  double diffX = distanceAtEnd / ballDistance * ballX;
  double diffY = distanceAtEnd / ballDistance * ballY;

  Vector2<double> destination(ballX - diffX, ballY - diffY);
  double distanceToDestination;
  distanceToDestination = fabs(ballDistance - distanceAtEnd);

  timeToReachX = fabs(destination.x) / maxSpeed;
  timeToReachY = fabs(destination.y) / maxSpeed;

  estimatedTimeToReachDestination = 
    (timeToReachX < timeToReachY) ? timeToReachY : timeToReachX;

  if 
  ( 
    ( !usingDash && fabs(toDegrees(ballAngle)) < 10 && distanceToDestination > 650.0 && SystemCall::getTimeSince(lastChangeTime) > 1000) ||
    (  usingDash && fabs(toDegrees(ballAngle)) < 40 && distanceToDestination > 250.0)
  )
  {
    motionRequest.motionType = MotionRequest::walk;
    motionRequest.walkType = MotionRequest::dash;
    motionRequest.walkParams.translation.x = (distanceToDestination > 400.0) ? 500.0 : distanceToDestination * 500.0 / 400.0;
    motionRequest.walkParams.rotation = angle;
    if (motionRequest.walkParams.rotation > fromDegrees(50.0)) motionRequest.walkParams.rotation = fromDegrees(50.0);
    if (motionRequest.walkParams.rotation < fromDegrees(-50.0)) motionRequest.walkParams.rotation = fromDegrees(-50.0);
    motionRequest.walkParams.translation.y = 0;
    if (!usingDash)
    {
      usingDash = true;
      lastChangeTime = SystemCall::getCurrentSystemTime();
    }
  }
  else if ((fabs(toDegrees(ballAngle))<10)&&(distanceToDestination<20))
  {
    motionRequest.motionType = MotionRequest::stand;
    usingDash = false;
  }
  else
  {
    motionRequest.motionType = MotionRequest::walk;
    motionRequest.walkType = MotionRequest::normal;

    motionRequest.walkParams.translation.x = 
      destination.x / estimatedTimeToReachDestination;
    motionRequest.walkParams.translation.y = 
      destination.y / estimatedTimeToReachDestination;

    motionRequest.walkParams.rotation = angle;

    if (motionRequest.walkParams.rotation > maxTurnSpeed)
      motionRequest.walkParams.rotation = maxTurnSpeed;
    if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
      motionRequest.walkParams.rotation = -maxTurnSpeed;
    if (usingDash)
    {
      usingDash = false;
      lastChangeTime = SystemCall::getCurrentSystemTime(); 
    }
  }
}

void DDD2004BasicBehaviorGetBehindBall::execute()
{
  // This was adapted from the XabslSkillGetBehindBall written by Max Risler.

  accelerationRestrictor.saveLastWalkParameters();

  double dir = fromDegrees(angle); 

  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);

  double destX = ball.x;
  double destY = ball.y;

  double posX = robotPose.getPose().translation.x;
  double posY = robotPose.getPose().translation.y;
  double posDir = robotPose.getPose().getAngle();

  Vector2<double> destPosDiff = ball - robotPose.getPose().translation;

  double absDestDir = atan2(destPosDiff.y, destPosDiff.x);
  double relDestDir = normalize(absDestDir - posDir);
  double destDist = destPosDiff.abs();

  if (destDist > 1000.0)
  {
    gotoPoint(destX,destY,maxSpeed);
  }
  else if (destDist >distance* 1.2)
  {
    // calculate destination point
    double px,py;
  
    px = destX -distance* cos(dir);
    py = destY -distance* sin(dir);
  
    double distToP = sqrt((posX-px)*(posX-px) + (posY-py)*(posY-py));
  
    if (destDist < distToP)
    {
      // dest is closer than destination point
      // walk to tangent
      double a = acos(distance/destDist);
    
      if ((leftRight == 0 && normalize(dir - absDestDir) > 0) ||
        leftRight > 0)
      {
        px = destX -distance* cos(absDestDir + a);
        py = destY -distance* sin(absDestDir + a);
      } else {
        px = destX -distance* cos(absDestDir - a);
        py = destY -distance* sin(absDestDir - a);
      }
      // distance is distance to tangent plus distance on perimeter
      distToP = sqrt((posX-px)*(posX-px) + (posY-py)*(posY-py));
      distToP +=distance* normalize(fabs((dir - absDestDir)));
    }
    gotoPoint(px,py,maxSpeed);
  }
  else
  {
    // circle around dest
    double diffAngle = normalize(posDir-dir);
    if (diffAngle > pi_2 && leftRight > 0) diffAngle -= pi2;
    if (diffAngle < -pi_2 && leftRight < 0) diffAngle += pi2;
  
    motionRequest.walkParams.translation.x = destDist - distance;
    motionRequest.walkParams.translation.y = diffAngle * distance;
    motionRequest.walkParams.rotation = relDestDir;
  }

    motionRequest.motionType = MotionRequest::walk;
    motionRequest.walkType = MotionRequest::normal;

  accelerationRestrictor.restrictAccelerations(300,300,180);
}

void DDD2004BasicBehaviorGetBehindBall::gotoPoint(double x, double y, double maxSpeed)
{
  double maxTurnSpeed = fromDegrees(60);

  double timeToReachX, timeToReachY, estimatedTimeToReachDestination;
  double distance = Geometry::distanceTo(robotPose.getPose(),Vector2<double>(x,y));
  double distanceX = distance
    * cos(Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)));
  double distanceY = distance
    * sin(Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)));

  if(distance == 0)
    distance = 1;
  double angle = Geometry::angleTo(robotPose.getPose(),ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted));

  Vector2<double> destination(distanceX, distanceY);

  timeToReachX = fabs(destination.x) / maxSpeed;
  timeToReachY = fabs(destination.y) / maxSpeed;

  estimatedTimeToReachDestination = 
    (timeToReachX < timeToReachY) ? timeToReachY : timeToReachX;

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;

  motionRequest.walkParams.translation.x = 
    destination.x / estimatedTimeToReachDestination;
  motionRequest.walkParams.translation.y = 
    destination.y / estimatedTimeToReachDestination;

  motionRequest.walkParams.rotation = angle;

  if (motionRequest.walkParams.rotation > maxTurnSpeed)
    motionRequest.walkParams.rotation = maxTurnSpeed;
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    motionRequest.walkParams.rotation = -maxTurnSpeed;
}



void DDD2004BasicBehaviorGoToPointAndAvoidObstacles::execute()
{
  double turnDistance = 400,
    angleToDestination = Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)),
		distanceToDestination = Geometry::distanceTo(robotPose.getPose(),Vector2<double>(x,y));
	double targetX, targetAngle;//, obstacleAvoidAngle;

  if (maxSpeed == 0) maxSpeed = 200;
	
  targetX = obstaclesModel.getDistanceInCorridor(0.0, 300);
	targetX -= 300; 
	if (targetX > maxSpeed)
		targetX = maxSpeed;
	if (targetX < -50)
		targetX = -50;
	if (obstaclesModel.getDistanceInCorridor(pi, 300) < 300)
		targetX = 0;
	
	speedPhi.setWeightP(0.7);
	speedX.setWeightP(0.7);
	targetAngle = 0.0;

  
	if (targetX > .4 * maxSpeed)
  {

		double leftForward = obstaclesModel.getTotalFreeSpaceInSector(.4, .79, turnDistance);
		double rightForward = obstaclesModel.getTotalFreeSpaceInSector(-.4, .79, turnDistance);

		// go in the direction of the next free sector ...
		targetAngle = obstaclesModel.getAngleOfNextFreeSector(pi/4, angleToDestination, (int )turnDistance);
		// ... but check if there's enough space for walking
		if (obstaclesModel.getDistanceInCorridor(targetAngle, 300) < turnDistance)
			targetAngle =	2*(leftForward - rightForward) / (leftForward + rightForward);

		if (fabs(targetAngle) < .2)
		{
			// only do this if no obstacle avoidance was necessary in the 
			// past 1 sec
			if (goToDestCounter > 10)
			{
				targetAngle = angleToDestination;
				//OUTPUT(idText, text, "DESTINATION: " << targetAngle);
			}
			// destination within reach
			else if ((obstaclesModel.getDistanceInCorridor(angleToDestination, 300) > distanceToDestination-100) &&
				(goToDestCounter > 5))
			{
				targetAngle = angleToDestination;
				//OUTPUT(idText, text, "GO FOR THE DESTINATION");
			}
			goToDestCounter++;
			targetAngle = (double)sgn(targetAngle) * .5;
		}
		else 
		{
			if (fabs(angleToDestination) < pi/1.8)
			{
				goToDestCounter = 0;
				//OUTPUT(idText, text, "AVOID: " << targetAngle);
			}
			if (fabs(targetAngle) > .5)
				targetAngle = (double)sgn(targetAngle) * .5;
		}
	}
  else 
  {
		if (stickToBackOffDirection > 15)
		{
			if (backOffAngle < 0)
				backOffAngle = -pi;
			else
				backOffAngle = pi;
			stickToBackOffDirection = 0;
		}
		targetAngle = backOffAngle;
		goToDestCounter = -10;

		//OUTPUT(idText, text, "BACKING OFF " << targetAngle);
  }
	stickToBackOffDirection++;
			
	//OUTPUT(idText, text, "... " << targetAngle);
	backOffAngle = targetAngle;

	if(obstaclesModel.getDistanceInCorridor((double)sgn(targetAngle) * pi/2, 300) < 300)
		targetAngle = 0;  
	
	motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = speedX.approximateVal(targetX);
  motionRequest.walkParams.translation.y = speedY.getVal();
  motionRequest.walkParams.rotation = speedPhi.approximateVal(targetAngle);
}








void DDD2004BasicBehaviorAvoidObstacles::execute()
{
  if(turnDistance == 0) turnDistance = 300;
  if(walkSpeed == 0) walkSpeed = 250;
  if(turnSpeed == 0) turnSpeed = 1.2;

  double distanceToObstacle = 200;
  if(stopDistance) distanceToObstacle = stopDistance;

  static PIDsmoothedValue speedPhi(0, .1, 0, 0, -pi/4, pi/4, pi/10);
  static PIDsmoothedValue speedX(0, .1, 0, 0, -walkSpeed, walkSpeed, 100);
  static PIDsmoothedValue speedY(0, .1, 0, 0, -walkSpeed/2, walkSpeed/2, 100);
  static double turnDirection = 0;
  speedX.setMax(walkSpeed);

  speedX.approximateVal(obstaclesModel.corridorInFront - distanceToObstacle);

/*    
  // if something is in the way but far away
  // or it is close and turning was not set...
  if(((obstaclesModel.distance[ObstaclesModel::front] < turnDistance) && 
    (obstaclesModel.distance[ObstaclesModel::front] >= distanceToObstacle)) 
    ||
    ((obstaclesModel.distance[ObstaclesModel::front] < turnDistance) && 
    (turnDirection == 0)))
  {
    if(obstaclesModel.distance[ObstaclesModel::frontLeft] < obstaclesModel.distance[ObstaclesModel::frontRight])
      turnDirection = -turnSpeed;
    else 
      turnDirection = turnSpeed;
  }
  // if nothing is in the way, walk forward
  else if (obstaclesModel.distance[ObstaclesModel::front] > turnDistance) 
  {
  }
*/  
  turnDirection = 0;
  if(obstaclesModel.corridorInFront < turnDistance) 
  {
    if(obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) < obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontRight))
      speedY.approximateVal(-walkSpeed);
    else if (obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) > obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft))
      speedY.approximateVal(walkSpeed);
    else if (obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) == obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontRight))
      speedY.approximateVal(0);
  }
  else  
    speedY.approximateVal(0);

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = speedX.getVal();
  motionRequest.walkParams.translation.y = speedY.getVal();
  motionRequest.walkParams.rotation = speedPhi.approximateVal(turnDirection);
}


void DDD2004BasicBehaviorGoToPoint::execute()
{
  accelerationRestrictor.saveLastWalkParameters();

  Vector2<double> destination(x,y);
  const Vector2<double>& self = robotPose.translation;

  if (maxSpeed==0) maxSpeed = 120;

  double maxTurnSpeed = fromDegrees(40);

  double distanceToDestination = Geometry::distanceTo(self,destination);

  double angleDifference = normalize(fromDegrees(destinationAngle) - robotPose.rotation);

  double angleToDestination = Geometry::angleTo(robotPose,destination);


  motionRequest.motionType = MotionRequest::walk;

  switch((int)walkType)
  {
    case 0: motionRequest.walkType = MotionRequest::normal; break;
    case 1: motionRequest.walkType = MotionRequest::upsideDown; break;
    case 2: motionRequest.walkType = MotionRequest::turnWithBall; break;
    case 3: motionRequest.walkType = MotionRequest::turnWithBall2; break;
    case 4: motionRequest.walkType = MotionRequest::turnWithBall3; break;
    case 5: motionRequest.walkType = MotionRequest::turnWithBall4; break;
    case 6: motionRequest.walkType = MotionRequest::turnWithBall5; break;
    case 7: motionRequest.walkType = MotionRequest::dash; break;
  }

  //this time does not necessarily include time for turning!:
  double estimatedTimeToReachDestination;
  if (distanceToDestination > 200)
  {
    estimatedTimeToReachDestination = (distanceToDestination+200)/ maxSpeed;
  
    motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
    motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  }
  else
  {
    estimatedTimeToReachDestination = 2*distanceToDestination / maxSpeed;
    if (distanceToDestination > 30)
    {
      motionRequest.walkParams.translation.x = 
        cos(angleToDestination) * maxSpeed*distanceToDestination/200;
      motionRequest.walkParams.translation.y = 
        sin(angleToDestination) * maxSpeed*distanceToDestination/200;
    }
    else
    {
      motionRequest.walkParams.translation.x = 0;
      motionRequest.walkParams.translation.y = 0;
    }
  }
  if (estimatedTimeToReachDestination==0)
  {
    estimatedTimeToReachDestination = 0.001;
  }

  if (fabs(toDegrees(angleDifference)) > 20)
  {
    motionRequest.walkParams.rotation = 
      angleDifference / estimatedTimeToReachDestination;
    if (motionRequest.walkParams.rotation > maxTurnSpeed)
    {
      motionRequest.walkParams.rotation = maxTurnSpeed;
    }
    if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    {
      motionRequest.walkParams.rotation = -maxTurnSpeed;
    }
  }
  else
  {
    motionRequest.walkParams.rotation = 2*angleDifference;
  }



  if ((fabs(toDegrees(angleDifference))<4)&&(distanceToDestination<25))
  {
    motionRequest.walkParams.translation.x = 0;
    motionRequest.walkParams.translation.y = 0;
    motionRequest.walkParams.rotation = 0;
  }

  accelerationRestrictor.restrictAccelerations(150,150,100);

  if (fabs(motionRequest.walkParams.translation.x) < 10
    && fabs(motionRequest.walkParams.translation.y) < 10
    && fabs(motionRequest.walkParams.rotation) < fromDegrees(10)
    && fabs(distanceToDestination) < 40
    && fabs(angleToDestination) < fromDegrees(5))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

void DDD2004BasicBehaviorGoToPointPrecisely::execute()
{
  Pose2D pose = robotPose + Pose2D(Vector2<double>(-105,0));

  accelerationRestrictor.saveLastWalkParameters();

  Vector2<double> destination = (Pose2D(fromDegrees(destinationAngle), Vector2<double>(x,y)) + Pose2D(Vector2<double>(-105,0))).translation;
  const Vector2<double>& self = pose.translation;

  if (maxSpeed==0) maxSpeed = 120;

  double maxTurnSpeed = fromDegrees(40);

  double distanceToDestination = Geometry::distanceTo(self,destination);

  double angleDifference = normalize(fromDegrees(destinationAngle) - pose.rotation);

  double angleToDestination = Geometry::angleTo(pose,destination);


  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;

  //this time does not necessarily include time for turning!:
  double estimatedTimeToReachDestination;
  if (distanceToDestination > 200)
  {
    estimatedTimeToReachDestination = (distanceToDestination+200)/ maxSpeed;
  
    motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
    motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  }
  else
  {
    estimatedTimeToReachDestination = 2*distanceToDestination / maxSpeed;
    motionRequest.walkParams.translation.x = 
      cos(angleToDestination) * maxSpeed*distanceToDestination/200;
    motionRequest.walkParams.translation.y = 
      sin(angleToDestination) * maxSpeed*distanceToDestination/200;
    const double minSpeed = 30;
    if(fabs(motionRequest.walkParams.translation.x) > fabs(motionRequest.walkParams.translation.y) &&
       fabs(motionRequest.walkParams.translation.x) < minSpeed)
    {
      motionRequest.walkParams.translation.x = motionRequest.walkParams.translation.x > 0 ? minSpeed : -minSpeed;
      motionRequest.walkParams.translation.x *= minSpeed / fabs(motionRequest.walkParams.translation.x);
    }
    else if(fabs(motionRequest.walkParams.translation.x) <= fabs(motionRequest.walkParams.translation.y) &&
            fabs(motionRequest.walkParams.translation.y) < minSpeed)
    {
      motionRequest.walkParams.translation.y = motionRequest.walkParams.translation.y > 0 ? minSpeed : -minSpeed;
      motionRequest.walkParams.translation.x *= minSpeed / fabs(motionRequest.walkParams.translation.y);
    }

  }
  if (estimatedTimeToReachDestination==0)
  {
    estimatedTimeToReachDestination = 0.001;
  }

  if (fabs(toDegrees(angleDifference)) > 20)
  {
    motionRequest.walkParams.rotation = 
      angleDifference / estimatedTimeToReachDestination;
    if (motionRequest.walkParams.rotation > maxTurnSpeed)
    {
      motionRequest.walkParams.rotation = maxTurnSpeed;
    }
    if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    {
      motionRequest.walkParams.rotation = -maxTurnSpeed;
    }
  }
  else
  {
    motionRequest.walkParams.rotation = 2*angleDifference;
  }

  accelerationRestrictor.restrictAccelerations(150,150,100);

  if (fabs(motionRequest.walkParams.translation.x) < 10
    && fabs(motionRequest.walkParams.translation.y) < 10
    && fabs(motionRequest.walkParams.rotation) < fromDegrees(10)
    && fabs(distanceToDestination) < 40
    && fabs(angleToDestination) < fromDegrees(5))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

/*
* Change Log
* 
* $Log: DDD2004SimpleBasicBehaviors.cpp,v $
* Revision 1.7  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.6  2004/04/07 13:44:33  risler
* ddd checkin after go04 - second part fixed walktype
*
* Revision 1.5  2004/04/07 12:28:57  risler
* ddd checkin after go04 - first part
*
* Revision 1.6  2004/04/04 08:46:11  risler
* dash turning added
*
* Revision 1.5  2004/04/02 23:43:16  risler
* slowdown before stopping dash
*
* Revision 1.4  2004/04/02 02:22:42  risler
* go-to-ball dash fixed and range increased
*
* Revision 1.3  2004/04/01 22:39:29  risler
* added go-to-point.walk-type
*
* Revision 1.2  2004/03/31 18:19:04  risler
* new go-to-ball basic behaviors using dash walk type
*
* Revision 1.1.1.1  2004/03/29 08:28:49  Administrator
* initial transfer from tamara
*
* Revision 1.4  2004/02/03 13:20:47  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.3  2003/10/28 13:57:45  dueffert
* unused evolution removed
*
* Revision 1.2  2003/10/28 13:25:52  dueffert
* spelling improved
*
* Revision 1.1  2003/10/26 22:49:38  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

