/**
* @file DDD2004BehaviorControl.cpp
* 
* Implementation of class DDD2004BehaviorControl.
*
* @author Martin Ltzsch
*/

#include "DDD2004BehaviorControl.h"
#include "Tools/Player.h"

DDD2004BehaviorControl::DDD2004BehaviorControl(const BehaviorControlInterfaces& interfaces)
: Xabsl2BehaviorControl(interfaces,SolutionRequest::ddd2004),
ballSymbols(interfaces),
configurationSymbols(interfaces,configuration),
headAndTailSymbols(interfaces),
joystickSymbols(interfaces),
ledAndSoundSymbols(interfaces),
mathFunctions(interfaces),
motionRequestSymbols(interfaces),
roboCupGameManagerSymbols(interfaces),
robotPoseSymbols(interfaces),
robotStateSymbols(interfaces),
specialVisionSymbols(interfaces),
strategySymbols(interfaces),
challengeSymbols(interfaces),
simpleBasicBehaviors(interfaces,errorHandler),
continuousBasicBehaviors(interfaces,configuration,errorHandler),
commonBasicBehaviors(interfaces,errorHandler),
lastSound(SoundRequest::none)
{
  Xabsl2FileInputSource file("Xabsl2/ddd04-ic.dat");
  init(file);
}

DDD2004BehaviorControl::~DDD2004BehaviorControl()
{
}

void DDD2004BehaviorControl::registerSymbolsAndBasicBehaviors()
{
  simpleBasicBehaviors.registerBasicBehaviors(*pEngine);
  continuousBasicBehaviors.registerBasicBehaviors(*pEngine);
  commonBasicBehaviors.registerBasicBehaviors(*pEngine);
  
  ballSymbols.registerSymbols(*pEngine);
  configurationSymbols.registerSymbols(*pEngine);
  headAndTailSymbols.registerSymbols(*pEngine);
  joystickSymbols.registerSymbols(*pEngine);
  ledAndSoundSymbols.registerSymbols(*pEngine);
  mathFunctions.registerSymbols(*pEngine);
  motionRequestSymbols.registerSymbols(*pEngine);
  roboCupGameManagerSymbols.registerSymbols(*pEngine);
  robotPoseSymbols.registerSymbols(*pEngine);
  robotStateSymbols.registerSymbols(*pEngine);
  specialVisionSymbols.registerSymbols(*pEngine);
  strategySymbols.registerSymbols(*pEngine);
  challengeSymbols.registerSymbols(*pEngine);
}

void DDD2004BehaviorControl::execute()
{
  for(int i = 0; i < 14; i++) ledRequest.faceLED[i] = LEDRequest::oooo;
  ledRequest.backFrontWhiteLED = LEDRequest::oooo;
  ledRequest.backMiddleWhiteLED = LEDRequest::oooo;
  ledRequest.backRearWhiteLED = LEDRequest::oooo;
  ledRequest.backMiddleOrangeLED = LEDRequest::oooo;

  ledRequest.backFrontBlueLED = 
    getPlayer().getTeamColor() == Player::blue ? LEDRequest::llll : LEDRequest::oooo;
  ledRequest.backRearRedLED = 
    getPlayer().getTeamColor() == Player::red ? LEDRequest::llll : LEDRequest::oooo;

  ballSymbols.update();
  configurationSymbols.update();
  roboCupGameManagerSymbols.update();
  robotPoseSymbols.update();
  strategySymbols.update();
  
  continuousBasicBehaviors.update();
  
  // set the outgoint behavior team message to none
  outgoingBehaviorTeamMessage.message = BehaviorTeamMessage::none;
  
    // Set the tail request depending on whether the ball was seen
  if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 120)
  {
    // ball seen now
    motionRequest.tailRequest = MotionRequest::wagHorizontalFast;
  }
  else if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 2000)
  {
    // ball seen
    motionRequest.tailRequest = MotionRequest::wagHorizontal;
  }
  else if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) 
    > BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
    && SystemCall::getTimeSince(ballPosition.communicated.timeWhenLastObserved) < 3000)
  {
    // ball known
    motionRequest.tailRequest = MotionRequest::wagVertical;
  }
  else
  {
    // ball not known
    motionRequest.tailRequest = MotionRequest::noTailWag;
  }
  
  // set head control mode lookToStars to detect missing settings
  headControlMode.headControlMode = HeadControlMode::lookToStars;

    
  // execute the engine
  executeEngine();

  if((robotState.getState() == RobotState::crashed)||
     (robotState.getState() == RobotState::rollLeft)||
     (robotState.getState() == RobotState::rollRight))
  {
    if ((motionRequest.motionType == MotionRequest::walk &&
      motionRequest.walkType != MotionRequest::upsideDown)||
      (motionRequest.motionType == MotionRequest::stand))
    {
      motionRequest.motionType = MotionRequest::getup;
      return;
    }
  }

  SoundRequest::SoundID currentSound = soundRequest.soundID;
  
  if (
    headControlMode.headControlMode == HeadControlMode::lookToStars ||
      (headControlMode.headControlMode != HeadControlMode::searchForBall &&
       headControlMode.headControlMode != HeadControlMode::searchAuto &&
       headControlMode.headControlMode != HeadControlMode::searchForObstacles &&
       headControlMode.headControlMode != HeadControlMode::lookStraightAhead &&
       headControlMode.headControlMode != HeadControlMode::lookBetweenFeet &&
       headControlMode.headControlMode != HeadControlMode::lookLeft &&
       headControlMode.headControlMode != HeadControlMode::lookRight &&
       headControlMode.headControlMode != HeadControlMode::catchBall)
    )
  {
    soundRequest.soundID = SoundRequest::rob101;
    OUTPUT(idText,text,HeadControlMode::getHeadControlModeName(headControlMode.headControlMode));
  }
  else if (
    strategySymbols.role == BehaviorTeamMessage::striker &&
    currentSound == lastSound)
  {
    if (strategySymbols.robotIsStuck)
    {
      soundRequest.soundID = SoundRequest::bing03;
    }
    else if (strategySymbols.obstaclesAreClose)
    {
      soundRequest.soundID = SoundRequest::bing01;
    }
    else if (robotState.getConsecutiveCollisionTimeAggregate() > 2000)
    {
      soundRequest.soundID = SoundRequest::bing05;
    }
  }

  lastSound = currentSound;

  // set the head led depening on the gamestate
  switch (outgoingBehaviorTeamMessage.gameState)
  {
    case BehaviorTeamMessage::initial:
    default:
      ledRequest.headWhiteLED = LEDRequest::oooo;
      ledRequest.headOrangeLED = LEDRequest::oooo;
      break;
    case BehaviorTeamMessage::ready:
      ledRequest.headWhiteLED = LEDRequest::llll;
      ledRequest.headOrangeLED = LEDRequest::oooo;
      break;
    case BehaviorTeamMessage::set:
      ledRequest.headWhiteLED = LEDRequest::lloo;
      ledRequest.headOrangeLED = LEDRequest::oooo;
      break;
    case BehaviorTeamMessage::playing:
      ledRequest.headWhiteLED = LEDRequest::oooo;
      ledRequest.headOrangeLED = LEDRequest::llll;
      break;
    case BehaviorTeamMessage::penalized:
      ledRequest.headWhiteLED = LEDRequest::oooo;
      ledRequest.headOrangeLED = LEDRequest::lloo;
      break;
      
  // set the back led depending on the dynamic role/kickoff
  if (outgoingBehaviorTeamMessage.gameState != BehaviorTeamMessage::initial &&
      outgoingBehaviorTeamMessage.gameState != BehaviorTeamMessage::ready &&
      outgoingBehaviorTeamMessage.gameState != BehaviorTeamMessage::set)
    switch (strategySymbols.role)
    {
      case BehaviorTeamMessage::striker:
        ledRequest.backMiddleWhiteLED = LEDRequest::oooo;
        ledRequest.backMiddleOrangeLED = LEDRequest::llll;
        break;
      case BehaviorTeamMessage::offensiveSupporter:
        ledRequest.backMiddleWhiteLED = LEDRequest::llll;
        ledRequest.backMiddleOrangeLED = LEDRequest::oooo;
        break;
      case BehaviorTeamMessage::defensiveSupporter:
        ledRequest.backMiddleWhiteLED = LEDRequest::lloo;
        ledRequest.backMiddleOrangeLED = LEDRequest::oooo;
        break;
      case BehaviorTeamMessage::goalie:
      default:
        ledRequest.backMiddleWhiteLED = LEDRequest::oooo;
        ledRequest.backMiddleOrangeLED = LEDRequest::oooo;
        break;
    }
  }
}

bool DDD2004BehaviorControl::handleMessage(InMessage& message)
{
  return (configuration.handleMessage(message) ||
    Xabsl2BehaviorControl::handleMessage(message));
}


/*
* Change log :
* 
* $Log: DDD2004BehaviorControl.cpp,v $
* Revision 1.7  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
* Revision 1.6  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.5  2004/04/08 10:08:41  risler
* fixed and unified team color symbols
*
* Revision 1.4  2004/04/07 12:28:54  risler
* ddd checkin after go04 - first part
*
* Revision 1.8  2004/04/02 23:44:39  dthomas
* added new invKinParams for turn with ball
* integrated into rotate-around-ball
* combined with updated catch-ball head-control-mode
*
* Revision 1.7  2004/04/02 00:52:17  risler
* fixed head-control-modes
*
* Revision 1.6  2004/04/01 22:41:29  risler
* removed led settings
*
* Revision 1.5  2004/04/01 20:32:31  dthomas
* added debug-message for wrong or missing head-control-mode
*
* Revision 1.4  2004/04/01 18:48:02  risler
* various led stuff
*
* Revision 1.3  2004/03/31 18:17:45  risler
* role led changed
*
* Revision 1.2  2004/03/31 00:10:13  risler
* implemented new kickoff rules
*
* Revision 1.1.1.1  2004/03/29 08:28:49  Administrator
* initial transfer from tamara
*
* Revision 1.3  2004/03/08 01:06:47  roefer
* Interfaces should be const
*
* Revision 1.2  2004/02/03 13:20:47  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.1  2003/10/26 22:49:37  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

