/** 
* @file GT2003ContinuousRules/AvoidBorder.h
*
* @author Max Risler
* @author Ronnie Brunn
*/

#ifndef __GT2003AvoidBorder_h_
#define __GT2003AvoidBorder_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace GT2003ContinuousRules
{

/**@class AvoidBorder
 * rule which avoids collision with field borders
 */
class AvoidBorder: public ContinuousRule{
public:

  /** constructor
   */
  AvoidBorder(const BehaviorControlInterfaces& interfaces,
              double maxForce = 1.5, 
              double maxForceLength = 0.0, 
              double influenceLength = 150.0)
      :ContinuousRule(interfaces, "AvoidBorder"), 
       maxForce(maxForce), 
       maxForceLength(maxForceLength),
       influenceLength(influenceLength) 
  {
    xPosInfluenceArea=((xPosOpponentGroundline+xPosOpponentSideCorner)/2.0)-sqrt((influenceLength*influenceLength)/2.0);
    yPosInfluenceArea=((yPosLeftGroundline+yPosLeftSideline)/2.0)-sqrt((influenceLength*influenceLength)/2.0);
  };

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
	 * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra,
                       double& rweight)
  {
    if ((fabs(robotPose.translation.x) <= xPosInfluenceArea) &&
        (fabs(robotPose.translation.y) <= yPosInfluenceArea))
    { // no influence 
      walk.x=walk.y=ra=rweight=0;
      return;
    }

    double distance = min (
      FieldDimensions::distanceToBorder ( robotPose.translation ),
      FieldDimensions::distanceToBorder ( robotPose*Vector2<double>(-bodyLength,0) ) );

    walk = -robotPose.translation;
    walk.normalize();

    walk *= getRepellingForce(distance, maxForce, maxForceLength, influenceLength);

    ra=0;
    rweight=0;
  }
  
protected:
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is maxForce (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;
  /** the maximum absolute x-Position up to which there is no influence by groundline, goal and corner */
  double xPosInfluenceArea;
  /** the maximum absolute y-Position up to which there is no influence by sideline and corner */
  double yPosInfluenceArea;

};

}

#endif 


/*
* Change log:
*
* $Log: AvoidBorder.h,v $
* Revision 1.3  2004/03/08 01:06:50  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/22 22:18:45  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:30  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.9  2003/06/04 11:17:35  risler
* reduced radius
*
* Revision 1.8  2003/05/22 06:48:02  dueffert
* extra ; warning removed
*
* Revision 1.7  2003/05/21 11:46:03  brunn
* added influence area
*
* Revision 1.6  2003/05/21 11:43:31  risler
* added namespaces to continuous rules to avoid problems with ambiguous class names
*
* Revision 1.5  2003/05/12 09:50:33  risler
* doxygen comments
*
* Revision 1.4  2003/05/12 00:03:28  dueffert
* doxygen bugs fixed
*
* Revision 1.3  2003/05/11 14:39:22  roefer
* Warnings removed
*
* Revision 1.2  2003/05/09 15:25:08  risler
* ContinuousBasicBehavior uses Xabsl2Array,
* ContinuousRules inherit from Xabsl2NamedItem
*
* Revision 1.1  2003/05/07 13:02:33  risler
* added first continuous basic behaviors to GT2003BehaviorControl
*
* Revision 1.2  2003/05/07 11:05:55  risler
* ContinuousBasicBehaviors:
* removed classes Attracting/RepellingFieldRule, replaced with getAttracting/RepellingForce functions
* some clean up
*
* Revision 1.1  2003/05/06 17:03:43  kallnik
* ContinuousBasicBehaviors code review
*
* Revision 1.2  2003/04/25 12:57:14  kallnik
* new version of ContinuousBasicBehavior released
*
* Revision 1.1  2003/04/15 15:52:11  risler
* DDD GO 2003 code integrated
*
* Revision 1.4  2003/04/07 22:01:12  dthomas
* updated: rules with linear function and fixed rotation limit
*
* Revision 1.3  2003/04/07 01:25:53  dthomas
* modified: one converted to double in distance
* modified: changed factor and influence in avoid teammate/opponents
* other rebalancing
*
* Revision 1.2  2003/04/04 10:59:17  max
* moved field dimensions function in separate class
* ddd-soccer: added option dribble-to-goal
*
* Revision 1.1  2003/04/03 14:43:43  max
* added border distance function to fielddimensions
* changed goto cont rules to use parabolic function
*
*/
