/** 
* @file GT2003ContinuousRules/AvoidObstaclesAheadTurning.h
*
* @author Matthias Jngel
*/

#ifndef __GT2003AvoidObstaclesAheadTurning_h_
#define __GT2003AvoidObstaclesAheadTurning_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/Debugging/DebugDrawings.h"

namespace GT2003ContinuousRules
{
  
/**@class AvoidObstaclesAheadTurning
* rule which avoids collision with the obstacles
  */
  class AvoidObstaclesAheadTurning: public ContinuousRule{
  public:
    
  /** constructor
    */
    AvoidObstaclesAheadTurning(const BehaviorControlInterfaces& interfaces,
      double maxForce = 1.5, 
      double maxForceLength = 100.0, 
      double influenceLength = 450.0)
      :ContinuousRule(interfaces, "AvoidObstaclesAheadTurning"),
      maxForce(maxForce),
      maxForceLength(maxForceLength),
      influenceLength(influenceLength)
    {};
    
    /**
    * Returns whether this rule generates absolute or robot relative coordinates.
    * This rule is robot relative.
    */
    virtual bool isRelative() {return true;}
    
    /** executes the basic behavior rule
    * @param robotPose the current robots pose at which the rule is to be evaluated
    * @param walk the direction and speed of the suggested motion coded as an vector (output)
    * @param ra the rotation angle, the direction the robot should be directed (output)
    * @param rweight the rotation weight the weight of the suggested rotation (output)
    */
    virtual void execute(const RobotPose& robotPose,
      Vector2<double>& walk,
      double& ra, double& rweight)
    {
      ra = 0;
      rweight = 0.0;
      walk.x = 0;
      walk.y = 0;
      if(obstaclesModel.corridorInFront < influenceLength) 
      {
        if(obstaclesModel.getAngleOfNextFreeSector(pi/4, 0, (int)influenceLength) < 0)
          ra = -pi_2;
        else
          ra = pi_2;
        
        walk.x = -getRepellingForce(obstaclesModel.corridorInFront, maxForce, maxForceLength, influenceLength);
        rweight = 1.0;
      }
    };
    
  protected:
    /** the maximum length of the result vector*/
    double maxForce;
    /** the distance below which the result vector length is maxForce (in mm)*/
    double maxForceLength;
    /** the distance above which the result vector length is 0 (in mm)*/
    double influenceLength;
  };
  
}


#endif 


/*
* Change log:
*
* $Log: AvoidObstaclesAheadTurning.h,v $
* Revision 1.3  2004/03/08 01:06:51  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/22 22:18:45  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/01 10:20:11  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/06/27 15:08:22  juengel
* Reduced influenceLength.
*
* Revision 1.5  2003/06/17 20:01:22  juengel
* Renamed getNextFreeSector to getAngleOfNextFreeSector.
*
* Revision 1.4  2003/06/12 16:52:05  juengel
* Increased influenceLength.
*
* Revision 1.3  2003/06/05 18:08:19  juengel
* Turns slower now.
*
* Revision 1.2  2003/06/05 08:06:49  juengel
* improved
*
* Revision 1.1  2003/06/03 13:08:03  juengel
* Added cont-walk-around-avoiding-obstacles.
*
*
*/
