/**
* @file GT2003ContinuousRules/GoToPoint.h
*
* @author Dirk Thomas
*/

#ifndef __GT2003GoToPoint_h_
#define __GT2003GoToPoint_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace GT2003ContinuousRules
{

/**@class GoToPoint
 */
class GoToPoint: public ContinuousRule{
private:
  /** reference to the point to go to */
  const double &pointX;
  const double &pointY;

  /** the point is in relative coordinates */
  bool relative;

  /** if the rule not constructed via a reference to the point
   * these variables are used to store the coordinates */
  double localPointX;
  double localPointY;

	/** the distance above which the result vector is maxForce in mm*/
  double maxForceLength;
  /** the distance below which the result vector is 0 in mm*/
  double minDistance;
  /** the maximum attracting force */
  double maxForce;


public:

  /** constructor
   */
  GoToPoint(
    const BehaviorControlInterfaces& interfaces,
    const Vector2<double>& point,
    bool relative = false,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0)
    :
    ContinuousRule(interfaces,"GoToPoint"),
    pointX(point.x),
    pointY(point.y),
    relative(relative),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength)
  {};

  /** constructor
   */
  GoToPoint(
    const BehaviorControlInterfaces& interfaces,
    const double& pointX,
    const double& pointY,
    bool relative = false,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0)
    :
    ContinuousRule(interfaces, "GoToPoint"),
    pointX(pointX),
    pointY(pointY),
    relative(relative),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength)
  {};

  /** constructor
   */
  GoToPoint(
    const BehaviorControlInterfaces& interfaces,
    const Vector2<int> point,
    bool relative = false,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0)
    :
    ContinuousRule(interfaces,"GoToPoint"),
    pointX(localPointX),
    pointY(localPointY),
    relative(relative),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength)
  {
      localPointX = (double)point.x;
      localPointY = (double)point.y;
  };

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   */
  virtual bool isRelative() {return relative;}

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {
    walk = Vector2<double>(pointX, pointY);

    if (!relative) 
      walk -= robotPose.translation;

    double distance = walk.abs();
    walk.normalize();

    walk *= getAttractingForce(distance, maxForce, minDistance, maxForceLength);

    rweight=0;
  }
};

}


#endif //__GoToPoint_h_


/*
* Change log:
*
* $Log: GoToPoint.h,v $
* Revision 1.3  2004/03/08 01:06:54  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/22 22:18:46  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.11  2003/06/17 18:28:45  thomas
* added: goalie cont-behaviours, return-state, etc.
*
* Revision 1.10  2003/06/03 10:51:24  risler
* ContinuousRules can be in relative coordinates
* no rotation rule active => no rotation
*
* Revision 1.9  2003/05/29 12:30:24  risler
* go to point should have default max force of 1.0
*
* Revision 1.8  2003/05/22 06:48:02  dueffert
* extra ; warning removed
*
* Revision 1.7  2003/05/21 11:43:31  risler
* added namespaces to continuous rules to avoid problems with ambiguous class names
*
* Revision 1.6  2003/05/14 09:05:53  risler
* added relative position to GoToPoint/TurnToPoint rules
*
* Revision 1.5  2003/05/13 15:39:56  risler
* goToPoint rule has no rotation
* added rules turnToPoint, turnToAngle
* added basic behavior go-forward-to-point
* added parameter angle to basic behavior go-to-point
*
* Revision 1.4  2003/05/12 09:50:33  risler
* doxygen comments
*
* Revision 1.3  2003/05/09 15:25:08  risler
* ContinuousBasicBehavior uses Xabsl2Array,
* ContinuousRules inherit from Xabsl2NamedItem
*
* Revision 1.2  2003/05/08 01:09:00  risler
* bugfix: gotopoint target point was no reference
*
* Revision 1.1  2003/05/07 13:02:33  risler
* added first continuous basic behaviors to GT2003BehaviorControl
*
* Revision 1.4  2003/05/07 11:05:55  risler
* ContinuousBasicBehaviors:
* removed classes Attracting/RepellingFieldRule, replaced with getAttracting/RepellingForce functions
* some clean up
*
* Revision 1.3  2003/05/07 10:46:38  kallnik
* warnings removed
*
* Revision 1.2  2003/05/07 10:02:08  kallnik
* ContinuousBasicBehavior:
* -DebugDrawings fixed
* -copyfiles: do no copy Config/ContBeh (files deleted)
*
* Revision 1.1  2003/05/06 17:03:43  kallnik
* ContinuousBasicBehaviors code review
*
* Revision 1.3  2003/05/01 17:09:07  loetzsch
* Redesign of ball modeling:
* - Modularized class BallPosition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*
* Revision 1.2  2003/04/25 12:57:13  kallnik
* new version of ContinuousBasicBehavior released
*
* Revision 1.1  2003/04/15 15:52:11  risler
* DDD GO 2003 code integrated
*
* Revision 1.9  2003/04/11 23:21:37  dialtone
* Added Kickoption for Offenciv Supporter, changed Supporter Config (removed Avoidball) and  minimum distance in goto Ball
*
* Revision 1.8  2003/04/11 04:14:25  max
* increased turn speed
*
* Revision 1.7  2003/04/10 21:46:20  max
* increased turn speed
*
* Revision 1.6  2003/04/07 22:01:12  dthomas
* updated: rules with linear function and fixed rotation limit
*
* Revision 1.5  2003/04/07 19:40:14  max
* constants optimized
*
* Revision 1.4  2003/04/07 18:41:41  max
* no message
*
* Revision 1.3  2003/04/07 18:26:59  max
* corrected rotation speed limit
*
* Revision 1.2  2003/04/07 17:22:16  max
* new gotoball
*
* Revision 1.1  2003/04/07 17:08:04  dthomas
* added gobehindballorthogonaltoborder, gotoball
*
*
*/
