/**
* @file GT2003ContinuousRules/TurnToPoint.h
*
* @author Max Risler
*/

#ifndef __GT2003TurnToPoint_h_
#define __GT2003TurnToPoint_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"

namespace GT2003ContinuousRules
{

/**@class TurnToPoint
 */
class TurnToPoint: public ContinuousRule{
private:
  /** reference to the point to turn to */
  const double &pointX;
  const double &pointY;

  /** the point is in relative coordinates */
  bool relative;
  
public:

  /** constructor
   */
  TurnToPoint(
    const BehaviorControlInterfaces& interfaces,
    const Vector2<double>& point,
    bool relative = false)
    :
    ContinuousRule(interfaces,"TurnToPoint"),
    pointX(point.x),
    pointY(point.y),
    relative(relative)
  {};

  /** constructor
   */
  TurnToPoint(
    const BehaviorControlInterfaces& interfaces,
    const double& pointX,
    const double& pointY,
    bool relative = false)
    :
    ContinuousRule(interfaces, "TurnToPoint"),
    pointX(pointX),
    pointY(pointY),
    relative(relative)
  {};

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   */
  virtual bool isRelative() {return relative;}

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    walk.x = walk.y = 0;

    if (relative)
      ra = Vector2<double>(pointX, pointY).angle();
    else
      ra = (Vector2<double>(pointX, pointY) - robotPose.translation).angle();

    rweight=1.0;
  }
};

}


#endif //__GT2003TurnToBall_h_


/*
* Change log:
*
* $Log: TurnToPoint.h,v $
* Revision 1.3  2004/03/08 01:06:55  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/22 22:18:46  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/06/03 10:51:24  risler
* ContinuousRules can be in relative coordinates
* no rotation rule active => no rotation
*
* Revision 1.5  2003/05/22 06:48:02  dueffert
* extra ; warning removed
*
* Revision 1.4  2003/05/21 11:43:31  risler
* added namespaces to continuous rules to avoid problems with ambiguous class names
*
* Revision 1.3  2003/05/19 10:26:19  risler
* added max speeds to continuousbasicbehaviors
*
* Revision 1.2  2003/05/14 09:05:53  risler
* added relative position to GoToPoint/TurnToPoint rules
*
* Revision 1.1  2003/05/13 15:39:56  risler
* goToPoint rule has no rotation
* added rules turnToPoint, turnToAngle
* added basic behavior go-forward-to-point
* added parameter angle to basic behavior go-to-point
*
*/
