/**
* @file GT2003ContinuousRules/TurnToPointGoalie.h
*
* @author Dirk Thomas
*/

#ifndef __GT2003TurnToPointGoalie_h_
#define __GT2003TurnToPointGoalie_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"

namespace GT2003ContinuousRules
{

/**@class TurnToPointGoalie
 * with maximum angle of 90 degrees
 */
class TurnToPointGoalie: public ContinuousRule{
private:
  /** reference to the point to turn to */
  const double &pointX;
  const double &pointY;

  /** the point is in relative coordinates */
  bool relative;
  
public:

  /** constructor
   */
  TurnToPointGoalie(
    const BehaviorControlInterfaces& interfaces,
    const Vector2<double>& point,
    bool relative = false)
    :
    ContinuousRule(interfaces,"TurnToPointGoalie"),
    pointX(point.x),
    pointY(point.y),
    relative(relative)
  {};

  /** constructor
   */
  TurnToPointGoalie(
    const BehaviorControlInterfaces& interfaces,
    const double& pointX,
    const double& pointY,
    bool relative = false)
    :
    ContinuousRule(interfaces, "TurnToPointGoalie"),
    pointX(pointX),
    pointY(pointY),
    relative(relative)
  {};

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   */
  virtual bool isRelative() {return relative;}

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    walk.x = walk.y = 0;

    if (relative)
      ra = Vector2<double>(pointX, pointY).angle();
    else
      ra = (Vector2<double>(pointX, pointY) - robotPose.translation).angle();

		if (ra > 0.0)
		{
			if (robotPose.getAngle() + ra > pi_2)
				ra = pi_2 - robotPose.getAngle();
		}
		else
		{
			if (robotPose.getAngle() + ra < -pi_2)
				ra = -pi_2 - robotPose.getAngle();
		}

    rweight=1.0;
  }
};

}


#endif //__GT2003TurnToBallGoalie_h_


/*
* Change log:
*
* $Log: TurnToPointGoalie.h,v $
* Revision 1.3  2004/03/08 01:06:55  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/22 22:18:46  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/08/08 11:37:30  dueffert
* doxygen docu corrected
*
* Revision 1.1  2003/07/07 09:28:03  thomas
* modified: goalie based on cont-goalie-clear and angle limit in guard-mode
*
*/
