/** 
* @file GT2003SimpleBasicBehaviors.h
*
* Declaration of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#ifndef __GT2003SimpleBasicBehaviors_h_
#define __GT2003SimpleBasicBehaviors_h_

#include "../../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2BasicBehavior.h"
#include "Tools/Actorics/WalkAccelerationRestrictor.h"
#include "Tools/Evolution/Population.h"
#include "Tools/Evolution/Parcour.h"
#include "Tools/Math/PIDsmoothedValue.h"


/**
* A Basic Behavior for going to the ball
* @author Matthias Jngel
* @author Martin Ltzsch
*/
class GT2003BasicBehaviorGoToBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2003BasicBehaviorGoToBall(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball.distance",distanceAtEnd);
    registerParameter("go-to-ball.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball.distance" */
  double distanceAtEnd;
  
  /** parameter "go-to-ball.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to the ball and reaching it with certain distance (x and y relative to robot) and direction
* @author Uwe Dffert
* @author Matthias Jngel
* @author Martin Ltzsch
*/
class GT2003BasicBehaviorGoToBallWithDirection : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2003BasicBehaviorGoToBallWithDirection(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball-with-direction", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball-with-direction.distance.x",distanceX);
    registerParameter("go-to-ball-with-direction.distance.y",distanceY);
    registerParameter("go-to-ball-with-direction.max-speed", maxSpeed);
    registerParameter("go-to-ball-with-direction.direction", direction);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball-with-direction.distance.x" */
  double distanceX;
  /** parameter "go-to-ball-with-direction.distance.y" */
  double distanceY;
  /** parameter "go-to-ball-with-direction.max-speed" */
  double maxSpeed;
  /** parameter "go-to-ball-with-direction.direction" */
  double direction;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};


/**
* A Basic Behavior for avoiding obstacles
* @author Matthias Jngel
*/
class GT2003BasicBehaviorAvoidObstacles : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2003BasicBehaviorAvoidObstacles(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("avoid-obstacles", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("avoid-obstacles.turn-distance",turnDistance);
    registerParameter("avoid-obstacles.stop-distance",stopDistance);
    registerParameter("avoid-obstacles.walk-speed", walkSpeed);
    registerParameter("avoid-obstacles.turn-speed", turnSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "avoid-obstacles.turn-distance" */
  double turnDistance;
  
  /** parameter "avoid-obstacles.stop-distance" */
  double stopDistance;
  
  /** parameter "avoid-obstacles.walk-speed" */
  double walkSpeed;
  
  /** parameter "avoid-obstacles.turn-speed" */
  double turnSpeed;
};


/**
* A Basic Behavior for going to a point on the field
* @author Uwe Dffert
* @author Martin Ltzsch
*/
class GT2003BasicBehaviorGoToPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2003BasicBehaviorGoToPoint(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point.x", x);
    registerParameter("go-to-point.y", y);
    registerParameter("go-to-point.destination-angle", destinationAngle);
    registerParameter("go-to-point.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point.x" */
  double x;
  
  /** parameter "go-to-point.y" */
  double y;
  
  /** parameter "go-to-point.destination-angle" */
  double destinationAngle;
  
  /** parameter "go-to-point.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
  
  friend class GT2003BasicBehaviorGetBehindBall;
};

/**
* A Basic Behavior for precisely going to a point on the field
* @author Uwe Dffert
* @author Martin Ltzsch
* @author Thomas Rfer
*/
class GT2003BasicBehaviorGoToPointPrecisely : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2003BasicBehaviorGoToPointPrecisely(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-precisely", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point-precisely.x", x);
    registerParameter("go-to-point-precisely.y", y);
    registerParameter("go-to-point-precisely.destination-angle", destinationAngle);
    registerParameter("go-to-point-precisely.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point-precisely.x" */
  double x;
  
  /** parameter "go-to-point-precisely.y" */
  double y;
  
  /** parameter "go-to-point-precisely.destination-angle" */
  double destinationAngle;
  
  /** parameter "go-to-point-precisely.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to a point on the field and avoiding obstacles on the way there.
* @author Jan Hoffmann
*/
class GT2003BasicBehaviorGoToPointAndAvoidObstacles : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2003BasicBehaviorGoToPointAndAvoidObstacles(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-and-avoid-obstacles", errorHandler),
    BehaviorControlInterfaces(interfaces),
    speedPhi(0, .1, 0, 0, -pi_4, pi_4, pi/10),
    speedX(0, .6, 0, 0, -80, 200, 100),
    speedY(0, .1, 0, 0, -100, 100, 100),
    goToDestCounter(0),
    stickToBackOffDirection(0)
  {
    registerParameter("go-to-point-and-avoid-obstacles.x", x);
    registerParameter("go-to-point-and-avoid-obstacles.y", y);
    registerParameter("go-to-point-and-avoid-obstacles.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-point-and-avoid-obstacles.x" */
  double x;
  
  /** parameter "go-to-point-and-avoid-obstacles.y" */
  double y;
  
  /** parameter "go-to-point-and-avoid-obstacles.max-speed" */
  double maxSpeed;
  
  /** smoothes the rotation */  
  PIDsmoothedValue speedPhi;
  
  /** smoothes the speed in forward direction */  
  PIDsmoothedValue speedX;
  
  /** smoothes the speed in sideways direction */  
  PIDsmoothedValue speedY;
  
  /** history that holds information in which direction the robot is/was turning */
  double turnDirection;
  
  /** history that holds information in which direction (left/right) the robot is/was moving  */
  double sideways;
  
  /** parameter "goToDestCounter" used for hysteresis*/
  long goToDestCounter;
  
  /** parameter "go-to-point-avoiding" */
  double backOffAngle;
  
  /** */
  long stickToBackOffDirection;
};

/**
* A Basic Behavior for getting behind the ball
* @author Max Risler
* @author Martin Ltzsch
*/
class GT2003BasicBehaviorGetBehindBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  GT2003BasicBehaviorGetBehindBall(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("get-behind-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("get-behind-ball.distance",distance);
    registerParameter("get-behind-ball.left-right", leftRight);
    registerParameter("get-behind-ball.max-speed", maxSpeed);
    registerParameter("get-behind-ball.angle", angle);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "get-behind-ball.distance" */
  double distance;
  
  /** parameter "get-behind-ball.left-right" */
  double leftRight;
  
  /** parameter "get-behind-ball.max-speed" */
  double maxSpeed;
  
  /** parameter "get-behind-ball.angle" */
  double angle;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
  
  /** Walks to a point */
  void gotoPoint(double x, double y, double maxSpeed);
};


/**
* @class GT2003SimpleBasicBehaviors
*
* Creates and registers simple basic behaviors
*/
class GT2003SimpleBasicBehaviors : public BehaviorControlInterfaces
{
public:
/**
* Constructor
  */
  GT2003SimpleBasicBehaviors(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : BehaviorControlInterfaces(interfaces),
    errorHandler(errorHandler),
    basicBehaviorAvoidObstacles(interfaces,errorHandler),
    basicBehaviorGetBehindBall(interfaces,errorHandler),
    basicBehaviorGoToBall(interfaces,errorHandler),
    basicBehaviorGoToBallWithDirection(interfaces,errorHandler),
    basicBehaviorGoToPoint(interfaces,errorHandler),
    basicBehaviorGoToPointPrecisely(interfaces,errorHandler),
    basicBehaviorGoToPointAndAvoidObstacles(interfaces,errorHandler)
  {}
  
  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine);
  
private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;
  
  //!@name Basic Behaviors
  //!@{
  GT2003BasicBehaviorAvoidObstacles  basicBehaviorAvoidObstacles;  
  GT2003BasicBehaviorGetBehindBall  basicBehaviorGetBehindBall;
  GT2003BasicBehaviorGoToBall  basicBehaviorGoToBall;
  GT2003BasicBehaviorGoToBallWithDirection  basicBehaviorGoToBallWithDirection;
  GT2003BasicBehaviorGoToPoint  basicBehaviorGoToPoint;  
  GT2003BasicBehaviorGoToPointPrecisely  basicBehaviorGoToPointPrecisely;  
  GT2003BasicBehaviorGoToPointAndAvoidObstacles  basicBehaviorGoToPointAndAvoidObstacles;
  //!@}
};

#endif // __GT2003SimpleBasicBehaviors_h_

/*
* Change Log
* 
* $Log: GT2003SimpleBasicBehaviors.h,v $
* Revision 1.24  2004/05/27 13:31:26  dueffert
* walking evolution stuff separated
*
* Revision 1.23  2004/05/20 17:18:35  dueffert
* automatic measurement (hopefully) finalized
*
* Revision 1.22  2004/05/19 07:59:44  dueffert
* automatic walk speed measurement significantly improved
*
* Revision 1.21  2004/05/14 10:15:49  dueffert
* exact measurement behavior with logging implemented
*
* Revision 1.20  2004/05/12 14:23:32  dueffert
* measurement basic behavior added
*
* Revision 1.19  2004/04/29 15:15:01  dueffert
* measurement is now frame instead of time based
*
* Revision 1.18  2004/04/16 14:56:36  dueffert
* cleanup for Martins data flow graphics
*
* Revision 1.17  2004/03/24 13:44:49  dueffert
* support for uniform noise mutation readded
*
* Revision 1.16  2004/03/19 16:37:27  dueffert
* blind measurement works for the first time
*
* Revision 1.15  2004/03/12 17:11:38  dueffert
* omni-evo uses parameters in package now; blind speed calculation improved
*
* Revision 1.14  2004/03/10 15:01:08  dueffert
* first simple fast turn (blind) measurement implemented
*
* Revision 1.13  2004/03/10 10:02:22  dueffert
* population size tweaked; statistics output added
*
* Revision 1.12  2004/03/09 10:28:10  dueffert
* now using const interface too
*
* Revision 1.11  2004/03/09 08:46:30  dueffert
* second measurement behavior added
*
* Revision 1.10  2004/03/08 01:06:49  roefer
* Interfaces should be const
*
* Revision 1.9  2004/03/01 15:06:27  dueffert
* measurement output beautified
*
* Revision 1.8  2004/02/23 16:48:11  dueffert
* several improvements for measurement of walking
*
* Revision 1.7  2004/02/18 14:49:20  dueffert
* behavior control can now change walking parameters
*
* Revision 1.6  2004/02/10 11:19:19  dueffert
* simple basic behavior for evolution added, another improved, beautified
*
* Revision 1.5  2003/12/09 15:15:20  dueffert
* WalkAccelerationRestrictor replaced by PIDsmoothedValue for evolving
*
* Revision 1.4  2003/11/19 13:49:39  dueffert
* better positioning
*
* Revision 1.3  2003/10/31 08:32:49  dueffert
* doxygen bugs fixed
*
* Revision 1.2  2003/10/28 13:27:22  dueffert
* spelling and evolution logic improved
*
* Revision 1.1  2003/10/22 22:18:45  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.11  2003/09/30 10:51:13  dueffert
* typos fixed
*
* Revision 1.10  2003/09/26 11:38:51  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.9  2003/08/09 14:57:06  dueffert
* some evolution improvements
*
* Revision 1.8  2003/08/08 15:43:25  dueffert
* some evolution implementation added
*
* Revision 1.7  2003/08/08 14:30:04  dueffert
* some evolution implementation added
*
* Revision 1.6  2003/08/04 07:46:39  roefer
* Challenge 2 modifications
*
* Revision 1.5  2003/07/30 14:52:03  dueffert
* walk evolution agent added
*
* Revision 1.4  2003/07/09 11:44:26  jhoffman
* obstacle avoidance simple behavior mode as used in challenge 3
* known bug: sometimes stops in front of obstacle (but eventually the situation is resolved)
*
* Revision 1.3  2003/07/06 20:34:30  dueffert
* GT2003BasicBehaviorGoToBallWithDirection added
*
* Revision 1.2  2003/07/04 16:24:24  jhoffman
* updated obstacle avoider (simple behavior) to be used in challenge
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/06/04 00:43:45  loetzsch
* put the GT2003WalkAccelerationRestrictor into two extra files in /Src/Tools
*
* Revision 1.2  2003/05/05 19:22:03  risler
* added classes Simple/ContinuousBasicBehaviors for registering basic behaviors
*
* Revision 1.1  2003/05/03 15:14:02  loetzsch
* GT2003BehaviorControl first draft
*
*
*/

