/**
* @file GT2003BehaviorConfiguration.h
* 
* Implementation of class GT2003BehaviorConfiguration.
*
* @author Max Risler
*/

#ifndef __GT2003BehaviorConfiguration_h_
#define __GT2003BehaviorConfiguration_h_

#include "Tools/Math/Pose2D.h"
#include "Tools/ContinuousBasicBehaviors/ContinuousBasicBehavior.h"
#include "Tools/MessageQueue/InMessage.h"
#include "Tools/Debugging/GenericDebugData.h"

/**
* @class GT2003BehaviorConfiguration
*
* This class contains all tuneable configuration parameters for GT2003BehaviorControl.
* 
* @author Max Risler
*/ 
class GT2003BehaviorConfiguration : public MessageHandler
{
public:
  
  /** number of different kickoff variants */
  enum { numOfKickoff = 2 };

  /** the max speed with which the robot walks to its kickoff position */
  double goToKickoffMaxSpeed;

  /** the goalie start position */
  Vector2<double> goalie;

  /** the defensive supporter start position */
  Vector2<double> defensiveSupporter;

  /** the offensive supporter start position at opponent kickoff */
  Vector2<double> offensiveSupporterOpponentKickoff;
  /** the striker start position at opponent kickoff */
  Vector2<double> strikerOpponentKickoff;

  /** the offensive supporter start position at own kickoff */
  Vector2<double> offensiveSupporterOwnKickoff[numOfKickoff];
  /** the orientation of the offensive supporter at own kickoff in degrees */
  double offensiveSupporterOwnKickoffAngle[numOfKickoff];
  /** the striker start position at own kickoff */
  Vector2<double> strikerOwnKickoff[numOfKickoff];
  /** the orientation of the striker at own kickoff in degrees */
  double strikerOwnKickoffAngle[numOfKickoff];


	/* The distance where behaviors near the ball should be activated */
  double ballNearDistance;
	
  /* The distance where behaviors far from the ball should be activated */
  double ballNotNearDistance;
	
  /* The distance where behaviors at the ball should be activated */
  double atBallDistance;
	
  /* The distance where behaviors near the ball should be activated */
  double notAtBallDistance;

  
  /* The evaluation mode for continuous basic behaviors.
   * This determines how continuous basic behaviors will be evaluated.
   * See ContinuousBasicBehavior.h for details.
   */
  ContinuousBasicBehavior::EvaluationMode contEvaluationMode;

  /** 
  * Constructor.
  */
  GT2003BehaviorConfiguration()
  {
    goToKickoffMaxSpeed = 250.0;

    // start position for goalie (both kickoffs)
    goalie.x = -2000.0;
    goalie.y = 50;
    // start position for defensive supporter (both kickoffs)
    defensiveSupporter.x = -1000.0;
    defensiveSupporter.y = -200.0;

    // start position for offensive supporter (opponent kickoffs)
    offensiveSupporterOpponentKickoff.x = -1000.0;
    offensiveSupporterOpponentKickoff.y = -800.0;
    // start position for striker (opponent kickoffs)
    strikerOpponentKickoff.x = -1000.0;
    strikerOpponentKickoff.y = 500.0;

    // long kickoff to the left
    offensiveSupporterOwnKickoff[0].x = -80.0;
    offensiveSupporterOwnKickoff[0].y = 1100.0;
    offensiveSupporterOwnKickoffAngle[0] = 0.0;
    strikerOwnKickoff[0].x = -130.0;
    strikerOwnKickoff[0].y = -130.0;
    strikerOwnKickoffAngle[0] = 45.0;

    // long kickoff to the right
    offensiveSupporterOwnKickoff[1].x = -80.0;
    offensiveSupporterOwnKickoff[1].y = -1100.0;
    offensiveSupporterOwnKickoffAngle[1] = 0.0;
    strikerOwnKickoff[1].x = -130.0;
    strikerOwnKickoff[1].y = 130.0;
    strikerOwnKickoffAngle[1] = -45.0;

    // direct long kickoff
    //offensiveSupporterOwnKickoff[2].x = -80.0;
    //offensiveSupporterOwnKickoff[2].y = -550.0;
    //offensiveSupporterOwnKickoffAngle[2] = 0;
    //strikerOwnKickoff[2].x = -170.0;
    //strikerOwnKickoff[2].y = 0.0;
    //strikerOwnKickoffAngle[2] = 0.0;

    ballNearDistance = 700.0;
    ballNotNearDistance = 1000.0;
    atBallDistance = 280.0;
    notAtBallDistance = 320.0;

    contEvaluationMode = ContinuousBasicBehavior::discrete;
  }

  /** 
  * Is called for every incoming debug message.
  * @param message An interface to read the message from the queue
  * @return if the message was read
  */
  virtual bool handleMessage(InMessage& message);
};

Out& operator<<(Out& stream, const GT2003BehaviorConfiguration& gt2003BehaviorConfiguration);

In& operator>>(In& stream,GT2003BehaviorConfiguration& gt2003BehaviorConfiguration);

#endif// __GT2003BehaviorConfiguration_h_

/*
 * Change log :
 * 
 * $Log: GT2003BehaviorConfiguration.h,v $
 * Revision 1.1  2003/10/06 13:39:29  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.3  2003/09/26 11:38:51  juengel
 * - sorted tools
 * - clean-up in DataTypes
 *
 * Revision 1.2  2003/07/03 10:55:16  risler
 * kickoff max speed increased
 *
 * Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.8  2003/06/22 14:25:01  roefer
 * Challenge 2 symbols added
 *
 * Revision 1.7  2003/06/21 19:38:49  thomas
 * modified: supporter on kickoff
 *
 * Revision 1.6  2003/06/21 18:26:04  roefer
 * Loading points.cfg
 *
 * Revision 1.5  2003/06/21 16:47:39  thomas
 * updated: coordinates of start positions and disabled middle-kickoff
 *
 * Revision 1.4  2003/06/21 15:01:12  risler
 * kickoff with more than one player possible
 * different kickoff variants added
 *
 * Revision 1.3  2003/05/27 12:23:42  risler
 * added GT2003BehaviorConfiguration debug message
 *
 * Revision 1.2  2003/05/26 13:45:02  loetzsch
 * some improvements
 *
 * Revision 1.1  2003/05/25 14:41:40  risler
 * added GT2003BehaviorConfiguration
 *
 *
 */
