/**
* @file GT2003BehaviorControl.cpp
* 
* Implementation of class GT2003BehaviorControl.
*
* @author Martin Ltzsch
* @author Matthias Jngel
* @author Max Risler
*/

#include "GT2003BehaviorControl.h"
#include "Tools/Player.h"

GT2003BehaviorControl::GT2003BehaviorControl(const BehaviorControlInterfaces& interfaces)
: Xabsl2BehaviorControl(interfaces,SolutionRequest::gt2003),
ballSymbols(interfaces),
configurationSymbols(interfaces,configuration),
headAndTailSymbols(interfaces),
joystickSymbols(interfaces),
ledAndSoundSymbols(interfaces),
mathFunctions(interfaces),
motionRequestSymbols(interfaces),
roboCupGameManagerSymbols(interfaces),
robotPoseSymbols(interfaces),
robotStateSymbols(interfaces),
specialVisionSymbols(interfaces),
strategySymbols(interfaces),
evolutionSymbols(interfaces, udParametersSet, udParametersCalibration, udCurrentIndex, measurementRequest, evolutionMode),
challengeSymbols(interfaces),
calibrationSymbols(interfaces),
simpleBasicBehaviors(interfaces,errorHandler),
evolutionBasicBehaviors(interfaces,errorHandler, udParametersSet, udExtraParameters, udParametersCalibration, udCurrentIndex, measurementRequest, evolutionMode),
continuousBasicBehaviors(interfaces,configuration,errorHandler),
commonBasicBehaviors(interfaces,errorHandler),
lastSound(SoundRequest::none),
udCurrentIndex((int)UDParametersSet::no_turn_0_fast),
evolutionMode(1)
{
  Xabsl2FileInputSource file("Xabsl2/gt03-ic.dat");
  init(file);
  
  measurementRequest = udParametersSet.getParameters(udCurrentIndex)->requestedMotion;
  for (int i=0;i<=127;i++)
  {
    udParametersCalibration[i]=Pose2D(0,0,0);
  }
  udExtraParameters.index=UDParametersSet::numberOfParameters;
}

GT2003BehaviorControl::~GT2003BehaviorControl()
{
}

void GT2003BehaviorControl::registerSymbolsAndBasicBehaviors()
{
  simpleBasicBehaviors.registerBasicBehaviors(*pEngine);
  evolutionBasicBehaviors.registerBasicBehaviors(*pEngine);
  continuousBasicBehaviors.registerBasicBehaviors(*pEngine);
  commonBasicBehaviors.registerBasicBehaviors(*pEngine);
  
  ballSymbols.registerSymbols(*pEngine);
  configurationSymbols.registerSymbols(*pEngine);
  calibrationSymbols.registerSymbols(*pEngine);
  headAndTailSymbols.registerSymbols(*pEngine);
  joystickSymbols.registerSymbols(*pEngine);
  ledAndSoundSymbols.registerSymbols(*pEngine);
  mathFunctions.registerSymbols(*pEngine);
  motionRequestSymbols.registerSymbols(*pEngine);
  roboCupGameManagerSymbols.registerSymbols(*pEngine);
  robotPoseSymbols.registerSymbols(*pEngine);
  robotStateSymbols.registerSymbols(*pEngine);
  specialVisionSymbols.registerSymbols(*pEngine);
  strategySymbols.registerSymbols(*pEngine);
  evolutionSymbols.registerSymbols(*pEngine);
  challengeSymbols.registerSymbols(*pEngine);
}

void GT2003BehaviorControl::execute()
{
  ballSymbols.update();
  configurationSymbols.update();
  roboCupGameManagerSymbols.update();
  robotPoseSymbols.update();
  strategySymbols.update();
  evolutionSymbols.update();
  
  continuousBasicBehaviors.update();
  
  // set the outgoint behavior team message to none
  outgoingBehaviorTeamMessage.message = BehaviorTeamMessage::none;
  
  // set the upper leds depending on the dynamic role.
  switch (strategySymbols.role)
  {
  case BehaviorTeamMessage::striker:
    ledRequest.redTopLEDs = LEDRequest::bothOn;
    break;
  case BehaviorTeamMessage::offensiveSupporter:
    ledRequest.redTopLEDs = LEDRequest::bothFastBlink;
    break;
  case BehaviorTeamMessage::defensiveSupporter:
  default:
    ledRequest.redTopLEDs = LEDRequest::bothOff;
  }
  
  // Set the tail request depending on whether the ball was seen
  if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 120)
  {
    // ball seen now
    motionRequest.tailRequest = MotionRequest::wagHorizontalFast;
  }
  else if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 2000)
  {
    // ball seen
    motionRequest.tailRequest = MotionRequest::wagHorizontal;
  }
  else if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) 
    > BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
    && SystemCall::getTimeSince(ballPosition.communicated.timeWhenLastObserved) < 3000)
  {
    // ball known
    motionRequest.tailRequest = MotionRequest::wagVertical;
  }
  else
  {
    // ball not known
    motionRequest.tailRequest = MotionRequest::noTailWag;
  }
  
  // set head control mode lookToStars to detect missing settings
  headControlMode.headControlMode = HeadControlMode::lookToStars;
  
  // execute the engine
  executeEngine();
  
   if((robotState.getState() == RobotState::crashed)||
     (robotState.getState() == RobotState::rollLeft)||
     (robotState.getState() == RobotState::rollRight))
  {
    if ((motionRequest.motionType == MotionRequest::walk &&
      motionRequest.walkType != MotionRequest::upsideDown)||
      (motionRequest.motionType == MotionRequest::stand))
    {
      motionRequest.motionType = MotionRequest::getup;
      return;
    }
  }
 SoundRequest::SoundID currentSound = soundRequest.soundID;
  
  if (headControlMode.headControlMode == HeadControlMode::lookToStars)
  {
    soundRequest.soundID = SoundRequest::rob101;
  }
  else if (
    strategySymbols.role == BehaviorTeamMessage::striker &&
    currentSound == lastSound)
  {
    if (strategySymbols.robotIsStuck)
    {
      soundRequest.soundID = SoundRequest::bing03;
    }
    else if (strategySymbols.obstaclesAreClose)
    {
      soundRequest.soundID = SoundRequest::bing01;
    }
    else if (robotState.getConsecutiveCollisionTimeAggregate() > 2000)
    {
      soundRequest.soundID = SoundRequest::bing05;
    }
  }
  
  lastSound = currentSound;
}

bool GT2003BehaviorControl::handleMessage(InMessage& message)
{
  switch (message.getMessageID())
  {
  case idUDEvolutionRequest:
    {
      int mode;
      message.bin >> mode;
      switch (mode)
      {
      case 1: // measure UDParameters #... or its mirror if that has higher index or a self choosen motionRequest
        {
          message.bin >> udCurrentIndex;
          message.bin >> evolutionMode;
          if (udCurrentIndex==(int)UDParametersSet::numberOfParameters)
          {
            message.bin >> measurementRequest;
            if (UDParametersSet::getSpeed(measurementRequest)<0.05)
            {
              UDParametersSet::setSpeed(measurementRequest,0.05);
            }
            udExtraParameters.requestedMotion=measurementRequest;
            OUTPUT(idText,text,"GT2003BehavContr: UDEvolutionRequest 'measure (" << measurementRequest.translation.x << ", " << measurementRequest.translation.y << ", " << measurementRequest.rotation << ")' received");
          }
          else
          {
            int mirror=UDParametersSet::getIndexOfMirror(udCurrentIndex);
            //use higher mirror number = lturn
            if ((mirror>=0)&&(mirror>udCurrentIndex))
            {
              udCurrentIndex=mirror;
            }
            measurementRequest=udParametersSet.getParameters(udCurrentIndex)->requestedMotion;
            OUTPUT(idText,text,"GT2003BehavContr: UDEvolutionRequest 'measure " << UDParametersSet::getIndexString(udCurrentIndex) << "' received (" << measurementRequest.translation.x << ", " << measurementRequest.translation.y << ", " << measurementRequest.rotation << ")");
          }
          break;
        }
      case 2: // save parametersSet
        udParametersSet.save();
        OUTPUT(idText,text,"GT2003BehavContr: UDEvolutionRequest 'save parametersSet' executed");
        break;
      case 3: // load parametersSet
        {
          udParametersSet.load();
          for (int i=0;i<(int)UDParametersSet::numberOfParameters;i++)
          {
            udParametersCalibration[i]=Pose2D(0,0,0);
          }
          OUTPUT(idText,text,"GT2003BehavContr: UDEvolutionRequest 'load parametersSet' executed and calibration resetted");
          break;
        }
      default:
        OUTPUT(idText,text,"GT2003BehavContr: unknown UDEvolutionRequest received");
        break;
      }
      return true;
    }
  case idUDParameters:
    {
      UDParameters param;
      message.bin >> param;
      if (param.index<UDParametersSet::numberOfParameters)
      {
        *udParametersSet.getParameters(param.index) = param;
        OUTPUT(idText,text,"GT2003BehavContr: UDParameters " << UDParametersSet::getIndexString(param.index) << " received (" << param.requestedMotion.translation.x << ", " << param.requestedMotion.translation.y << ", " << param.requestedMotion.rotation << ")");
      }
      else
      {
        udExtraParameters = param;
        OUTPUT(idText,text,"GT2003BehavContr: extra UDParameters received (" << param.requestedMotion.translation.x << ", " << param.requestedMotion.translation.y << ", " << param.requestedMotion.rotation << ")");
      }
      udParametersCalibration[param.index]=Pose2D(0,0,0);
      //and send it to Motion immediately:
      udParameters = param;
      walkParameterTimeStamp = SystemCall::getCurrentSystemTime();
      return true;
    }
  default:
    return (configuration.handleMessage(message) ||
      Xabsl2BehaviorControl::handleMessage(message));
  }
}


/*
* Change log :
* 
* $Log: GT2003BehaviorControl.cpp,v $
* Revision 1.15  2004/06/21 11:29:45  dueffert
* backport of initialization bug fix
*
* Revision 1.14  2004/05/27 13:31:26  dueffert
* walking evolution stuff separated
*
* Revision 1.13  2004/05/23 13:31:22  dueffert
* mirroring corrected
*
* Revision 1.12  2004/05/19 08:08:43  dueffert
* automatic walk speed measurement significantly improved
*
* Revision 1.11  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
* Revision 1.10  2004/05/14 10:16:06  dueffert
* exact measurement behavior with logging implemented
*
* Revision 1.9  2004/05/12 14:24:05  dueffert
* minimum speed for measurements
*
* Revision 1.8  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.7  2004/04/16 14:56:36  dueffert
* cleanup for Martins data flow graphics
*
* Revision 1.6  2004/03/08 01:06:56  roefer
* Interfaces should be const
*
* Revision 1.5  2004/02/03 13:20:48  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.4  2003/10/26 22:49:39  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
* Revision 1.3  2003/10/22 22:18:45  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.2  2003/10/12 11:53:22  juengel
* Added calibration-symbols.
*
* Revision 1.1  2003/10/06 13:39:29  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.5  2003/07/03 14:43:17  risler
* new sounds
*
* Revision 1.4  2003/07/03 11:29:23  juengel
* no message
*
* Revision 1.3  2003/07/03 11:17:14  risler
* no ring
*
* Revision 1.2  2003/07/03 10:34:50  loetzsch
* added symbol "not-stuck-but..."
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.24  2003/06/22 15:59:22  loetzsch
* sets some sound requests now
*
* Revision 1.23  2003/06/22 14:25:01  roefer
* Challenge 2 symbols added
*
* Revision 1.22  2003/06/21 18:26:04  roefer
* Loading points.cfg
*
* Revision 1.21  2003/06/21 15:01:12  risler
* kickoff with more than one player possible
* different kickoff variants added
*
* Revision 1.20  2003/06/20 20:21:43  goehring
* CollisionSounds added for all players
*
* Revision 1.19  2003/06/20 16:57:58  goehring
* CollisionDetectorValues improved
*
* Revision 1.18  2003/06/20 10:33:19  goehring
* Added CollisionSounds
*
* Revision 1.17  2003/06/19 18:34:35  juengel
* head control mode lookToStars to detect missing settings
*
* Revision 1.16  2003/06/18 15:38:49  risler
* set lookBetweenFeet if no other head control mode was set
*
* Revision 1.15  2003/06/02 14:44:36  loetzsch
* added ballSymbols.update() to ::execute
*
* Revision 1.14  2003/05/27 16:52:30  loetzsch
* outgoingBehaviorTeammessage is set to "none" before the execution of the engine
*
* Revision 1.13  2003/05/27 12:23:42  risler
* added GT2003BehaviorConfiguration debug message
*
* Revision 1.12  2003/05/26 23:10:26  loetzsch
* some more improvements in the supporting behaviors
*
* Revision 1.11  2003/05/25 22:37:02  loetzsch
* finished the game state options of GT2003
*
* Revision 1.10  2003/05/25 14:41:40  risler
* added GT2003BehaviorConfiguration
*
* Revision 1.9  2003/05/07 18:19:00  juengel
* Added missing symbols to GT2003BehaviorControl.
*
* Revision 1.8  2003/05/07 17:27:59  risler
* added MotionRequestSymbols
*
* Revision 1.7  2003/05/07 13:02:34  risler
* added first continuous basic behaviors to GT2003BehaviorControl
*
* Revision 1.6  2003/05/06 16:28:19  loetzsch
* added class StrategySymbols
*
* Revision 1.5  2003/05/06 16:03:05  loetzsch
* added class RobotPoseSymbols
*
* Revision 1.4  2003/05/06 14:46:12  loetzsch
* added registration of generated GT2003BehaviorControl symbols
*
* Revision 1.3  2003/05/05 19:22:03  risler
* added classes Simple/ContinuousBasicBehaviors for registering basic behaviors
*
* Revision 1.2  2003/05/04 11:41:40  loetzsch
* added class BallSymbols
*
* Revision 1.1  2003/05/03 15:14:02  loetzsch
* GT2003BehaviorControl first draft
*
*/

