/**
* @file MSH2004KickEngine.h
* 
* Implementation of class MSH2004KickEngine.
*
* @author Carsten Schumann (schumann@tempus-vivit.net)
*/
#include "Tools/Math/Pose2D.h"
#include "MSH2004KickEngineEntry.h"
#include "Representations/Cognition/BallModel.h"
#include "Representations/Motion/MotionRequest.h"

#ifndef __MSH2004KickEngine_h_
#define __MSH2004KickEngine_h_

const int MSH2004KickEngineXMin=-4;
const int MSH2004KickEngineXMax=13;
const int MSH2004KickEngineYMin=-13; // note: YMin must be symetrically to YMax
const int MSH2004KickEngineYMax=13;
const int MSH2004KickEngineSectors=(MSH2004KickEngineXMax-MSH2004KickEngineXMin+1)*(MSH2004KickEngineYMax-MSH2004KickEngineYMin+1);

/**
* @class MSH2004KickEngine
*
* Container and Database class for the MSH2004KickEngine
*
* Aim of the kick engine is a fast solution for a query with 
* the following parameters
*
* - current Ball position
* - desired ball destination
* - movement abilities and remaining time
* - tolerance regarding the destination
*/
class MSH2004KickEngine{
 
public:
  MSH2004KickEngineEntry* ballPosition[MSH2004KickEngineXMax-MSH2004KickEngineXMin+1][MSH2004KickEngineYMax-MSH2004KickEngineYMin+1]; //==28x56cm 
  short unsigned int possibleBallPosition[MSH2004KickEngineXMax-MSH2004KickEngineXMin+1][MSH2004KickEngineYMax-MSH2004KickEngineYMin+1]; //==28x56cm 
  unsigned int possibleBallPositionCount;
  MSH2004KickEngine(const char* kickEngineFile);
  ~MSH2004KickEngine();


  void clearDistribution();
  void addKick(double ballXmm, double ballYmm, double kickAngle, double kickDistance,MotionRequest::SpecialActionID kick);
  void save();
  void load();

  MSH2004KickEngineSearchResult getPreferredKick(double currentBallXmm,
                                                  double currentBallYmm,
                                                  double kickAngle, 
                                                  double kickDistance,
                                                  double maxKickTolerance);

  MSH2004KickEngineSearchResult getNeighbourhoodKick(double currentBallXmm,
                                                      double currentBallYmm,
                                                      double kickAngle, 
                                                      double kickDistance, 
                                                      double maxSearchDistance,
                                                      double maxKickTolerance);

  MSH2004KickEngineSearchResult getUnknownPosition(MotionRequest::SpecialActionID kick);
  bool resultKnown(short int x, short int y,MotionRequest::SpecialActionID kick);
  bool resultKnownInNeighbourhood(short int cx, short int cy, int searchDistance,MotionRequest::SpecialActionID kick);
  void registerTry(int x,int y,MotionRequest::SpecialActionID kick);
  bool inBounds(short int x,short int y);
  bool goodResultKnown(short int x, short int y, MotionRequest::SpecialActionID kick);
  void calculateClusterPoints(MSH2004KickEngineSearchResult& result);
  /* calculates movement, so that the  ball gets from ballAngle/ballDistance to sectorTo/distanceTo*/


  void setSearchDistanceMetrics(double forBallX, double forBallY);
  void setSearchToleranceMetrics(double forKickSector, double forKickDistance);
  double getDistance(int x1, int y1, int x2, int y2);

  short int getBallXSector(double x);
  short int getBallYSector(double y);
  short int getBallXSectorWithoutBounds(double x);
  short int getBallYSectorWithoutBounds(double y);
  short unsigned int getKickSector(double angle);
  short unsigned int getKickDistance(double distance);

  void reportPossibleBallPosition(double x, double y);
  bool ballPositionPossible(int sectorX, int sectorY);

 
private:
  bool knownKicks[MotionRequest::numOfSpecialAction];
  char kickEngineFile[60];


  double distanceForBallX;
  double distanceForBallY;
  double distanceForKickDistance;
  double distanceForKickSector;
  double unsavedChanges;
};


#endif
/*
 * Change log :
 * 
 * $Log: MSH2004KickEngine.h,v $
 * Revision 1.9  2004/04/08 15:33:04  wachter
 * GT04 checkin of Microsoft-Hellounds
 *
 * Revision 1.12  2004/03/29 14:44:21  pg_cars
 * Kick engine works now ;-)
 * Use PID Ball Locator and MSH2004-test behavior for testing
 *
 * Revision 1.11  2004/03/26 15:31:34  pg_cars
 * Improved data file format, improved debugging
 *
 * Revision 1.10  2004/03/25 21:05:24  pg_cars
 * change of datafile, before backup
 *
 * Revision 1.9  2004/03/22 17:53:08  pg_cars
 * improved training for kick engine
 *
 * Revision 1.8  2004/03/15 15:59:46  schumann
 * splitted symbols for kick engine from other symbols
 *
 * Revision 1.7  2004/03/10 09:01:52  schumann
 * changed origin of coordinate system
 *
 * Revision 1.6  2004/03/09 14:13:32  schumann
 * changed training for kickengine
 *
 * Revision 1.5  2004/03/04 08:37:40  schumann
 * added neighbourhood search
 *
 * Revision 1.4  2004/02/27 16:20:45  schumann
 * improved kickengine training
 *
 * Revision 1.3  2004/02/25 11:53:20  schumann
 * extended range of kick engine
 *
 * Revision 1.2  2004/02/24 13:51:05  schumann
 * Changed structure of kick engine
 *
 * Revision 1.1  2004/02/23 12:51:37  schumann
 * Added MSH2004KickEngine
 *
*/

