/**
* @file MSH2004KickEngineKick.cpp
* 
* Implementation of class MSH2004KickEngineKick.
*
* @author Carsten Schumann (schumann@tempus-vivit.net)
*/
 
#include "MSH2004KickEngineKick.h"
#include "Tools/Streams/InStreams.h"
#include "Tools/Streams/OutStreams.h"
#include "Tools/Debugging/Debugging.h"
//----------------------------------------------------------------------------
MSH2004KickEngineKick::MSH2004KickEngineKick(MotionRequest::SpecialActionID kickID)
{
  this->kickResults=new List<MSH2004KickEngineKickResult>();
  this->kickID=kickID;
  this->kickCount=0;
  this->trainingTries=0;
}
//----------------------------------------------------------------------------
MSH2004KickEngineKick::MSH2004KickEngineKick(MotionRequest::SpecialActionID kickID, unsigned short int trainingTries)
{
  this->kickResults=new List<MSH2004KickEngineKickResult>();
  this->kickID=kickID;
  this->kickCount=0;
  this->trainingTries=trainingTries;
}
//----------------------------------------------------------------------------
MSH2004KickEngineKick::~MSH2004KickEngineKick()
{
  this->kickResults->clear();
  delete(this->kickResults);
}
//----------------------------------------------------------------------------
MotionRequest::SpecialActionID MSH2004KickEngineKick::getKickID()
{
  return this->kickID;
}
//----------------------------------------------------------------------------
void MSH2004KickEngineKick::addKick(unsigned short int sector, 
                                    unsigned short int distance)
{
  if (distance==0)
  {
    sector=0;
  }
  
  List<MSH2004KickEngineKickResult>::Pos iterator= kickResults->getFirst();

  bool found=false;
  while (iterator)
  {
    if ((*kickResults)[iterator].distance==distance && (*kickResults)[iterator].sector==sector)
    {
      (*kickResults)[iterator].count++;
      found=true;
    }
    iterator++;
  }

  if (!found)
  {
    MSH2004KickEngineKickResult k;
    k.count=1;
    k.sector=sector;
    k.distance=distance;
    this->kickResults->insert(k);
  }
  this->kickCount++;
}
//----------------------------------------------------------------------------
double MSH2004KickEngineKick::getSuccessRate(unsigned short int kickSector,
                                             unsigned short int kickDistance)
{
  List<MSH2004KickEngineKickResult>::Pos iterator= kickResults->getFirst();

  while (iterator)
  {
    if ((*kickResults)[iterator].distance==kickDistance && (*kickResults)[iterator].sector==kickSector)
    {
      if (this->kickCount > 0 && (*kickResults)[iterator].count > 0)
      {
        double successRate=(double)(*kickResults)[iterator].count / (double)this->kickCount;
        return successRate;
      } else {
        return 0.0;
      }
    }
    iterator++;
  }
  return 0.0;

}
//----------------------------------------------------------------------------
double MSH2004KickEngineKick::getTolerance(int alter_ks, int alter_kd, double distanceForKickSector, double distanceForKickDistance)
{
  if (alter_ks>=16)
  {
    alter_ks-=32;
  }
  if (alter_ks<=-16)
  {
    alter_ks+=32;
  }
  return abs(alter_ks)*distanceForKickSector+abs(alter_kd)*distanceForKickDistance;
}
//----------------------------------------------------------------------------
double MSH2004KickEngineKick::getSuccessRate(unsigned short int kickSector,
                                             unsigned short int kickDistance, 
                                             double maxKickTolerance,
                                             double distanceForKickSector, 
                                             double distanceForKickDistance)
{
  double summedRate=0;
  List<MSH2004KickEngineKickResult>::Pos iterator= kickResults->getFirst();

  while (iterator)
  {
    double tolerance=getTolerance((*kickResults)[iterator].sector-kickSector,(*kickResults)[iterator].distance-kickDistance,distanceForKickSector, distanceForKickDistance);
    if ((tolerance < maxKickTolerance) && ((*kickResults)[iterator].distance>0))
    {
      summedRate+=(double)(*kickResults)[iterator].count;
    }
    iterator++;
  }
  if (this->kickCount > 0 && summedRate > 0)
  {
    double successRate=summedRate / (double)this->kickCount;
    return successRate;
  } else {
    return 0.0;
  }
}
//----------------------------------------------------------------------------
bool MSH2004KickEngineKick::kickKnown()
{
  if (this->trainingTries>=1) return true;
  if (this->kickResults->getSize()>0) return true;
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004KickEngineKick::goodResultKnown()
{
  if (this->trainingTries>=1) return false;
  if (this->kickResults->getSize()==0) return false;

  List<MSH2004KickEngineKickResult>::Pos iterator= kickResults->getFirst();
  int goodCount=0;
  int badCount=0;

  while (iterator)
  {
    if ((*kickResults)[iterator].distance>0)
    {
      goodCount+=(*kickResults)[iterator].count;
    } else {
      badCount+=(*kickResults)[iterator].count;
    }
    iterator++;
  }

  
  return goodCount>=badCount;
}
//----------------------------------------------------------------------------
void MSH2004KickEngineKick::registerTry()
{
  this->trainingTries++;
  INFO(sendKickEngineInfo, idText,text,"Tries="<<trainingTries);
}

void MSH2004KickEngineKick::loadTries(In& stream, char* buffer)
{
  stream >> this->trainingTries;
  stream >> buffer;
}

void MSH2004KickEngineKick::loadResults(In& stream, char* buffer)
{
  do{
    MSH2004KickEngineKickResult k;
    stream >> k.distance;
    stream >> k.sector;
    stream >> k.count;
    this->kickCount+=k.count;
    this->kickResults->insert(k);
    stream >> buffer;
  } while (!strcmp(buffer,"Result"));
}
//----------------------------------------------------------------------------
In& operator>>(In& stream,MSH2004KickEngineKick& kick)
{
#define KickEngineTextConfigFile
#ifdef KickEngineTextConfigFile
  MSH2004KickEngineKickResult k;
  stream>>k.distance;
  stream>>k.sector;
  stream>>k.count;
  kick.kickCount+=k.count;
  kick.kickResults->insert(k);

#else
  short unsigned int size;
  kick.kickCount=0;
  stream >> kick.kickID;
  INFO(sendKickEngineInfo, idText,text,"  ReadingKick:"<<kick.kickID);
  stream >> size;
  stream >> kick.trainingTries;
  for (int i=0; i<size; i++)
  {
    MSH2004KickEngineKickResult k;
    stream >> k.distance;
    stream >> k.sector;
    stream >> k.count;
    kick.kickCount+=k.count;
    INFO(sendKickEngineInfo, idText,text,"    Distance:"<<k.distance<< ",  Sector:"<<k.sector<<", Count:"<<k.count);
    kick.kickResults->insert(k);
  }

#endif
  return stream;
}
//----------------------------------------------------------------------------
void MSH2004KickEngineKick::save(Out& stream)
{
  stream << "  Tries "<<trainingTries<<"\r\n";
  List<MSH2004KickEngineKickResult>::Pos iterator= kickResults->getFirst();
  while (iterator)
  {
    stream << "  Result ";
    stream << (*kickResults)[iterator].distance;
    stream << " ";
    stream << (*kickResults)[iterator].sector;
    stream << " ";
    stream << (*kickResults)[iterator].count;
    stream << "\r\n";
    iterator++;
  }
}
//----------------------------------------------------------------------------
Out& operator<<(Out& stream, const MSH2004KickEngineKick& kick)
{
#define KickEngineTextConfigFile
#ifdef KickEngineTextConfigFile
  stream << "  Kick "<<kick.kickID<<" "<<kick.trainingTries<<"\r\n";
  List<MSH2004KickEngineKickResult>::Pos iterator= kick.kickResults->getFirst();
  while (iterator)
  {
    stream << "    Result ";
    stream << (*kick.kickResults)[iterator].distance;
    stream << " ";
    stream << (*kick.kickResults)[iterator].sector;
    stream << " ";
    stream << (*kick.kickResults)[iterator].count;
    stream << "\r\n";
    iterator++;
  }
    
#else
  stream << kick.kickID;
  stream << (short unsigned int)kick.kickResults->getSize();
  stream << kick.trainingTries;
  List<MSH2004KickEngineKickResult>::Pos iterator= kick.kickResults->getFirst();
  while (iterator)
  {
    stream << (*kick.kickResults)[iterator].distance;
    stream << (*kick.kickResults)[iterator].sector;
    stream << (*kick.kickResults)[iterator].count;
    iterator++;
  }
#endif  
  return stream;
}


/*
 * Change log :
 * 
 * $Log: MSH2004KickEngineKick.cpp,v $
 * Revision 1.7  2004/04/08 15:33:04  wachter
 * GT04 checkin of Microsoft-Hellounds
 *
 * Revision 1.13  2004/04/01 14:45:29  kindler
 * - added debug keys for walking- and kick-engine
 *
 * Revision 1.12  2004/03/29 14:44:21  pg_cars
 * Kick engine works now ;-)
 * Use PID Ball Locator and MSH2004-test behavior for testing
 *
  * Revision 1.11  2004/03/27 13:58:44  pg_cars
 * Bug fixed
 *
 * Revision 1.10  2004/03/26 15:31:34  pg_cars
 * Improved data file format, improved debugging
 *
 * Revision 1.9  2004/03/26 09:21:49  pg_cars
 * datafile changed, aggressive kick engine
 *
 * Revision 1.8  2004/03/25 21:05:24  pg_cars
 * change of datafile, before backup
 *
 * Revision 1.7  2004/03/22 17:53:08  pg_cars
 * improved training for kick engine
 *
 * Revision 1.6  2004/03/15 15:59:46  schumann
 * splitted symbols for kick engine from other symbols
 *
 * Revision 1.5  2004/03/09 14:13:32  schumann
 * changed training for kickengine
 *
 * Revision 1.4  2004/03/04 08:37:41  schumann
 * added neighbourhood search
 *

 * Revision 1.3  2004/02/27 16:20:45  schumann
 * improved kickengine training
 *
 * Revision 1.2  2004/02/24 16:03:04  schumann
 * New behavior for training
 *
 * Revision 1.1  2004/02/24 13:51:05  schumann
 * Changed structure of kick engine
 *
*/

