/** 
* @file DefaultCollisionDetector.cpp
* 
* This file contains a class for Sensor Data Processing.
*/
//10/16/2003 for gt2003 inv kin 
#include "DefaultCollisionDetector.h"
#include "Tools/Debugging/Debugging.h"

    double maxfl1 = 0;
    double maxfl2 = 0;
    double maxfr1 = 0;
    double maxfr2 = 0;
    double maxhl1 = 0;
    double maxhl2 = 0;
    double maxhr1 = 0;
    double maxhr2 = 0;
 


DefaultCollisionDetector::DefaultCollisionDetector
(const CollisionDetectorInterfaces& interfaces)
: CollisionDetector(interfaces)
{
  infin = 1000000000;
  index = 0;
  min =  5;   // minimum delay value
  max = 12;   // maximum delay value
  oldVal = 0;
  newVal = 0;

  frontMin =  0;//20
  frontMax = infin;  //400 or 180
  backMin =  -0;//-20
  backMax = -infin;
  sideMin =   0;  //20
  sideMax =  infin; //400 or left 120
  rotMin = 0.00;
  rotMax = infin; //2.5
    
  
  // Behaviour motion change recognition  
  currentMotionX = 0;
  currentMotionY = 0;
  currentRotation = 0;
  lastMotionX = 0;
  lastMotionY = 0;
  lastRotation = 0;
  motionChangeThresholdX = 50;
  motionChangeThresholdY = 50;
  motionChangeThresholdR = 0.2;
  calmDown = 32; 
  lastMotionChange = calmDown + 1;
  lastTooBigMotionChange = 0;
  motionChangeThresholdFactor = 1;
  xRatio = 0;
  yRatio = 0;
  rRatio = 0;
  q = 12;



  // Robot sepecific parameters for direction weights
  scaleThresholdFore = scaleThresholdBack = scaleThresholdSide = 1;
  scaleThresholdRot = 2;

  walkingMode = 0;
  
  //Debugging variables  
 //* 
  //*/
  // reset the Ringbuffer and the FloatingAvg
  // i = number of joints, l = quadruple element, d = delay index
  
  for (int i = 0; i < SensorData::numOfSensor_ERS210; i++)
  { 
    for (int l = 0; l < SensorDataBuffer::maxNumOfFrames; l++)
    {
      for (int d = 0; d < 16; d++)
      {
        floatingAvg[i][l][d] = 0;
      }
      minAvg[i][l] = 0;
    }
    
    
    // i = number of joints, j = Ringbuffer element, k = sensor or actor data
    // t = lower or upper threshold bound
    
    for (int j = 0; j < 128; j++)
    {
      for (int k = 0; k < 2; k++) 
      {
        ringBuffer[i][j][k] = 0;
      }
    }
    for (int t = 0; t < 3; t++)
    {
      threshold[i][t] = 0;
    }
  }
  
  // unused angle values
  threshold[SensorData::legFL3][2] = infin;
  threshold[SensorData::legFR3][2] = infin;
  threshold[SensorData::legHL3][2] = infin;
  threshold[SensorData::legHR3][2] = infin;
  
}


void DefaultCollisionDetector::calculateThresholds() 
{
  //Head thresholds are indifferent to motions and directions
  walkingMode = 0;
  
  threshold[SensorData::legFL1][2] = 1;
  threshold[SensorData::legHL1][2] = 1;
  threshold[SensorData::legFL2][2] = 1;
  threshold[SensorData::legHL2][2] = 1;
  
  //Threshold calculation for straight forward walking only
  if  ((currentMotionX > frontMin) && (currentMotionX <= frontMax))
  {
    walkingMode = 1; //foreward
   
    if (currentMotionX <= 5)
    {
      threshold[SensorData::legFL1][1] = 800; threshold[SensorData::legFL2][1]  =  500;  
      threshold[SensorData::legHL1][1] = 1000; threshold[SensorData::legHL2][1] = 1200; 
    }
    else if  (currentMotionX <= 30)
    {
      threshold[SensorData::legFL1][1] =  500; threshold[SensorData::legFL2][1] = 2000;  
      threshold[SensorData::legHL1][1] = 1200; threshold[SensorData::legHL2][1] = 2000; 
    }
    else if (currentMotionX <= 60)
    {
      threshold[SensorData::legFL1][1] =  500; threshold[SensorData::legFL2][1] = 4000;  
      threshold[SensorData::legHL1][1] = 1200; threshold[SensorData::legHL2][1] = 4000; 
    }
    else if (currentMotionX <= 90)
    {
      threshold[SensorData::legFL1][1] =  500; threshold[SensorData::legFL2][1] = 5000;
      threshold[SensorData::legHL1][1] = 1200; threshold[SensorData::legHL2][1] = 5000; 
    }
    else if (currentMotionX <= 120)
    {
      threshold[SensorData::legFL1][1] = 1000; threshold[SensorData::legFL2][1] = 5000;
      threshold[SensorData::legHL1][1] = 1500; threshold[SensorData::legHL2][1] = 5000; 
    }
    else if (currentMotionX <= 150)
    {
      threshold[SensorData::legFL1][1] = 1500; threshold[SensorData::legFL2][1] = 5000;
      threshold[SensorData::legHL1][1] = 1500; threshold[SensorData::legHL2][1] = 5000; 
    }
    else if (currentMotionX <= 180)
    {
      threshold[SensorData::legFL1][1] = 2500; threshold[SensorData::legFL2][1] = 5000;
      threshold[SensorData::legHL1][1] = 1500; threshold[SensorData::legHL2][1] = 6000; 
    }
    else if (currentMotionX <= 210)
    {
      threshold[SensorData::legFL1][1] = 3000; threshold[SensorData::legFL2][1] = 5000;  
      threshold[SensorData::legHL1][1] = 1500; threshold[SensorData::legHL2][1] = 6000;  
    }
    else if (currentMotionX <= 240)
    {
      threshold[SensorData::legFL1][1] = 6000; threshold[SensorData::legFL2][1] = 5000;
      threshold[SensorData::legHL1][1] = 3000; threshold[SensorData::legHL2][1] = 5000; 
    }
    else 
    {
      threshold[SensorData::legFL1][1] = 8000; threshold[SensorData::legFL2][1] = 5000;  
      threshold[SensorData::legHL1][1] = 3000; threshold[SensorData::legHL2][1] = 5000; 
    }
  
  threshold[SensorData::legFL1][2] += int(threshold[SensorData::legFL1][1] * scaleThresholdFore);
  threshold[SensorData::legFL2][2] += int(threshold[SensorData::legFL2][1] * scaleThresholdFore);
  threshold[SensorData::legHL1][2] += int(threshold[SensorData::legHL1][1] * scaleThresholdFore);
  threshold[SensorData::legHL2][2] += int(threshold[SensorData::legHL2][1] * scaleThresholdFore);
 }
  
  
  //Threshold calculation for straight backward walking only
  else if  ((currentMotionX < backMin) && (currentMotionX >= backMax))
  {
    walkingMode = 2;
    
    if  (currentMotionX >= -5)
    {
      threshold[SensorData::legFL1][1] = 1500; threshold[SensorData::legFL2][1] = 2000;
      threshold[SensorData::legHL1][1] = 1500; threshold[SensorData::legHL2][1] = 2000;
    }
    else if (currentMotionX >= -30)
    {
      threshold[SensorData::legFL1][1] = 1000; threshold[SensorData::legFL2][1] = 4000; 
      threshold[SensorData::legHL1][1] = 2000; threshold[SensorData::legHL2][1] = 4000; 
    }
    else if (currentMotionX >= -60)
    {
      threshold[SensorData::legFL1][1] = 1500; threshold[SensorData::legFL2][1] = 6000; 
      threshold[SensorData::legHL1][1] = 2500; threshold[SensorData::legHL2][1] = 4000; 
    }
    else if (currentMotionX >= -90)
    {
      threshold[SensorData::legFL1][1] = 1000; threshold[SensorData::legFL2][1] = 4000;
      threshold[SensorData::legHL1][1] = 4000; threshold[SensorData::legHL2][1] = 4000;
    }
    else if (currentMotionX >= -120)
    {
      threshold[SensorData::legFL1][1] = 2000; threshold[SensorData::legFL2][1] = 2500;
      threshold[SensorData::legHL1][1] = 3000; threshold[SensorData::legHL2][1] = 4000;
    }
    else if (currentMotionX >= -150)
    {
      threshold[SensorData::legFL1][1] = 1500; threshold[SensorData::legFL2][1] = 2000; 
      threshold[SensorData::legHL1][1] = 4000; threshold[SensorData::legHL2][1] = 1500; 
    }
    else if (currentMotionX >= -180)
    {
      threshold[SensorData::legFL1][1] = 2500; threshold[SensorData::legFL2][1] = 2000;
      threshold[SensorData::legHL1][1] = 4000; threshold[SensorData::legHL2][1] = 1500;
    }
    else
    {
      threshold[SensorData::legFL1][1] = 9000;
      threshold[SensorData::legFL2][1] = 5000;
      threshold[SensorData::legHL1][1] = 7000;
      threshold[SensorData::legHL2][1] = 9000;
    }
  threshold[SensorData::legFL1][2] += int(threshold[SensorData::legFL1][1] * scaleThresholdBack);
  threshold[SensorData::legFL2][2] += int(threshold[SensorData::legFL2][1] * scaleThresholdBack);
  threshold[SensorData::legHL1][2] += int(threshold[SensorData::legHL1][1] * scaleThresholdBack);
  threshold[SensorData::legHL2][2] += int(threshold[SensorData::legHL2][1] * scaleThresholdBack);
  }
 
  //Threshold calculation for left and right walking 
  
  if (
    (fabs(currentMotionY) > sideMin) &&
    (fabs(currentMotionY) <= sideMax)
    )
  {
    walkingMode = walkingMode * 10 + 3;
    
    if (fabs(currentMotionY) <= 5)
    {
      threshold[SensorData::legFL1][1] = 1000; threshold[SensorData::legFL2][1] = 1500; 
      threshold[SensorData::legHL1][1] = 2000; threshold[SensorData::legHL2][1] = 3000;
    }
    else if 
      (fabs(currentMotionY) <= 30)
    {
      threshold[SensorData::legFL1][1] = 2000; threshold[SensorData::legFL2][1] = 7000;
      threshold[SensorData::legHL1][1] = 4000; threshold[SensorData::legHL2][1] = 6000; 
    }
    else if 
      (fabs(currentMotionY) <= 60)
    {
      threshold[SensorData::legFL1][1] = 4000; threshold[SensorData::legFL2][1] = 8000;
      threshold[SensorData::legHL1][1] = 7000; threshold[SensorData::legHL2][1] = 8000;
    }
    else if 
      (fabs(currentMotionY) <= 90)
    {
      threshold[SensorData::legFL1][1] = 3000; threshold[SensorData::legFL2][1] = 7000;
      threshold[SensorData::legHL1][1] = 5000; threshold[SensorData::legHL2][1] = 6000; 
    }
    else if 
      (fabs(currentMotionY) <= 120)
    {
      threshold[SensorData::legFL1][1] = 1500; threshold[SensorData::legFL2][1] =12000;
      threshold[SensorData::legHL1][1] = 2000; threshold[SensorData::legHL2][1] = 7000; 
    }
    else if 
      (fabs(currentMotionY) <= 150)
    {
      threshold[SensorData::legFL1][1] = 1500; threshold[SensorData::legFL2][1] =15000;
      threshold[SensorData::legHL1][1] = 4000; threshold[SensorData::legHL2][1] = 7000; 
    }
    else if 
      (fabs(currentMotionY) <= 180)
    {
      threshold[SensorData::legFL1][1] = 1500; threshold[SensorData::legFL2][1] =300000; 
      threshold[SensorData::legHL1][1] =15000; threshold[SensorData::legHL2][1] =  7000; 
    }
    else if 
      (fabs(currentMotionY) <= 210)
    {
      threshold[SensorData::legFL1][1] = 2000; threshold[SensorData::legFL2][1] =600000; 
      threshold[SensorData::legHL1][1] =20000; threshold[SensorData::legHL2][1] = 25000; 
    }
    else if 
      (fabs(currentMotionY) <= 240)
    {
      threshold[SensorData::legFL1][1] = 4000; threshold[SensorData::legFL2][1] =800000;
      threshold[SensorData::legHL1][1] =20000; threshold[SensorData::legHL2][1] = 30000;
    }
    else if 
      (fabs(currentMotionY) <= 250)
    {
      threshold[SensorData::legFL1][1] = 4000; threshold[SensorData::legFL2][1] =800000; 
      threshold[SensorData::legHL1][1] =20000; threshold[SensorData::legHL2][1] = 40000; 
    }
    else
    {
      threshold[SensorData::legFL1][1] = 4000; threshold[SensorData::legFL2][1] =80000;
      threshold[SensorData::legHL1][1] =20000; threshold[SensorData::legHL2][1] =50000;
    }
  } //sideThresholds ended

  threshold[SensorData::legFL1][2] += int(threshold[SensorData::legFL1][1] * scaleThresholdSide);
  threshold[SensorData::legFL2][2] += int(threshold[SensorData::legFL2][1] * scaleThresholdSide);
  threshold[SensorData::legHL1][2] += int(threshold[SensorData::legHL1][1] * scaleThresholdSide);
  threshold[SensorData::legHL2][2] += int(threshold[SensorData::legHL2][1] * scaleThresholdSide);
  
  //Threshold calculation for rotation no matter if there is any other direction
  // rotation only
    if ((fabs(currentRotation) > rotMin) && (fabs(currentRotation) <= rotMax)) 
    {
      walkingMode = walkingMode * 10 + 5;     
      
      if (fabs(currentRotation) <= 0.05)
      {
        threshold[SensorData::legFL1][1] =   800; threshold[SensorData::legFL2][1] =  1500; 
        threshold[SensorData::legHL1][1] =  2000; threshold[SensorData::legHL2][1] =  2500; 
      }
      else if (fabs(currentRotation) <= 0.3)
      {
        threshold[SensorData::legFL1][1] =   800; threshold[SensorData::legFL2][1] =  7000; 
        threshold[SensorData::legHL1][1] =  1000; threshold[SensorData::legHL2][1] =  2500; 
      }
      else if (fabs(currentRotation) <= 0.6)
      {
        threshold[SensorData::legFL1][1] =  1000; threshold[SensorData::legFL2][1] =  4000;
        threshold[SensorData::legHL1][1] =  1500; threshold[SensorData::legHL2][1] =  2500; 
      }
      else if (fabs(currentRotation) <= 0.9)
      {
        threshold[SensorData::legFL1][1] =  1000; threshold[SensorData::legFL2][1] =  3000; 
        threshold[SensorData::legHL1][1] =  2000; threshold[SensorData::legHL2][1] =  3000; 
      }
      else if (fabs(currentRotation) <= 1.2)
      {
        threshold[SensorData::legFL1][1] =  1000; threshold[SensorData::legFL2][1] =  3000; 
        threshold[SensorData::legHL1][1] =  2000; threshold[SensorData::legHL2][1] =  3000; 
      }
      else if (fabs(currentRotation) <= 1.5)
      {
        threshold[SensorData::legFL1][1] =  1000; threshold[SensorData::legFL2][1] =  3000; 
        threshold[SensorData::legHL1][1] =  3000; threshold[SensorData::legHL2][1] =  3000; 
      }
      else if (fabs(currentRotation) <= 1.8)
      {
        threshold[SensorData::legFL1][1] =  1000; threshold[SensorData::legFL2][1] = 6000; 
        threshold[SensorData::legHL1][1] =  2500; threshold[SensorData::legHL2][1] = 3500; 
      }
      else if (fabs(currentRotation) <= 2.1)
      {
        threshold[SensorData::legFL1][1] =  1000; threshold[SensorData::legFL2][1] = 6000; 
        threshold[SensorData::legHL1][1] =  2500; threshold[SensorData::legHL2][1] = 3500; 
      }
      else if (fabs(currentRotation) <= 2.4)
      {
        threshold[SensorData::legFL1][1] =  2000; threshold[SensorData::legFL2][1] = 7000; 
        threshold[SensorData::legHL1][1] =  3500; threshold[SensorData::legHL2][1] = 3500; 
      }
      
      else //max
      {
        threshold[SensorData::legFL1][1] =  3000; threshold[SensorData::legFL2][1] =15000;
        threshold[SensorData::legHL1][1] =  5000; threshold[SensorData::legHL2][1] = 4000;
      }

  threshold[SensorData::legFL1][2] += int(threshold[SensorData::legFL1][1] * scaleThresholdRot);
  threshold[SensorData::legFL2][2] += int(threshold[SensorData::legFL2][1] * scaleThresholdRot);
  threshold[SensorData::legHL1][2] += int(threshold[SensorData::legHL1][1] * scaleThresholdRot);
  threshold[SensorData::legHL2][2] += int(threshold[SensorData::legHL2][1] * scaleThresholdRot);
  }
 
  
///To Test 
if (
    (
     (fabs(currentRotation) >= 0.75) && 
      (
       (fabs(currentMotionX) >= 100) ||
       (fabs(currentMotionY) >= 100 )
      )
     ) 
     ||
     (
      (fabs(currentMotionX) >= 100) &&
      (fabs(currentMotionY) >= 100)
     )
    )
  {
   threshold[SensorData::legFL1][2] *= 2;
   threshold[SensorData::legFL2][2] *= 2;
   threshold[SensorData::legHL1][2] *= 2;
   threshold[SensorData::legHL2][2] *= 2;
  }
   
      
  
  //default
  if (walkingMode == 0)
  {
    threshold[SensorData::legFL1][2] = infin;
    threshold[SensorData::legFL2][2] = infin;
    threshold[SensorData::legHL1][2] = infin;
    threshold[SensorData::legHL2][2] = infin;
  }
  /*
  threshold[SensorData::legFL1][2] =  1000;
  threshold[SensorData::legFL2][2] =  1000;
  threshold[SensorData::legHL1][2] =  1000;
  threshold[SensorData::legHL2][2] =  1000;
  */
  
  
  /* Head values */
  
  threshold[SensorData::headTilt][2] = 20000;
  threshold[SensorData::headPan][2]  = 300000;
  threshold[SensorData::headRoll][2] = 20000;

 
  /* Symmetric exchange */
  
  threshold[SensorData::legFR1][2] = threshold[SensorData::legFL1][2];
  threshold[SensorData::legFR2][2] = threshold[SensorData::legFL2][2];
  threshold[SensorData::legHR1][2] = threshold[SensorData::legHL1][2];
  threshold[SensorData::legHR2][2] = threshold[SensorData::legHL2][2];
  
}



void DefaultCollisionDetector::execute()
{     
  collisionPercept.setFrameNumber(sensorDataBuffer.frame[0].frameNumber);
  currentMotionX = executedMotionRequest.walkParams.translation.x; // get the motion values
  currentMotionY = executedMotionRequest.walkParams.translation.y;
  currentRotation = executedMotionRequest.walkParams.rotation;
  walkingMode = 0;                       // reset walking Mode
  
  collisionPercept.state[JointData::legFL3] = 0; // 3rd angles are not calculated
  collisionPercept.state[JointData::legHL3] = 0;
  collisionPercept.state[JointData::legFR3] = 0;
  collisionPercept.state[JointData::legHR3] = 0;
  
  
  for (int i = 0; i < SensorData::numOfSensor_ERS210; i++) 
  { // for all sensors    
    
    if ((i == SensorData::headTilt) || (i == SensorData::headPan) ||
      (i == SensorData::headRoll) || 
      (i == SensorData::legFL1) || (i == SensorData::legFL2) ||
      (i == SensorData::legHL1) || (i == SensorData::legHL2) ||
      (i == SensorData::legFR1) || (i == SensorData::legFR2) ||
      (i == SensorData::legHR1) || (i == SensorData::legHR2)) //consider just 2*4 leg angles and 3 head angles
    {
      for (int quarter = 0; quarter < sensorDataBuffer.numOfFrames; quarter++) // for all SensorDataElements values in the buffer
      { 
        int current = index + quarter; // current is the actual frame
        
        min = 5;  //minimum delay value
        max = 12; //maximum delay value 
        minAvg[i][current%SensorDataBuffer::maxNumOfFrames] = min;
        
        
        for (int d = min; d <= max; d++) // for all possible delays
        {
          
          oldVal = (int)(((ringBuffer[i][(current + 128 - q)%128][0] - 
            ringBuffer[i][(current + 128 - q - d)%128][1])*
            (ringBuffer[i][(current + 128 - q)%128][0] - 
            ringBuffer[i][(current + 128 - q - d)%128][1]))/12); // the old value off the Queue
          
          
          
          ringBuffer[i][current%128][0] = (int)(sensorDataBuffer.frame[quarter].data[i]/32);
          ringBuffer[i][current%128][1] = (int)(sensorDataBuffer.frame[quarter].refValue[i]/32);
          
          if (current < 16) 
          {
            ringBuffer[i][current%128][0] = 0;
          }
          
          //unbedingt nach der Ringbufferuebergabe !!!
          
          newVal = (int)(((ringBuffer[i][current%128][0] - 
            ringBuffer[i][(current + 128 - d)%128][1])*
            (ringBuffer[i][current%128][0] - 
            ringBuffer[i][(current + 128 - d)%128][1]))/12); // the new value for the Queue
          
          
          floatingAvg[i][current%SensorDataBuffer::maxNumOfFrames][d] =  newVal - oldVal + 
            floatingAvg[i][(current+SensorDataBuffer::maxNumOfFrames-1)%SensorDataBuffer::maxNumOfFrames][d];  
          
          
          if (floatingAvg[i][current%SensorDataBuffer::maxNumOfFrames][d] < floatingAvg[i][current%SensorDataBuffer::maxNumOfFrames][minAvg[i][current%SensorDataBuffer::maxNumOfFrames]])
          {
            minAvg[i][current%SensorDataBuffer::maxNumOfFrames] = d;
          }
        } // all delays
      } // all SensorDataBufferElements
    } // just head and Legs
  } //all angles
  
  
  xRatio = fabs(currentMotionX-lastMotionX);
  yRatio = fabs(currentMotionY-lastMotionY);
  rRatio = fabs(currentRotation-lastRotation);
  
  collisionPercept.reset();
  
  
  if  // calculates the behaviour threshold
    (
    (motionChangeThresholdX < xRatio) ||
    (motionChangeThresholdY < yRatio) ||
    (motionChangeThresholdR < rRatio)
    )  
  {
    lastMotionChange = 0;
    
    /* 09112003
    if (rRatio * 50 / motionChangeThresholdR > lastTooBigMotionChange)
    {lastTooBigMotionChange = rRatio * 50 / motionChangeThresholdR;}
    
    if (yRatio * 50 / motionChangeThresholdY > lastTooBigMotionChange)
    {lastTooBigMotionChange = yRatio * 50 / motionChangeThresholdY;}
    
    if (xRatio * 50 / motionChangeThresholdX> lastTooBigMotionChange)
    { lastTooBigMotionChange = xRatio * 50 / motionChangeThresholdX;}
    
    if (lastTooBigMotionChange > 100) { lastTooBigMotionChange = 100; } // lastTooBigMotionChange lies between 50 and 100    
    */

  }
  
  
  if (lastMotionChange <= calmDown) //calculates the behaviour cleaned thresholds
  {
    /* 09112003
    motionChangeThresholdFactor = lastTooBigMotionChange/100 * 200; // between 99 and 1
    motionChangeThresholdFactor = (motionChangeThresholdFactor - 1)*
      (1 - lastMotionChange/calmDown) + 1;
    */

    motionChangeThresholdFactor = 1000000;
    if (lastMotionChange == calmDown)
    {
      motionChangeThresholdFactor = 1;
    }

  }
  
  if  ((executedMotionRequest.motionType==MotionRequest::walk)) // no calming down anymore
  {
    calculateThresholds(); // calculates the threshold values considering speed and direction
    
    collisionPercept.state[JointData::headTilt]  = floatingAvg[SensorData::headTilt][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::headTilt][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::headTilt][2];  //Head tilt
    collisionPercept.state[JointData::headPan]   = floatingAvg[SensorData::headPan][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::headPan][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::headPan][2];  //Head pan
    collisionPercept.state[JointData::headRoll]  = floatingAvg[SensorData::headRoll][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::headRoll][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::headRoll][2];  //Head roll
    
    collisionPercept.state[JointData::legFL1] = floatingAvg[SensorData::legFL1][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFL1][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFL1][2] / motionChangeThresholdFactor; //FrontLeft1
    collisionPercept.state[JointData::legFR1] = floatingAvg[SensorData::legFR1][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFR1][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFR1][2] / motionChangeThresholdFactor; //FrontRight1
    collisionPercept.state[JointData::legHL1] = floatingAvg[SensorData::legHL1][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHL1][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHL1][2] / motionChangeThresholdFactor; //HindLeft1
    collisionPercept.state[JointData::legHR1] = floatingAvg[SensorData::legHR1][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHR1][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHR1][2] / motionChangeThresholdFactor; //HindRight1
    
    collisionPercept.state[JointData::legFL2] = floatingAvg[SensorData::legFL2][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFL2][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFL2][2] / motionChangeThresholdFactor; //FrontLeft2
    collisionPercept.state[JointData::legFR2] = floatingAvg[SensorData::legFR2][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFR2][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFR2][2] / motionChangeThresholdFactor; //FrontRight2
    collisionPercept.state[JointData::legHL2] = floatingAvg[SensorData::legHL2][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHL2][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHL2][2] / motionChangeThresholdFactor; //HindLeft2
    collisionPercept.state[JointData::legHR2] = floatingAvg[SensorData::legHR2][index%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHR2][index%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHR2][2] / motionChangeThresholdFactor; //HindRight2
    
    
    // look for bigger values of the SensorDataBuffer
    for (int current = index + 1; current < (index + sensorDataBuffer.numOfFrames); current++)
    {
      if (collisionPercept.state[JointData::headTilt] < floatingAvg[SensorData::headTilt][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::headTilt][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::headTilt][2])  
      {collisionPercept.state[JointData::headTilt] =  floatingAvg[SensorData::headTilt][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::headTilt][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::headTilt][2];} //Head tilt
      
      if (collisionPercept.state[JointData::headPan]  < floatingAvg[SensorData::headPan][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::headPan][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::headPan][2])
      {collisionPercept.state[JointData::headPan] =  floatingAvg[SensorData::headPan][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::headPan][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::headPan][2];}  //Head pan
      
      if (collisionPercept.state[JointData::headRoll] < floatingAvg[SensorData::headRoll][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::headRoll][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::headRoll][2])
      {collisionPercept.state[JointData::headRoll] =  floatingAvg[SensorData::headRoll][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::headRoll][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::headRoll][2];}//Head roll
      
      
      
      if (collisionPercept.state[JointData::legFL1] < floatingAvg[SensorData::legFL1][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFL1][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFL1][2] / motionChangeThresholdFactor)
      {collisionPercept.state[JointData::legFL1] = floatingAvg[SensorData::legFL1][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFL1][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFL1][2] / motionChangeThresholdFactor;}//FrontLeft1
      
      if (collisionPercept.state[JointData::legFR1] < floatingAvg[SensorData::legFR1][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFR1][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFR1][2] / motionChangeThresholdFactor) 
      {collisionPercept.state[JointData::legFR1] = floatingAvg[SensorData::legFR1][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFR1][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFR1][2] / motionChangeThresholdFactor;}//FrontRight1
      
      if (collisionPercept.state[JointData::legHL1] < floatingAvg[SensorData::legHL1][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHL1][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHL1][2] / motionChangeThresholdFactor) 
      {collisionPercept.state[JointData::legHL1] = floatingAvg[SensorData::legHL1][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHL1][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHL1][2] / motionChangeThresholdFactor;}//HindLeft1
      
      if (collisionPercept.state[JointData::legHR1] < floatingAvg[SensorData::legHR1][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHR1][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHR1][2] / motionChangeThresholdFactor) 
      {collisionPercept.state[JointData::legHR1] = floatingAvg[SensorData::legHR1][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHR1][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHR1][2] / motionChangeThresholdFactor;}//HindRight1
      
      
      
      if (collisionPercept.state[JointData::legFL2] < floatingAvg[SensorData::legFL2][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFL2][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFL2][2] / motionChangeThresholdFactor)  
      {collisionPercept.state[JointData::legFL2] =  floatingAvg[SensorData::legFL2][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFL2][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFL2][2] / motionChangeThresholdFactor;}//FrontLeft2
      
      if (collisionPercept.state[JointData::legFR2] < floatingAvg[SensorData::legFR2][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFR2][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFR2][2] / motionChangeThresholdFactor) 
      {collisionPercept.state[JointData::legFR2] =  floatingAvg[SensorData::legFR2][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legFR2][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legFR2][2] / motionChangeThresholdFactor;}//FrontRight2
      
      if (collisionPercept.state[JointData::legHL2] < floatingAvg[SensorData::legHL2][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHL2][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHL2][2] / motionChangeThresholdFactor) 
      {collisionPercept.state[JointData::legHL2] =  floatingAvg[SensorData::legHL2][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHL2][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHL2][2] / motionChangeThresholdFactor;}//HindLeft2
      
      if (collisionPercept.state[JointData::legHR2] < floatingAvg[SensorData::legHR2][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHR2][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHR2][2] / motionChangeThresholdFactor) 
      {collisionPercept.state[JointData::legHR2] =  floatingAvg[SensorData::legHR2][current%SensorDataBuffer::maxNumOfFrames][minAvg[SensorData::legHR2][current%SensorDataBuffer::maxNumOfFrames]] / threshold[SensorData::legHR2][2] / motionChangeThresholdFactor;}//HindRight2
    }

      /* DEBUG START*/

      //if (lastMotionChange < calmDown) 
   /*   {
      OUTPUT(idText, text, motionChangeThresholdFactor << " \r\n " <<
        "   FL1: " << collisionPercept.state[JointData::legFL1] << 
        " FL2: " << collisionPercept.state[JointData::legFL2] << " \r\n " <<
        "   HL1: " << collisionPercept.state[JointData::legHL1] << 
        " HL2: " << collisionPercept.state[JointData::legHL2] << " \r\n " <<
        "   FR1: " << collisionPercept.state[JointData::legFR1] << 
        " FR2: " << collisionPercept.state[JointData::legFR2] << " \r\n " <<
        "   HR1: " << collisionPercept.state[JointData::legHR1] << 
        " HR2: " << collisionPercept.state[JointData::legHR2] << " \r\n " <<  
        "  index:  " << index << " X:" << executedMotionRequest.walkParams.translation.x<< 
        " Y:" << executedMotionRequest.walkParams.translation.y <<
        " R: " << executedMotionRequest.walkParams.rotation << "   \r\n \r\n ");
      }  
     */  
    
      /* RELEASE DEBUG
      if (collisionPercept.state[JointData::legFL1] > 700) {OUTPUT(idText, text, "FL1  " << collisionPercept.state[JointData::legFL1] << "  index:  " 
        << index << " X:" << executedMotionRequest.walkParams.translation.x << " Y: " << executedMotionRequest.walkParams.translation.y << " R: " 
        << executedMotionRequest.walkParams.rotation);}
      if (collisionPercept.state[JointData::legFL2] > 700) {OUTPUT(idText, text, "FL2  " << collisionPercept.state[JointData::legFL2] << "  index:  " 
        << index << " X:" << executedMotionRequest.walkParams.translation.x << " Y: " << executedMotionRequest.walkParams.translation.y << " R: " 
        << executedMotionRequest.walkParams.rotation);}
      if (collisionPercept.state[JointData::legHL1] > 700) {OUTPUT(idText, text, "HL1  " << collisionPercept.state[JointData::legHL1] << "  index:  " 
        << index << " X:" << executedMotionRequest.walkParams.translation.x << " Y: " << executedMotionRequest.walkParams.translation.y << " R: " 
        << executedMotionRequest.walkParams.rotation);}
      if (collisionPercept.state[JointData::legHL2] > 700) {OUTPUT(idText, text, "HL2  " << collisionPercept.state[JointData::legHL2] << "  index:  " 
        << index << " X:" << executedMotionRequest.walkParams.translation.x << " Y: " << executedMotionRequest.walkParams.translation.y << " R: " 
        << executedMotionRequest.walkParams.rotation);}
      
      if (collisionPercept.state[JointData::legFR1] > 700) {OUTPUT(idText, text, "FR1  " << collisionPercept.state[JointData::legFR1] << "  index:  " 
        << index << " X:" << executedMotionRequest.walkParams.translation.x << " Y: " << executedMotionRequest.walkParams.translation.y << " R: " 
        << executedMotionRequest.walkParams.rotation);}
      if (collisionPercept.state[JointData::legFR2] > 700) {OUTPUT(idText, text, "FR2  " << collisionPercept.state[JointData::legFR2] << "  index:  " 
        << index << " X:" << executedMotionRequest.walkParams.translation.x << " Y: " << executedMotionRequest.walkParams.translation.y << " R: " 
        << executedMotionRequest.walkParams.rotation);}
      if (collisionPercept.state[JointData::legHR1] > 700) {OUTPUT(idText, text, "HR1  " << collisionPercept.state[JointData::legHR1] << "  index:  " 
        << index << " X:" << executedMotionRequest.walkParams.translation.x << " Y: " << executedMotionRequest.walkParams.translation.y << " R: " 
        << executedMotionRequest.walkParams.rotation );}
      if (collisionPercept.state[JointData::legHR2] > 700) {OUTPUT(idText, text, "HR2  " << collisionPercept.state[JointData::legHR2] << "  index:  " 
        << index << " X:" << executedMotionRequest.walkParams.translation.x << " Y: " << executedMotionRequest.walkParams.translation.y << " R: " 
        << executedMotionRequest.walkParams.rotation );}
      */
      
      
      
      
      /*
      if (!(index%400))
      {     
        
        if (
          (currentMotionX) ||
          (currentMotionY) ||
          (currentRotation)
          )
          
        {
          OUTPUT(idText, text, "FL1 " << maxfl1 << "\r\n" <<  
            "FR1 " << maxfr1 << "\r\n" <<  
            "FL2 " << maxfl2 << "\r\n" <<  
            "FR2 " << maxfr2 << "\r\n" <<  
            "HL1 " << maxhl1 << "\r\n" <<  
            "HR1 " << maxhr1 << "\r\n" <<  
            "HL2 " << maxhl2 << "\r\n" <<  
            "HR2 " << maxhr2 << "\r\n" <<  
            "***********************" << "\r\n" << 
            index << " Mode: " << walkingMode << "\r\n" << 
            " X-sp " << currentMotionX << " X-Ratio " << xRatio << "\r\n" << 
            " Y-sp " << currentMotionY << " Y-Ratio " << yRatio << "\r\n" << 
            " R-sp " << currentRotation/3.1415*180 <<" r-Ratio " << rRatio << "\r\n" <<
            "*++++++++++++++*");       
        } 
        
        maxfl1 = 0;
        maxfl2 = 0;
        maxfr1 = 0;
        maxfr2 = 0;
        maxhl1 = 0;
        maxhl2 = 0;
        maxhr1 = 0;
        maxhr2 = 0;
        
      }
      
      if ((walkingMode))
      {
        
        if (maxfl1 < collisionPercept.state[JointData::legFL1])
        {maxfl1 = collisionPercept.state[JointData::legFL1];}
        if (maxfl2 < collisionPercept.state[JointData::legFL2])
        {maxfl2 = collisionPercept.state[JointData::legFL2];}
        if (maxfr1 < collisionPercept.state[JointData::legFR1])
        {maxfr1 = collisionPercept.state[JointData::legFR1];}
        if (maxfr2 < collisionPercept.state[JointData::legFR2])
        {maxfr2 = collisionPercept.state[JointData::legFR2];}
        if (maxhl1 < collisionPercept.state[JointData::legHL1])
        {maxhl1 = collisionPercept.state[JointData::legHL1];}
        if (maxhl2 < collisionPercept.state[JointData::legHL2])
        {maxhl2 = collisionPercept.state[JointData::legHL2];}
        if (maxhr1 < collisionPercept.state[JointData::legHR1])
        {maxhr1 = collisionPercept.state[JointData::legHR1];}
        if (maxhr2 < collisionPercept.state[JointData::legHR2])
        {maxhr2 = collisionPercept.state[JointData::legHR2];}
        
      }
      */ // DEBUG END
      
      
      
      }
      index += sensorDataBuffer.numOfFrames; // increase index value
      lastMotionChange += sensorDataBuffer.numOfFrames; // increase value since Motion Change
      lastMotionX = currentMotionX;          // remember last Motion X, Y and Rotation
      lastMotionY = currentMotionY;
      lastRotation= currentRotation;
}


/*
* Change log :
* 
* $Log: DefaultCollisionDetector.cpp,v $
* Revision 1.7  2004/04/19 10:31:38  goehring
* no message
*
* Revision 1.6  2004/04/14 16:04:43  goehring
* Thresholds modified to ERS7
*
* Revision 1.5  2004/03/08 01:38:49  roefer
* Interfaces should be const
*
* Revision 1.4  2003/12/31 20:16:13  roefer
* SensorData for ERS-7
*
* Revision 1.3  2003/11/14 19:02:26  goehring
* frameNumber added
*
* Revision 1.2  2003/10/16 13:00:03  goehring
* RotationThreshold doubled, TooBigMotionChangeThreshold changed from continous to discrete
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.9  2003/09/26 21:23:20  loetzsch
* renamed class JointState to CollisionPercept
*
* Revision 1.8  2003/07/15 12:49:14  goehring
* Speed optimized
*
* Revision 1.7  2003/07/08 00:42:25  goehring
* Review, Debug
*
* Revision 1.6  2003/07/06 19:43:52  goehring
* First Release Version
*
* Revision 1.5  2003/07/06 07:26:31  goehring
* CollisionModell changed to equidistant threshold intervalls
*
* Revision 1.4  2003/07/05 23:47:45  goehring
* Debug
*
* Revision 1.3  2003/07/05 23:23:51  goehring
* Debugging and Improvement, Adaptation to GT2003 Walking Engine
*
* Revision 1.2  2003/07/04 09:24:09  goehring
* Improved the collision threshold modell
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.17  2003/06/21 16:46:48  goehring
* Review
*
* Revision 1.16  2003/06/20 20:20:43  goehring
* ExecutedMotionReques instead of MotionRequest implemented
*
* Revision 1.15  2003/06/20 16:57:45  goehring
* CollisionDetectorValues improved
*
* Revision 1.14  2003/06/20 10:29:27  goehring
* MotionCombination thresholds implemented, Review
*
* Revision 1.13  2003/06/18 18:12:07  goehring
* Thresholds improved
*
* Revision 1.12  2003/06/13 12:39:51  goehring
* Collision algorithm improved
*
* Revision 1.11  2003/06/10 15:05:32  goehring
* Review
*
* Revision 1.10  2003/06/05 23:01:22  goehring
* Several bugfixes  and value smoothing implemented
*
* Revision 1.9  2003/06/04 18:24:15  goehring
* Debugging of the Interface for the HistoryViewer
*
* Revision 1.8  2003/06/04 14:44:14  goehring
* Debug
*
* Revision 1.7  2003/06/04 12:22:27  goehring
* Debug
*
* Revision 1.6  2003/06/03 09:27:54  goehring
* HistoryValues implemented
*
* Revision 1.5  2003/06/02 09:40:13  goehring
* Review
*
* Revision 1.4  2003/05/23 13:08:26  goehring
* Number of Sensorvalues changed from 4 to numOfValues (BETA)
*
* Revision 1.3  2003/05/23 09:36:51  goehring
* MotionStabilizer gets all 4 Sensor Values
*
* Revision 1.2  2003/05/16 08:09:34  goehring
* Testroutines implemented
*
* Revision 1.1  2003/04/25 19:48:08  goehring
* Added new module CollisionDetector
*
*/
