/**
* @file DefaultGetupEngine.cpp
* 
* This file contains the default getup engine.
* 
* @author Max Risler
* @author Andreas Heinze
*/

#include "DefaultGetupEngine.h"
#include "Platform/SystemCall.h"
#include "Tools/Debugging/Debugging.h"
#include "Tools/RobotConfiguration.h"


/* GT2001 Mof sequence
* ~ ~ ~   ~ ~ ~   2000 0 2500      2000 0 2500      2000 0 2500      2000 0 2500       0 80
* ~ ~ ~   ~ ~ ~   1500 1500 2500   1500 1500 2500   1500 1500 2500   1500 1500 2500    0 30
* ~ ~ ~   ~ ~ ~   -1600 1500 2500  -1600 1500 2500  -1400 1600 2500  -1400 1600 2500   0 50
* ~ ~ ~   ~ ~ ~   -1300 800 2100   -1300 800 2100   -1200 1000 2300  -1200 1000 2300   0 10
* ~ ~ ~   ~ ~ ~   -426 356 1626   -426 356 1626   -662 456 1784   -662 456 1784   1 25
*
* current sequence 
* ~ ~ ~   ~ ~ ~   2000 0 2500      2000 0 2500      2000 0 2500      2000 0 2500       0 80
* ~ ~ ~   ~ ~ ~   1500 1500 2500   1500 1500 2500   1500 1500 2500   1500 1500 2500    0 30
* ~ ~ ~   ~ ~ ~   -1600 1500 2500  -1600 1500 2500  -1400 1600 2500  -1400 1600 2500   0 50
* ~ ~ ~   ~ ~ ~   0 800 2100   0 800 2100   -1200 1000 2300  -1200 1000 2300   0 10
*/


/* GT2004 ERS7 Mof sequence

  * -------------for lying on one side------------------------
  *
  * right legs transformation:
  * -159  -230   540   ~ ~ ~    2112  -119  1683      -5   199  1916    1985   -91  2053    -488    39  1373   1 30
  * -159  -230   540   ~ ~ ~    2112  -119  1683      -5   199  1916    1985   -91  2053    -488    39  1373   1 50
  *  -82     7   507   ~ ~ ~    1498  -142  2120     -28   233  1911    1222  -125  2113    -464     5  1411   0 25
  * -142   -89   468   ~ ~ ~    1487  -142  2142     -28  1571  1905    1406  -125  2118      78  1423  2102   0 25
  *
  *
  * or
  *
  *
  * left legs transformation:
  * -159   230  -540   ~ ~ ~      -5   199  1916    2112  -119  1683    -488    39  1373    1985   -91  2053   1 30
  * -159   230  -540   ~ ~ ~      -5   199  1916    2112  -119  1683    -488    39  1373    1985   -91  2053   1 50
  * -82    -7  -507   ~ ~ ~     -28   233  1911    1498  -142  2120    -464     5  1411    1222  -125  2113   0 25
  * -142    89  -468   ~ ~ ~     -28  1571  1905    1487  -142  2142      78  1423  2102    1406  -125  2118   0 25
  *
  *
  *
  *
  * -----------for sitting upright-----------------------------
  * sit down:
  * -5 151 638  ~ ~ ~  -129 267 183  37 56 -79  -811 216 258 -699 11 -17  1 50
  *
  * legs to sides:
  * -82 125 553  ~ ~ ~  -73 239 120  93 68 -34  -823 1321 258 -724 1418 -62  1 25
  * -65 114 553  ~ ~ ~  -90 301 109  81 5 -17  816 1344 227 799 1383 -79  1 25
  *
  *
  *----------------------make a roll-----------------------
  *
  * to back:
  * -1346 1564 371  ~ ~ ~  -112 1623 114  -5 1623 219  -54 1520 221 0 1457 362  0 50
  *
  * legs up:
  * -1000 438 364  ~ ~ ~  0 -182 91  -84 -182 184  -73 -250 184 24 -250 338  0 60
  *
  
    
      *---------------------headstand to stand--------------------
      * move legs:
      * 16 -70 872  ~ ~ ~  -73 244 2125  -164 324 2044  -97 273 184 72 153 239  1 50
      
        * head press:
        * -1346 -3 586  ~ ~ ~  -73 273 2040  -90 353 2077  -73 267 184 109 148 307  1 70
        *
        *
        *-------------------------unknown--------------------------
        * stand:
        * -38 18 781 ~ ~ ~  -62 273 1902  -62 273 1902  -589 216 1385 -581 171 1378  0 60
        * -38 18 781 ~ ~ ~  -62 273 1902  -62 273 1902  -589 216 1385 -581 171 1378  0 60
        *
        * sit down:
        * -49 -14 573  ~ ~ ~  -140 261 160  24 -79 -153  -683 432 -45 -693 222 -51  0 70
        *
        * lay down
        * -141 98 646  ~ ~ ~  978 45 977  997 5 917  -1499 290 460 -1396 318 218  1 50
        *
        * move up:
        * -175 99 579  ~ ~ ~  1232 45 2125  1143 85 2066  -2038 261 559 -1872 313 104  0 50
        * -175 106 586  ~ ~ ~  894 28 2125  816 74 2038  -2033 381 1531 -1921 307 1552  0 50
        * -175 95 592  ~ ~ ~  905 56 2142  829 91 2077  -1036 102 2167 -1149 -11 2167  0 50
        *
        * stand:
        * -38 18 781 ~ ~ ~  -62 273 1902  -62 273 1902  -589 216 1385 -581 171 1378  0 60
        * -38 18 781 ~ ~ ~  -62 273 1902  -62 273 1902  -589 216 1385 -581 171 1378  0 60
        *
        *
        * --------new roll type--------------------
        * move right legs:
        * -1173 -23 -220  ~ ~ ~  1416 -68 2006  6 -239 1355  970 170 2047 -546 -159 1449  0 100
        * -1129 -116 -122  ~ ~ ~  2048 -102 2108  -22 -136 -28  1601 -79 460 96 -256 184  0 25
        *
        * move left legs
        * -1173    23   220   ~ ~ ~       6  -239  1355    1416   -68  2006    -546  -159  1449     970   170  2047   0 100
        * -1129   116   122   ~ ~ ~     -22  -136   -28    2048  -102  2108      96  -256   184    1601   -79   460   0 50
        
*/

DefaultGetupEngine::DefaultGetupEngine(const GetupEngineInterfaces& interfaces) 
: GetupEngine(interfaces),
stateStartTime(SystemCall::getCurrentSystemTime()),
lastGetupTime(0),
state(standing),
lastState(standing)
{
  
#define I jointDataInvalidValue
  
  
  sequence[unknown].numOfEntries = 3;
  sequence[unknown].loop = false;
  sequence[unknown].entry[0] = JointData(0,-1500000,0,I,I,I, 2000000,0,2500000, 2000000,0,2500000, 2000000,0,2500000, 2000000,0,2500000);
  sequence[unknown].interpolate[0] = false; 
  sequence[unknown].repeats[0] = 80;
  sequence[unknown].entry[1] = JointData(0,-1500000,0,I,I,I, 1500000,1500000,2500000, 1500000,1500000,2500000, 1500000,1500000,2500000, 1500000,1500000,2500000);
  sequence[unknown].interpolate[1] = false; 
  sequence[unknown].repeats[1] = 30;
  sequence[unknown].entry[2] = JointData(0,-1500000,0,I,I,I, -1600000,1500000,2500000, -1600000,1500000,2500000, -1400000,1600000,2500000, -1400000,1600000,2500000);
  sequence[unknown].interpolate[2] = false; 
  sequence[unknown].repeats[2] = 100;
  
  sequence[leftSideDown].numOfEntries = 2;
  sequence[leftSideDown].loop = false;
  sequence[leftSideDown].entry[0] = JointData(0,-1500000,0,I,I,I, 0,0,0, 2000000,0,0, 0,0,0, 2000000,0,0);
  sequence[leftSideDown].interpolate[0] = false; 
  sequence[leftSideDown].repeats[0] = 120;
  sequence[leftSideDown].entry[1] = JointData(0,-1500000,0,I,I,I, -1600000,1500000,2500000, -1600000,1500000,2500000, -1400000,1600000,2500000, -1400000,1600000,2500000);
  sequence[leftSideDown].interpolate[1] = false; 
  sequence[leftSideDown].repeats[1] = 100;
  
  sequence[rightSideDown].numOfEntries = 2;
  sequence[rightSideDown].loop = false;
  sequence[rightSideDown].entry[0] = JointData(0,1500000,0,I,I,I, 2000000,0,0, 0,0,0, 2000000,0,0, 0,0,0);
  sequence[rightSideDown].interpolate[0] = false;
  sequence[rightSideDown].repeats[0] = 120;
  sequence[rightSideDown].entry[1] = JointData(0,1500000,0,I,I,I, -1600000,1500000,2500000, -1600000,1500000,2500000, -1400000,1600000,2500000, -1400000,1600000,2500000);
  sequence[rightSideDown].interpolate[1] = false;
  sequence[rightSideDown].repeats[1] = 100;
  
  sequence[getupFromUpright].numOfEntries = 1;
  sequence[getupFromUpright].loop = false;
  sequence[getupFromUpright].entry[0] = JointData(I,I,I,I,I,I, 0,800000,2100000, 0,800000,2100000, -1200000,1000000,2300000, -1200000,1000000,2300000);
  sequence[getupFromUpright].interpolate[0] = false; 
  sequence[getupFromUpright].repeats[0] = 30;
  
  sequence[rollRight].numOfEntries = 10;
  sequence[rollRight].loop = false;
  sequence[rollRight].entry[0] = JointData(-1500000, 0, 0, 0, I, I, 0, 300000, 2000000, 0, 300000, 2000000, -600000, 200000, 1400000, -600000, 200000, 1400000);
  sequence[rollRight].interpolate[0]=true;
  sequence[rollRight].repeats[0]=30;
  sequence[rollRight].entry[1] = JointData(-1500000, 0, 0, 0, I, I, 300000, -200000, 1300000, -200000, -200000, 300000, 0, -200000, 1700000, -400000, -200000, 800000);
  sequence[rollRight].interpolate[1]=true;
  sequence[rollRight].repeats[1]=40;
  sequence[rollRight].entry[2] = JointData(-1500000, 0, 0, 0, I, I, 300000, -200000, 1300000, -200000, -200000, 300000, 0, -200000, 1700000, -400000, -200000, 800000);
  sequence[rollRight].interpolate[2]=false;
  sequence[rollRight].repeats[2]=40;
  sequence[rollRight].entry[3] = JointData(-1500000, 0, 0, 0, I, I, 2000000, 400000, 1300000, 2000000, 400000, 1300000, 2000000, 400000, 1300000, 2000000, 400000, 1300000);
  sequence[rollRight].interpolate[3]=true;
  sequence[rollRight].repeats[3]=20;
  sequence[rollRight].entry[4] = JointData(0, 1500000, 0, I, I, I, 2000000, 300000, 0, -100000, 0, 0, 2000000, 300000, 0, -100000, 0, 0);
  sequence[rollRight].interpolate[4]=true;
  sequence[rollRight].repeats[4]=40;
  sequence[rollRight].entry[5] = JointData(-1500000, 0, 0, 0, I, I, 0, -200000, 0, 100000, 0, 0, 0, -200000, 0, 100000, 0, 0);
  sequence[rollRight].interpolate[5]=true;
  sequence[rollRight].repeats[5]=40;
  sequence[rollRight].entry[6] = JointData(-1500000, 0, 0, 0, I, I, 0, -200000, 0, 2000000, 0, 0, 0, -200000, 0, 2000000, 0, 0);
  sequence[rollRight].interpolate[6]=true;
  sequence[rollRight].repeats[6]=40;
  sequence[rollRight].entry[7] = JointData(-1500000, 0, 0, 0, I, I, 0, -200000, 0, 2000000, 400000, 0, 0, -200000, 0, 2000000, 400000, 0);
  sequence[rollRight].interpolate[7]=true;
  sequence[rollRight].repeats[7]=20;
  sequence[rollRight].entry[8] = JointData(-1500000, 0, 0, 0, I, I, 0, 300000, 2000000, 0, 300000, 2000000, -600000, 200000, 1400000, -600000, 200000, 1400000);
  sequence[rollRight].interpolate[8]=true;
  sequence[rollRight].repeats[8]=30;
  sequence[rollRight].entry[9] = JointData(-1500000, 0, 0, 0, I, I, 0, 300000, 2000000, 0, 300000, 2000000, -600000, 200000, 1400000, -600000, 200000, 1400000);
  sequence[rollRight].interpolate[9]=false;
  sequence[rollRight].repeats[9]=60;
  
  sequence[rollLeft].numOfEntries = 10;
  sequence[rollLeft].loop = false;
  sequence[rollLeft].entry[0] = JointData(-1500000, 0, 0, 0, I, I, 0, 300000, 2000000, 0, 300000, 2000000, -600000, 200000, 1400000, -600000, 200000, 1400000);
  sequence[rollLeft].interpolate[0]=true;
  sequence[rollLeft].repeats[0]=30;
  sequence[rollLeft].entry[1] = JointData(-1500000, 0, 0, 0, I, I, -200000, -200000, 300000, 300000, -200000, 1300000, -400000, -200000, 800000, 0, -200000, 1700000);
  sequence[rollLeft].interpolate[1]=true;
  sequence[rollLeft].repeats[1]=40;
  sequence[rollLeft].entry[2] = JointData(-1500000, 0, 0, 0, I, I, -200000, -200000, 300000, 300000, -200000, 1300000, -400000, -200000, 800000, 0, -200000, 1700000);
  sequence[rollLeft].interpolate[2]=false;
  sequence[rollLeft].repeats[2]=40;
  sequence[rollLeft].entry[3] = JointData(-1500000, 0, 0, 0, I, I, 2000000, 400000, 1300000, 2000000, 400000, 1300000, 2000000, 400000, 1300000, 2000000, 400000, 1300000);
  sequence[rollLeft].interpolate[3]=true;
  sequence[rollLeft].repeats[3]=20;
  sequence[rollLeft].entry[4] = JointData(0, -1500000, 0, I, I, I, -100000, 0, 0, 2000000, 300000, 0, -100000, 0, 0, 2000000, 300000, 0);
  sequence[rollLeft].interpolate[4]=true;
  sequence[rollLeft].repeats[4]=40;
  sequence[rollLeft].entry[5] = JointData(-1500000, 0, 0, 0, I, I, 100000, 0, 0, 0, -200000, 0, 100000, 0, 0, 0, -200000, 0);
  sequence[rollLeft].interpolate[5]=true;
  sequence[rollLeft].repeats[5]=40;
  sequence[rollLeft].entry[6] = JointData(-1500000, 0, 0, 0, I, I, 2000000, 0, 0, 0, -200000, 0, 2000000, 0, 0, 0, -200000, 0);
  sequence[rollLeft].interpolate[6]=true;
  sequence[rollLeft].repeats[6]=40;
  sequence[rollLeft].entry[7] = JointData(-1500000, 0, 0, 0, I, I, 2000000, 400000, 0, 0, -200000, 0, 2000000, 400000, 0, 0, -200000, 0);
  sequence[rollLeft].interpolate[7]=true;
  sequence[rollLeft].repeats[7]=20;
  sequence[rollLeft].entry[8] = JointData(-1500000, 0, 0, 0, I, I, 0, 300000, 2000000, 0, 300000, 2000000, -600000, 200000, 1400000, -600000, 200000, 1400000);
  sequence[rollLeft].interpolate[8]=true;
  sequence[rollLeft].repeats[8]=30;
  sequence[rollLeft].entry[9] = JointData(-1500000, 0, 0, 0, I, I, 0, 300000, 2000000, 0, 300000, 2000000, -600000, 200000, 1400000, -600000, 200000, 1400000);
  sequence[rollLeft].interpolate[9]=false;
  sequence[rollLeft].repeats[9]=60;
  
  
  
  
  /*---------------------movement for ERS7---------------------------------------------------------------*/
  
  //just a stand
  sequenceERS7[unknown].numOfEntries = 9;
  sequenceERS7[unknown].loop = false;
  
  sequenceERS7[unknown].entry[0] = JointData(-38000,18000,781000,I,I,I, -62000,273000,1902000, -62000,273000,1902000, -589000,216000,1385000, -589000,216000,1385000);
  sequenceERS7[unknown].interpolate[0] = false; 
  sequenceERS7[unknown].repeats[0] = 60;
  
  sequenceERS7[unknown].entry[1] = JointData(-38000,18000,781000,I,I,I, -62000,273000,1902000, -62000,273000,1902000, -589000,216000,1385000, -589000,216000,1385000);
  sequenceERS7[unknown].interpolate[1] = false; 
  sequenceERS7[unknown].repeats[1] = 60;
  
  //sit down
  sequenceERS7[unknown].entry[2] = JointData(-38000,18000,781000,I,I,I, 24000,-79000,-153000, 24000,-79000,-153000, -693000,222000,-51000, -693000,222000,-51000);
  sequenceERS7[unknown].interpolate[2] = false; 
  sequenceERS7[unknown].repeats[2] = 70;
  
  //lay down
  sequenceERS7[unknown].entry[3] = JointData(-38000,18000,781000,I,I,I, 997000,5000,917000, 997000,5000,917000, -1396000,318000,218000, -1396000,318000,218000);
  sequenceERS7[unknown].interpolate[3] = true; 
  sequenceERS7[unknown].repeats[3] = 50;
  
  //move up
  sequenceERS7[unknown].entry[4] = JointData(-38000,18000,781000,I,I,I, 1143000,85000,2066000, 1143000,85000,2066000, -1872000,313000,-104000, -1872000,313000,-104000);
  sequenceERS7[unknown].interpolate[4] = false;
  sequenceERS7[unknown].repeats[4] = 50;
  
  sequenceERS7[unknown].entry[5] = JointData(-38000,18000,781000,I,I,I, 816000,74000,2038000, 816000,74000,2038000, -1921000,307000,1552000, -1921000,307000,1552000);
  sequenceERS7[unknown].interpolate[5] = false; 
  sequenceERS7[unknown].repeats[5] = 50;
  
  sequenceERS7[unknown].entry[6] = JointData(-38000,18000,781000,I,I,I, 829000,91000,2077000, 829000,91000,2077000, -1149000,-11000,2167000, -1149000,-11000,2167000);
  sequenceERS7[unknown].interpolate[6] = false; 
  sequenceERS7[unknown].repeats[6] = 50;
  
  
  //back to stand
  sequenceERS7[unknown].entry[7] = JointData(-38000,18000,781000,I,I,I, -62000,273000,1902000, -62000,273000,1902000, -589000,216000,1385000, -589000,216000,1385000);
  sequenceERS7[unknown].interpolate[7] = false; 
  sequenceERS7[unknown].repeats[7] = 60;
  
  sequenceERS7[unknown].entry[8] = JointData(-38000,18000,781000,I,I,I, -62000,273000,1902000, -62000,273000,1902000, -589000,216000,1385000, -589000,216000,1385000);
  sequenceERS7[unknown].interpolate[8] = false; 
  sequenceERS7[unknown].repeats[8] = 60;
  
  
  /* standing */
  sequenceERS7[standing].numOfEntries = 2;
  sequenceERS7[standing].loop = false;
  
  sequenceERS7[standing].entry[0] = JointData(-38000,18000,781000,I,I,I, -62000,273000,1902000, -62000,273000,1902000, -589000,216000,1385000, -589000,216000,1385000);
  sequenceERS7[standing].interpolate[0] = false; 
  sequenceERS7[standing].repeats[0] = 60;
  
  sequenceERS7[standing].entry[1] = JointData(-38000,18000,781000,I,I,I, -62000,273000,1902000, -62000,273000,1902000, -589000,216000,1385000, -589000,216000,1385000);
  sequenceERS7[standing].interpolate[1] = false; 
  sequenceERS7[standing].repeats[1] = 60;
  
  
  
  /* lying on one side */
  //left legs transformation
  sequenceERS7[leftSideDown].numOfEntries = 4;
  sequenceERS7[leftSideDown].loop = false;
  
  sequenceERS7[leftSideDown].entry[0] = JointData(-159000,-230000,540000,I,I,I, -5000,199000,1916000,2112000,-119000,1683000, -488000,39000,1373000, 1985000,-91000,2053000);
  sequenceERS7[leftSideDown].interpolate[0] = true; 
  sequenceERS7[leftSideDown].repeats[0] = 30;
  
  
  sequenceERS7[leftSideDown].entry[1] = JointData(-159000,-230000,540000,I,I,I, -5000,199000,1916000,2112000,-119000,1683000, -488000,39000,1373000, 1985000,-91000,2053000);
  sequenceERS7[leftSideDown].interpolate[1] = true; 
  sequenceERS7[leftSideDown].repeats[1] = 50;
  
  sequenceERS7[leftSideDown].entry[2] = JointData(-82000,7000,507000,I,I,I,  -28000,233000,1911000, 1498000,-142000,2120000, -464000,5000,1411000, 1222000,-125000,2113000);
  sequenceERS7[leftSideDown].interpolate[2] = false;
  sequenceERS7[leftSideDown].repeats[2] = 25;
  
  
  sequenceERS7[leftSideDown].entry[3] = JointData(-142000,-89000,468000,I,I,I, -28000,1571000,1905000, -1487000,-142000,2142000, 78000,1423000,2102000, -1406000,-125000,2118000);
  sequenceERS7[leftSideDown].interpolate[3] = false;
  sequenceERS7[leftSideDown].repeats[3] = 25;
  
  
  
  //right legs transformation
  sequenceERS7[rightSideDown].numOfEntries = 4;
  sequenceERS7[rightSideDown].loop = false;
  
  sequenceERS7[rightSideDown].entry[0] = JointData(-159000,-230000,540000,I,I,I, 2112000,-119000,1683000, -5000,199000,1916000, 1985000,-91000,2053000, -488000,39000,1373000);
  sequenceERS7[rightSideDown].interpolate[0] = true; 
  sequenceERS7[rightSideDown].repeats[0] = 30;
  
  
  sequenceERS7[rightSideDown].entry[1] = JointData(-159000,-230000,540000,I,I,I, 2112000,-119000,1683000, -5000,199000,1916000, 1985000,-91000,2053000, -488000,39000,1373000);
  sequenceERS7[rightSideDown].interpolate[1] = true; 
  sequenceERS7[rightSideDown].repeats[1] = 50;
  
  sequenceERS7[rightSideDown].entry[2] = JointData(-82000,7000,507000,I,I,I, 1498000,-142000,2120000, -28000,233000,1911000, 1222000,-125000,2113000, -464000,5000,1411000);
  sequenceERS7[rightSideDown].interpolate[2] = false;
  sequenceERS7[rightSideDown].repeats[2] = 25;
  
  
  sequenceERS7[rightSideDown].entry[3] = JointData(-142000,-89000,468000,I,I,I, -1487000,-142000,2142000, -28000,1571000,1905000, -1406000,-125000,2118000, 78000,1423000,2102000);
  sequenceERS7[rightSideDown].interpolate[3] = false;
  sequenceERS7[rightSideDown].repeats[3] = 25;
  
  
  /*	// make a roll
  
    //to back
    sequenceERS7[rollLeft].numOfEntries = 2;
    sequenceERS7[rollLeft].loop = false;
    
      sequenceERS7[rollLeft].entry[0] = JointData(-1346000,1564000,371000,0,I,I, -112000,1623000,114000, -5000,1623000,-219000, -54000,1520000,221000, 0,1457000,362000);
      sequenceERS7[rollLeft].interpolate[0] = false; 
      sequenceERS7[rollLeft].repeats[0] = 50;
      
        //legs up
        sequenceERS7[rollLeft].entry[1] = JointData(-1000000,438000,364000,0,I,I, 0,-182000,91000, -84000,-182000,184000, -73000,250000,184000, 24000,-250000,338000);
        sequenceERS7[rollLeft].interpolate[1] = false; 
        sequenceERS7[rollLeft].repeats[1] = 60;
        
          
            //to back
            sequenceERS7[rollRight].numOfEntries = 2;
            sequenceERS7[rollRight].loop = false;
            
              sequenceERS7[rollRight].entry[0] = JointData(-1346000,1564000,371000,0,I,I, -112000,1623000,114000, -5000,1623000,-219000, -54000,1520000,221000, 0,1457000,362000);
              sequenceERS7[rollRight].interpolate[0] = false; 
              sequenceERS7[rollRight].repeats[0] = 50;
              
                //legs up
                sequenceERS7[rollRight].entry[1] = JointData(-1000000,438000,364000,0,I,I, 0,-182000,91000, -84000,-182000,184000, -73000,250000,184000, 24000,-250000,338000);
                sequenceERS7[rollRight].interpolate[1] = false; 
                sequenceERS7[rollRight].repeats[1] = 60;
  */
  
  // make a roll
  
  //for rollLeft
  //to back
  /*	sequenceERS7[rollLeft].numOfEntries = 2;
  sequenceERS7[rollLeft].loop = false;
  
    sequenceERS7[rollLeft].entry[0] = JointData(-1173000,23000,220000,0,I,I, 6000,-239000, 1355000,1416000,-68000, 2006000,-546000, -159000, 1449000,970000, 170000,2047000);
    sequenceERS7[rollLeft].interpolate[0] = false; 
    sequenceERS7[rollLeft].repeats[0] = 100;
    
      //legs up
      sequenceERS7[rollLeft].entry[1] = JointData(-1129000,116000,122000,0,I,I, -22000,-136000, -28000,2048000,-102000, 2108000,960000,-256000,184000, 1601000,-79000,460000);
      sequenceERS7[rollLeft].interpolate[1] = false; 
      sequenceERS7[rollLeft].repeats[1] = 50;
      
        
          //for rollRight
          //to back
          sequenceERS7[rollRight].numOfEntries = 2;
          sequenceERS7[rollRight].loop = false;
          
            sequenceERS7[rollRight].entry[0] = JointData(-1173000,-23000,-220000,0,I,I, 1416000,-68000, 2006000, 6000,-239000, 1355000,970000, 170000,2047000,-546000, -159000, 1449000);
            sequenceERS7[rollRight].interpolate[0] = false; 
            sequenceERS7[rollRight].repeats[0] = 100;
            
              //legs up
              sequenceERS7[rollRight].entry[1] = JointData(-1000000,438000,364000,0,I,I, 2048000,-102000, 2108000,-22000,-136000, -28000, 1601000,-79000,460000,960000,-256000,184000);
              sequenceERS7[rollRight].interpolate[1] = false; 
              sequenceERS7[rollRight].repeats[1] = 50;
              
                
  */	
  
  
  /*
  //headstand to stand 
  
    //move legs
    sequenceERS7[makeRoll7].numOfEntries = 2;
    sequenceERS7[makeRoll7].loop = false;
    
      sequenceERS7[makeRoll7].entry[0] = JointData(16000,-70000,872000,I,I,I, -73000,244000,2125000, -164000,324000,2044000, -97000,273000,184000, 72000,153000,239000);
      sequenceERS7[makeRoll7].interpolate[0] = true; 
      sequenceERS7[makeRoll7].repeats[0] = 50;
      
        //head press
        sequenceERS7[makeRoll7].entry[1] = JointData(-1000000,438000,364000,I,I,I, 0,-182000,91000, -84000,-182000,184000, -73000,250000,184000, 24000,-250000,338000);
        sequenceERS7[makeRoll7].interpolate[1] = true; 
        sequenceERS7[makeRoll7].repeats[1] = 70;
        
  */
#undef I
  
  countTry=0;
}

bool DefaultGetupEngine::executeParameterized(JointData& jointData)
{
  const RobotConfiguration& r = getRobotConfiguration();
  
  // do nothing if no valid acceleration values are set
  if (
    sensorDataBuffer.numOfFrames == 0 ||
    (sensorDataBuffer.lastFrame().data[SensorData::accelerationX]==0 &&
    sensorDataBuffer.lastFrame().data[SensorData::accelerationY]==0 && 
    sensorDataBuffer.lastFrame().data[SensorData::accelerationZ]==0))
    return true;
  
  if (r.getRobotDesign()==RobotDesign::ERS210)
  {
    
    headState.neckHeight = 160;
    headState.stableMotion = (state==standing);
    executedMotionRequest.motionType = MotionRequest::getup; 
    
    
    if (lastMotionType == MotionRequest::getup && sequencer.isRunning())
    {
      sequencer.nextSequenceData(jointData);
      lastGetupTime=SystemCall::getCurrentSystemTime();
      return true;
    }
    
    if (headState.bodyRoll <= -defaultGetupEngineFalldownRollAngle)
    {
      // left side down
      state = leftSideDown;
      
    }
    else if (headState.bodyRoll >= defaultGetupEngineFalldownRollAngle)
    {
      // right side down
      state = rightSideDown;
      
    }
    else if (headState.bodyTilt > defaultGetupEngineFalldownTiltAngle || headState.bodyTilt < -defaultGetupEngineFalldownTiltAngle)
    { 
      // rolled by x axis
      state = unknown;
    }
    else if (headState.bodyRoll >= defaultGetupEngineRollOnWallAngle)
      
    {
      
      state=rollRight;
      
    }
    
    else if (headState.bodyRoll <= -defaultGetupEngineRollOnWallAngle)
    {
      
      state=rollLeft;
      
    }
    else if (state == getupFromUpright || state == standing)
    {
      state = standing;
      return false;
    }
    else
    {
      state = getupFromUpright;
    }
    
    if (state!=lastState)
    {
      stateStartTime=SystemCall::getCurrentSystemTime();
    }
    
    if ((SystemCall::getTimeSince(lastGetupTime)<200)&&
      (SystemCall::getTimeSince(stateStartTime)>3000))
    {
      if (state==rightSideDown)
      {
        
        state=rollRight;
        
      }
      else if (state==leftSideDown)
      {
        
        state=rollLeft;
        
      }
    }
    
  }
  else
  {
    
    //-------------------begin for ERS7---------------------
    
    
    headState.neckHeight = 160;
    headState.stableMotion = (state==standing);
    executedMotionRequest.motionType = MotionRequest::getup; 
    
    
    if (lastMotionType == MotionRequest::getup && sequencer.isRunning())
    {
      sequencer.nextSequenceData(jointData);
      lastGetupTime=SystemCall::getCurrentSystemTime();
      return true;
    }
    
    
    if ((headState.bodyRoll >= defaultGetupEngineRollOnWallAngle7)&&
      (countTry>2)&&
      (state == rightSideDown))
    {
      //	OUTPUT(idText,text,"ERS7: rollRight (Angle)");
      state=leftSideDown;
      countTry=0;
      
    }
    else if ((headState.bodyRoll <= -defaultGetupEngineRollOnWallAngle7)&&
      (countTry>2)&&
      (state == leftSideDown))
    {
      //	OUTPUT(idText,text,"ERS7: rollLeft (Angle)");
      state = rightSideDown;
      countTry = 0;
    }
				else if ((headState.bodyRoll <= -defaultGetupEngineFalldownRollAngle7)&&
          (countTry < 3))
        {
          // left side down
          //	OUTPUT(idText,text,"ERS7: leftSideDown");
          state = leftSideDown;
          countTry++;
        }
        else if (headState.bodyRoll >= defaultGetupEngineFalldownRollAngle7)
        {
          // right side down
          //	OUTPUT(idText,text,"ERS7: rightSideDown");
          state = rightSideDown;
          countTry++;
        }
        else if ((headState.bodyTilt > defaultGetupEngineFalldownTiltAngle7 || headState.bodyTilt < defaultGetupEngineFalldownTiltAngleBack7)&&
          (countTry<3))
        { 
          // rolled by x axis
          //	OUTPUT(idText,text,"ERS7: unknown");
          state = unknown;
          countTry++;
        }
        else if (state == unknown || state == standing || state == leftSideDown || state == rightSideDown)
        {
          state = standing;
          countTry = 0;
          return false;
        }
        
        /*	else if (state == getupFromUpright || state == standing)
        {
        state = standing;
        return false;
        }
        
          if (state!=lastState)
          {
          stateStartTime=SystemCall::getCurrentSystemTime();
          }
          
            
              if ((SystemCall::getTimeSince(lastGetupTime)<1000)&&
              (SystemCall::getTimeSince(lastGetupTime)>600)&&
              (SystemCall::getTimeSince(stateStartTime)>4000))
              {
              if (state==rightSideDown)
              {
              OUTPUT(idText,text,"ERS7: rollRight (GetUpTime)");
              // state=rollRight;
              //	state = leftSideDown;
              //	state = leftSideDown;
              
                }
                else if (state==leftSideDown)
                {
                OUTPUT(idText,text,"ERS7: rollLeft (GetUpTime)");
                //	state=rollLeft;
                //	state = rightSideDown;
                //	state = rightSideDown;
                
                  }
                  }
                  
                    
        */		
        //-------------------end for ERS7-----------------------
  }
  
  
  
  
  
  
  
  
  
  
  lastGetupTime=SystemCall::getCurrentSystemTime();
  lastState=state;
  
  if (r.getRobotDesign()==RobotDesign::ERS210)
  {
    //	OUTPUT(idText,text,"ERS210");
    sequencer = sequence[state];
    sequencer.start();
    sequencer.nextSequenceData(jointData);
    
    
  }
  else
  {
    //	OUTPUT(idText,text,"ERS7");
    sequencer = sequenceERS7[state];
    sequencer.start();
    sequencer.nextSequenceData(jointData);
  }
  return true;
}



/*
* Change log :
* 
* $Log: DefaultGetupEngine.cpp,v $
* Revision 1.18  2004/04/09 11:35:51  roefer
* Bremen Byters German Open check-in
*
* Revision 1.17  2004/04/08 15:33:05  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.17  2004/03/29 13:18:49  pg_joha
* renamend new BycycleKick and added a mirrored variant of it
*
* Revision 1.16  2004/03/19 11:25:30  juengel
* Bug fixed.
*
* Revision 1.15  2004/03/17 10:06:51  dueffert
* shut mouth while rolling
*
* Revision 1.14  2004/03/08 01:38:51  roefer
* Interfaces should be const
*
* Revision 1.13  2004/03/03 18:35:15  roefer
* Acceleration sensors magically work again
*
* Revision 1.12  2004/02/29 18:32:07  heinze
* changed some comments
*
* Revision 1.11  2004/02/29 14:57:37  roefer
* Removed static variable, initialized state, added support for robot with broken acceleration sensors
*
* Revision 1.10  2004/02/27 20:32:07  brueckne
* comment by heinze: getup-engine works for all common cases, specials aren't included yet
*
* Revision 1.9  2004/02/27 19:19:29  heinze
* took out absurd behavior
*
* Revision 1.8  2004/02/27 18:01:10  heinze
* still improving the engine for ERS7
*
* Revision 1.7  2004/02/27 17:32:27  heinze
* improved getup-behavior for ERS7
*
* Revision 1.5  2004/02/26 11:47:27  heinze
* bug removed
*
* Revision 1.4  2004/02/26 10:56:35  heinze
* worked at the getUp-Engine for ERS7
*
* Revision 1.3  2004/02/26 09:44:23  heinze
* took out some OUTPUTS
*
* Revision 1.2  2004/02/25 16:03:50  heinze
* added/began getup-motion for ERS7
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/07/06 09:31:02  risler
* wait state added at end of roll, preventing double roll
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.9  2003/06/20 15:32:32  dueffert
* getting from down from wall by rolling added
*
* Revision 1.8  2003/06/19 15:53:40  dueffert
* rolling if stuck added, beautified
*
* Revision 1.7  2003/06/18 18:12:11  dueffert
* getup stuck recognition added
*
* Revision 1.6  2003/05/02 18:26:18  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.5  2003/04/06 19:41:01  dueffert
* a little slower but better getup
*
* Revision 1.4  2003/03/06 12:05:43  dueffert
* execute with parameters renamed to avoid inheritance warnings
*
* Revision 1.3  2002/09/17 23:55:20  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.2  2002/09/11 00:06:58  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:14  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1.1.1  2002/05/10 12:40:14  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.14  2002/05/06 16:03:28  risler
* corrected bodyRoll angle
*
* Revision 1.13  2002/05/05 15:14:23  risler
* changed stand implementation
*
* Revision 1.12  2002/05/04 18:23:43  risler
* added calculation of executedMotionRequest
*
* Revision 1.11  2002/04/23 10:38:31  risler
* renamed headOdometry to headState
*
* Revision 1.10  2002/04/17 17:04:41  risler
* Darmstadt GO
*
* Revision 1.9  2002/04/08 17:49:24  risler
* removed body tilt from odometry
*
* Revision 1.8  2002/04/02 14:01:39  dueffert
* minor odometry enhancements
*
* Revision 1.7  2002/04/02 13:10:19  dueffert
* big change: odometryData and cameraMatrix in image now, old logfiles may be obsolete
*
* Revision 1.6  2002/03/27 15:54:28  risler
* added neckHeight and stableMotion in motion modules
*
* Revision 1.5  2002/02/26 20:55:58  risler
* getup on startup after first sensordata received
*
* Revision 1.4  2002/02/26 20:54:20  risler
* getup on startup after first sensordata received
*
* Revision 1.3  2002/02/23 17:07:22  risler
* code cleaned up
*
* Revision 1.2  2002/02/23 16:33:07  risler
* finished GetupEngine
*
* Revision 1.1  2002/02/20 16:40:50  risler
* added GetupEngine
*
*
*/
