/**
* @file GetupEngine.h
* 
* Definition of class GetupEngine
* 
* @author Max Risler
*/

#ifndef __GetupEngine_h_
#define __GetupEngine_h_

#include "Tools/Module/Module.h"

#include "Representations/Motion/MotionRequest.h"
#include "Representations/Perception/SensorDataBuffer.h"
#include "Representations/Motion/JointData.h"
#include "Representations/Motion/PIDData.h"
#include "Representations/Motion/OdometryData.h"
#include "Representations/Motion/HeadState.h"

/**
* @class GetupEngineInterfaces
* 
* Interfaces of the GetupEngine module.
*/
class GetupEngineInterfaces
{
public:
  /** Constructor */
  GetupEngineInterfaces(
    const MotionRequest::MotionID& lastMotionType,
    const SensorDataBuffer& sensorDataBuffer,
    const MotionRequest& motionRequest,
    PIDData& pidData,
    OdometryData& odometryData,
    HeadState& headState,
    MotionRequest& executedMotionRequest)
    : lastMotionType(lastMotionType),
    sensorDataBuffer(sensorDataBuffer),
    motionRequest(motionRequest),
    pidData(pidData),
    odometryData(odometryData),
    headState(headState),
    executedMotionRequest(executedMotionRequest)
  {}
protected:  
  /** The motion type of the previous frame */
  const MotionRequest::MotionID& lastMotionType;
  
  /** The current sensor data */
  const SensorDataBuffer& sensorDataBuffer;
  
  /** The current motion request */
  const MotionRequest& motionRequest;
  
  /** The current PID servo gains */
  PIDData& pidData;

  /** Calculated odometry data */
  OdometryData& odometryData;

  /** The neck height and the body tilt to be estimated by the getup engine */
  HeadState& headState;

  /** The motion request that is currently executed by the getup engine */
  MotionRequest& executedMotionRequest;

};

/**
* @class GetupEngine
* 
* A generic class for getup engine modules.
*
* The getup engine lets the robot stand up from any position
* by generating appropriate joint data.
*
* @author Max Risler
*/
class GetupEngine : public Module, protected GetupEngineInterfaces
{
public:
/*
* Constructor.
* @param interfaces The paramters of the GetupEngine module.
  */
  GetupEngine(const GetupEngineInterfaces& interfaces)
    : GetupEngineInterfaces(interfaces)
  {}
  
  /**
  * calculates the next joint data set
  * @param jointData stores calculated frame
  * @return true if next set should be calculated by GetupEngine
  *         false if change to other module can occur
  */
  virtual bool executeParameterized(JointData& jointData) = 0;
  
  /** Destructor */
  virtual ~GetupEngine() {}
  
protected:
  /** That function is not used but has to be implemented */
  virtual void execute() {};
};

#endif //__GetupEngine_h_

/*
* Change log :
* 
* $Log: GetupEngine.h,v $
* Revision 1.2  2004/03/08 01:38:52  roefer
* Interfaces should be const
*
* Revision 1.1  2003/10/06 14:10:14  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.2  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.8  2003/05/02 18:26:18  risler
* SensorDataBuffer added
* replaced SensorData with SensorDataBuffer
* full SensorData resolution now accessible
*
* Revision 1.7  2003/03/06 12:06:50  dueffert
* execute with parameters renamed to avoid inheritance warnings
*
* Revision 1.6  2002/11/20 18:02:29  risler
* added PID values to GT2001MotionNetSpecialActions
* added playDead motion
*
* Revision 1.5  2002/09/17 23:55:20  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.4  2002/09/12 13:17:27  loetzsch
* made member variables of module interfaces protected
*
* Revision 1.3  2002/09/11 20:06:19  loetzsch
* continued experiments with modules/solutions
*
* Revision 1.2  2002/09/11 00:06:58  loetzsch
* continued change of module/solution mechanisms
*
* Revision 1.1  2002/09/10 15:36:14  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.2  2002/08/22 14:41:04  risler
* added some doxygen comments
*
* Revision 1.1.1.1  2002/05/10 12:40:14  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.2  2002/02/23 16:33:07  risler
* finished GetupEngine
*
* Revision 1.1  2002/02/20 16:40:50  risler
* added GetupEngine
*
*
*/
