/**
* @file ATH2004ERS7HeadControl.cpp
*
* Implementation of class ATH2004ERS7HeadControl
*/

#ifdef _WIN32
static const unsigned int timeAfterWhichBallIsConsideredLost = 1000;
#else
static const unsigned int timeAfterWhichBallIsConsideredLost = 300;
#endif


#include "Platform/GTAssert.h"
#include "ATH2004ERS7HeadControl.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Actorics/Kinematics.h"
#include "Tools/Math/Common.h"
#include "Platform/SystemCall.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Debugging/DebugDrawings.h"

void ATH2004ERS7HeadControl::HeadPathPlanner::init(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration)
{
  const RobotDimensions& robotDimensions = getRobotConfiguration().getRobotDimensions();
  if (numberOfVectors>maxNumberOfPoints-1) numberOfVectors=maxNumberOfPoints-1;
  if (duration<8) duration=8;
  
  //start from current position
  points[0].x = lastTilt;
  points[0].y = lastPan;
  points[0].z = lastRoll;
  currentPoint = 0;
  numberOfPoints = numberOfVectors;
  currentFrame = 0;
  numberOfFrames = duration/8;
  
  const Range<long> jointRangeHeadTilt(toMicroRad(robotDimensions.jointLimitHeadTiltN),toMicroRad(robotDimensions.jointLimitHeadTiltP));
  const Range<long> jointRangeHeadPan( toMicroRad(robotDimensions.jointLimitHeadPanN), toMicroRad(robotDimensions.jointLimitHeadPanP));
  const Range<long> jointRangeHeadRoll(toMicroRad(robotDimensions.jointLimitHeadRollN),toMicroRad(robotDimensions.jointLimitHeadRollP));
  
  //calculate total distance and speed
  double distance=0;
  int i;
  for (i=0;i<numberOfVectors;i++)
  {
    //clip arcs to physical limits
    points[i+1].x = jointRangeHeadTilt.limit(vectors[i].x);
    points[i+1].y = jointRangeHeadPan.limit(vectors[i].y);
    points[i+1].z = jointRangeHeadRoll.limit(vectors[i].z);
    
    distance += sqrt(
      (double)(points[i+1].x-points[i].x)*(double)(points[i+1].x-points[i].x)+
      (double)(points[i+1].y-points[i].y)*(double)(points[i+1].y-points[i].y)+
      (double)(points[i+1].z-points[i].z)*(double)(points[i+1].z-points[i].z));
  }
  double speed = distance/(double)numberOfFrames; //in urad per frame
  if (speed<minHeadSpeed) speed=minHeadSpeed;
  
  //calculate duration for each part of the route
  double sum=0;
  for (i=0;i<=numberOfVectors;i++)
  {
    firstFrame[i] = sum/speed;
    sum += sqrt(
      (double)(points[i+1].x-points[i].x)*(double)(points[i+1].x-points[i].x)+
      (double)(points[i+1].y-points[i].y)*(double)(points[i+1].y-points[i].y)+
      (double)(points[i+1].z-points[i].z)*(double)(points[i+1].z-points[i].z));
  }
}

void ATH2004ERS7HeadControl::HeadPathPlanner::initLoop(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration)
{
  Vector3<long> newVec[maxNumberOfPoints];
  double min=1e20;
  int minIndex=0;
  int i;
  if (numberOfVectors>maxNumberOfPoints-1) numberOfVectors=maxNumberOfPoints-1;
  for (i=0;i<numberOfVectors;i++)
  {
    double diff2 =
      (double)(vectors[i].x-lastTilt)*(double)(vectors[i].x-lastTilt)+
      (double)(vectors[i].y-lastPan)*(double)(vectors[i].y-lastPan)+
      (double)(vectors[i].z-lastRoll)*(double)(vectors[i].z-lastRoll);
    if (diff2<min)
    {
      min=diff2;
      minIndex=i;
    }
  }
  /** @todo check if opposite loop direction wouldnt be better after visit of first point from current position */
  for (i=0;i<=numberOfVectors;i++)
  {
    newVec[i]=vectors[(minIndex+i)%numberOfVectors];
  }
  init(newVec,numberOfVectors+1,duration);
}

bool ATH2004ERS7HeadControl::HeadPathPlanner::getAngles(long& tilt, long& pan, long& roll)
{
  if (currentFrame<numberOfFrames)
  {
    currentFrame++;
    while ((currentFrame>firstFrame[currentPoint+1])&&
           (currentPoint<numberOfPoints-1)&&
           (currentFrame<numberOfFrames))
    {
      currentPoint++;
    }
    
    double dist=0;
    if (currentPoint<numberOfPoints)
    {
      dist=firstFrame[currentPoint+1]-firstFrame[currentPoint];
    }
    if (dist==0)
    {
      tilt = points[currentPoint].x;
      pan  = points[currentPoint].y;
      roll = points[currentPoint].z;
    }
    else
    {
      double leftFactor = (firstFrame[currentPoint+1]-currentFrame)/dist;
      double rightFactor = 1-leftFactor;
      tilt = (long)(leftFactor*points[currentPoint].x + rightFactor*points[currentPoint+1].x);
      pan  = (long)(leftFactor*points[currentPoint].y + rightFactor*points[currentPoint+1].y);
      roll = (long)(leftFactor*points[currentPoint].z + rightFactor*points[currentPoint+1].z);
    }
  }
  return (currentFrame>=numberOfFrames);
}




ATH2004ERS7HeadControl::ATH2004ERS7HeadControl(HeadControlInterfaces& interfaces)
: HeadControl(interfaces),headControlState(otherModes),lastHeadControlState(otherModes),
lastScanWasLeft(true), timeOfLastStateChange(0),lastOdometryData(currentOdometryData),lastTimeOfGoodSL(0)
{ 
  headPathPlanner.lastTilt = sensorDataBuffer.lastFrame().data[SensorData::headTilt];
  headPathPlanner.lastPan  = sensorDataBuffer.lastFrame().data[SensorData::headPan];
  headPathPlanner.lastRoll = sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  leftWatch= true;
}

void ATH2004ERS7HeadControl::setJoints(long tilt, long pan, long roll, long speed, long mouth)
{
  headMotionRequest.tilt = tilt;
  headMotionRequest.pan  = pan;
  headMotionRequest.roll = roll;
  headMotionRequest.mouth = mouth;
}


void ATH2004ERS7HeadControl::moveHead(long tilt, long pan, long roll, long speed, long mouth)
{
  //tilt = tilt + sensorDataBuffer.lastFrame().data[SensorData::headTilt];
  //pan = pan + sensorDataBuffer.lastFrame().data[SensorData::headPan];
  //roll = roll + sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  //mouth = mouth + sensorDataBuffer.lastFrame().data[SensorData::mouth];
  
  if (speed >= 400)
  {
    if (headMotionRequest.tilt != tilt) {
      if (headMotionRequest.tilt > toMicroRad(jointLimitHeadTiltP))
        headMotionRequest.tilt = toMicroRad(jointLimitHeadTiltP);
      else {
        if (headMotionRequest.tilt < toMicroRad(jointLimitHeadTiltN))
          headMotionRequest.tilt = toMicroRad(jointLimitHeadTiltN);
        else
          headMotionRequest.tilt = tilt;
      }
    }
    if (headMotionRequest.pan != pan) {
      if (headMotionRequest.pan > toMicroRad(jointLimitHeadPanP))
        headMotionRequest.pan = toMicroRad(jointLimitHeadPanP);
      else {
        if (headMotionRequest.pan < toMicroRad(jointLimitHeadPanN))
          headMotionRequest.pan = toMicroRad(jointLimitHeadPanN);
        else
          headMotionRequest.pan = pan;
      }
    }
    if (headMotionRequest.roll != roll) {
      if (headMotionRequest.roll > toMicroRad(jointLimitHeadRollP))
        headMotionRequest.roll = toMicroRad(jointLimitHeadRollP);
      else {
        if (headMotionRequest.roll > toMicroRad(jointLimitHeadRollN))
          headMotionRequest.roll = toMicroRad(jointLimitHeadRollN);
        else
          headMotionRequest.roll = roll;
      }
    }
    if (headMotionRequest.mouth != mouth)
      headMotionRequest.mouth = mouth;
  }
  else
  {
    if (headPathPlanner.isLastPathFinished())
    {
      Vector3<long> point(tilt,pan,roll);
      Vector3<long> points[1]={point};
      headPathPlanner.init(points,1, 2000);
    }
    headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
  }
}


void ATH2004ERS7HeadControl::lookAtPoint(const Vector3<double> &pos,const Vector2<int> &offset,double& tilt,double& pan,double& roll)
{
  Vector3<double> posInCameraCoordinates;
  posInCameraCoordinates = cameraMatrix2.rotation.invert() * (pos - cameraMatrix2.translation);
  Vector2<int> pointToLookTo;
  Vector2<double> angles;
  angles.x = atan2(posInCameraCoordinates.x, posInCameraCoordinates.y);
  angles.y = atan2(posInCameraCoordinates.x, posInCameraCoordinates.z);
  Geometry::calculatePointByAngles(angles, cameraMatrix2, cameraInfo, pointToLookTo);
  //debugging...
  LINE(sketch, offset.x, 0, offset.x, 144, 1, Drawings::ps_solid, Drawings::gray);
  LINE(sketch, 0,offset.y, 176, offset.y, 1, Drawings::ps_solid, Drawings::gray);
  DOT(sketch, pointToLookTo.x - 88, pointToLookTo.y - 72, Drawings::white, Drawings:: black);
  DEBUG_DRAWING_FINISHED(sketch);
  /*  
  PAINT_TO_DEBUGDRAWING(crosshairs,
  crosshairsDrawing.line(0, offset.y + 71, 175, offset.y + 71, DebugDrawing::ps_solid, 0, DebugDrawing::Color(255,255,255) );
  crosshairsDrawing.line(offset.x + 87, 0, offset.x + 87, 143, DebugDrawing::ps_solid, 0, DebugDrawing::Color(255,255,255) );
  );
  
  GTMath::Vector2<int> onScreen;
  onScreen.x = (int)(- transformedPoint.y*fx/(transformedPoint.x));
  onScreen.y = (int)(- transformedPoint.z*fy/(transformedPoint.x));
  
  PAINT_TO_DEBUGDRAWING(crosshairs,
  crosshairsDrawing.dot(87+onScreen.x, 71+onScreen.y, DebugDrawing::Color(255,255,255), DebugDrawing::Color(127,127,255));
  );
  
  SEND_DEBUGDRAWING(crosshairs);
  */
  
  int xoff = offset.x;
  int yoff = offset.y;
  
  //Is the offset in range?
  if (yoff > cameraInfo.resolutionHeight / 2) yoff = cameraInfo.resolutionHeight / 2;
  else if (yoff < -cameraInfo.resolutionHeight / 2) yoff = -cameraInfo.resolutionHeight / 2;
  
  if (xoff > cameraInfo.resolutionWidth / 2) xoff = cameraInfo.resolutionWidth / 2;
  else if (xoff < -cameraInfo.resolutionWidth / 2) xoff = -cameraInfo.resolutionWidth / 2;
  
  Pose2D robotPose = this->robotPose.getPose();
  Vector3<double> aim1(
    pos.x-robotPose.translation.x,
    pos.y-robotPose.translation.y,
    pos.z-headState.neckHeight
    );
  
  RotationMatrix rot;
  Vector3<double> aim;
  rot.rotateY(-headState.bodyTilt).
    rotateX(headState.bodyRoll).
    rotateZ(-robotPose.getAngle());
  aim = rot*aim1;
  
  //now we have rotated the coordinates in a way that the neck
  //is (0,0,0) and should look without any body rotation to aim:
  //aim.x in front, aim.y left, aim.z above
  
  //Looking with a certain xoffset and yoffset gives us a ray from camera into space.
  //we want the coordinates of all points on that ray to be a simple funtion of a single
  //argument, the distance from camera to an image plane
  double xtan = -tan(cameraInfo.openingAngleWidth / 2)*xoff/(cameraInfo.resolutionWidth / 2);
  double ytan = tan(cameraInfo.openingAngleHeight / 2)*yoff/(cameraInfo.resolutionHeight / 2);
  
  //due to ERS7 anatomy we can now change tilt1 and/or tilt2 to reach our aim. There
  //are 3 possibly useful strategies for that:
  //1) change tilt1 and tilt2 synchronously: this would look cool but will be uncalculatable
  //2) maximize tilt1: try to find tilt2 solution for max tilt1, if there is none try to find
  //   tilt1 solution for minimal tilt2
  //3) maximize tilt2: try to find tilt1 solution for max tilt2, if there is none try to find
  //   tilt2 solution for minimal tilt1
  
  //this is method 2):
  tilt=0;
  if (!lookAtPointFixTilt1(aim,xtan,ytan,tilt,pan,roll))
  {
//    Vector2<double> pointOnPlane(pos.x, pos.y);
//    if(fabs(pointOnPlane.angle()) < .7)
//    {
      roll=jointLimitHeadRollN;
      lookAtPointFixTilt2(aim,xtan,ytan,tilt,pan,roll);
//    }
  }
  
  /*
  // just a check to see that everything went right on ERS7:
  RotationMatrix rot2;
  rot2.rotateY(tilt);
  Vector3<double> aim2 = rot2*aim;
  RotationMatrix rot3;
  rot3.rotateZ(-pan);
  Vector3<double> aim3 = rot3*aim2;
  Vector3<double> aim4 = aim3-Vector3<double>(0,0,distanceNeckToPanCenter);
  RotationMatrix rot4;
  rot4.rotateY(roll);
  Vector3<double> aim5 = rot4*aim4;
  Vector3<double> aim6 = aim5-Vector3<double>(distancePanCenterToCameraX,0,0);
  double realxoff=-aim6.y/aim6.x*cameraInfo.resolutionWidth / 2/tan(cameraInfo.openingAngleWidth/2);
  double realyoff=aim6.z/aim6.x*cameraInfo.resolutionHeight / 2/tan(cameraInfo.openingAngleHeight/2);
  if ((aim5.x<=0)||(fabs(realxoff-xoff)>0.1)||(fabs(realyoff-yoff)>0.1))
  {
  bool mistake=true;
  }
  */
  const Range<double> jointRangeHeadTilt2(jointLimitHeadRollN, jointLimitHeadRollP);
  roll = jointRangeHeadTilt2.limit(roll);
  
  const Range<double> jointRangeHeadTilt(jointLimitHeadTiltN,jointLimitHeadTiltP);
  const Range<double> jointRangeHeadPan(jointLimitHeadPanN, jointLimitHeadPanP);
  tilt = jointRangeHeadTilt.limit(tilt);
  pan = jointRangeHeadPan.limit(pan);
}

bool ATH2004ERS7HeadControl::lookAtPointFixTilt2(const Vector3<double> &aim, const double& xtan, const double& ytan, double& tilt, double& pan, const double& tilt2)
{
  //d shall be the distance from camera to object plane.
  
  //then we look at a point with these coordinates from tilt2 joint:
  //x = pcc+d
  //y = xtan*d
  //z = ytan*d
  //after rotation with tilt2 we get in coordinates from tilt1 joint:
  //x = Cos[t2]*(pcc + d) - Sin[t2]*(ytan*d)
  //y = xtan*d
  //z = npc + Sin[t2]*(pcc + d) + Cos[t2]*(ytan*d)
  
  //we know that we want to look at (aimx,aimy,aimz). Thats Sqrt[aimx^2+aimy^2+aimz^2] away from the neck.
  //The matching point (x,y,z) on our ray of look must have the same distance. this can be resolved to d
  //(thanks to Mathematica, see lookAtPointUDFinal.nb):
  
  //every visible d is positive, so with some helping variables we get:
  double a = 1 + xtan*xtan+ytan*ytan;
  double ns2 = distanceNeckToPanCenter*sin(tilt2);
  double nc2 = distanceNeckToPanCenter*cos(tilt2);
  double b = distancePanCenterToCameraX + nc2*ytan + ns2;
  double c = aim.x*aim.x + aim.y*aim.y + aim.z*aim.z - distanceNeckToPanCenter*distanceNeckToPanCenter
    - distancePanCenterToCameraX*distancePanCenterToCameraX - 2*distancePanCenterToCameraX*ns2;
  double radikant = b*b + a*c;
  double d;
  if (radikant<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt=headPathPlanner.lastTilt;
    pan=headPathPlanner.lastPan;
    return false;
  }
  else
  {
    d = -(distancePanCenterToCameraX + ytan*nc2 + ns2 - sqrt(radikant))/a;
  }
  if (d<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt=headPathPlanner.lastTilt;
    pan=headPathPlanner.lastPan;
    return false;
  }
  
  //Now we can calculate x,y,z from d:
  double x = cos(tilt2)*(distancePanCenterToCameraX + d) - sin(tilt2)*(ytan*d);
  double y = xtan*d;
  double z = distanceNeckToPanCenter + sin(tilt2)*(distancePanCenterToCameraX + d) + cos(tilt2)*(ytan*d);
  
  //we currently look at point (x,y,z). it has the correct distance, but has to be rotated twice
  //to match (aimx,aimy,aimz). First we want to get the y coordinate to aimy by rotating with pan.
  //This is useful, because pan is closer to camera and changing tilt afterwards does not change
  //y coordinate anymore. As this is done around z-axis, lets look into the xy-plane through (x,y,z).
  //We look at a point that is Sqrt[x^2+y^2] away from the z-axis and has an arc of ArcTan[x,y] (=atan2(y,x)).
  //We would like to look at a point in the same distance at y coordinate aimy. That would form an
  //arc of ArcSin[aimy/Sqrt[x^2+y^2]]. Therefore we have to rotate with:
  
  pan = asin(aim.y/sqrt(x*x+y*y))-atan2(y,x);
  if (pan>=pi)
  {
    pan -= pi2;
  }
  else if (pan<-pi)
  {
    pan += pi2;
  }
  
  //There is a second solution here: the arc we want the point to have above the ground can also
  //be Pi-ArcSin[aimy/Sqrt[x^2+y^2]]. Then it is definitely behind us, but that is possible and
  //useful in some cases.
  
  double pantemp = pi-asin(aim.y/sqrt(x*x+y*y))-atan2(y,x);
  if (pantemp>=pi)
  {
    pantemp -= pi2;
  }
  else if (pantemp<-pi)
  {
    pantemp += pi2;
  }
  
  //If the point is behind us, then we prefer to test the second solution first, because its nicer
  //to look at those points with upright head. Explanation: Imagine looking with the lower left pixel.
  //We want to look at a point left and a little behind of us. We find a correct y left a little
  //infront and a little behind us. If we take the one infront, we can still manage to rotate to the
  //one behind with very low tilt. Therefore we try the one behind us first in this case, so:
  if (aim.x<0)
  {
    double temp=pan;
    pan=pantemp;
    pantemp=temp;
  }
  
  //(If pan is in bounds) and after rotating with pan we look at:
  double xt=cos(pan)*x-sin(pan)*y;
  //yt=sin(pan)*x+cos(pan)*y=aimy
  //zt=z
  
  //and then we only need to find the correct tilt1:
  tilt = atan2(aim.z,aim.x)-atan2(z,xt);
  if ((tilt<jointLimitHeadTiltN)||(tilt>jointLimitHeadTiltP)||(pan<jointLimitHeadPanN)||(pan>jointLimitHeadPanP))
  {
    pan=pantemp;
    xt = cos(pan)*x - sin(pan)*y;
    tilt = atan2(aim.z,aim.x)-atan2(z,xt);
    if ((tilt<jointLimitHeadTiltN)||(tilt>jointLimitHeadTiltP)||(pan<jointLimitHeadPanN)||(pan>jointLimitHeadPanP))
    {
      return false;
    }
  }
  return true;
}

bool ATH2004ERS7HeadControl::lookAtPointFixTilt1(const Vector3<double> &aim, const double& xtan, const double& ytan, const double& tilt, double& pan, double& tilt2)
{
  //first convert from neck coordinates to before-tilt2 cordinates:
  Vector3<double> aim2(
    cos(tilt)*aim.x - sin(tilt)*aim.z,
    aim.y,
    sin(tilt)*aim.x + cos(tilt)*aim.z - distanceNeckToPanCenter
    );
  //and than we start the same way as in the fixTilt2 case, but much easier:
  //x = pcc + d
  //y = xtan*d
  //z = ytan*d
  
  //we know that we want to look at (aim2x,aim2y,aim2z). Thats Sqrt[aim2x^2+aim2y^2+aim2z^2] away from the tilt2 axis center.
  //The matching point (x,y,z) on our ray of look must have the same distance. this can be resolved to d
  //(thanks to Mathematica, see lookAtPointUDFinal.nb):
  
  //every visible d is positive, so with some helping variables we get:
  double a = 1 + xtan*xtan+ytan*ytan;
  double radikant = a*(aim2.x*aim2.x+aim2.y*aim2.y+aim2.z*aim2.z - distancePanCenterToCameraX*distancePanCenterToCameraX)
    + distancePanCenterToCameraX*distancePanCenterToCameraX;
  double d;
  if (radikant<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt2=headPathPlanner.lastRoll;
    pan=headPathPlanner.lastPan;
    return false;
  }
  else
  {
    d = (-distancePanCenterToCameraX + sqrt(radikant))/a;
  }
  if (d<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt2=headPathPlanner.lastRoll;
    pan=headPathPlanner.lastPan;
    return false;
  }
  
  //Now we can calculate x,y,z from d:
  double x = distancePanCenterToCameraX+d;
  double y = xtan*d;
  double z = ytan*d;
  
  //we currently look at point (x,y,z). it has the correct distance, but has to be rotated twice
  //to match (aim2x,aim2y,aim2z). First we want to get tilt2. This one is closer to camera and
  //using pan afterwards doesnt change z anymore
  
  tilt2 = asin(aim2.z/sqrt(x*x+z*z))-atan2(z,x);
  if (tilt2>=pi)
  {
    tilt2 -= pi2;
  }
  else if (tilt2<-pi)
  {
    tilt2 += pi2;
  }
  
  //after rotating with tilt1 we look at:
  double xt=cos(tilt2)*x-sin(tilt2)*z;
  //yt=y=ytan*d
  //zt=sin(tilt2)*x+cos(tilt2)*z=aimz
  
  //and then we only need to find the correct pan:
  pan = atan2(aim2.y,aim2.x)-atan2(y,xt);
  if (pan>=pi)
  {
    pan -= pi2;
  }
  else if (pan<-pi)
  {
    pan += pi2;
  }
  if ((tilt2<jointLimitHeadRollN)||(tilt2>jointLimitHeadRollP)||(pan<jointLimitHeadPanN)||(pan>jointLimitHeadPanP))
  {
    return false;
  }
  return true;
}


void ATH2004ERS7HeadControl::execute()
{
  updateBodyPose();
  if(headIsBlockedBySpecialActionOrWalk) 
  {
    headControlModeExecutedLastTime.headControlMode = HeadControlMode::none;
    headControlState=otherModes;
    lastHeadControlState=otherModes;
    headPathPlanner.lastTilt = sensorDataBuffer.lastFrame().data[SensorData::headTilt];
    headPathPlanner.lastPan  = sensorDataBuffer.lastFrame().data[SensorData::headPan];
    headPathPlanner.lastRoll = sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  }
  
  buildCameraMatrix(sensorDataBuffer.lastFrame(),headState);
  
  if (robotPose.getValidity()>0.5)
  {
    lastTimeOfGoodSL=SystemCall::getCurrentSystemTime();
  }
  long timeSinceBallSeenLast = SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);
  
  //Transitions between states
  switch (headControlMode.headControlMode)
  {
  case HeadControlMode::searchForBall:
  case HeadControlMode::searchAuto:
  case HeadControlMode::lookToStarsWhenStepIsFinished:
    switch (headControlState)
    {
    case otherModes:
      if(headControlMode.headControlMode == HeadControlMode::lookToStarsWhenStepIsFinished)
      {
        if(headControlModeExecutedLastTime.headControlMode != HeadControlMode::lookToStarsWhenStepIsFinished)
        {
          headControlState = waitUntilStepIsFinished;
          OUTPUT(idText, text, executedMotionRequest.positionInWalkCycle);
        }
        break;
      }
      else if (timeSinceBallSeenLast<1000)
      {
        headControlState=returnToBall;
      }
      else
      {
        headControlState=ballLost;
      }
      break;
    case lookToStarsWhenStepIsFinished:
        headControlState = lookToStarsWhenStepIsFinished;
      break;
    case waitUntilStepIsFinished:
      if(
        executedMotionRequest.positionInWalkCycle > 0.27 &&
        executedMotionRequest.positionInWalkCycle < 0.35 )
        headControlState = lookToStarsWhenStepIsFinished;
      break;
    case lookAtBall:
      if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) > timeAfterWhichBallIsConsideredLost)
      {
        headControlState=ballJustLost;
      }
      else if (headControlMode.headControlMode == HeadControlMode::searchAuto)
      {
        long consecTime = ballPosition.seen.getConsecutivelySeenTime();
        if (consecTime>300)//|| fabs(ballPosition.seen.angle()) > pi_2)
        {
          headControlState=searchAutoUpperScan;
        }
      }
      break;
    case ballJustLost:
      if (timeSinceBallSeenLast<250)
      {
        headControlState=lookAtBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=ballLost;
      }
      break;
    case ballLost:
      if (timeSinceBallSeenLast<250)
      {
        headControlState=lookAtBall;
      }	
      break;
    case returnToBall:
      if (timeSinceBallSeenLast<250)
      {
        headControlState=lookAtBall;
      }
      else if (SystemCall::getTimeSince(timeOfLastStateChange) > 500)
      {
        headControlState=ballJustLost;
      }
      break;
    case searchAutoUpperScan:
      if (headControlMode.headControlMode == HeadControlMode::searchForBall)
      {
        headControlState=returnToBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=searchAutoLowerScan;
      }
      break;
    case searchAutoLowerScan:
      if (timeSinceBallSeenLast<250)
      {
        headControlState=lookAtBall;
      }
      else if (headControlMode.headControlMode == HeadControlMode::searchForBall)
      {
        headControlState=returnToBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=ballJustLost;
      }
      break;
    }
    break;
    default:
      headControlState=otherModes;
  }
  
  //Execution
  switch (headControlState)
  {
  case lookAtBall:
    executeLookAtBall();
    break;
  case ballJustLost:
    executeBallJustLost();
    break;
  case ballLost:
    executeBallLost();
    break;
  case returnToBall:
    executeReturnToBall();
    break;
  case searchAutoUpperScan:
    executeSearchAutoUpperScan();
    break;
  case searchAutoLowerScan:
    executeSearchAutoLowerScan();
    break;
  case waitUntilStepIsFinished:
    //catchBall:
    setJoints(-1064000, 0, 439000,400,0);
    break;
  case lookToStarsWhenStepIsFinished:
    setJoints(toMicroRad(jointLimitHeadTiltP),0,toMicroRad(jointLimitHeadRollP),200, 0);
    break;
  case otherModes:
    {
      switch (headControlMode.headControlMode)
      {
      case HeadControlMode::lookLeft:
        lastScanWasLeft = true;
        lookLeft();
        break;  
      case HeadControlMode::lookRight:
        lastScanWasLeft = false;
        lookRight();
        break;
      case HeadControlMode::searchForLandmarks:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForLandmarks();
        break;  
      case HeadControlMode::searchForLines:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForLines();
        break;  
      case HeadControlMode::searchForSpecialBall:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForSpecialBall();
        break;  
      case HeadControlMode::searchForObstacles:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForObstacles();
        break;  
      case HeadControlMode::searchForObstaclesInFront:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForObstaclesInFront();
        break;  
      case HeadControlMode::lookBetweenFeet:
        setJoints(-1000000, 0, 0, 200);
        break;  
      case HeadControlMode::lookAroundWhenBallCaught:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        lookAroundWhenBallCaught();
        break;  
      case HeadControlMode::lookAroundWhenBallJustCaught:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        lookAroundWhenBallJustCaught();
        break;  
      case HeadControlMode::lookStraightAhead:
        setJoints(0,0,0,200);
        break;  
      case HeadControlMode::kickRight:
        lastScanWasLeft = false;
        kickRight();
        break;  
      case HeadControlMode::kickLeft:
        lastScanWasLeft = true;
        kickRight();
        break;  
      case HeadControlMode::catchBall:
        setJoints(-1064000, 0, 439000,400,0);
        break;
      case HeadControlMode::stayAsForced:
        stayAsForced();
        break;  
      case HeadControlMode::followTail:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        followTail();
        break;  
      case HeadControlMode::lookToStars:
        setJoints(toMicroRad(jointLimitHeadTiltP),0,toMicroRad(jointLimitHeadRollP),200, -700000);
        break;
      case HeadControlMode::lookAtPoint:
        {
          double tilt,pan,roll;
          Vector3<double> point(headControlMode.point.x,headControlMode.point.y,headControlMode.point.z);
          lookAtPoint(point,headControlMode.offset,tilt,pan,roll);
          setJoints(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll), 200,0);
          break;
        }
      case HeadControlMode::watchOrigin:
        {
          double tilt,pan,roll;
          Vector3<double> point(0,0,180);
          //its OK to start immediately here, because we scan smooth and in a way that we can always see it again:
          if (robotPose.getValidity()<0.5)
          {
            //if we dont know where to look, we look there smooth:
            if(headPathPlanner.isLastPathFinished())
            {
              unsigned long blindTime=SystemCall::getTimeSince(lastTimeOfGoodSL);
              double odoRot=currentOdometryData.rotation-lastOdometryData.rotation;
              if ((fabs(robotPose.rotation)<1.3)&&
                  ((blindTime<500)||
                   ((fabs(odoRot)<0.004)&&(blindTime<1300))
                  )
                 )
              {
                //scan for origin
                if(headPathPlanner.isLastPathFinished())
                {
                  if(lastScanWasLeft)
                  {
                    lookAtPoint(point,Vector2<int>(-65,0),tilt,pan,roll);
                    Vector3<long> right(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll));
                    Vector3<long> points[1]={right};
                    headPathPlanner.init(points,1, 550);
                  }
                  else
                  {
                    lookAtPoint(point,Vector2<int>(65,0),tilt,pan,roll);
                    Vector3<long> left(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll));
                    Vector3<long> points[1]={left};
                    headPathPlanner.init(points,1, 550);
                  }
                  lastScanWasLeft=!lastScanWasLeft;
                }
              }
              else if (odoRot==0)
              {
                //2do: somethimg usefull
              }
              else if (odoRot>0)
              {
                //if we can/will see the origin, then left
                
                lastScanWasLeft=true;
                static Vector3<long> point7(50000,1500000 ,50000);
                Vector3<long> points[1]={point7};
                headPathPlanner.init(points,1, 450);
              }
              else
              {
                //if we can/will see the origin, then right
                lastScanWasLeft=false;
                static Vector3<long> point7(50000,-1500000 ,50000);
                Vector3<long> points[1]={point7};
                headPathPlanner.init(points,1, 450);
              }
            }
            headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
          }
          else
          {
            //if we know where to look, we look there immediately:
            if (robotPose.rotation<-1.45)
            {
              setJoints(50000,1500000 ,50000, 1,0);
            }
            else if (robotPose.rotation>1.45)
            {
              setJoints(50000,-1500000 ,50000, 1,0);
            }
            else
            {
              double rota=-robotPose.speedbyDistanceToGoal*30;
              rota /= 2;
              lookAtPoint(point,Vector2<int>((int)rota,0),tilt,pan,roll);
              setJoints(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll), 200,0);
            }
          }
          break;
        }
      case HeadControlMode::lookParallelToGround:
        if (headState.bodyTilt>0)
        {
          setJoints(0,0,toMicroRad(headState.bodyTilt));
        }
        else
        {
          setJoints(toMicroRad(headState.bodyTilt),0,0);
        }
        break;
      case HeadControlMode::scanMediumWidth:
        setJoints(
          toMicroRad(headState.bodyTilt), 
          toMicroRad(cameraInfo.openingAngleWidth/4*sin((double )SystemCall::getCurrentSystemTime()/200.0)), 
          0);
        break;
      case HeadControlMode::lookJustBelowHorizon:
        setJoints(
          toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight*.5), 
          toMicroRad(cameraInfo.openingAngleWidth/4*sin((double )SystemCall::getCurrentSystemTime()/200.0)), 
          0);
        break;
      case HeadControlMode::direct:
        moveHead(headControlMode.tilt, headControlMode.pan, headControlMode.roll, headControlMode.speed, headControlMode.mouth);
      default: 
        /* if none is selected, do nothing so direction can be set manually through the head motion tester */ 
        break;
      }
    }
  }
  
  headControlModeExecutedLastTime.headControlMode = headControlMode.headControlMode;
  if (lastHeadControlState != headControlState)
  {
    timeOfLastStateChange = SystemCall::getCurrentSystemTime();
  }
  lastHeadControlState = headControlState;
  lastOdometryData = currentOdometryData;

  headPathPlanner.lastTilt=headMotionRequest.tilt;
  headPathPlanner.lastPan=headMotionRequest.pan;
  headPathPlanner.lastRoll=headMotionRequest.roll;
}


void ATH2004ERS7HeadControl::getLookAtBallAngles(double& tilt,double& pan,double& roll)
{
  double distanceToBall = Geometry::distanceTo(robotPose.getPose(), ballPosition.seen);
  double angleToBall = Geometry::angleTo(robotPose.getPose(),ballPosition.seen);

  if (angleToBall >  pi_2) angleToBall =  pi_2;
  if (angleToBall < -pi_2) angleToBall = -pi_2;
  
  if (distanceToBall > 1000) distanceToBall = 1000;

  // clip ball distances "inside" robot
  if (distanceToBall <  100) 
  {
    distanceToBall =  100;   
    angleToBall = 0;
  }


  double ballX = distanceToBall * cos(angleToBall);
  double ballY = distanceToBall * sin(angleToBall);

  Vector2<double> ballOnField = Geometry::relative2FieldCoord(robotPose, ballX, ballY);

  lookAtPoint(
    Vector3<double>(ballOnField.x, ballOnField.y, 2 * ballRadius), 
    Vector2<int>(0, 0), 
    tilt, pan, roll);
}

void ATH2004ERS7HeadControl::executeLookAtBall()
{
  double tilt, pan, roll;
  
  getLookAtBallAngles(tilt,pan,roll);
  // clipped to a minimum angle to prevent from touching the ball
  if (tilt < -0.86) tilt=-0.86;
  setJoints(toMicroRad(tilt), toMicroRad(pan), toMicroRad(roll));
}

void ATH2004ERS7HeadControl::executeBallJustLost()
{
  if (lastHeadControlState!=headControlState)
  {
    // scan around propagated ball position can look pretty ugly in easy cases, therefore we scan around seen ball
    //Vector2<double> propagatedBallPos = ballPosition.propagated.getPropagatedPosition(SystemCall::getCurrentSystemTime());
    double tilt,pan,roll;
    getLookAtBallAngles(tilt,pan,roll);
    Vector3<long> littleLeft( toMicroRad(tilt), toMicroRad(pan)+300000,toMicroRad(roll));
    Vector3<long> littleRight(toMicroRad(tilt), toMicroRad(pan)-300000,toMicroRad(roll));
    Vector3<long> littleUp(   toMicroRad(tilt), toMicroRad(pan),toMicroRad(roll)+120000);
    Vector3<long> littleDown( toMicroRad(tilt), toMicroRad(pan),toMicroRad(roll)-280000);
    Vector3<long> littlePoints[5]={littleDown,littleLeft,littleUp,littleRight,littleDown};
    headPathPlanner.init(littlePoints,5, 500);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void ATH2004ERS7HeadControl::executeBallLost()
{ 
  Vector3<long> left(0, 1400000, -350000);
  Vector3<long> right(0, -1400000, -350000);
  Vector3<long> leftTop(0,   1400000, 0);
  Vector3<long> rightTop(0, -1400000, 0);
  Vector3<long> halfLeftDown(-150000, 800000, -350000);
  Vector3<long> halfRightDown(-150000, -800000, -350000);
  Vector3<long> up(0,0,125000);
  Vector3<long> down(-400000,0,-350000);

  if ((lastHeadControlState!=headControlState)||headPathPlanner.isLastPathFinished())
  {
    if (lastScanWasLeft)
    {
      Vector3<long> points[9]={down,halfRightDown,right,rightTop,up,leftTop,left,halfLeftDown,down};
      headPathPlanner.init(points, 9, 1400);
    }
    else
    {
      Vector3<long> points[9]={down,halfLeftDown,left,leftTop,up,rightTop,right,halfRightDown,down};
      headPathPlanner.init(points, 9, 1400);
    }
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
  //lastScanWasLeft = (headMotionRequest.pan>0);
}

void ATH2004ERS7HeadControl::executeReturnToBall()
{
  if (lastHeadControlState!=headControlState)
  {
    double tilt, pan, roll;
    getLookAtBallAngles(tilt,pan,roll);
    if (tilt < -0.86) tilt=-0.86;
    
    Vector3<long> ball(toMicroRad(tilt), toMicroRad(pan),toMicroRad(roll));
    //calculated time proportional to effort to reach ball angles: 400ms for rotation of pi:
    long time = (long)(sqrt(
      (double)(headPathPlanner.lastTilt-tilt)*(double)(headPathPlanner.lastTilt-tilt)+
      (double)(headPathPlanner.lastPan-pan)*(double)(headPathPlanner.lastPan-pan)+
      (double)(headPathPlanner.lastRoll)*(double)(headPathPlanner.lastRoll))/pi/1000000*400);
    headPathPlanner.init(&ball,1, time);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void ATH2004ERS7HeadControl::executeSearchAutoUpperScan()
{
  if (lastHeadControlState!=headControlState)
  {
    if (lastScanWasLeft)
    {
      //initialize upper right scan, maximum 90
//      Vector3<long> centerTop(    0, headPathPlanner.lastPan-70000,0);
//      Vector3<long> rightTop(     0, max(-1000000,headPathPlanner.lastPan-1000000),0);
//      Vector3<long> rightBottom(  0, max(-1000000,headPathPlanner.lastPan-1000000), -100000);
      Vector3<long> centerTop(   0, headPathPlanner.lastPan-70000, 250000);
      Vector3<long> rightTop(    0, max(-1200000,headPathPlanner.lastPan-1500000),0);
      Vector3<long> rightBottom( 0, max(-1200000,headPathPlanner.lastPan-1500000),-250000);
      Vector3<long> points[3]={centerTop,rightTop,rightBottom};
      headPathPlanner.init(points,3, 550);
    }
    else
    {
      //initialize upper left scan, maximum 90
//      Vector3<long> centerTop(    0, headPathPlanner.lastPan+70000,0);
//      Vector3<long> leftTop(      0, min(1000000,headPathPlanner.lastPan+1000000),0);
//      Vector3<long> leftBottom(   0, min(1000000,headPathPlanner.lastPan+1000000),-100000);
      Vector3<long> centerTop(   0, headPathPlanner.lastPan+70000,min(headPathPlanner.lastRoll+200000,100000));
      Vector3<long> leftTop(     0, min(1200000,headPathPlanner.lastPan+1500000),0);
      Vector3<long> leftBottom(  0, min(1200000,headPathPlanner.lastPan+1500000),-250000);
      Vector3<long> points[3]={centerTop,leftTop,leftBottom};
      headPathPlanner.init(points,3, 550);
    }
    lastScanWasLeft = !lastScanWasLeft;
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void ATH2004ERS7HeadControl::executeSearchAutoLowerScan()
{
  if (lastHeadControlState!=headControlState)
  {
    double tilt,pan,roll;
    getLookAtBallAngles(tilt,pan,roll);
    if(lastScanWasLeft) // == (headPathPlanner.lastPan > 0)
    {
      //initialize lower left scan
//      Vector3<long> centerBottom(0, toMicroRad(pan)+70000,-100000);
      Vector3<long> left(-100000, headPathPlanner.lastPan-100000,-350000);
      Vector3<long> centerBottom(-250000, toMicroRad(pan)+70000,-350000);
      Vector3<long> ball(toMicroRad(tilt),toMicroRad(pan)-50000,toMicroRad(roll));
      Vector3<long> points[3]={left,centerBottom,ball};
      headPathPlanner.init(points,3, 400);
    }
    else
    {
      //initialize lower right scan
//      Vector3<long> centerBottom(0, toMicroRad(pan)-70000,-100000);
      Vector3<long> right(-100000, headPathPlanner.lastPan+100000,-350000);
      Vector3<long> centerBottom(-250000, toMicroRad(pan)-70000,-350000);
      Vector3<long> ball(toMicroRad(tilt),toMicroRad(pan)+50000,toMicroRad(roll));
      Vector3<long> points[3]={right,centerBottom,ball};
      headPathPlanner.init(points,3, 400);
    }
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}


void ATH2004ERS7HeadControl::kickLeft()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> down(-1200000,headPathPlanner.lastPan,0);
    Vector3<long> left(-1300000,1300000,0);
    Vector3<long> points[2]={down,left};
    headPathPlanner.init(points,2, 400);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void ATH2004ERS7HeadControl::kickRight()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> down(-1200000,headPathPlanner.lastPan,0);
    Vector3<long> right(-1300000,-1300000,0);
    Vector3<long> points[2]={down,right};
    headPathPlanner.init(points,2, 400);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}


void ATH2004ERS7HeadControl::lookLeft()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> down(max(headPathPlanner.lastTilt, -250000),headPathPlanner.lastPan, -300000);
    Vector3<long> left(0,1400000,-350000);
    Vector3<long> points[2]={down,left};
    headPathPlanner.init(points,2, 200);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void ATH2004ERS7HeadControl::lookRight()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> down(max(headPathPlanner.lastTilt, -250000),headPathPlanner.lastPan, -300000);
    Vector3<long> right(0,-1400000,-350000);
    Vector3<long> points[2]={down,right};
    headPathPlanner.init(points,2, 200);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}


void ATH2004ERS7HeadControl::searchForLandmarks()
{
  //just like last year, but
  /** @todo may be we would like to see lines too */
  
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0),  1500000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[2]={right,left};
      headPathPlanner.init(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, 800);
    }
    else
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, 800);
    }
  }
}

void ATH2004ERS7HeadControl::searchForLines()
{
  Vector3<long> left(0, 1500000, 0);
  Vector3<long> right(0, -1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[2]={right,left};
      headPathPlanner.init(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, 800);
    }
    else
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, 800);
    }
  }
}

void ATH2004ERS7HeadControl::searchForSpecialBall()
{
  
  //Vector3<long> centerTop(    160000, headPathPlanner.lastPan-70000,0);
  Vector3<long> rightTop(     160000, -1500000,0);
  Vector3<long> rightCenter( -120000, -1500000,0);
  Vector3<long> down(        -900000,        0,0);
  Vector3<long> leftCenter(  -120000,  1500000,0);
  Vector3<long> leftTop(      160000,  1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[3]={rightTop,down,leftTop};
      headPathPlanner.init(points,3, 8000);
    }
    else
    {
      Vector3<long> points[3]={leftTop,down,rightTop};
      headPathPlanner.init(points,3, 8000);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[3]={leftTop,down,rightTop};
      headPathPlanner.init(points,3, 8000);
    }
    else
    {
      Vector3<long> points[3]={rightTop,down,leftTop};
      headPathPlanner.init(points,3, 8000);
    }
  }
}

void ATH2004ERS7HeadControl::searchForObstacles()
{
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0),  1500000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1500000,0);
  Vector3<long> center(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 2.0), 0,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points, 2, 800);
    }
    else
    {
      Vector3<long> points[3]={right, center, left};
      headPathPlanner.init(points, 3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points, 1, 800);
    }
    else
    {
      Vector3<long> points[2]={center, left};
      headPathPlanner.init(points, 2, 800);
    }
  }
}

void ATH2004ERS7HeadControl::searchForObstaclesInFront()
{
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), 1400000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1400000,0);
  Vector3<long> center(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 1.5), 0,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points, 2, 800);
    }
    else
    {
      Vector3<long> points[3]={right, center, left};
      headPathPlanner.init(points, 3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points, 1, 800);
    }
    else
    {
      Vector3<long> points[2]={center, left};
      headPathPlanner.init(points, 2, 800);
    }
  }
}


void ATH2004ERS7HeadControl::lookAroundWhenBallCaught()
{
  Vector3<long> left(  300000, 1200000,0);
  Vector3<long> middle(     0,       0,0);
  Vector3<long> right( 300000,-1200000,0);
  //The mode is active the first time after another one
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    //If current pan is left:
    if(headPathPlanner.lastPan > 0)
    {
      //from current position to left, center and right.
      Vector3<long> points[3]={left,middle,right};
      headPathPlanner.init(points,3, 800);
    }
    else
    {
      //from current position to right, center and left.
      Vector3<long> points[3]={right,middle,left};
      headPathPlanner.init(points,3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    //init the next loop
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={middle,right};
      headPathPlanner.init(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={middle,left};
      headPathPlanner.init(points,2, 800);
    }
  }
}

void ATH2004ERS7HeadControl::lookAroundWhenBallJustCaught()
{
  Vector3<long> left(  100000, 500000,0);
  Vector3<long> right( 100000,-500000,0);
  
  //The mode is active the first time after another one
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    //If current pan is left:
    if(headPathPlanner.lastPan > 0)
    {
      //from current position to left, center and right.
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, 350);
    }
    else
    {
      //from current position to right, center and left.
      Vector3<long> points[3]={right,left};
      headPathPlanner.init(points,2, 350);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    //init the next loop
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, 350);
    }
    else
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, 350);
    }
  }
}

void ATH2004ERS7HeadControl::stayAsForced()
{
  const int tiltTolerance = 60000;
  const int panTolerance  = 60000;
  const int rollTolerance = 50000;
  
  long tiltDiff = sensorDataBuffer.lastFrame().data[SensorData::headTilt] - headMotionRequest.tilt;
  long panDiff  = sensorDataBuffer.lastFrame().data[SensorData::headPan]  - headMotionRequest.pan;
  long rollDiff = sensorDataBuffer.lastFrame().data[SensorData::headRoll] - headMotionRequest.roll;
  
  if (tiltDiff > tiltTolerance)
  {
    headMotionRequest.tilt += tiltDiff-tiltTolerance;
  }
  else if (tiltDiff < -tiltTolerance)
  {
    headMotionRequest.tilt += tiltDiff+tiltTolerance;
  }
  
  if (panDiff > panTolerance)
  {
    headMotionRequest.pan += panDiff-panTolerance;
  }
  else if (panDiff < -panTolerance)
  {
    headMotionRequest.pan += panDiff+panTolerance;
  }
  
  if (rollDiff > rollTolerance)
  {
    headMotionRequest.roll += rollDiff-rollTolerance;
  }
  else if (rollDiff < -rollTolerance)
  {
    headMotionRequest.roll += rollDiff+rollTolerance;
  }
}

void ATH2004ERS7HeadControl::followTail()
{
  long tilt, pan, tilt2;
  if(getRobotConfiguration().getRobotDesign() == RobotConfiguration::ERS7)
  {
    tilt2 = sensorDataBuffer.lastFrame().data[SensorData::headTilt2];
    if(sensorDataBuffer.lastFrame().data[SensorData::backF] > 15)
    {
      if(sensorDataBuffer.lastFrame().data[SensorData::backR] > 15) tilt2 = 0;
      else tilt2 -= toMicroRad(fromDegrees(20));
    }
    else if(sensorDataBuffer.lastFrame().data[SensorData::backR] > 15) tilt2 += toMicroRad(fromDegrees(20));
  }
  else // not ERS7
  {
    tilt2 = 0;
  }
  
  pan = toMicroRad(
    fromMicroRad(sensorDataBuffer.lastFrame().data[SensorData::tailPan]) / jointLimitTailPanN * jointLimitHeadPanN);
  
  tilt = toMicroRad(
    (fromMicroRad(sensorDataBuffer.lastFrame().data[SensorData::tailTilt]) - jointLimitTailTiltN) /
    (jointLimitTailTiltP - jointLimitTailTiltN) * (jointLimitHeadTiltP - jointLimitHeadTiltN) 
    +jointLimitHeadTiltN
    );
  
  setJoints(tilt, pan, tilt2);
}

void ATH2004ERS7HeadControl::buildCameraMatrix(const SensorData& data, const HeadState& headState)
{
  if(lastFrame != data.frameNumber)
  {
    lastFrame = data.frameNumber;
    currentTilt = fromMicroRad(data.data[SensorData::headTilt]);
    currentPan  = fromMicroRad(data.data[SensorData::headPan]);
    currentRoll = fromMicroRad(data.data[SensorData::headRoll]);
    
    Geometry::calculateCameraMatrix(currentTilt, currentPan, currentRoll, headState, cameraMatrix2);
  }
}

void ATH2004ERS7HeadControl::fromWorld2CameraCoordinates
(
 const Vector3<double> pointIn3D,  
 Vector3<double> &pointInCameraCoordinates
 )
{
  RotationMatrix bodyRotation;
  Vector3<double> bodyTranslation;
  //  double phi = -this->robotPose.getPose().getAngle();
  bodyRotation.fromKardanRPY(0, 0, -robotPose.rotation);
  bodyTranslation.x = robotPose.translation.x; 
  bodyTranslation.y = robotPose.translation.y;
  bodyTranslation.z = 0;
  
  pointInCameraCoordinates = 
    cameraMatrix2.rotation.invert()*((bodyRotation.invert()*(pointIn3D - bodyTranslation) - cameraMatrix2.translation));
}

void ATH2004ERS7HeadControl::fromWorld2RobotCoordinates
(
 const Vector3<double> pointIn3D,  
 Vector3<double> &pointInRobotCoordinates
 )
{
  RotationMatrix bodyRotation;
  Vector3<double> bodyTranslation;
  
  bodyRotation.fromKardanRPY(0, 0, -robotPose.rotation);
  
  bodyTranslation.x = robotPose.translation.x; 
  bodyTranslation.y = robotPose.translation.y;
  bodyTranslation.z = 0;
  
  pointInRobotCoordinates = bodyRotation.invert()*(pointIn3D - bodyTranslation);
}

void ATH2004ERS7HeadControl::updateBodyPose()
{
  if(sensorDataBuffer.lastFrame().frameNumber > robotVerticesBuffer[0].frameNumber)
  {
    for(int i = 0; i < sensorDataBuffer.numOfFrames; ++i)
    {
      RobotVertices rob;
      Kinematics::calcNeckAndLegPositions(sensorDataBuffer.frame[i], rob);
      robotVerticesBuffer.add(rob);
    }
    headState.bodyTilt = robotVerticesBuffer[0].bodyTilt;
    headState.bodyRoll = robotVerticesBuffer[0].bodyRoll;
    headState.neckHeight = robotVerticesBuffer[0].neckHeight;
    headState.calculated = true;
		Kinematics::calculateCameraMatrix(sensorDataBuffer.lastFrame(), headState, cameraMatrix2);
  }
}


/*
* Change log :
* 
* $Log: ATH2004ERS7HeadControl.cpp,v $
* Revision 1.10  2004/05/17 19:21:50  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.9  2004/05/11 16:37:04  dueffert
* ERS7 psd and behindMe stuff in watchOrigin improved
*
* Revision 1.8  2004/05/11 14:08:32  juengel
* Mouth closed for lookAtStarsWhenStepFinished.
*
* Revision 1.7  2004/05/10 19:10:16  juengel
* Added states waitUntilStepIsFinished and lookToStarsWhenStepIsFinished
*
* Revision 1.6  2004/05/10 10:30:33  juengel
* Added executedMotionRequest to HeadControlInterfaces.
*
* Revision 1.5  2004/05/04 11:11:03  dueffert
* port of improved watchOrigin from GT2003 finished
*
* Revision 1.4  2004/05/04 09:05:01  dueffert
* watchOrigin improved; headPathPlanner bug fixed in all copies
*
* Revision 1.3  2004/04/05 17:56:47  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.4  2004/04/02 21:11:57  dueffert
* upper-/lower-scan bug fixed; ballLost-scan always starts down
*
* Revision 1.3  2004/04/02 15:30:17  hoffmann
* look at ball parameters improved
* search auto parameters improved
*
* Revision 1.2  2004/04/01 11:25:34  juengel
* Changed state ball-lost
*
* Revision 1.1.1.1  2004/03/31 11:16:44  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.2  2004/03/29 15:12:45  jhoffman
* works now
*
* Revision 1.1  2004/03/28 14:06:41  jhoffman
* renamed headcontrolmode ATH2004 in ATH2004ERS7
*
* Revision 1.11  2004/03/27 19:44:40  juengel
* Parameter changed.
*
* Revision 1.10  2004/03/27 19:39:31  juengel
* Clipping for lookAtBall added.
*
* Revision 1.9  2004/03/27 17:21:32  juengel
* Removed executeLookAtBallGoalie.
*
* Revision 1.8  2004/03/26 13:54:18  juengel
* Old versions of look at point and search auto lower/upper scan.
*
* Revision 1.7  2004/03/25 09:23:41  jhoffman
* all modes parametrized to work with ERS7
*
* Revision 1.6  2004/03/24 18:39:41  juengel
* Added updateBodyPose()
*
* Revision 1.5  2004/03/24 14:42:45  jhoffman
* removed bug that made "ball lost" look funny
*
* Revision 1.4  2004/03/23 16:58:16  loetzsch
* improved
*
* Revision 1.3  2004/03/17 19:43:02  juengel
* Added kickLeft and kickRight.
*
* Revision 1.2  2004/03/17 09:33:59  jhoffman
* changes to state machine, look at ball improved, look just below horizon recovered from revision 1.23
*
* Revision 1.1  2004/03/16 14:00:21  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.1  2004/03/15 17:11:39  hoffmann
* - added ATH2004ERS7HeadControl
* - added ATH2004LEDControl
* - headmotiontester shows "tilt2"
* - motion process updates odometry while no new robotPose is received, added to motion request
* - some ui adjustments
* - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
*
*
*
*/
