/**
* @file DDD2004HeadControl.cpp
*
* Implementation of class DDD2004HeadControl
*/

#ifdef _WIN32
static const unsigned int timeAfterWhichBallIsConsideredLost = 1000;
#else
static const unsigned int timeAfterWhichBallIsConsideredLost = 300;
#endif


#include "Platform/GTAssert.h"
#include "DDD2004HeadControl.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"
#include "Platform/SystemCall.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Debugging/DebugDrawings.h"

void DDD2004HeadControl::HeadPathPlanner::init(const Vector3<long>* vectors, long* durations,int numberOfVectors, bool optimizeTimings)
{
  const RobotDimensions& robotDimensions = getRobotConfiguration().getRobotDimensions();
  if (numberOfVectors>maxNumberOfPoints-1) numberOfVectors=maxNumberOfPoints-1;
	
	enum {FRAMEQUOTIENT = 8};
	
  const Range<long> jointRangeHeadTilt(toMicroRad(robotDimensions.jointLimitHeadTiltN),toMicroRad(robotDimensions.jointLimitHeadTiltP));
  const Range<long> jointRangeHeadPan( toMicroRad(robotDimensions.jointLimitHeadPanN), toMicroRad(robotDimensions.jointLimitHeadPanP));
  const Range<long> jointRangeHeadRoll(toMicroRad(robotDimensions.jointLimitHeadRollN),toMicroRad(robotDimensions.jointLimitHeadRollP));

	//start from current position
  points[0].x = jointRangeHeadTilt.limit(lastTilt);
  points[0].y = jointRangeHeadPan.limit(lastPan);
  points[0].z = jointRangeHeadRoll.limit(lastRoll);
	// sometimes the values are weird, so the first frame should be ignored
	if (points[0].x != lastTilt || points[0].y != lastPan || points[0].z != lastRoll)
	{
		points[0].x = jointRangeHeadTilt.limit(vectors[0].x);
		points[0].y = jointRangeHeadPan.limit(vectors[0].y);
    points[0].z = jointRangeHeadRoll.limit(vectors[0].z);
		durations[0]=8;
	}
	
  currentPoint = 0;
  numberOfPoints = numberOfVectors;
  currentFrame = 0;
 
  //calculate total distance and speed
   int i;
  for (i=0;i<numberOfVectors;i++)
  {
    //clip arcs to physical limits
    points[i+1].x = jointRangeHeadTilt.limit(vectors[i].x);
    points[i+1].y = jointRangeHeadPan.limit(vectors[i].y);
    points[i+1].z = jointRangeHeadRoll.limit(vectors[i].z);
		if (optimizeTimings)
		{
			long tempTime = calculateHeadTiming(points[i],points[i+1]);
			if (durations[i] < tempTime)
				durations[i] = tempTime;
		}
  }
	
	long overAllDuration = calculateDurationsSum(durations, numberOfVectors);
	numberOfFrames = overAllDuration / FRAMEQUOTIENT;

  //double speed = distance/(double)numberOfFrames; //in urad per frame
  //if (speed<minHeadSpeed) speed=minHeadSpeed;
  
  //calculate duration for each part of the route
	//OUTPUT (idText,text,"------------------");
	//OUTPUT (idText,text,"Planning headpath with " << numberOfVectors << " Points - Duration:" << overAllDuration);
	firstFrame[0] = 0;
	for (i=0;i<numberOfVectors;i++)
  {
		firstFrame[i + 1] = firstFrame[i] + (durations[i] / FRAMEQUOTIENT);
		/*
		OUTPUT (idText,text,"Point:" << i << " - Duration:" << durations[i]);
		OUTPUT(idText,text,"T1/P/T2:(" << points[i].x << ")/(" << points[i].y << ")/(" << points[i].z << ")");
		OUTPUT (idText,text,"Frame  :" << firstFrame[i]);
		OUTPUT (idText,text,"------------");
		*/
		
  }
		/*
		OUTPUT (idText,text,"Point:" << i );
		OUTPUT(idText,text,"T1/P/T2:(" << points[i].x << ")/(" << points[i].y << ")/(" << points[i].z << ")");
		OUTPUT (idText,text,"------------");
		*/
}

long DDD2004HeadControl::HeadPathPlanner::calculateDurationsSum(long* duration, int durations)
{
	long sum=0;
	for (int i=0;i<durations;i++)
	{
		// correcting, if duration is shorter than min movement time
		if (duration[i]<8) duration[i]=8;
		sum+=duration[i];
	}
	return sum;
}


bool DDD2004HeadControl::HeadPathPlanner::headPositionReached(Vector3<long> pos)
{
	double grad2Rad = toMicroRad(pi / 180) ;
	if (   abs(sensorDataBuffer.lastFrame().data[SensorData::headTilt]-pos.x) < 4 * grad2Rad
		  && abs(sensorDataBuffer.lastFrame().data[SensorData::headPan]-pos.y)  < 4 * grad2Rad
			&& abs(sensorDataBuffer.lastFrame().data[SensorData::headRoll]-pos.z) < 5 * grad2Rad)
		return true;
	return false;
}

bool DDD2004HeadControl::HeadPathPlanner::getAngles(long& tilt, long& pan, long& roll)
{
	if (currentFrame<numberOfFrames)
  {
		currentFrame++;
    while ((currentFrame>firstFrame[currentPoint+1])&&
           (currentPoint<numberOfPoints-1)&&
           (currentFrame<numberOfFrames))
    {
      currentPoint++;
    }
		    
    double distanceInFrames=0;
    if (currentPoint<numberOfPoints)
      distanceInFrames = firstFrame[currentPoint+1]-firstFrame[currentPoint];
    
    if (distanceInFrames==0)
    {
      tilt = points[currentPoint].x;
      pan  = points[currentPoint].y;
      roll = points[currentPoint].z;
    }
    else
    {
      double leftFactor = (firstFrame[currentPoint+1]-currentFrame)/distanceInFrames;
      double rightFactor = 1-leftFactor;
      debugTilt1 = tilt = (long)(leftFactor*points[currentPoint].x + rightFactor*points[currentPoint+1].x);
      debugPan = pan  = (long)(leftFactor*points[currentPoint].y + rightFactor*points[currentPoint+1].y);
			debugTilt2 = roll = (long)(leftFactor*points[currentPoint].z + rightFactor*points[currentPoint+1].z);
			/*    
			OUTPUT(idText,text,"firstFrame:" << firstFrame[currentPoint+1]);
			OUTPUT(idText,text,"Pan:" << debugPan << " rightFactor:" << rightFactor);
			OUTPUT(idText,text,"DistanceInFrame:" << distanceInFrames);
			OUTPUT(idText,text,"currentPoint:" << currentPoint );
			OUTPUT(idText,text,"currentFrame:" << currentFrame );
			OUTPUT(idText,text,"T1/P/T2:(" << tilt << ")/(" << pan << ")/(" << roll << ")");
			*/
    }
  }
	else
	{
		if (debugOut)
		{
			/*
			OUTPUT(idText,text,"getAngles: LastFrame:" << currentFrame);
			OUTPUT(idText,text,"T1/P/T2:(" << tilt << ")/(" << pan << ")/(" << roll << ")");
			*/
			debugOut = false;
		}
  }
	return (currentFrame>=numberOfFrames);
}

long DDD2004HeadControl::HeadPathPlanner::calculateHeadTiming(Vector3<long> &pos1,Vector3<long> &pos2)
{
	Vector3<long> minSpeed;
	minSpeed.x = abs(pos1.x-pos2.x)/headPathSpeedTilt1;
	minSpeed.y = abs(pos1.y-pos2.y)/headPathSpeedPan;
	minSpeed.z = abs(pos1.z-pos2.z)/headPathSpeedTilt2;

	// explain slowest speed
	return max(max(minSpeed.x,minSpeed.y),minSpeed.z);
}

// ----------------------------------------------------------
// ---------------------------------------- Begin HeadControl
// ----------------------------------------------------------

DDD2004HeadControl::DDD2004HeadControl(const HeadControlInterfaces& interfaces)
: HeadControl(interfaces),headControlState(otherModes),lastHeadControlState(otherModes),
lastScanWasLeft(true),headPathPlanner(sensorDataBuffer)
{ 
  headPathPlanner.lastTilt = sensorDataBuffer.lastFrame().data[SensorData::headTilt];
  headPathPlanner.lastPan  = sensorDataBuffer.lastFrame().data[SensorData::headPan];
  headPathPlanner.lastRoll = sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  scanPan = 0.0;
  leftWatch= true;

	//setup the main looking directions
	setupMainAngles();
	lastAutoScanWasLeft = true;
	communicatedBallTrustedCount = 0;

	// autowatch landmark init
	usedLandmarkIndex=0;
	lastUsedLandmarkIndex=100;
	usedLandmarkTime = lastSweepTime = SystemCall::getCurrentSystemTime();

	// these values are the default settings for standard pid values
	speedTilt1 = 1500;
	speedPan   = 5350;
	speedTilt2 = 2350;

	// setting speed in headpath planner
	headPathPlanner.headPathSpeedTilt1 = (long) speedTilt1;
	headPathPlanner.headPathSpeedPan   = (long) speedPan;
	headPathPlanner.headPathSpeedTilt2 = (long) speedTilt2;

	calibrationReset();

}

void DDD2004HeadControl::setJoints(long tilt, long pan, long roll, long speed, long mouth)
{
  headMotionRequest.tilt = tilt;
  headMotionRequest.pan  = pan;
  headMotionRequest.roll = roll;
  headMotionRequest.mouth = mouth;
}


void DDD2004HeadControl::moveHead(long tilt, long pan, long roll, long speed, long mouth)
{
  //tilt = tilt + sensorDataBuffer.lastFrame().data[SensorData::headTilt];
  //pan = pan + sensorDataBuffer.lastFrame().data[SensorData::headPan];
  //roll = roll + sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  //mouth = mouth + sensorDataBuffer.lastFrame().data[SensorData::mouth];
  
  if (speed >= 400)
  {
    if (headMotionRequest.tilt != tilt) {
      if (headMotionRequest.tilt > toMicroRad(jointLimitHeadTiltP))
        headMotionRequest.tilt = toMicroRad(jointLimitHeadTiltP);
      else {
        if (headMotionRequest.tilt < toMicroRad(jointLimitHeadTiltN))
          headMotionRequest.tilt = toMicroRad(jointLimitHeadTiltN);
        else
          headMotionRequest.tilt = tilt;
      }
    }
    if (headMotionRequest.pan != pan) {
      if (headMotionRequest.pan > toMicroRad(jointLimitHeadPanP))
        headMotionRequest.pan = toMicroRad(jointLimitHeadPanP);
      else {
        if (headMotionRequest.pan < toMicroRad(jointLimitHeadPanN))
          headMotionRequest.pan = toMicroRad(jointLimitHeadPanN);
        else
          headMotionRequest.pan = pan;
      }
    }
    if (headMotionRequest.roll != roll) {
      if (headMotionRequest.roll > toMicroRad(jointLimitHeadRollP))
        headMotionRequest.roll = toMicroRad(jointLimitHeadRollP);
      else {
        if (headMotionRequest.roll > toMicroRad(jointLimitHeadRollN))
          headMotionRequest.roll = toMicroRad(jointLimitHeadRollN);
        else
          headMotionRequest.roll = roll;
      }
    }
    if (headMotionRequest.mouth != mouth)
      headMotionRequest.mouth = mouth;
  }
  else
  {
    if (headPathPlanner.isLastPathFinished())
    {
      Vector3<long> point(tilt,pan,roll);
      Vector3<long> points[1]={point};
      headPathPlanner.oldInit(points,1,2000);
    }
    headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
  }
}


void DDD2004HeadControl::lookAtPoint(const Vector3<double> &pos,const Vector2<int> &offset,double& tilt,double& pan,double& roll)
{
  Vector3<double> posInCameraCoordinates;
  posInCameraCoordinates = cameraMatrix2.rotation.invert() * (pos - cameraMatrix2.translation);
  Vector2<int> pointToLookTo;
  Vector2<double> angles;
  angles.x = atan2(posInCameraCoordinates.x, posInCameraCoordinates.y);
  angles.y = atan2(posInCameraCoordinates.x, posInCameraCoordinates.z);
  Geometry::calculatePointByAngles(angles, cameraMatrix2, cameraInfo, pointToLookTo);
  //debugging...
  LINE(sketch, offset.x, 0, offset.x, 144, 1, Drawings::ps_solid, Drawings::gray);
  LINE(sketch, 0,offset.y, 176, offset.y, 1, Drawings::ps_solid, Drawings::gray);
  DOT(sketch, pointToLookTo.x - 88, pointToLookTo.y - 72, Drawings::white, Drawings:: black);
  DEBUG_DRAWING_FINISHED(sketch);
  /*  
  PAINT_TO_DEBUGDRAWING(crosshairs,
  crosshairsDrawing.line(0, offset.y + 71, 175, offset.y + 71, DebugDrawing::ps_solid, 0, DebugDrawing::Color(255,255,255) );
  crosshairsDrawing.line(offset.x + 87, 0, offset.x + 87, 143, DebugDrawing::ps_solid, 0, DebugDrawing::Color(255,255,255) );
  );
  
  GTMath::Vector2<int> onScreen;
  onScreen.x = (int)(- transformedPoint.y*fx/(transformedPoint.x));
  onScreen.y = (int)(- transformedPoint.z*fy/(transformedPoint.x));
  
  PAINT_TO_DEBUGDRAWING(crosshairs,
  crosshairsDrawing.dot(87+onScreen.x, 71+onScreen.y, DebugDrawing::Color(255,255,255), DebugDrawing::Color(127,127,255));
  );
  
  SEND_DEBUGDRAWING(crosshairs);
  */
  
  int xoff = offset.x;
  int yoff = offset.y;
  
  //Is the offset in range?
  if (yoff > cameraInfo.resolutionHeight / 2) yoff = cameraInfo.resolutionHeight / 2;
  else if (yoff < -cameraInfo.resolutionHeight / 2) yoff = -cameraInfo.resolutionHeight / 2;
  
  if (xoff > cameraInfo.resolutionWidth / 2) xoff = cameraInfo.resolutionWidth / 2;
  else if (xoff < -cameraInfo.resolutionWidth / 2) xoff = -cameraInfo.resolutionWidth / 2;
  
  Pose2D robotPose = this->robotPose.getPose();
  Vector3<double> aim1(
    pos.x-robotPose.translation.x,
    pos.y-robotPose.translation.y,
    pos.z-headState.neckHeight
    );
  
  RotationMatrix rot;
  Vector3<double> aim;
  rot.rotateY(-headState.bodyTilt).
    rotateX(headState.bodyRoll).
    rotateZ(-robotPose.getAngle());
  aim = rot*aim1;
  
  //now we have rotated the coordinates in a way that the neck
  //is (0,0,0) and should look without any body rotation to aim:
  //aim.x in front, aim.y left, aim.z above
  
  //Looking with a certain xoffset and yoffset gives us a ray from camera into space.
  //we want the coordinates of all points on that ray to be a simple funtion of a single
  //argument, the distance from camera to an image plane
  double xtan = -tan(cameraInfo.openingAngleWidth / 2)*xoff/(cameraInfo.resolutionWidth / 2);
  double ytan = tan(cameraInfo.openingAngleHeight / 2)*yoff/(cameraInfo.resolutionHeight / 2);
  
  if (getRobotConfiguration().getRobotDesign()==RobotDesign::ERS7)
  {
    //due to ERS7 anatomy we can now change tilt1 and/or tilt2 to reach our aim. There
    //are 3 possibly useful strategies for that:
    //1) change tilt1 and tilt2 synchronously: this would look cool but will be uncalculatable
    //2) maximize tilt1: try to find tilt2 solution for max tilt1, if there is none try to find
    //   tilt1 solution for minimal tilt2
    //3) maximize tilt2: try to find tilt1 solution for max tilt2, if there is none try to find
    //   tilt2 solution for minimal tilt1
    
    //this is method 2):
    tilt=0;
    if (!lookAtPointFixTilt1(aim,xtan,ytan,tilt,pan,roll))
    {
      roll=jointLimitHeadRollN;
      lookAtPointFixTilt2(aim,xtan,ytan,tilt,pan,roll);
    }
    
    /*
    // just a check to see that everything went right on ERS7:
    RotationMatrix rot2;
    rot2.rotateY(tilt);
    Vector3<double> aim2 = rot2*aim;
    RotationMatrix rot3;
    rot3.rotateZ(-pan);
    Vector3<double> aim3 = rot3*aim2;
    Vector3<double> aim4 = aim3-Vector3<double>(0,0,distanceNeckToPanCenter);
    RotationMatrix rot4;
    rot4.rotateY(roll);
    Vector3<double> aim5 = rot4*aim4;
    Vector3<double> aim6 = aim5-Vector3<double>(distancePanCenterToCameraX,0,0);
    double realxoff=-aim6.y/aim6.x*cameraInfo.resolutionWidth / 2/tan(cameraInfo.openingAngleWidth/2);
    double realyoff=aim6.z/aim6.x*cameraInfo.resolutionHeight / 2/tan(cameraInfo.openingAngleHeight/2);
    if ((aim5.x<=0)||(fabs(realxoff-xoff)>0.1)||(fabs(realyoff-yoff)>0.1))
    {
    bool mistake=true;
    }
    */
    const Range<double> jointRangeHeadTilt2( jointLimitHeadRollN, jointLimitHeadRollP);
    roll = jointRangeHeadTilt2.limit(roll);
  }
  else
  {
    roll=0;
    lookAtPointFixTilt2(aim,xtan,ytan,tilt,pan,roll);
    /*
    // just a check to see that everything went right on ERS210:
    RotationMatrix rot2;
    rot2.rotateY(tilt);
    Vector3<double> aim2 = rot2*aim;
    RotationMatrix rot3;
    rot3.rotateZ(-pan);
    Vector3<double> aim3 = rot3*aim2;
    Vector3<double> aim4 = aim3-Vector3<double>(distancePanCenterToCameraX,0,distanceNeckToPanCenter);
    double realxoff=-aim4.y/aim4.x*cameraInfo.resolutionWidth / 2/tan(cameraInfo.openingAngleWidth/2);
    double realyoff=aim4.z/aim4.x*cameraInfo.resolutionHeight / 2/tan(cameraInfo.openingAngleHeight/2);
    if ((aim3.x<=0)||(fabs(realxoff-xoff)>0.1)||(fabs(realyoff-yoff)>0.1))
    {
    bool mistake=true;
    }
    */
  }
  
  const Range<double> jointRangeHeadTilt(jointLimitHeadTiltN,jointLimitHeadTiltP);
  const Range<double> jointRangeHeadPan( jointLimitHeadPanN, jointLimitHeadPanP);
  tilt = jointRangeHeadTilt.limit(tilt);
  pan = jointRangeHeadPan.limit(pan);
}

bool DDD2004HeadControl::lookAtPointFixTilt2(const Vector3<double> &aim, const double& xtan, const double& ytan, double& tilt, double& pan, const double& tilt2)
{
  //d shall be the distance from camera to object plane.
  
  //then we look at a point with these coordinates from tilt2 joint:
  //x = pcc+d
  //y = xtan*d
  //z = ytan*d
  //after rotation with tilt2 we get in coordinates from tilt1 joint:
  //x = Cos[t2]*(pcc + d) - Sin[t2]*(ytan*d)
  //y = xtan*d
  //z = npc + Sin[t2]*(pcc + d) + Cos[t2]*(ytan*d)
  
  //we know that we want to look at (aimx,aimy,aimz). Thats Sqrt[aimx^2+aimy^2+aimz^2] away from the neck.
  //The matching point (x,y,z) on our ray of look must have the same distance. this can be resolved to d
  //(thanks to Mathematica, see lookAtPointUDFinal.nb):
  
  //every visible d is positive, so with some helping variables we get:
  double a = 1 + xtan*xtan+ytan*ytan;
  double ns2 = distanceNeckToPanCenter*sin(tilt2);
  double nc2 = distanceNeckToPanCenter*cos(tilt2);
  double b = distancePanCenterToCameraX + nc2*ytan + ns2;
  double c = aim.x*aim.x + aim.y*aim.y + aim.z*aim.z - distanceNeckToPanCenter*distanceNeckToPanCenter
    - distancePanCenterToCameraX*distancePanCenterToCameraX - 2*distancePanCenterToCameraX*ns2;
  double radikant = b*b + a*c;
  double d;
  if (radikant<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt=headPathPlanner.lastTilt;
    pan=headPathPlanner.lastPan;
    return false;
  }
  else
  {
    d = -(distancePanCenterToCameraX + ytan*nc2 + ns2 - sqrt(radikant))/a;
  }
  if (d<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt=headPathPlanner.lastTilt;
    pan=headPathPlanner.lastPan;
    return false;
  }
  
  //Now we can calculate x,y,z from d:
  double x = cos(tilt2)*(distancePanCenterToCameraX + d) - sin(tilt2)*(ytan*d);
  double y = xtan*d;
  double z = distanceNeckToPanCenter + sin(tilt2)*(distancePanCenterToCameraX + d) + cos(tilt2)*(ytan*d);
  
  //we currently look at point (x,y,z). it has the correct distance, but has to be rotated twice
  //to match (aimx,aimy,aimz). First we want to get the y coordinate to aimy by rotating with pan.
  //This is useful, because pan is closer to camera and changing tilt afterwards does not change
  //y coordinate anymore. As this is done around z-axis, lets look into the xy-plane through (x,y,z).
  //We look at a point that is Sqrt[x^2+y^2] away from the z-axis and has an arc of ArcTan[x,y] (=atan2(y,x)).
  //We would like to look at a point in the same distance at y coordinate aimy. That would form an
  //arc of ArcSin[aimy/Sqrt[x^2+y^2]]. Therefore we have to rotate with:
  
  pan = asin(aim.y/sqrt(x*x+y*y))-atan2(y,x);
  if (pan>=pi)
  {
    pan -= pi2;
  }
  else if (pan<-pi)
  {
    pan += pi2;
  }
  
  //There is a second solution here: the arc we want the point to have above the ground can also
  //be Pi-ArcSin[aimy/Sqrt[x^2+y^2]]. Then it is definitely behind us, but that is possible and
  //useful in some cases.
  
  double pantemp = pi-asin(aim.y/sqrt(x*x+y*y))-atan2(y,x);
  if (pantemp>=pi)
  {
    pantemp -= pi2;
  }
  else if (pantemp<-pi)
  {
    pantemp += pi2;
  }
  
  //If the point is behind us, then we prefer to test the second solution first, because its nicer
  //to look at those points with upright head. Explanation: Imagine looking with the lower left pixel.
  //We want to look at a point left and a little behind of us. We find a correct y left a little
  //infront and a little behind us. If we take the one infront, we can still manage to rotate to the
  //one behind with very low tilt. Therefore we try the one behind us first in this case, so:
  if (aim.x<0)
  {
    double temp=pan;
    pan=pantemp;
    pantemp=temp;
  }
  
  //(If pan is in bounds) and after rotating with pan we look at:
  double xt=cos(pan)*x-sin(pan)*y;
  //yt=sin(pan)*x+cos(pan)*y=aimy
  //zt=z
  
  //and then we only need to find the correct tilt1:
  tilt = atan2(aim.z,aim.x)-atan2(z,xt);
  if (tilt>=pi)
  {
    tilt -= pi2;
  }
  else if (tilt<-pi)
  {
    tilt += pi2;
  }
  double pansave=pan;
  double tiltsave=tilt;
  if ((tilt<jointLimitHeadTiltN)||(tilt>jointLimitHeadTiltP)||(pan<jointLimitHeadPanN)||(pan>jointLimitHeadPanP))
  {
    pan=pantemp;
    xt = cos(pan)*x - sin(pan)*y;
    tilt = atan2(aim.z,aim.x)-atan2(z,xt);
    if (tilt>=pi)
    {
      tilt -= pi2;
    }
    else if (tilt<-pi)
    {
      tilt += pi2;
    }
    if ((tilt<jointLimitHeadTiltN)||(tilt>jointLimitHeadTiltP)||(pan<jointLimitHeadPanN)||(pan>jointLimitHeadPanP))
    {
      //if we cant see the point, its better to look with upright head:
      pan=pansave;
      tilt=tiltsave;
      return false;
    }
  }
  return true;
}

bool DDD2004HeadControl::lookAtPointFixTilt1(const Vector3<double> &aim, const double& xtan, const double& ytan, const double& tilt, double& pan, double& tilt2)
{
  //first convert from neck coordinates to before-tilt2 cordinates:
  Vector3<double> aim2(
    cos(tilt)*aim.x - sin(tilt)*aim.z,
    aim.y,
    sin(tilt)*aim.x + cos(tilt)*aim.z - distanceNeckToPanCenter
    );
  //and than we start the same way as in the fixTilt2 case, but much easier:
  //x = pcc + d
  //y = xtan*d
  //z = ytan*d
  
  //we know that we want to look at (aim2x,aim2y,aim2z). Thats Sqrt[aim2x^2+aim2y^2+aim2z^2] away from the tilt2 axis center.
  //The matching point (x,y,z) on our ray of look must have the same distance. this can be resolved to d
  //(thanks to Mathematica, see lookAtPointUDFinal.nb):
  
  //every visible d is positive, so with some helping variables we get:
  double a = 1 + xtan*xtan+ytan*ytan;
  double radikant = a*(aim2.x*aim2.x+aim2.y*aim2.y+aim2.z*aim2.z - distancePanCenterToCameraX*distancePanCenterToCameraX)
    + distancePanCenterToCameraX*distancePanCenterToCameraX;
  double d;
  if (radikant<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt2=headPathPlanner.lastRoll;
    pan=headPathPlanner.lastPan;
    return false;
  }
  else
  {
    d = (-distancePanCenterToCameraX + sqrt(radikant))/a;
  }
  if (d<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt2=headPathPlanner.lastRoll;
    pan=headPathPlanner.lastPan;
    return false;
  }
  
  //Now we can calculate x,y,z from d:
  double x = distancePanCenterToCameraX+d;
  double y = xtan*d;
  double z = ytan*d;
  
  //we currently look at point (x,y,z). it has the correct distance, but has to be rotated twice
  //to match (aim2x,aim2y,aim2z). First we want to get tilt2. This one is closer to camera and
  //using pan afterwards doesnt change z anymore
  
  tilt2 = asin(aim2.z/sqrt(x*x+z*z))-atan2(z,x);
  if (tilt2>=pi)
  {
    tilt2 -= pi2;
  }
  else if (tilt2<-pi)
  {
    tilt2 += pi2;
  }
  
  //after rotating with tilt1 we look at:
  double xt=cos(tilt2)*x-sin(tilt2)*z;
  //yt=y=ytan*d
  //zt=sin(tilt2)*x+cos(tilt2)*z=aimz
  
  //and then we only need to find the correct pan:
  pan = atan2(aim2.y,aim2.x)-atan2(y,xt);
  if (pan>=pi)
  {
    pan -= pi2;
  }
  else if (pan<-pi)
  {
    pan += pi2;
  }
  if ((tilt2<jointLimitHeadRollN)||(tilt2>jointLimitHeadRollP)||(pan<jointLimitHeadPanN)||(pan>jointLimitHeadPanP))
  {
    return false;
  }
  return true;
}

void DDD2004HeadControl::executeGoaliePlanning(long timeSinceBallSeenLast)
{

	//OUTPUT(idText,text,"ExecutingGolaiePlanning");
  // Transitions between states
	// This statemachine is confusing
	// First the next headmovement is planned, after that, it will be executed by
	// anonther statemachine
	// ----------------------------------- Planing ------------------------------------
	long timeBallConsecutivelySeenTime =  ballPosition.seen.getConsecutivelySeenTime();

	switch (headControlMode.headControlMode)
  {
		case HeadControlMode::searchForBall:
		switch (headControlState)
		{
			case otherModes:
				calibrationReset();
				//OUTPUT(idText,text,"In otherModes/Planning");
				if (timeSinceBallSeenLast>500)
					headControlState=searchAutoScan; //returnToBall;
				else
					headControlState=ballLost;
				break;
			case searchAutoScan:
			case lookAtBall:
				if (timeSinceBallSeenLast > 250)
					headControlState=ballJustLost;
				else
					headControlState=searchAutoScan;
				break;
			case doSweep:
				if (headPathPlanner.isLastPathFinished())
					headControlState=lookAtBall;
				break;
			case ballJustLost:
				// do a sweep until ball was found or path has ended
				if (timeSinceBallSeenLast < 250)
						headControlState=searchAutoScan;
				else if (headPathPlanner.isLastPathFinished())
				{
					headControlState=ballLost;
					communicatedBallTrustedCount=0;
				}
				break;
			case ballLost:
				// until ball seen -> ballLost
				if (timeSinceBallSeenLast < 250)
					headControlState=searchAutoScan;
				break;
			
		} //end 2.switch
		break;
		case HeadControlMode::searchAuto:
		case HeadControlMode::searchAutoForGoalie:
		
		
		switch (headControlState)
		{			
			case otherModes:
				calibrationReset();
					headControlState=searchAutoScan; //returnToBall
				break;
			case ballJustLost:
				// do a sweep until ball was found or path has ended
				usedLandmarkTime = usedLandmarkTime-10000;
				lostBall = true;
				if (timeSinceBallSeenLast < 300)
					headControlState=searchAutoScan;
				else if (headPathPlanner.isLastPathFinished())
				{
					headControlState=ballLost;
					communicatedBallTrustedCount = 0;
				}
				break;
			case ballLost:
				// until ball seen -> ballLost
				if (timeSinceBallSeenLast < 250)
					headControlState=lookAtBall;
				break;
			case lookAtBall:
				if (timeSinceBallSeenLast < 300 && timeBallConsecutivelySeenTime > 300)
				{
						headControlState=searchAutoScan;
						communicatedBallTrustedCount = 0;
				}
				else
					headControlState=ballJustLost;
				break;
			case searchAutoScan:
				if (timeSinceBallSeenLast > 300)
				{
					headControlState=ballJustLost;
				}
				else if (SystemCall::getTimeSince(lastSweepTime)>5000)
				{
					headControlState=doSweep;
					lastSweepTime=SystemCall::getCurrentSystemTime();
				}
				break;
			case doSweep:
				if (headPathPlanner.isLastPathFinished())
					headControlState=searchAutoScan;
				break;

		} //end 2.switch
		break;
		default:
				headControlState=otherModes;
  } // end 1. Switch
	
	//OUTPUT(idText,text,"End Planning");
	
}

void DDD2004HeadControl::executePlayerPlanning(long timeSinceBallSeenLast)
{
	//OUTPUT(idText,text,"ExecutingGolaiePlanning");
  // Transitions between states
	// This statemachine is confusing
	// First the next headmovement is planned, after that, it will be executed by
	// anonther statemachine
	// ----------------------------------- Planing ------------------------------------
	long timeBallConsecutivelySeenTime =  ballPosition.seen.getConsecutivelySeenTime();
	double dist;
	switch (headControlMode.headControlMode)
  {
		case HeadControlMode::searchForBall:
		switch (headControlState)
		{
			case otherModes:
				//OUTPUT(idText,text,"In otherModes/Planning");
				calibrationReset();
				if (timeSinceBallSeenLast>500)
					headControlState=lookAtBall; //returnToBall;
				else
					headControlState=ballLost;
				break;
			case searchAutoScan:
			case lookAtBall:
				if (timeSinceBallSeenLast < 300 && timeBallConsecutivelySeenTime > 300)
				{
						headControlState=searchAutoScan;
						communicatedBallTrustedCount = 0;
				}
				else
					headControlState=ballJustLost;
				break;
			case doSweep:
				if (headPathPlanner.isLastPathFinished())
					headControlState=lookAtBall;
				break;
			case ballJustLost:
				// do a sweep until ball was found or path has ended
				if (timeSinceBallSeenLast < 250)
						headControlState=lookAtBall;
				else if (headPathPlanner.isLastPathFinished())
					headControlState=ballLost;
				break;
			case ballLost:
				// until ball seen -> ballLost
				if (timeSinceBallSeenLast < 250)
				{
					headControlState=lookAtBall;
					communicatedBallTrustedCount = 0;
				}
				break;
			
		} //end 2.switch
		break;
		case HeadControlMode::searchAuto:
		case HeadControlMode::searchAutoForGoalie:
		
		
		switch (headControlState)
		{			
			case otherModes:
				calibrationReset();
					headControlState=lookAtBall; //returnToBall
				break;
			case ballJustLost:
				// do a sweep until ball was found or path has ended
				usedLandmarkTime = usedLandmarkTime-10000;
				lostBall = true;
				if (timeSinceBallSeenLast < 300)
					headControlState=lookAtBall;
				else if (headPathPlanner.isLastPathFinished())
					headControlState=ballLost;
				break;
			case ballLost:
				// until ball seen -> ballLost
				if (timeSinceBallSeenLast < 250)
				{
						headControlState=lookAtBall;
						communicatedBallTrustedCount = 0;
				}
				break;
			case lookAtBall:
				if (timeSinceBallSeenLast < 300 && timeBallConsecutivelySeenTime > 300)
				{
						headControlState=searchAutoScan;
						communicatedBallTrustedCount = 0;
				}
				else
					headControlState=ballJustLost;
				break;
			case searchAutoScan:
				if (timeSinceBallSeenLast > 300)
					headControlState=ballJustLost;
				dist = (ballPosition.seen-robotPose.translation).abs();
				if (dist>1300
						&& SystemCall::getTimeSince(lastSweepTime)>5000)
				{
					headControlState=doSweep;
					lastSweepTime=SystemCall::getCurrentSystemTime();
				}
				break;
			case doSweep:
				if (headPathPlanner.isLastPathFinished())
					headControlState=lookAtBall;
				break;

		} //end 2.switch
		break;
		default:
				headControlState=otherModes;
  } // end 1. Switch
	
	//OUTPUT(idText,text,"End Planning");
} // end executeGoaliePlanning


void DDD2004HeadControl::execute()
{
	// is there any global function, which tells the role of the robot
	
	// Check if special action blocks the head
	if(headIsBlockedBySpecialActionOrWalk) 
  {
    headControlModeExecutedLastTime.headControlMode = HeadControlMode::none;
    headControlState=otherModes;
    lastHeadControlState=otherModes;
    headPathPlanner.lastTilt = sensorDataBuffer.lastFrame().data[SensorData::headTilt];
    headPathPlanner.lastPan  = sensorDataBuffer.lastFrame().data[SensorData::headPan];
    headPathPlanner.lastRoll = sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  }

  // do some other stuff
	headMotionRequest.mouth=0;  
  buildCameraMatrix(sensorDataBuffer.lastFrame(),headState);
  long timeSinceBallSeenLast = SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);
	// decide if player or goalie
	if (getPlayer().getPlayerNumber()==Player::one)
		executeGoaliePlanning(timeSinceBallSeenLast);
	else
		executePlayerPlanning(timeSinceBallSeenLast);
	
 
  
  // ----------------------------------- Execution ------------------------------------
 
	//if (headControlState != lastHeadControlState)
		//OUTPUT (idText,text, "headControlState is " << headControlState);
  switch (headControlState)
  {
  case lookAtBall:
    executeLookAtBall();
    break;
  case ballJustLost:
    executeBallJustLost();
    break;
  case ballLost:
    executeBallLost();
    break;
  case searchAutoScan:
		executeLookAtBallAuto();
    break;
	case doSweep:
		executeSearchAutoScan();
		break;
	case otherModes:
    {
			//OUTPUT(idText,text,"In otherMode in Execution - headControlMode:" << headControlMode.headControlMode);
			// ---------------------- Sub Case
      switch (headControlMode.headControlMode)
      {
      case HeadControlMode::lookLeft:
        lastScanWasLeft = true;
        lookLeft();
        break;  
      case HeadControlMode::lookRight:
        lastScanWasLeft = false;
        lookRight();
        break;
      case HeadControlMode::searchForLandmarks:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForLandmarks();
        break;  
      case HeadControlMode::searchForLines:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForLines();
        break;  
      case HeadControlMode::searchForSpecialBall:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForSpecialBall();
        break;  
      case HeadControlMode::searchForObstacles:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForObstacles();
        break;  
      case HeadControlMode::searchForObstaclesInFront:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForObstaclesInFront();
        break;  
      case HeadControlMode::lookBetweenFeet:
        setJoints(-1000000, 0, 0, 200);
        break;  
      case HeadControlMode::lookAroundWhenBallCaught:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        lookAroundWhenBallCaught();
        break;  
      case HeadControlMode::lookAroundWhenBallJustCaught:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        lookAroundWhenBallJustCaught();
        break;  
			case HeadControlMode::none:
      case HeadControlMode::lookStraightAhead:
        setJoints(0,0,0,200);
        break;  
      case HeadControlMode::catchBall:
        if (getRobotConfiguration().getRobotDesign()==RobotDesign::ERS7)
					catchBall();
        else //ERS210
					setJoints(-1064000, 0, 439000,400,0);
				break;
      case HeadControlMode::stayAsForced:
        stayAsForced();
        break;  
      case HeadControlMode::followTail:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        followTail();
        break;  
      case HeadControlMode::lookToStars:
				lookToTheStars();
        break;
      case HeadControlMode::lookAtPoint:
        {
          double tilt,pan,roll;
          Vector3<double> point(headControlMode.point.x,headControlMode.point.y,headControlMode.point.z);
          lookAtPoint(point,headControlMode.offset,tilt,pan,roll);
          setJoints(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll), 200,0);
          break;
        }
      case HeadControlMode::lookParallelToGround:
        if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
        {
          if (headState.bodyTilt>0)
          {
            setJoints(0,0,toMicroRad(headState.bodyTilt));
          }
          else
          {
            setJoints(toMicroRad(headState.bodyTilt),0,0);
          }
        }
        else
        {
          setJoints(toMicroRad(headState.bodyTilt),0,-toMicroRad(headState.bodyRoll));
        }
        break;

      case HeadControlMode::lookJustBelowHorizon:
        setJoints(
          toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight*.5), 
          toMicroRad(cameraInfo.openingAngleWidth/4*sin((double )SystemCall::getCurrentSystemTime()/200.0)), 
          0);
        break;
      case HeadControlMode::watchOrigin:
      case HeadControlMode::direct:
        moveHead(headControlMode.tilt, headControlMode.pan, headControlMode.roll, headControlMode.speed, headControlMode.mouth);
				break;
      case HeadControlMode::calibrateHeadSpeed:
				if (calibrateHeadSpeedIsReady())
					lookToTheStars();
				else
					calibrateHeadSpeed();
				break;
      case HeadControlMode::scanMediumWidth:
      default: 
        lookToTheStars();
	
				//OUTPUT(idText,text,"DDD2004HeadControl:An unknown/unhandled HeadControlMode was set:" << headControlMode.headControlMode << "Name:" << HeadControlMode::getHeadControlModeName(headControlMode.headControlMode));
        /* if none is selected, do nothing so direction can be set manually through the head motion tester */ 
        break;
      }
			// END Sub Case
    }
  }
  // ------------------------- END Execution ------------------------------------
	headControlModeExecutedLastTime.headControlMode = headControlMode.headControlMode;
  lastHeadControlState      = headControlState;
  headPathPlanner.lastTilt  = headMotionRequest.tilt;
  headPathPlanner.lastPan   = headMotionRequest.pan;
  headPathPlanner.lastRoll  = headMotionRequest.roll;
} // End Execute


void DDD2004HeadControl::getLookAtBallAnglesAuto(double& tilt,double& pan,double& roll)
{
  
  // calculate where the ball should be in the image:
  // far away balls are small and close to the horizon,
  // close balls appear under horizon...   

  int yOffset = 25;// = (int)(35 - 60.0*(500 - distanceToBall)/500.0);

	//Logic:
	//if the ball is in sight -> look 2sek on ball
	//	if possible look at landmark and ball
	//		if looklandmark look 3 sek on the same until ball it is not possible anymore
	//												or ball is lost
	//else lookatball

	orderLandmarksByDistanceInAngle(ballPosition.seen);
	Vector3<double> ball3d (ballPosition.seen.x,ballPosition.seen.y,0);
	
	Vector3<double> pointToLook = isInView(ball3d,
												Vector3<double> (landmarks[usedLandmarkIndex].x,
																				landmarks[usedLandmarkIndex].y,
																				0));
	
	// use the same landmark for a time x
	if (SystemCall::getTimeSince(usedLandmarkTime)>2000 || lostBall)
	{
		usedLandmarkTime = SystemCall::getCurrentSystemTime();
		lastUsedLandmarkIndex = usedLandmarkIndex;
		usedLandmarkIndex = landmarkIndexs[0];
		//OUTPUT (idText,text,"Using Landmark:" << usedLandmarkIndex);
		lostBall = false;
	}

	/*
	// Ohh.. Ball lost, dont look to landmarks until landmark time is out
	if (ball3d != pointToLook && !lostBall)
	{
	*/
		lookAtPoint(
			Vector3<double>(pointToLook.x, pointToLook.y, ballRadius), 
			Vector2<int>(0, yOffset), 
			tilt, pan, roll);
	/*
	}
	else
	{
		lostBall = true;
		lookAtPoint(
			ball3d, 
			Vector2<int>(0, 0), 
			tilt, pan, roll);
	}
	*/
	/*
	OUTPUT(idText,text,"X/Y/Z:(" << tilt<< ")/(" << pan << ")/(" << roll << ")");
	if (ball3d == pointToLook)
	{
		OUTPUT(idText,text,"Landmark:"<< usedLandmarkIndex << " - LookBall " << SystemCall::getTimeSince(usedLandmarkTime));
	}
	else
	{
		OUTPUT(idText,text,"Landmark:"<< usedLandmarkIndex << " - LookBetween" << SystemCall::getTimeSince(usedLandmarkTime));
	}
	*/
}


void DDD2004HeadControl::watchOrigin()
{
	double tilt,pan,roll;
  Vector3<double> point(0,0,180);
  //its OK to start immediately here, because we scan smooth and in a way that we can always see it again:
  if (robotPose.getValidity()<0.5)
  {
    //if we dont know where to look, we look there smooth:
    if(headPathPlanner.isLastPathFinished())
    {
      if (robotPose.rotation<-0.9)
      {
        //if we can/will see the origin, then left
        lastScanWasLeft=true;
        static Vector3<long> point210(300000,1500000 ,0);
        static Vector3<long> point7(0,1500000 ,0);
        if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
        {
          Vector3<long> points[1]={point7};
          headPathPlanner.oldInit(points,1, 450);
        }
        else
        {
          Vector3<long> points[1]={point210};
          headPathPlanner.oldInit(points,1, 450);
        }
      }
      else if (robotPose.rotation>0.9)
      {
        //if we can/will see the origin, then right
        lastScanWasLeft=false;
        static Vector3<long> point210(300000,-1500000 ,0);
        static Vector3<long> point7(0,-1500000 ,0);
        if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
        {
          Vector3<long> points[1]={point7};
          headPathPlanner.oldInit(points,1, 450);
        }
        else
        {
          Vector3<long> points[1]={point210};
          headPathPlanner.oldInit(points,1, 450);
        }
      }
      else
      {
        //scan for origin
        if(lastScanWasLeft)
        {
          lookAtPoint(point,Vector2<int>(-65,0),tilt,pan,roll);
          Vector3<long> right(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll));
          Vector3<long> points[1]={right};
          headPathPlanner.oldInit(points,1, 550);
        }
        else
        {
          lookAtPoint(point,Vector2<int>(65,0),tilt,pan,roll);
          Vector3<long> left(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll));
          Vector3<long> points[1]={left};
          headPathPlanner.oldInit(points,1, 550);
        }
        lastScanWasLeft=!lastScanWasLeft;
      }
    }
    headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
  }
  else
  {
    //if we know where to look, we look there immediately:
    headPathPlanner.init();
    double rota=-robotPose.speedbyDistanceToGoal*30;
    if(getRobotConfiguration().getRobotDesign()!= RobotConfiguration::ERS7)
    {
      //old head is much to fast with *30, new would be too slow with *0/*15
      rota /= 2;
    }
    lookAtPoint(point,Vector2<int>((int)rota,0),tilt,pan,roll);
    setJoints(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll), 200,0);
  }
}


void DDD2004HeadControl::executeLookAtBallAuto()
{
  double tilt, pan, roll;
  
  getLookAtBallAnglesAuto(tilt,pan,roll);
  // clipped to a minimum angle to prevent from touching the ball
  if (tilt < -0.3) tilt = -0.3;
  setJoints(toMicroRad(tilt), toMicroRad(pan), toMicroRad(roll),0);
}

void DDD2004HeadControl::executeLookAtBall()
{
  double tilt, pan, roll;
  
  getLookAtBallAngles(tilt,pan,roll);
  // clipped to a minimum angle to prevent from touching the ball
  if (tilt < -0.3) tilt = -0.3;
  setJoints(toMicroRad(tilt), toMicroRad(pan+scanPan), toMicroRad(roll));
}


void DDD2004HeadControl::getLookAtBallAngles(double& tilt,double& pan,double& roll)
{
  double distanceToBall = Geometry::distanceTo(robotPose.getPose(), ballPosition.seen);
  
  // calculate where the ball should be in the image:
  // far away balls are small and close to the horizon,
  // close balls appear under horizon...   
  if (distanceToBall > 500) distanceToBall = 500;
  int yOffset = 0; //(int)(35 - 60.0*(500 - distanceToBall)/500.0);
  
  lookAtPoint(
    Vector3<double>(ballPosition.seen.x, ballPosition.seen.y, ballRadius), 
    Vector2<int>(0, yOffset), 
    tilt, pan, roll);

}

void DDD2004HeadControl::lookToTheStars()
{
  if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
  {
    setJoints(toMicroRad(jointLimitHeadTiltP),0,toMicroRad(jointLimitHeadRollP),200, -700000);
  }
  else
  {        
    setJoints(toMicroRad(jointLimitHeadTiltP),0,0,200, -700000);
  }
}
void DDD2004HeadControl::executeBallJustLost()
{
	
  if (lastHeadControlState!=headControlState)
  {
		//OUTPUT(idText,text,"ExecuteBallJustLost: Calculating Headmotion");
		// look for propergated ball and down, if the ball lies below
		// calculate 2D realtive field position to absolute 3d
		double propagatedAbs2dBall = Geometry::distanceTo(robotPose.getPose(), ballPosition.propagated);

		Vector3<double> propagated3dBall (ballPosition.seen.x,ballPosition.seen.y,0);
		// center the ball view in the middle of the image
		// if the ball is near, the ball should be seen, if we look halfway down
		
		int yOffset = 25;
		// constante definition of distance to ball
		enum { ballNearBy = 500 };
		/*
		int distanceToBall = (int)(ballPosition.propagated-robotPose.translation).abs();
		if (distanceToBall < ballNearBy && distanceToBall != 0)
		{
			yOffset = (- cameraInfo.resolutionHeight / 2) * distanceToBall / ballNearBy;
		}
		*/
		Vector2<int> cameraImageOffset(0,yOffset);
		double tilt,pan,roll;
		lookAtPoint(propagated3dBall,cameraImageOffset,tilt,pan,roll);
		
		Vector3<long> propagated3dBallAngles(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll));
		// first down,then propergated
		Vector3<long> lookAt;
		Vector2<double> toBall = ballPosition.propagated - robotPose.translation;
		double angleToBall = normalize(atan2(toBall.y,toBall.x))-robotPose.rotation;
		if (angleToBall>0)
			lookAt = leftDown;
		else
			lookAt = rightDown;

		if (propagatedAbs2dBall < 400)
		{
			Vector3<long> littlePoints[]={down,lookAt};
			long durations[] = {100,100};
			headPathPlanner.init(littlePoints,durations, (sizeof(littlePoints)/sizeof(Vector3<long>)));
		}
		else
		{
			Vector3<long> littlePoints[]={propagated3dBallAngles,down};
			long durations[] = {150,100};
			headPathPlanner.init(littlePoints,durations, (sizeof(littlePoints)/sizeof(Vector3<long>)));
		}

	}
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void DDD2004HeadControl::executeBallLost()
{ 

 	if (lastHeadControlState!=headControlState || headPathPlanner.isLastPathFinished())
	{
		//OUTPUT(idText,text,"In executeBallLost() - Last:");
		// First look to the communicated position 
		//   this would only be wise, if the ball was seen in the near past
		//   ballmodel.timeWhenLastObserved
		// After this, look around
		Vector3<long> communicated3dBallAngles;
		Vector3<long> leftRightSweepTop,leftRightSweepBottom,halfLeftRightSweep;

	
		bool useCommunicatedBall = false;
		bool ballIsLeftSide = false;
		long communicatedBallTime = SystemCall::getTimeSince(ballPosition.communicated.timeWhenLastObserved); 
		if ( communicatedBallTime < 1500
			&& ballPosition.seen.timeWhenLastSeen < ballPosition.communicated.timeWhenLastObserved)
		{
			lastUsedCommunicatedBallTime = ballPosition.communicated.timeWhenLastObserved;
			// Calculate communicated ball angels for neck
			Vector3<double> communicated3dBall (ballPosition.communicated.x,ballPosition.communicated.y,0);
			// calculating the ball side
			Vector2<double> toBall = ballPosition.communicated - robotPose.translation;
			double angleToBall = normalize(atan2(toBall.y,toBall.x))-robotPose.rotation;
			if (angleToBall>0)
				ballIsLeftSide=true;
			// center the ball view in the middle of the image
			// if the ball is near, the ball should be seen, if we look halfway down
			int yOffset = 25; // -cameraInfo.resolutionHeight /2;
			// constante definition of distance to ball
			enum { ballNearBy = 500 };
			/*
			int distanceToBall = (int)toBall.abs();
			if (distanceToBall < ballNearBy && distanceToBall != 0)
			{
				yOffset = (- cameraInfo.resolutionHeight / 2) * distanceToBall / ballNearBy;
			}
			*/
			Vector2<int> cameraImageOffset(0,yOffset);
			double tilt,pan,roll;
			lookAtPoint(communicated3dBall,cameraImageOffset,tilt,pan,roll);
			
			communicated3dBallAngles = Vector3<long> (toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll));

			// do not use the communicated ball, if it was trusted often enough
			if (communicatedBallTrustedCount<1)
			{	
				communicatedBallTrustedCount++;
				useCommunicatedBall = true;
			}

			//OUTPUT(idText,text,"In executeBallLost() - Using Communicated Ball");

		}

		bool leftSide;
		// using communicated ball for making the decision in 
		// which direction we have first to look
		
		if (useCommunicatedBall)
			leftSide = ballIsLeftSide;
		else
			leftSide = lastScanWasLeft;

		if (leftSide)
		{
			leftRightSweepTop = left;
			leftRightSweepBottom = leftDown;
			halfLeftRightSweep = middleLeft;
		}
		else
		{
			leftRightSweepTop = right;
			leftRightSweepBottom = rightDown;
			halfLeftRightSweep = middleRight;
		}

		if (useCommunicatedBall)
		{
			Vector3<long> points[]={communicated3dBallAngles,communicated3dBallAngles,leftRightSweepBottom,leftRightSweepTop,halfLeftRightSweep,up,down};
			long durations[] = {0,100,160,120,160,100,80};
			headPathPlanner.init(points,durations,sizeof(points)/sizeof(Vector3<long>));
		}
		else
		{
			// if the ball is farer away
			double communicatedBallDistance = (ballPosition.communicated-robotPose.translation).abs();
			//OUTPUT (idText,text,"CommBallDist" << communicatedBallDistance);
			if (communicatedBallDistance > 600)
			{
				Vector3<long> points[]={up,down,halfLeftRightSweep,leftRightSweepBottom,leftRightSweepTop,halfLeftRightSweep};
				long durations[] =     {100,80,120,200,120,200};
				//OUTPUT(idText,text,"In executeBallLost() - ball far away");
				headPathPlanner.init(points,durations,sizeof(points)/sizeof(Vector3<long>));
			}
			else
			{
				Vector3<long> points[]={down,down,halfLeftRightSweep,leftRightSweepBottom,leftRightSweepTop,halfLeftRightSweep,up};
				long durations[] =   {100 ,80,  80,                120,                 80,              200,               120};
				//OUTPUT(idText,text,"In executeBallLost() - ball near away");
				headPathPlanner.init(points,durations,sizeof(points)/sizeof(Vector3<long>));
			}
			
		}
		lastScanWasLeft = !leftSide;
	}
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void DDD2004HeadControl::executeReturnToBall()
{
  if (lastHeadControlState!=headControlState)
  {
		//OUTPUT(idText,text,"In executeReturnToBall");
    double tilt, pan, roll;
    getLookAtBallAngles(tilt,pan,roll);
    if (tilt < -0.3) tilt = -0.3;
    
    Vector3<long> ball(toMicroRad(tilt), toMicroRad(pan),toMicroRad(roll));
		long durations[] = {0};
    headPathPlanner.init(&ball,durations, 1,true);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void DDD2004HeadControl::executeSearchAutoScan()
{
	
  if (lastHeadControlState!=headControlState)
  {
		Vector3<long> leftRightSweepTop,leftRightSweepBottom,halfLeftRightSweep;
		if (lastAutoScanWasLeft)
		{
			leftRightSweepTop = left;
			leftRightSweepBottom = leftDown;
			halfLeftRightSweep = middleLeft;
		}
		else
		{
			leftRightSweepTop = right;
			leftRightSweepBottom = rightDown;
			halfLeftRightSweep = middleRight;
		}

		// Calculating the angle to bestFlag. if on sweep path, look at it
		Vector3<double> flag;
		if (getPlayer().getPlayerNumber()==Player::one)
			flag = get3dVector2BestFlag();
		else
			flag = get3dVector2BestFlagForGoalie();


		Vector2<double> toFlag (flag.x,flag.y);
		toFlag = toFlag - robotPose.translation;
		//toFlag.y = toFlag.y - robotPose.y;
		double angle = atan2(toFlag.y,toFlag.x)-robotPose.rotation;

		double flagTilt1,flagPan,flagTilt2;
		Vector2<int> offset (0,0);
		lookAtPoint(flag,offset,flagTilt1,flagPan,flagTilt2);
		// convert to microrad
		Vector3<long> flagInMicroRad ((long)toMicroRad(flagTilt1),(long)toMicroRad(flagPan),(long)toMicroRad(flagTilt2));

		if (fabs(angle)<pi/2 && 
			 ((angle>=0 && lastAutoScanWasLeft) 
			  || (angle<=0 && !lastAutoScanWasLeft) ))
		{
			// sweep with Landmark 
			Vector3<long> points[]={flagInMicroRad,flagInMicroRad,leftRightSweepTop,leftRightSweepTop,
														halfLeftRightSweep,up};
			long durations[] = {0,300,200,100,100,80};
			headPathPlanner.init(points,durations, sizeof(points)/sizeof(Vector3<long>));
		}
		else
		{
			//sweep without landmark
			Vector3<long> points[]={halfLeftRightSweep,leftRightSweepTop,leftRightSweepTop,
														halfLeftRightSweep,up};
			long durations[] = {0,120,120,200,80,80};
			headPathPlanner.init(points,durations, sizeof(points)/sizeof(Vector3<long>));
		}

    lastAutoScanWasLeft = !lastAutoScanWasLeft;
		lastScanWasLeft = lastAutoScanWasLeft;
  }

  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);

}

Vector3<double> DDD2004HeadControl::get3dVector2BestFlag()
{
	// check all Flags
	Vector2<double> flags[4] = {landmarkFlagFrontLeft,landmarkFlagFrontRight,
															landmarkFlagRearLeft,landmarkFlagRearRight};
	double angle,bestAngle=pi;
	int bestVectorIndex = 0;
	for (int i=0;i<4;i++)
	{
		Vector2<double> toFlag = flags[i]-robotPose.translation;
		angle = atan2(toFlag.y,toFlag.x) - robotPose.rotation;
		
		if (fabs(angle) < fabs(bestAngle))
		{
			bestAngle = angle;
			bestVectorIndex = i;
		}
	}
	//OUTPUT(idText,text,"flag:" << bestVectorIndex);
	return Vector3<double> (flags[bestVectorIndex].x,flags[bestVectorIndex].y,flagHeight);
}

Vector3<double> DDD2004HeadControl::get3dVector2BestFlagForGoalie()
{
	// check only the flags in the rear
	Vector2<double> flags[2] = {landmarkFlagRearLeft,landmarkFlagRearRight};
	double angle,bestAngle=pi;
	int bestVectorIndex = 0;
	for (int i=0;i<2;i++)
	{
		Vector2<double> toFlag = flags[i]-robotPose.translation;
		angle = atan2(toFlag.y,toFlag.x) - robotPose.rotation;
		
		if (fabs(angle) < fabs(bestAngle))
		{
			bestAngle = angle;
			bestVectorIndex = i;
		}
	}
	//OUTPUT(idText,text,"flag:" << bestVectorIndex);
	return Vector3<double> (flags[bestVectorIndex].x,flags[bestVectorIndex].y,flagHeight);
}


Vector3<double> DDD2004HeadControl::get3dVector2BestGoal()
{
	// check all Flags
	Vector2<double> opponent   ((double)xPosOpponentGoal,0);
	Vector2<double> own  ((double)xPosOwnGoal,0);
	Vector2<double> goals[2] = {opponent,own};

	double angle,bestAngle=pi;
	int bestVectorIndex = 0;
	for (int i=0;i<2;i++)
	{
		Vector2<double> toGoal = goals[i]-robotPose.translation;
		angle = atan2(toGoal.y,toGoal.x)  - robotPose.rotation;

		if (fabs(angle) < fabs(bestAngle))
		{
			bestAngle = angle;
			bestVectorIndex = i;
		}
	}
	return Vector3<double> (goals[bestVectorIndex].x,goals[bestVectorIndex].y,0);
}

void DDD2004HeadControl::orderLandmarksByDistanceInAngle(Vector2<double> pointOnField)
{
	double angles[numberOfLandmarks],tempDouble;
	int i,tempInt;
	Vector2<double> transLandmark,transPoint;
	for (i=0;i<numberOfLandmarks;i++)
	{
		transLandmark = landmarks[i]-robotPose.translation;
		transPoint = pointOnField-robotPose.translation;
		angles[i] = normalize(transLandmark.angle() - transPoint.angle());
		// spread the angles to stop jumping between landmarks

		if (angles[i]> pi || angles[i]< -pi) angles[i]=pi;
		landmarkIndexs[i] = i;
	}

	// doing a bubblesort -- BUBBLE BOBBLE, i love dragons
	// if i start writing such a shit we are at the german open
	// and have sleept less than two hours
	bool sortOperation = true;
	int count=0;
	while(sortOperation && count<numberOfLandmarks+1)
	{
		sortOperation = false;
		for(i=0;i<numberOfLandmarks-1;i++)
			if (fabs(angles[i])>(fabs(angles[i+1])))
			{
				tempDouble = angles[i+1];
				angles[i+1] = angles[i];
				angles[i] = tempDouble;
				
				tempInt = landmarkIndexs[i+1];
				landmarkIndexs[i+1] = landmarkIndexs[i];
				landmarkIndexs[i] = tempInt;
				sortOperation = true;
			}
		count++;
	}
	//OUTPUT(idText,text,"Using Landmark " << landmarkIndexs[0]);
}

Vector3<double> DDD2004HeadControl::isInView(Vector3<double> primary,Vector3<double> secondary)
{
	enum {MAXDISTANCE = 2000, MINDISTANCE = 400};
	Vector2<double> horz1 (primary.x,primary.y);
	Vector2<double> horz2 (secondary.x,secondary.y);
	
	horz1 -= robotPose.translation;
	horz2 -= robotPose.translation;
	double horizontalAngle = normalize(horz1.angle()-horz2.angle());
	
	Vector2<double> vert1 (horz1.x,primary.z);
	Vector2<double> vert2 (horz2.x,secondary.z);

	double verticalAngle = normalize(vert1.angle() - vert2.angle());

  	double distanceToBall = Geometry::distanceTo(robotPose.getPose(), ballPosition.seen);
 
	if (distanceToBall > MAXDISTANCE) distanceToBall = MAXDISTANCE;
	if (distanceToBall < MINDISTANCE) distanceToBall = MINDISTANCE;
	double factor = (distanceToBall - MINDISTANCE) /(MAXDISTANCE-MINDISTANCE);
	// look at 75% of field of Field
	if (fabs(verticalAngle) < (3*cameraInfo.openingAngleWidth/4) * factor
		&& fabs(horizontalAngle) < (3*cameraInfo.openingAngleHeight/4) * factor)
	{
		primary.x -= robotPose.translation.x;
		primary.y -= robotPose.translation.y;
		secondary.x -= robotPose.translation.x;
		secondary.y -= robotPose.translation.y;

	  	Vector3<double> ret ((primary.x-secondary.x)/2+secondary.x + robotPose.translation.x,
														(primary.y-secondary.y)/2+secondary.y + robotPose.translation.y,
														(primary.z-secondary.z)/2);
		return ret;

	}
	else 
	{
		return primary;
	}
}


void DDD2004HeadControl::calibrateHeadSpeed()
{
	// First look up-left,wait some time (roboter has to get up)
	// second look from left to right time (time this)
	// third look up down, using tilt1 (time this)
	// fourth look up down, using tilt2 (time this)

	Vector3<long> calibrationTilt1Down (-1385604,0,40000);
	Vector3<long> calibrationTilt1Up (52359,0,40000);

	Vector3<long> calibrationTilt2Down (0,0,-326175);
	Vector3<long> calibrationTilt2Up (0,0,750630);

	Vector3<long> calibrationLeft  (52400, 1608598,-120000);
	Vector3<long> calibrationRight (52400,-1608598,-120000);
	
	enum {calibrationRounds = 3};

	if (headPathPlanner.isLastPathFinished() || calibrationState == calibrationStateStart)
	{
		switch(calibrationState)
		{
		case calibrationStateStart:
		{
			Vector3<long> points[]={up,up};
			long durations[] = {0,1000};
			
			headPathPlanner.init(points,durations,sizeof(points)/sizeof(Vector3<long>));
			calibrationState = calibrationStateLeft;
			calibrationRoundCount = 0;
			calibrationSuccessfulRounds = 0;
			speedTilt1 = 0;
			speedPan = 0;
			speedTilt2 = 0;
			break;
		}
		case calibrationStateLeft:
		{
			Vector3<long> points[]={calibrationLeft,calibrationLeft};
			long durations[] = {2000,200};
			headPathPlanner.init(points,durations,sizeof(points)/sizeof(Vector3<long>),false);
			calibrationState = calibrationStateLeftWait;
			calibrationTime = SystemCall::getCurrentSystemTime();
			break;
		}
		case calibrationStateLeftWait:
			if (headPositionReached(calibrationLeft) || isTimedOut())
				calibrationState = calibrationStateRight;
			break;
		case calibrationStateRight:
		{
			Vector3<long> points[]={calibrationRight};
			long durations[] = {0};
			headPathPlanner.init(points,durations,sizeof(points)/sizeof(Vector3<long>),false);
			calibrationState = calibrationStateRightWait;
			calibrationTime = SystemCall::getCurrentSystemTime();
			break;
		}
		case calibrationStateRightWait:
			if (headPositionReached(calibrationRight) || isTimedOut())
			{
				if (!isTimedOut())
				{
					speedPan += abs(calibrationLeft.y-calibrationRight.y) / (SystemCall::getTimeSince(calibrationTime)+0.0001);
					calibrationSuccessfulRounds++;
				}
				else
					calibrationTimeOutsPan++;
				if (calibrationRoundCount < calibrationRounds-1)
				{
					calibrationState = calibrationStateLeft;
					calibrationRoundCount++;
				}
				else
				{
					// calibration of joint ready
					speedPan /= calibrationSuccessfulRounds;
					calibrationSuccessfulRounds = 0; 
					calibrationRoundCount = 0;
					calibrationState = calibrationStateDownTilt1;
				}
			}
			break;
		case calibrationStateDownTilt1:
		{
			Vector3<long> points[]={calibrationTilt1Down,calibrationTilt1Down};
			long durations[] = {1500,200};
			headPathPlanner.init(points,durations,sizeof(points)/sizeof(Vector3<long>),false);
			calibrationState = calibrationStateDownTilt1Wait;
			calibrationTime = SystemCall::getCurrentSystemTime();
			break;
		}
		case calibrationStateDownTilt1Wait:
			if (headPositionReached(calibrationTilt1Down) || isTimedOut())
				calibrationState = calibrationStateUpTilt1;
			break;
		case calibrationStateUpTilt1:
		{
			Vector3<long> points[]={calibrationTilt1Up};
			long durations[] = {0};
			headPathPlanner.init(points,durations,sizeof(points)/sizeof(Vector3<long>),false);
			calibrationState = calibrationStateUpTilt1Wait;
			calibrationTime = SystemCall::getCurrentSystemTime();
			break;
		}
		case calibrationStateUpTilt1Wait:
			if (headPositionReached(calibrationTilt1Up) || isTimedOut())
			{
				if (!isTimedOut())
				{
					speedTilt1 += abs(calibrationTilt1Down.x-calibrationTilt1Up.x) / (SystemCall::getTimeSince(calibrationTime)+0.0001);
					calibrationSuccessfulRounds++;
				}
				else
					calibrationTimeOutsTilt1++;
				if (calibrationRoundCount < calibrationRounds-1)
				{
					calibrationState = calibrationStateDownTilt1;
					calibrationRoundCount++;
				}
				else
				{
					// calibration of joint ready
					speedTilt1 /= calibrationSuccessfulRounds;
					calibrationSuccessfulRounds = 0; 
					calibrationRoundCount = 0;
					calibrationState = calibrationStateDownTilt2;
				}
			}
			break;
		case calibrationStateDownTilt2:
		{
			Vector3<long> points[]={calibrationTilt2Down,calibrationTilt2Down};
			long durations[] = {1500,200};
			headPathPlanner.init(points,durations,sizeof(points)/sizeof(Vector3<long>),false);
			calibrationState = calibrationStateDownTilt2Wait;
			calibrationTime = SystemCall::getCurrentSystemTime();
			break;
		}
		case calibrationStateDownTilt2Wait:
			if (headPositionReached(calibrationTilt2Down) || isTimedOut())
				calibrationState = calibrationStateUpTilt2;
			break;
		case calibrationStateUpTilt2:
		{
			Vector3<long> points[]={calibrationTilt2Up};
			long durations[] = {0};
			headPathPlanner.init(points,durations,sizeof(points)/sizeof(Vector3<long>),false);
			calibrationState = calibrationStateUpTilt2Wait;
			calibrationTime = SystemCall::getCurrentSystemTime();
			break;
		}
		case calibrationStateUpTilt2Wait:
			if (headPositionReached(calibrationTilt2Up) || isTimedOut())
			{
				if (!isTimedOut())
				{
					speedTilt2 += abs(calibrationTilt2Down.z-calibrationTilt2Up.z) / (SystemCall::getTimeSince(calibrationTime)+0.0001);
					calibrationSuccessfulRounds++;
				}
				else
					calibrationTimeOutsTilt2++;
				if (calibrationRoundCount < calibrationRounds-1)
				{
					calibrationState = calibrationStateDownTilt2;
					calibrationRoundCount++;
				}
				else
				{
					// calibration of joint ready
					speedTilt2 /= calibrationSuccessfulRounds;
					calibrationSuccessfulRounds = 0; 
					calibrationRoundCount = 0;
					calibrationState = calibrationStateUseResults;
				}
			}

			break;
		case calibrationStateUseResults:
			if (   calibrationTimeOutsTilt1 == 0
					&& calibrationTimeOutsPan   == 0
					&& calibrationTimeOutsTilt2 == 0)
			{
				OUTPUT(idText,text,"Headspeed calibration was successful");
			}
			else
			{
				OUTPUT(idText,text,"Headspeed calibration failed. ");
				OUTPUT(idText,text,"Check the values of function headPositionReached or buy a new head !");
				OUTPUT(idText,text,"Rounds: " << calibrationRounds << "; Timeouts: Tilt1 = " << calibrationTimeOutsTilt1 << "; Pan = " << calibrationTimeOutsPan << "; Tilt2 = " << calibrationTimeOutsTilt2 );
			}
			// if too much timeout  occured, set to standard values
			if (calibrationTimeOutsTilt1-calibrationRounds == 0) speedTilt1 = 1500;
			if (calibrationTimeOutsPan-calibrationRounds == 0)   speedPan   = 5350;
			if (calibrationTimeOutsTilt2-calibrationRounds == 0) speedTilt2 = 2350;

			OUTPUT(idText,text,"speedTilt1:" << speedTilt1);
			OUTPUT(idText,text,"speedPan:" << speedPan);
			OUTPUT(idText,text,"speedTilt2:" << speedTilt2);


			// setting speed in headpath planner
			headPathPlanner.headPathSpeedTilt1 = (long) speedTilt1;
			headPathPlanner.headPathSpeedPan   = (long) speedPan;
			headPathPlanner.headPathSpeedTilt2 = (long) speedTilt2;
			calibrationState = calibrationStateReady;
			break;
		default:
		case calibrationStateReady:
			break;
		}
	}
	headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}


void DDD2004HeadControl::lookLeft()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> points[]={up,leftDown};
		long durations[] = {0,100};
    headPathPlanner.init(points,durations,2);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void DDD2004HeadControl::lookRight()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> points[]={up,rightDown};
		long durations[] = {0,100};
    headPathPlanner.init(points,durations,2);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void DDD2004HeadControl::catchBall()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> stretchNeck(-800000,0,800000);
		Vector3<long> holdingWithChin(-1050000,0,700000);

		Vector3<long> points[]={stretchNeck,holdingWithChin};
		long durations[] = {100,100};
    headPathPlanner.init(points,durations,2);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);

}
void DDD2004HeadControl::searchForLandmarks()
{
  //just like last year, but
  /** @todo may be we would like to see lines too */
  
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0),  1500000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[2]={right,left};
      headPathPlanner.oldInit(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.oldInit(points,2, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[1]={left};
      headPathPlanner.oldInit(points,1, 800);
    }
    else
    {
      Vector3<long> points[1]={right};
      headPathPlanner.oldInit(points,1, 800);
    }
  }
}

void DDD2004HeadControl::searchForLines()
{
  Vector3<long> left(0, 1500000, 0);
  Vector3<long> right(0, -1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[2]={right,left};
      headPathPlanner.oldInit(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.oldInit(points,2, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[1]={left};
      headPathPlanner.oldInit(points,1, 800);
    }
    else
    {
      Vector3<long> points[1]={right};
      headPathPlanner.oldInit(points,1, 800);
    }
  }
}

void DDD2004HeadControl::searchForSpecialBall()
{
  
  //Vector3<long> centerTop(    160000, headPathPlanner.lastPan-70000,0);
  Vector3<long> rightTop(     160000, -1500000,0);
  Vector3<long> rightCenter( -120000, -1500000,0);
  Vector3<long> down(        -900000,        0,0);
  Vector3<long> leftCenter(  -120000,  1500000,0);
  Vector3<long> leftTop(      160000,  1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[3]={rightTop,down,leftTop};
      headPathPlanner.oldInit(points,3, 8000);
    }
    else
    {
      Vector3<long> points[3]={leftTop,down,rightTop};
      headPathPlanner.oldInit(points,3, 8000);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[3]={leftTop,down,rightTop};
      headPathPlanner.oldInit(points,3, 8000);
    }
    else
    {
      Vector3<long> points[3]={rightTop,down,leftTop};
      headPathPlanner.oldInit(points,3, 8000);
    }
  }
}

void DDD2004HeadControl::searchForObstacles()
{
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan > 0)
    {
			Vector3<long> points[]={left, middleLeft, up, middleRight, right};
			long durations[] = {0 , 280, 200, 200, 200 , 280};
      headPathPlanner.init(points, durations, sizeof(points)/sizeof(Vector3<long>));
    }
    else
    {
      Vector3<long> points[]={right, middleRight, up, middleLeft,left};
			long durations[] = {0 , 280, 200, 200, 200 , 280};
      headPathPlanner.init(points, durations, sizeof(points)/sizeof(Vector3<long>));
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.oldInit(points, 1, 800);
    }
    else
    {
      Vector3<long> points[2]={up, left};
      headPathPlanner.oldInit(points, 2, 800);
    }
  }
}

void DDD2004HeadControl::searchForObstaclesInFront()
{
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), 1400000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1400000,0);
  Vector3<long> center(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 1.5), 0,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.oldInit(points, 2, 800);
    }
    else
    {
      Vector3<long> points[3]={right, center, left};
      headPathPlanner.oldInit(points, 3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.oldInit(points, 1, 800);
    }
    else
    {
      Vector3<long> points[2]={center, left};
      headPathPlanner.oldInit(points, 2, 800);
    }
  }
}


void DDD2004HeadControl::lookAroundWhenBallCaught()
{
  Vector3<long> left(  300000, 1200000,0);
  Vector3<long> middle(     0,       0,0);
  Vector3<long> right( 300000,-1200000,0);
  //The mode is active the first time after another one
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    //If current pan is left:
    if(headPathPlanner.lastPan > 0)
    {
      //from current position to left, center and right.
      Vector3<long> points[3]={left,middle,right};
      headPathPlanner.oldInit(points,3, 800);
    }
    else
    {
      //from current position to right, center and left.
      Vector3<long> points[3]={right,middle,left};
      headPathPlanner.oldInit(points,3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    //init the next loop
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={middle,right};
      headPathPlanner.oldInit(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={middle,left};
      headPathPlanner.oldInit(points,2, 800);
    }
  }
}

void DDD2004HeadControl::lookAroundWhenBallJustCaught()
{
  Vector3<long> left(  100000, 500000,0);
  Vector3<long> right( 100000,-500000,0);
  
  //The mode is active the first time after another one
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    //If current pan is left:
    if(headPathPlanner.lastPan > 0)
    {
      //from current position to left, center and right.
      Vector3<long> points[2]={left,right};
      headPathPlanner.oldInit(points,2, 350);
    }
    else
    {
      //from current position to right, center and left.
      Vector3<long> points[3]={right,left};
      headPathPlanner.oldInit(points,2, 350);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    //init the next loop
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.oldInit(points,1, 350);
    }
    else
    {
      Vector3<long> points[1]={left};
      headPathPlanner.oldInit(points,1, 350);
    }
  }
}

void DDD2004HeadControl::stayAsForced()
{
  const int tiltTolerance = 60000;
  const int panTolerance  = 60000;
  const int rollTolerance = 50000;
  
  long tiltDiff = sensorDataBuffer.lastFrame().data[SensorData::headTilt] - headMotionRequest.tilt;
  long panDiff  = sensorDataBuffer.lastFrame().data[SensorData::headPan]  - headMotionRequest.pan;
  long rollDiff = sensorDataBuffer.lastFrame().data[SensorData::headRoll] - headMotionRequest.roll;
  
  if (tiltDiff > tiltTolerance)
  {
    headMotionRequest.tilt += tiltDiff-tiltTolerance;
  }
  else if (tiltDiff < -tiltTolerance)
  {
    headMotionRequest.tilt += tiltDiff+tiltTolerance;
  }
  
  if (panDiff > panTolerance)
  {
    headMotionRequest.pan += panDiff-panTolerance;
  }
  else if (panDiff < -panTolerance)
  {
    headMotionRequest.pan += panDiff+panTolerance;
  }
  
  if (rollDiff > rollTolerance)
  {
    headMotionRequest.roll += rollDiff-rollTolerance;
  }
  else if (rollDiff < -rollTolerance)
  {
    headMotionRequest.roll += rollDiff+rollTolerance;
  }
}

void DDD2004HeadControl::followTail()
{
  long tilt, pan, tilt2;
  if(getRobotConfiguration().getRobotDesign() == RobotConfiguration::ERS7)
  {
    tilt2 = sensorDataBuffer.lastFrame().data[SensorData::headTilt2];
    if(sensorDataBuffer.lastFrame().data[SensorData::backF] > 15)
    {
      if(sensorDataBuffer.lastFrame().data[SensorData::backR] > 15) tilt2 = 0;
      else tilt2 -= toMicroRad(fromDegrees(20));
    }
    else if(sensorDataBuffer.lastFrame().data[SensorData::backR] > 15) tilt2 += toMicroRad(fromDegrees(20));
  }
  else // not ERS7
  {
    tilt2 = 0;
  }
  
  pan = toMicroRad(
    fromMicroRad(sensorDataBuffer.lastFrame().data[SensorData::tailPan]) / jointLimitTailPanN * jointLimitHeadPanN);
  
  tilt = toMicroRad(
    (fromMicroRad(sensorDataBuffer.lastFrame().data[SensorData::tailTilt]) - jointLimitTailTiltN) /
    (jointLimitTailTiltP - jointLimitTailTiltN) * (jointLimitHeadTiltP - jointLimitHeadTiltN) 
    +jointLimitHeadTiltN
    );
  
  setJoints(tilt, pan, tilt2);
}

void DDD2004HeadControl::buildCameraMatrix(const SensorData& data, const HeadState& headState)
{
  if(lastFrame != data.frameNumber)
  {
    lastFrame = data.frameNumber;
    currentTilt = fromMicroRad(data.data[SensorData::headTilt]);
    currentPan  = fromMicroRad(data.data[SensorData::headPan]);
    currentRoll = fromMicroRad(data.data[SensorData::headRoll]);
    
    Geometry::calculateCameraMatrix(currentTilt, currentPan, currentRoll, headState, cameraMatrix2);
  }
}

void DDD2004HeadControl::fromWorld2CameraCoordinates
(
 const Vector3<double> pointIn3D,  
 Vector3<double> &pointInCameraCoordinates
 )
{
  RotationMatrix bodyRotation;
  Vector3<double> bodyTranslation;
  //  double phi = -this->robotPose.getPose().getAngle();
  bodyRotation.fromKardanRPY(0, 0, -robotPose.rotation);
  bodyTranslation.x = robotPose.translation.x; 
  bodyTranslation.y = robotPose.translation.y;
  bodyTranslation.z = 0;
  
  pointInCameraCoordinates = 
    cameraMatrix2.rotation.invert()*((bodyRotation.invert()*(pointIn3D - bodyTranslation) - cameraMatrix2.translation));
}

void DDD2004HeadControl::fromWorld2RobotCoordinates
(
 const Vector3<double> pointIn3D,  
 Vector3<double> &pointInRobotCoordinates
 )
{
  RotationMatrix bodyRotation;
  Vector3<double> bodyTranslation;
  
  bodyRotation.fromKardanRPY(0, 0, -robotPose.rotation);
  
  bodyTranslation.x = robotPose.translation.x; 
  bodyTranslation.y = robotPose.translation.y;
  bodyTranslation.z = 0;
  
  pointInRobotCoordinates = bodyRotation.invert()*(pointIn3D - bodyTranslation);
}


void DDD2004HeadControl::setupMainAngles()
{		
		if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
		{
		
		  left.x = 52400; 
			left.y = 1573200; 
			left.z = -120000;

			leftDown.x = 52400;
			leftDown.y = 1573200;
			leftDown.z = -350000;
    
		  right.x = 52400; 
			right.y = -1573200; 
			right.z = -120000;

			rightDown.x = 52400;
			rightDown.y = -1573200;
			rightDown.z = -350000;

			middleLeft.x = 52400;
			middleLeft.y = 560000;
			middleLeft.z = 62000;

			middleRight.x = 52400;
			middleRight.y = -560000;
			middleRight.z = 62000;

			up.x = 52400;
			up.y = 0;
			up.z = 34500;

			down.x = -270000;
			down.y = 0;
			down.z = -350000;
		}
		else 
		{
			left.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 4.0);
			left.y = 1500000;
			left.z = 0;
    
			right.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 4.0); 
			right.y = -1500000;
			right.z = 0;
    
			up.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0);
			up.y = 0;
			up.z = 0;
    
			down.x = -960000;
			down.y = 0;
			down.z = 0;

			middleLeft = left;
			middleRight = right;

			rightDown = right;
			leftDown = left;
		}

	//flags
	landmarks[0] = landmarkFlagFrontLeft =  Vector2<double> ((double)xPosFrontFlags,(double)yPosLeftFlags);
	landmarks[1] = landmarkFlagFrontRight = Vector2<double> ((double)xPosFrontFlags,(double)yPosRightFlags);
	landmarks[2] = landmarkFlagRearLeft   = Vector2<double> ((double)xPosBackFlags,(double)yPosLeftFlags);
	landmarks[3] = landmarkFlagRearRight  = Vector2<double> ((double)xPosBackFlags,(double)yPosRightFlags);
	
	
	//own penalty area
	landmarks[4]   = Vector2<double> ((double)xPosOwnPenaltyArea,(double)yPosLeftPenaltyArea);
	landmarks[5]   = Vector2<double> ((double)xPosOwnPenaltyArea,(double)yPosRightPenaltyArea);
	
	//opponent penalty area
	landmarks[6]   = Vector2<double> ((double)xPosOpponentPenaltyArea,(double)yPosLeftPenaltyArea);
	landmarks[7]   = Vector2<double> ((double)xPosOpponentPenaltyArea,(double)yPosRightPenaltyArea);

	// halfway
	landmarks[8]   = Vector2<double> ((double)xPosHalfWayLine,(double)0);
	landmarks[9]   = Vector2<double> ((double)xPosHalfWayLine,(double)yPosLeftGroundline);
	landmarks[10]   = Vector2<double> ((double)xPosHalfWayLine,(double)yPosRightGroundline);

	//goals
	landmarks[11] = ownGoal  = Vector2<double> ((double)xPosOpponentGoal,(double)0);
	landmarks[12] = opponentGoal  = Vector2<double> ((double)xPosOwnGoal,(double)0);

}

/*
* Change log :
* 
* $Log: DDD2004HeadControl.cpp,v $
* Revision 1.7  2004/05/17 19:21:51  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.6  2004/05/07 14:00:30  thomas
* modifications for berlin-event
*
* Revision 1.5  2004/05/04 09:05:34  dueffert
* headPathPlanner bug fixed in all copies
*
* Revision 1.4  2004/05/03 20:22:53  dassler
* Bug Fixed
*
* Revision 1.3  2004/05/03 19:18:53  dassler
* if ball just lost, the headcontrol looks more down
*
* Revision 1.2  2004/05/03 15:59:53  dassler
* automatic timing optimiting
* head calibration
*
* Revision 1.1  2004/04/07 12:28:57  risler
* ddd checkin after go04 - first part
*
* Revision 1.18  2004/04/04 12:12:43  Marc
* bugfix
*
* Revision 1.17  2004/04/04 10:42:34  Marc
* Timings improved
*
* Revision 1.16  2004/04/04 08:41:45  Marc
* Goalie Timing and flags chenaged
*
* Revision 1.15  2004/04/04 03:01:35  Marc
* Goalie Behavior installed
*
* Revision 1.14  2004/04/03 23:43:22  Marc
* Headcontrol improved
* looking always on landmarks and ball
* if ball far away, take a look to closest landmark
*
* Revision 1.11  2004/04/03 01:14:06  Marc
* New HeadControl
* Look for ball and landmark simultainly
*
* Revision 1.9  2004/04/02 12:07:28  Marc
* Bugfix
*
* Revision 1.8  2004/04/02 10:33:23  Marc
* Added CatchBall
*
* Revision 1.7  2004/04/02 02:29:49  Marc
* Looking know in searchAuto at Landmark
*
* Revision 1.6  2004/04/01 14:11:16  Marc
* Bug Bug fixed
*
* Revision 1.5  2004/04/01 12:15:48  Marc
* BugFix
*
* Revision 1.4  2004/04/01 11:22:57  Marc
* SearchAuto implemented
*
* Revision 1.3  2004/03/31 17:45:11  Marc
* Search for Ball implemented
* New headpath planer implemented
* New Values set for standard head angles
*
* Revision 1.2  2004/03/29 15:20:35  Marc
* DDD2004 Headcontrol installed
*
* Revision 1.1  2004/03/29 11:32:27  Marc
* Registered
*
*
*/
