/**
* @file DDD2004HeadControl.h
* 
* This file contains a class for head-behavior.
*
* @author Jan Hoffmann
*/

#ifndef __DDD2004HeadControl_h__
#define __DDD2004HeadControl_h__

#include "HeadControl.h"
#include "Tools/Math/PIDsmoothedValue.h"
#include "Tools/FieldDimensions.h"


//in urad per frame: 4000 = 28.6/s
#define minHeadSpeed 4000

/**
* @class DDD2004HeadControl
*
* A solution of the HeadControl module.
*
* Development targets are:
* - improve the ballfinding procedure for reducing the time of not watching the ball
* - reducing the time of searching the ball by using the propergated ball position
* - look smooth at landmarks, only if needed
*
*
*/
class DDD2004HeadControl : public HeadControl
{
public:
/** 
* Constructor.
* @param interfaces The paramters of the HeadControl module.
  */
  DDD2004HeadControl(const HeadControlInterfaces& interfaces);
  
  /** Destructor. */
  ~DDD2004HeadControl() {}
  
  /** Executes the module */
  virtual void execute();
  
  enum headControlStates
  {
    otherModes,
      lookAtBall,
      lookAtBallGoalie,
      ballJustLost,
      ballLost,
      returnToBall,
      searchAutoScan,
			doSweep,
      numOfStates
  } headControlState, lastHeadControlState;
  
  bool lastScanWasLeft,leftWatch,lastAutoScanWasLeft;
  double scanPan;
	// speed in microRad/millisec for all joints
	double speedTilt1,speedPan,speedTilt2;

  
private:
  CameraInfo cameraInfo;
  
	Vector3<long> left;
	Vector3<long> right;
	Vector3<long> middleLeft;
	Vector3<long> middleRight;
	Vector3<long> rightDown;
	Vector3<long> leftDown;
	Vector3<long> up;
	Vector3<long> down;

	// Landmarks

	Vector2<double> landmarkFlagFrontLeft; 
	Vector2<double> landmarkFlagFrontRight;
	Vector2<double> landmarkFlagRearLeft; 
	Vector2<double> landmarkFlagRearRight;

	Vector2<double> ownGoal; 
	Vector2<double> opponentGoal;


	enum {numberOfLandmarks = 13};

	int landmarkIndexs[numberOfLandmarks];
	Vector2<double> landmarks[numberOfLandmarks];

	long usedLandmarkTime,lastSweepTime;
	int usedLandmarkIndex,lastUsedLandmarkIndex;
	bool lostBall;

	long lastUsedCommunicatedBallTime;
	int	communicatedBallTrustedCount;

	/* Plans the player head motion. Sets up the headControlState */
  void executePlayerPlanning(long timeSinceBallSeenLast);
	/* Plans the goalie head motion. Sets up the headControlState */
	void executeGoaliePlanning(long timeSinceBallSeenLast);

	/* Orders the landmarks by angle to 2dVector */
	void orderLandmarksByDistanceInAngle(Vector2<double> pointOnField);

	/* Checks if the two Vectors can be seen by the camera */
	Vector3<double> isInView(Vector3<double> primary,Vector3<double> secondary);

	void executeLookAtBall();
	void executeLookAtBallAuto();
  void executeLookAtBallGoalie();
  void executeBallJustLost();
  void executeBallLost();
  void executeReturnToBall();
  void executeSearchAutoScan();
  
  void searchForLandmarks();
  void searchForLines();
  void searchForSpecialBall();
  void searchForObstacles();
  void searchForObstaclesInFront();
  void lookAroundWhenBallJustCaught();
  void lookAroundWhenBallCaught();
  void stayAsForced();
  void followTail();
  void lookLeft();
  void lookRight();
	void catchBall();
	void watchOrigin();
  void lookToTheStars();
  /** should be obsolete
  2do: CameraMatrix woanders herbekommen!
  */
  void buildCameraMatrix(const SensorData& data, const HeadState& headState);
  unsigned long lastFrame;
  

	/* this calculates the speed of the head motors.
		 later the speeds where used for the headPathPlanner for calculating realistic headmotions */
	void calibrateHeadSpeed();
	enum
	{
		calibrationStateStart,
		calibrationStateLeft,
		calibrationStateLeftWait,
		calibrationStateRight,
		calibrationStateRightWait,
		calibrationStateDownTilt1,
		calibrationStateDownTilt1Wait,
		calibrationStateUpTilt1,
		calibrationStateUpTilt1Wait,
		calibrationStateDownTilt2,
		calibrationStateDownTilt2Wait,
		calibrationStateUpTilt2,
		calibrationStateUpTilt2Wait,
		calibrationStateUseResults,
		calibrationStateReady
	}	calibrationState;

	long calibrationTime;
	int calibrationTimeOutsTilt1,calibrationTimeOutsPan,calibrationTimeOutsTilt2;
	int calibrationRoundCount,calibrationSuccessfulRounds;

	void calibrationReset()
	{
		calibrationTimeOutsTilt1 = 0;
		calibrationTimeOutsPan   = 0;
		calibrationTimeOutsTilt2 = 0;
		calibrationState = calibrationStateStart;
	}
	/* return true, if the position is reached more or less */
	bool headPositionReached(Vector3<long> pos)
	{
		return headPathPlanner.headPositionReached(pos);
	}

	/* return true, if the calibration move is over the time to reach the given position */
	bool isTimedOut()
	{
		return (SystemCall::getTimeSince(calibrationTime)>2500);
	}
	/* return true, if the calibration process is ready */
	bool calibrateHeadSpeedIsReady()
	{
		return (calibrationState==calibrationStateReady);
	}
  /** 3d transformation from world coordinates to camera coordinates using the camera matrix */
  void fromWorld2CameraCoordinates(const Vector3<double> pointIn3D, Vector3<double> &pointInCameraCoordinates);
  
  /** 3d transformation from world coordinates to robot coordinates */
  void fromWorld2RobotCoordinates(const Vector3<double> pointIn3D, Vector3<double> &pointInRobotCoordinates);
  
  /** look at 3d-point on field with offset point in camera image */
  void lookAtPoint(const Vector3<double> &pos,const Vector2<int> &offset,double& tilt,double& pan,double& roll);
  /** look at 3d-point subroutine trying to find tilt1 solution for given tilt2/roll. this is useful on ERS210 and ERS7
  *@return true when matching angles were found */
  bool lookAtPointFixTilt2(const Vector3<double> &aim, const double& xtan, const double& ytan, double& tilt, double& pan, const double& tilt2);
  /** look at 3d-point subroutine trying to find tilt2 solution for given tilt1. this is only useful on ERS7
  *@return true when matching angles were found */
  bool lookAtPointFixTilt1(const Vector3<double> &aim, const double& xtan, const double& ytan, const double& tilt, double& pan, double& tilt2);
  
  /** calculate the angles for looking at the seen ball with a certain camera pixel depending on ball distance */
  void getLookAtBallAngles(double& tilt,double& pan,double& roll);
  
  /** calculate the angles for looking at the seen ball with a certain camera pixel depending on ball distance
	    Its looks automaticly to the next Landmark on the field and the ball just in time */
  void getLookAtBallAnglesAuto(double& tilt,double& pan,double& roll);
 
	/** just a fake for compatibility, no smoothing yet (speed is ignored) */
  void setJoints(long tilt, long pan, long roll, long speed=100, long mouth=0);
  
  /* direct HeadControl with speed parameter (smooting) */
  void moveHead(long tilt=0, long pan=0, long roll=0, long speed=400, long mouth=0);
  
  /* setup the jointangles for the main looking directions */
	void setupMainAngles();
	
	/* returns the best flag in sight */
	Vector3<double> get3dVector2BestFlag();
	Vector3<double> get3dVector2BestFlagForGoalie();
	/* return the best goal in sight */
	Vector3<double> get3dVector2BestGoal();


	/** Indicates if this head control mode was active the last time execute was called */
  HeadControlMode headControlModeExecutedLastTime;
  
  /** This struct can calculate a smooth series of head joint angles from a gives set of way points and an overall duration */
  struct HeadPathPlanner
  {
  public:
  /**
  * Resets/Initializes a set of points to visit in a certain time
  * @param vectors set of arcs to visit
  * @param numberOfVectors number of Vector3s in param vectors
  * @param duration time in ms to reach the last position in param vecors
    */
    void init(const Vector3<long>* vectors=0, long* durations=0, int numberOfVectors=0,bool optimizeTimings=true);
    void oldInit(const Vector3<long>* vectors=0,int numberOfVectors=0, long duration=0)
		{
			if (numberOfVectors==1)
			{
				long durations[1]={duration};
				init(vectors,durations,1);
			}
			else
			{
				// divide in equal chunks
				if (numberOfVectors!=0)
				{
					// division by zero work around
					long chunkedDuration = duration / numberOfVectors;
					long durations[maxNumberOfPoints+1];
					// first time
					durations[0]=0;
					for (int i=1;i<=numberOfVectors;i++)
						durations[i]=chunkedDuration;
					init(vectors,durations,numberOfVectors);
				}
			}
		}
    /**
    * constructor
    */
    HeadPathPlanner(const SensorDataBuffer& sensorDataBuffer):lastTilt(0),lastPan(0),lastRoll(0),currentPoint(0),currentFrame(0),numberOfFrames(0),numberOfPoints(0),sensorDataBuffer(sensorDataBuffer) {}
    
    bool debugOut;
		long debugPan,debugTilt1,debugTilt2;
    /**
    * Calculates the angles for tilt pan and roll
    * @return true if the last point is reached.
    */
    bool getAngles(long& tilt, long& pan, long& roll);
    
		/* returns true, if the head has reached the given position */
		bool headPositionReached(Vector3<long> pos);

    /**
    * Return whether the last initialized path is already finished
    * @return true if last path is finished.
    */
    inline bool isLastPathFinished() { return (currentFrame>=numberOfFrames); }
    
		/* returns the minimum time which is needed between the given headposition */
		long calculateHeadTiming(Vector3<long> &pos1,Vector3<long> &pos2);
    
    long lastTilt;
    long lastPan;
    long lastRoll;
    
		long headPathSpeedTilt1,headPathSpeedPan,headPathSpeedTilt2;
		const SensorDataBuffer& sensorDataBuffer;
    
  private:
		/**
		* Return the whole Duration of the Headpath
		* @return the sum of duration of the headpath
		*/
		long calculateDurationsSum(long* duration, int durations);
		long currentPoint;
    long currentFrame;
    long numberOfFrames;
    
    enum {maxNumberOfPoints = 20};
    long numberOfPoints;
    Vector3<long> points[maxNumberOfPoints];
    
		double firstFrame[maxNumberOfPoints];
  };
  
  HeadPathPlanner headPathPlanner;
  
  // needed for transformations and for simple look at point
  CameraMatrix cameraMatrix2;
  double currentTilt, currentPan, currentRoll;
  
};

#endif //__DDD2004HeadControl_h__

/*
* Change log :
* 
* $Log: DDD2004HeadControl.h,v $
* Revision 1.3  2004/05/17 19:21:51  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.2  2004/05/03 15:59:53  dassler
* automatic timing optimiting
* head calibration
*
* Revision 1.1  2004/04/07 12:28:57  risler
* ddd checkin after go04 - first part
*
* Revision 1.10  2004/04/04 08:41:45  Marc
* Goalie Timing and flags chenaged
*
* Revision 1.9  2004/04/04 03:01:36  Marc
* Goalie Behavior installed
*
* Revision 1.8  2004/04/03 23:43:23  Marc
* Headcontrol improved
* looking always on landmarks and ball
* if ball far away, take a look to closest landmark
*
* Revision 1.7  2004/04/03 01:14:06  Marc
* New HeadControl
* Look for ball and landmark simultainly
*
* Revision 1.6  2004/04/02 10:33:24  Marc
* Added CatchBall
*
* Revision 1.5  2004/04/02 02:29:50  Marc
* Looking know in searchAuto at Landmark
*
* Revision 1.4  2004/04/01 14:11:17  Marc
* Bug Bug fixed
*
* Revision 1.3  2004/04/01 11:22:57  Marc
* SearchAuto implemented
*
* Revision 1.2  2004/03/31 17:45:11  Marc
* Search for Ball implemented
* New headpath planer implemented
* New Values set for standard head angles
*
* Revision 1.1  2004/03/29 11:32:27  Marc
* Registered
*

*
*/
