/**
* @file GT2003HeadControl.cpp
*
* Implementation of class GT2003HeadControl
*/

#ifdef _WIN32
static const unsigned int timeAfterWhichBallIsConsideredLost = 1000;
#else
static const unsigned int timeAfterWhichBallIsConsideredLost = 300;
#endif


#include "Platform/GTAssert.h"
#include "GT2003HeadControl.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"
#include "Platform/SystemCall.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Debugging/DebugDrawings.h"

void GT2003HeadControl::HeadPathPlanner::init(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration)
{
  const RobotDimensions& robotDimensions = getRobotConfiguration().getRobotDimensions();
  if (numberOfVectors>maxNumberOfPoints-1) numberOfVectors=maxNumberOfPoints-1;
  if (duration<8) duration=8;
  
  //start from current position
  points[0].x = lastTilt;
  points[0].y = lastPan;
  points[0].z = lastRoll;
  currentPoint = 0;
  numberOfPoints = numberOfVectors;
  currentFrame = 0;
  numberOfFrames = duration/8;
  
  const Range<long> jointRangeHeadTilt(toMicroRad(robotDimensions.jointLimitHeadTiltN),toMicroRad(robotDimensions.jointLimitHeadTiltP));
  const Range<long> jointRangeHeadPan( toMicroRad(robotDimensions.jointLimitHeadPanN), toMicroRad(robotDimensions.jointLimitHeadPanP));
  const Range<long> jointRangeHeadRoll(toMicroRad(robotDimensions.jointLimitHeadRollN),toMicroRad(robotDimensions.jointLimitHeadRollP));
  
  //calculate total distance and speed
  double distance=0;
  int i;
  for (i=0;i<numberOfVectors;i++)
  {
    //clip arcs to physical limits
    points[i+1].x = jointRangeHeadTilt.limit(vectors[i].x);
    points[i+1].y = jointRangeHeadPan.limit(vectors[i].y);
    points[i+1].z = jointRangeHeadRoll.limit(vectors[i].z);
    
    distance += sqrt(
      (double)(points[i+1].x-points[i].x)*(double)(points[i+1].x-points[i].x)+
      (double)(points[i+1].y-points[i].y)*(double)(points[i+1].y-points[i].y)+
      (double)(points[i+1].z-points[i].z)*(double)(points[i+1].z-points[i].z));
  }
  double speed = distance/(double)numberOfFrames; //in urad per frame
  if (speed<minHeadSpeed) speed=minHeadSpeed;
  
  //calculate duration for each part of the route
  double sum=0;
  for (i=0;i<=numberOfVectors;i++)
  {
    firstFrame[i] = sum/speed;
    sum += sqrt(
      (double)(points[i+1].x-points[i].x)*(double)(points[i+1].x-points[i].x)+
      (double)(points[i+1].y-points[i].y)*(double)(points[i+1].y-points[i].y)+
      (double)(points[i+1].z-points[i].z)*(double)(points[i+1].z-points[i].z));
  }
}

void GT2003HeadControl::HeadPathPlanner::initLoop(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration)
{
  Vector3<long> newVec[maxNumberOfPoints];
  double min=1e20;
  int minIndex=0;
  int i;
  if (numberOfVectors>maxNumberOfPoints-1) numberOfVectors=maxNumberOfPoints-1;
  for (i=0;i<numberOfVectors;i++)
  {
    double diff2 =
      (double)(vectors[i].x-lastTilt)*(double)(vectors[i].x-lastTilt)+
      (double)(vectors[i].y-lastPan)*(double)(vectors[i].y-lastPan)+
      (double)(vectors[i].z-lastRoll)*(double)(vectors[i].z-lastRoll);
    if (diff2<min)
    {
      min=diff2;
      minIndex=i;
    }
  }
  /** @todo check if opposite loop direction wouldnt be better after visit of first point from current position */
  for (i=0;i<=numberOfVectors;i++)
  {
    newVec[i]=vectors[(minIndex+i)%numberOfVectors];
  }
  init(newVec,numberOfVectors+1,duration);
}

bool GT2003HeadControl::HeadPathPlanner::getAngles(long& tilt, long& pan, long& roll)
{
  if (currentFrame<numberOfFrames)
  {
    currentFrame++;
    while ((currentFrame>firstFrame[currentPoint+1])&&
           (currentPoint<numberOfPoints-1)&&
           (currentFrame<numberOfFrames))
    {
      currentPoint++;
    }
    
    double dist=0;
    if (currentPoint<numberOfPoints)
    {
      dist=firstFrame[currentPoint+1]-firstFrame[currentPoint];
    }
    if (dist==0)
    {
      tilt = points[currentPoint].x;
      pan  = points[currentPoint].y;
      roll = points[currentPoint].z;
    }
    else
    {
      double leftFactor = (firstFrame[currentPoint+1]-currentFrame)/dist;
      double rightFactor = 1-leftFactor;
      tilt = (long)(leftFactor*points[currentPoint].x + rightFactor*points[currentPoint+1].x);
      pan  = (long)(leftFactor*points[currentPoint].y + rightFactor*points[currentPoint+1].y);
      roll = (long)(leftFactor*points[currentPoint].z + rightFactor*points[currentPoint+1].z);
    }
  }
  return (currentFrame>=numberOfFrames);
}




GT2003HeadControl::GT2003HeadControl(const HeadControlInterfaces& interfaces)
: HeadControl(interfaces),headControlState(otherModes),lastHeadControlState(otherModes),
lastScanWasLeft(true),lastOdometryData(currentOdometryData),lastTimeOfGoodSL(0)
{ 
  headPathPlanner.lastTilt = sensorDataBuffer.lastFrame().data[SensorData::headTilt];
  headPathPlanner.lastPan  = sensorDataBuffer.lastFrame().data[SensorData::headPan];
  headPathPlanner.lastRoll = sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  scanPan = 0.0;
  leftWatch= true;
}

void GT2003HeadControl::setJoints(long tilt, long pan, long roll, long speed, long mouth)
{
  headMotionRequest.tilt = tilt;
  headMotionRequest.pan  = pan;
  headMotionRequest.roll = roll;
  headMotionRequest.mouth = mouth;
}


void GT2003HeadControl::moveHead(long tilt, long pan, long roll, long speed, long mouth)
{
  //tilt = tilt + sensorDataBuffer.lastFrame().data[SensorData::headTilt];
  //pan = pan + sensorDataBuffer.lastFrame().data[SensorData::headPan];
  //roll = roll + sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  //mouth = mouth + sensorDataBuffer.lastFrame().data[SensorData::mouth];
  
  if (speed >= 400)
  {
    if (headMotionRequest.tilt != tilt) {
      if (headMotionRequest.tilt > toMicroRad(jointLimitHeadTiltP))
        headMotionRequest.tilt = toMicroRad(jointLimitHeadTiltP);
      else {
        if (headMotionRequest.tilt < toMicroRad(jointLimitHeadTiltN))
          headMotionRequest.tilt = toMicroRad(jointLimitHeadTiltN);
        else
          headMotionRequest.tilt = tilt;
      }
    }
    if (headMotionRequest.pan != pan) {
      if (headMotionRequest.pan > toMicroRad(jointLimitHeadPanP))
        headMotionRequest.pan = toMicroRad(jointLimitHeadPanP);
      else {
        if (headMotionRequest.pan < toMicroRad(jointLimitHeadPanN))
          headMotionRequest.pan = toMicroRad(jointLimitHeadPanN);
        else
          headMotionRequest.pan = pan;
      }
    }
    if (headMotionRequest.roll != roll) {
      if (headMotionRequest.roll > toMicroRad(jointLimitHeadRollP))
        headMotionRequest.roll = toMicroRad(jointLimitHeadRollP);
      else {
        if (headMotionRequest.roll > toMicroRad(jointLimitHeadRollN))
          headMotionRequest.roll = toMicroRad(jointLimitHeadRollN);
        else
          headMotionRequest.roll = roll;
      }
    }
    if (headMotionRequest.mouth != mouth)
      headMotionRequest.mouth = mouth;
  }
  else
  {
    if (headPathPlanner.isLastPathFinished())
    {
      Vector3<long> point(tilt,pan,roll);
      Vector3<long> points[1]={point};
      headPathPlanner.init(points,1, 2000);
    }
    headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
  }
}


void GT2003HeadControl::lookAtPoint(const Vector3<double> &pos,const Vector2<int> &offset,double& tilt,double& pan,double& roll)
{
  Vector3<double> posInCameraCoordinates;
  posInCameraCoordinates = cameraMatrix2.rotation.invert() * (pos - cameraMatrix2.translation);
  Vector2<int> pointToLookTo;
  Vector2<double> angles;
  angles.x = atan2(posInCameraCoordinates.x, posInCameraCoordinates.y);
  angles.y = atan2(posInCameraCoordinates.x, posInCameraCoordinates.z);
  Geometry::calculatePointByAngles(angles, cameraMatrix2, cameraInfo, pointToLookTo);
  //debugging...
  LINE(sketch, offset.x, 0, offset.x, 144, 1, Drawings::ps_solid, Drawings::gray);
  LINE(sketch, 0,offset.y, 176, offset.y, 1, Drawings::ps_solid, Drawings::gray);
  DOT(sketch, pointToLookTo.x - 88, pointToLookTo.y - 72, Drawings::white, Drawings:: black);
  DEBUG_DRAWING_FINISHED(sketch);
  /*  
  PAINT_TO_DEBUGDRAWING(crosshairs,
  crosshairsDrawing.line(0, offset.y + 71, 175, offset.y + 71, DebugDrawing::ps_solid, 0, DebugDrawing::Color(255,255,255) );
  crosshairsDrawing.line(offset.x + 87, 0, offset.x + 87, 143, DebugDrawing::ps_solid, 0, DebugDrawing::Color(255,255,255) );
  );
  
  GTMath::Vector2<int> onScreen;
  onScreen.x = (int)(- transformedPoint.y*fx/(transformedPoint.x));
  onScreen.y = (int)(- transformedPoint.z*fy/(transformedPoint.x));
  
  PAINT_TO_DEBUGDRAWING(crosshairs,
  crosshairsDrawing.dot(87+onScreen.x, 71+onScreen.y, DebugDrawing::Color(255,255,255), DebugDrawing::Color(127,127,255));
  );
  
  SEND_DEBUGDRAWING(crosshairs);
  */
  
  int xoff = offset.x;
  int yoff = offset.y;
  
  //Is the offset in range?
  if (yoff > cameraInfo.resolutionHeight / 2) yoff = cameraInfo.resolutionHeight / 2;
  else if (yoff < -cameraInfo.resolutionHeight / 2) yoff = -cameraInfo.resolutionHeight / 2;
  
  if (xoff > cameraInfo.resolutionWidth / 2) xoff = cameraInfo.resolutionWidth / 2;
  else if (xoff < -cameraInfo.resolutionWidth / 2) xoff = -cameraInfo.resolutionWidth / 2;
  
  Pose2D robotPose = this->robotPose.getPose();
  Vector3<double> aim1(
    pos.x-robotPose.translation.x,
    pos.y-robotPose.translation.y,
    pos.z-headState.neckHeight
    );
  
  RotationMatrix rot;
  Vector3<double> aim;
  rot.rotateY(-headState.bodyTilt).
    rotateX(headState.bodyRoll).
    rotateZ(-robotPose.getAngle());
  aim = rot*aim1;
  
  //now we have rotated the coordinates in a way that the neck
  //is (0,0,0) and should look without any body rotation to aim:
  //aim.x in front, aim.y left, aim.z above
  
  //Looking with a certain xoffset and yoffset gives us a ray from camera into space.
  //we want the coordinates of all points on that ray to be a simple funtion of a single
  //argument, the distance from camera to an image plane
  double xtan = -tan(cameraInfo.openingAngleWidth / 2)*xoff/(cameraInfo.resolutionWidth / 2);
  double ytan = tan(cameraInfo.openingAngleHeight / 2)*yoff/(cameraInfo.resolutionHeight / 2);
  
  if (getRobotConfiguration().getRobotDesign()==RobotDesign::ERS7)
  {
    //due to ERS7 anatomy we can now change tilt1 and/or tilt2 to reach our aim. There
    //are 3 possibly useful strategies for that:
    //1) change tilt1 and tilt2 synchronously: this would look cool but will be uncalculatable
    //2) maximize tilt1: try to find tilt2 solution for max tilt1, if there is none try to find
    //   tilt1 solution for minimal tilt2
    //3) maximize tilt2: try to find tilt1 solution for max tilt2, if there is none try to find
    //   tilt2 solution for minimal tilt1
    
    //this is method 2):
    tilt=0;
    if (!lookAtPointFixTilt1(aim,xtan,ytan,tilt,pan,roll))
    {
      roll=jointLimitHeadRollN;
      lookAtPointFixTilt2(aim,xtan,ytan,tilt,pan,roll);
    }
    
    /*
    // just a check to see that everything went right on ERS7:
    RotationMatrix rot2;
    rot2.rotateY(tilt);
    Vector3<double> aim2 = rot2*aim;
    RotationMatrix rot3;
    rot3.rotateZ(-pan);
    Vector3<double> aim3 = rot3*aim2;
    Vector3<double> aim4 = aim3-Vector3<double>(0,0,distanceNeckToPanCenter);
    RotationMatrix rot4;
    rot4.rotateY(roll);
    Vector3<double> aim5 = rot4*aim4;
    Vector3<double> aim6 = aim5-Vector3<double>(distancePanCenterToCameraX,0,0);
    double realxoff=-aim6.y/aim6.x*cameraInfo.resolutionWidth / 2/tan(cameraInfo.openingAngleWidth/2);
    double realyoff=aim6.z/aim6.x*cameraInfo.resolutionHeight / 2/tan(cameraInfo.openingAngleHeight/2);
    if ((aim5.x<=0)||(fabs(realxoff-xoff)>0.1)||(fabs(realyoff-yoff)>0.1))
    {
    bool mistake=true;
    }
    */
    const Range<double> jointRangeHeadTilt2( jointLimitHeadRollN, jointLimitHeadRollP);
    roll = jointRangeHeadTilt2.limit(roll);
  }
  else
  {
    roll=0;
    lookAtPointFixTilt2(aim,xtan,ytan,tilt,pan,roll);
    /*
    // just a check to see that everything went right on ERS210:
    RotationMatrix rot2;
    rot2.rotateY(tilt);
    Vector3<double> aim2 = rot2*aim;
    RotationMatrix rot3;
    rot3.rotateZ(-pan);
    Vector3<double> aim3 = rot3*aim2;
    Vector3<double> aim4 = aim3-Vector3<double>(distancePanCenterToCameraX,0,distanceNeckToPanCenter);
    double realxoff=-aim4.y/aim4.x*cameraInfo.resolutionWidth / 2/tan(cameraInfo.openingAngleWidth/2);
    double realyoff=aim4.z/aim4.x*cameraInfo.resolutionHeight / 2/tan(cameraInfo.openingAngleHeight/2);
    if ((aim3.x<=0)||(fabs(realxoff-xoff)>0.1)||(fabs(realyoff-yoff)>0.1))
    {
    bool mistake=true;
    }
    */
  }
  
  const Range<double> jointRangeHeadTilt(jointLimitHeadTiltN,jointLimitHeadTiltP);
  const Range<double> jointRangeHeadPan( jointLimitHeadPanN, jointLimitHeadPanP);
  tilt = jointRangeHeadTilt.limit(tilt);
  pan = jointRangeHeadPan.limit(pan);
}

bool GT2003HeadControl::lookAtPointFixTilt2(const Vector3<double> &aim, const double& xtan, const double& ytan, double& tilt, double& pan, const double& tilt2)
{
  //d shall be the distance from camera to object plane.
  
  //then we look at a point with these coordinates from tilt2 joint:
  //x = pcc+d
  //y = xtan*d
  //z = ytan*d
  //after rotation with tilt2 we get in coordinates from tilt1 joint:
  //x = Cos[t2]*(pcc + d) - Sin[t2]*(ytan*d)
  //y = xtan*d
  //z = npc + Sin[t2]*(pcc + d) + Cos[t2]*(ytan*d)
  
  //we know that we want to look at (aimx,aimy,aimz). Thats Sqrt[aimx^2+aimy^2+aimz^2] away from the neck.
  //The matching point (x,y,z) on our ray of look must have the same distance. this can be resolved to d
  //(thanks to Mathematica, see lookAtPointUDFinal.nb):
  
  //every visible d is positive, so with some helping variables we get:
  double a = 1 + xtan*xtan+ytan*ytan;
  double ns2 = distanceNeckToPanCenter*sin(tilt2);
  double nc2 = distanceNeckToPanCenter*cos(tilt2);
  double b = distancePanCenterToCameraX + nc2*ytan + ns2;
  double c = aim.x*aim.x + aim.y*aim.y + aim.z*aim.z - distanceNeckToPanCenter*distanceNeckToPanCenter
    - distancePanCenterToCameraX*distancePanCenterToCameraX - 2*distancePanCenterToCameraX*ns2;
  double radikant = b*b + a*c;
  double d;
  if (radikant<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt=headPathPlanner.lastTilt;
    pan=headPathPlanner.lastPan;
    return false;
  }
  else
  {
    d = -(distancePanCenterToCameraX + ytan*nc2 + ns2 - sqrt(radikant))/a;
  }
  if (d<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt=headPathPlanner.lastTilt;
    pan=headPathPlanner.lastPan;
    return false;
  }
  
  //Now we can calculate x,y,z from d:
  double x = cos(tilt2)*(distancePanCenterToCameraX + d) - sin(tilt2)*(ytan*d);
  double y = xtan*d;
  double z = distanceNeckToPanCenter + sin(tilt2)*(distancePanCenterToCameraX + d) + cos(tilt2)*(ytan*d);
  
  //we currently look at point (x,y,z). it has the correct distance, but has to be rotated twice
  //to match (aimx,aimy,aimz). First we want to get the y coordinate to aimy by rotating with pan.
  //This is useful, because pan is closer to camera and changing tilt afterwards does not change
  //y coordinate anymore. As this is done around z-axis, lets look into the xy-plane through (x,y,z).
  //We look at a point that is Sqrt[x^2+y^2] away from the z-axis and has an arc of ArcTan[x,y] (=atan2(y,x)).
  //We would like to look at a point in the same distance at y coordinate aimy. That would form an
  //arc of ArcSin[aimy/Sqrt[x^2+y^2]]. Therefore we have to rotate with:
  
  pan = asin(aim.y/sqrt(x*x+y*y))-atan2(y,x);
  if (pan>=pi)
  {
    pan -= pi2;
  }
  else if (pan<-pi)
  {
    pan += pi2;
  }
  
  //There is a second solution here: the arc we want the point to have above the ground can also
  //be Pi-ArcSin[aimy/Sqrt[x^2+y^2]]. Then it is definitely behind us, but that is possible and
  //useful in some cases.
  
  double pantemp = pi-asin(aim.y/sqrt(x*x+y*y))-atan2(y,x);
  if (pantemp>=pi)
  {
    pantemp -= pi2;
  }
  else if (pantemp<-pi)
  {
    pantemp += pi2;
  }
  
  //If the point is behind us, then we prefer to test the second solution first, because its nicer
  //to look at those points with upright head. Explanation: Imagine looking with the lower left pixel.
  //We want to look at a point left and a little behind of us. We find a correct y left a little
  //infront and a little behind us. If we take the one infront, we can still manage to rotate to the
  //one behind with very low tilt. Therefore we try the one behind us first in this case, so:
  if (aim.x<0)
  {
    double temp=pan;
    pan=pantemp;
    pantemp=temp;
  }
  
  //(If pan is in bounds) and after rotating with pan we look at:
  double xt=cos(pan)*x-sin(pan)*y;
  //yt=sin(pan)*x+cos(pan)*y=aimy
  //zt=z
  
  //and then we only need to find the correct tilt1:
  tilt = atan2(aim.z,aim.x)-atan2(z,xt);
  if (tilt>=pi)
  {
    tilt -= pi2;
  }
  else if (tilt<-pi)
  {
    tilt += pi2;
  }
  double pansave=pan;
  double tiltsave=tilt;
  if ((tilt<jointLimitHeadTiltN)||(tilt>jointLimitHeadTiltP)||(pan<jointLimitHeadPanN)||(pan>jointLimitHeadPanP))
  {
    pan=pantemp;
    xt = cos(pan)*x - sin(pan)*y;
    tilt = atan2(aim.z,aim.x)-atan2(z,xt);
    if (tilt>=pi)
    {
      tilt -= pi2;
    }
    else if (tilt<-pi)
    {
      tilt += pi2;
    }
    if ((tilt<jointLimitHeadTiltN)||(tilt>jointLimitHeadTiltP)||(pan<jointLimitHeadPanN)||(pan>jointLimitHeadPanP))
    {
      //if we cant see the point, its better to look with upright head:
      pan=pansave;
      tilt=tiltsave;
      return false;
    }
  }
  return true;
}

bool GT2003HeadControl::lookAtPointFixTilt1(const Vector3<double> &aim, const double& xtan, const double& ytan, const double& tilt, double& pan, double& tilt2)
{
  //first convert from neck coordinates to before-tilt2 cordinates:
  Vector3<double> aim2(
    cos(tilt)*aim.x - sin(tilt)*aim.z,
    aim.y,
    sin(tilt)*aim.x + cos(tilt)*aim.z - distanceNeckToPanCenter
    );
  //and than we start the same way as in the fixTilt2 case, but much easier:
  //x = pcc + d
  //y = xtan*d
  //z = ytan*d
  
  //we know that we want to look at (aim2x,aim2y,aim2z). Thats Sqrt[aim2x^2+aim2y^2+aim2z^2] away from the tilt2 axis center.
  //The matching point (x,y,z) on our ray of look must have the same distance. this can be resolved to d
  //(thanks to Mathematica, see lookAtPointUDFinal.nb):
  
  //every visible d is positive, so with some helping variables we get:
  double a = 1 + xtan*xtan+ytan*ytan;
  double radikant = a*(aim2.x*aim2.x+aim2.y*aim2.y+aim2.z*aim2.z - distancePanCenterToCameraX*distancePanCenterToCameraX)
    + distancePanCenterToCameraX*distancePanCenterToCameraX;
  double d;
  if (radikant<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt2=headPathPlanner.lastRoll;
    pan=headPathPlanner.lastPan;
    return false;
  }
  else
  {
    d = (-distancePanCenterToCameraX + sqrt(radikant))/a;
  }
  if (d<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt2=headPathPlanner.lastRoll;
    pan=headPathPlanner.lastPan;
    return false;
  }
  
  //Now we can calculate x,y,z from d:
  double x = distancePanCenterToCameraX+d;
  double y = xtan*d;
  double z = ytan*d;
  
  //we currently look at point (x,y,z). it has the correct distance, but has to be rotated twice
  //to match (aim2x,aim2y,aim2z). First we want to get tilt2. This one is closer to camera and
  //using pan afterwards doesnt change z anymore
  
  tilt2 = asin(aim2.z/sqrt(x*x+z*z))-atan2(z,x);
  if (tilt2>=pi)
  {
    tilt2 -= pi2;
  }
  else if (tilt2<-pi)
  {
    tilt2 += pi2;
  }
  
  //after rotating with tilt1 we look at:
  double xt=cos(tilt2)*x-sin(tilt2)*z;
  //yt=y=ytan*d
  //zt=sin(tilt2)*x+cos(tilt2)*z=aimz
  
  //and then we only need to find the correct pan:
  pan = atan2(aim2.y,aim2.x)-atan2(y,xt);
  if (pan>=pi)
  {
    pan -= pi2;
  }
  else if (pan<-pi)
  {
    pan += pi2;
  }
  if ((tilt2<jointLimitHeadRollN)||(tilt2>jointLimitHeadRollP)||(pan<jointLimitHeadPanN)||(pan>jointLimitHeadPanP))
  {
    return false;
  }
  return true;
}


void GT2003HeadControl::execute()
{
  if(headIsBlockedBySpecialActionOrWalk) 
  {
    headControlModeExecutedLastTime.headControlMode = HeadControlMode::none;
    headControlState=otherModes;
    lastHeadControlState=otherModes;
    headPathPlanner.lastTilt = sensorDataBuffer.lastFrame().data[SensorData::headTilt];
    headPathPlanner.lastPan  = sensorDataBuffer.lastFrame().data[SensorData::headPan];
    headPathPlanner.lastRoll = sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  }
  
  headMotionRequest.mouth=0;
  if (robotPose.getValidity()>0.5)
  {
    lastTimeOfGoodSL=SystemCall::getCurrentSystemTime();
  }
  
  buildCameraMatrix(sensorDataBuffer.lastFrame(),headState);
  
  long timeSinceBallSeenLast = SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);
  
  //Transitions between states
  switch (headControlMode.headControlMode)
  {
  case HeadControlMode::searchForBall:
  case HeadControlMode::searchAuto:
  case HeadControlMode::searchAutoForGoalie:
    switch (headControlState)
    {
    case otherModes:
      if (timeSinceBallSeenLast<1000)
      {
        headControlState=lookAtBall; //returnToBall;
      }
      else
      {
        headControlState=ballLost;
      }
      break;
    case lookAtBall:
    case lookAtBallGoalie:
      if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) > timeAfterWhichBallIsConsideredLost)
      {
        headControlState=ballJustLost;
      }
      else if (headControlMode.headControlMode == HeadControlMode::searchAuto)
      {
        long consecTime = ballPosition.seen.getConsecutivelySeenTime();
        if (/*(consecTime>2000) || ((*/consecTime>300/*)&&(ballPosition.seen.speed.abs()<150))*/)
        {
          headControlState=searchAutoUpperScan;
        }
      }
      else if (headControlMode.headControlMode == HeadControlMode::searchAutoForGoalie)
      {
        long consecTime = ballPosition.seen.getConsecutivelySeenTime();
        if ((consecTime>3000) && (ballPosition.seen.x > xPosHalfWayLine))
        {
          headControlState=searchAutoUpperScan;
        }
        else if (consecTime>6000)
        {
          headControlState=searchAutoUpperScan;
        }
      }
      break;
    case ballJustLost:
      if (timeSinceBallSeenLast<250)
      {
        if (headControlMode.headControlMode == HeadControlMode::searchAutoForGoalie)
          headControlState=lookAtBallGoalie;
        else
          headControlState=lookAtBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=ballLost;
      }
      break;
    case ballLost:
      if (timeSinceBallSeenLast<250)
      {
        if (headControlMode.headControlMode == HeadControlMode::searchAutoForGoalie)
          headControlState=lookAtBallGoalie;
        else
          headControlState=lookAtBall;
      }	
      break;
    case returnToBall:
      if (timeSinceBallSeenLast<250)
      {
        if (headControlMode.headControlMode == HeadControlMode::searchAutoForGoalie)
          headControlState=lookAtBallGoalie;
        else
          headControlState=lookAtBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=ballJustLost;
      }
      break;
    case searchAutoUpperScan:
      if (headControlMode.headControlMode == HeadControlMode::searchForBall)
      {
        headControlState=returnToBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=searchAutoLowerScan;
      }
      break;
    case searchAutoLowerScan:
      if (timeSinceBallSeenLast<250)
      {
        headControlState=lookAtBall;
      }
      else if (headControlMode.headControlMode == HeadControlMode::searchForBall)
      {
        headControlState=returnToBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=ballJustLost;
      }
      break;
    }
    break;
    default:
      headControlState=otherModes;
  }
  
  //Execution
  switch (headControlState)
  {
  case lookAtBall:
    executeLookAtBall();
    break;
  case ballJustLost:
    executeBallJustLost();
    break;
  case ballLost:
    executeBallLost();
    break;
  case returnToBall:
    executeReturnToBall();
    break;
  case searchAutoUpperScan:
    executeSearchAutoUpperScan();
    break;
  case searchAutoLowerScan:
    executeSearchAutoLowerScan();
    break;
  case lookAtBallGoalie:
    executeLookAtBallGoalie();
    break;
  case otherModes:
    {
      switch (headControlMode.headControlMode)
      {
      case HeadControlMode::none:
        /* if none is selected, do nothing so direction can be set manually through the head motion tester */ 
        break;
      case HeadControlMode::lookLeft:
        lastScanWasLeft = true;
        lookLeft();
        break;  
      case HeadControlMode::lookRight:
        lastScanWasLeft = false;
        lookRight();
        break;
      case HeadControlMode::searchForLandmarks:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForLandmarks();
        break;  
      case HeadControlMode::searchForLines:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForLines();
        break;  
      case HeadControlMode::searchForSpecialBall:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForSpecialBall();
        break;  
      case HeadControlMode::searchForObstacles:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForObstacles();
        break;  
      case HeadControlMode::searchForObstaclesInFront:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForObstaclesInFront();
        break;  
      case HeadControlMode::lookBetweenFeet:
        setJoints(-1000000, 0, 0, 200);
        break;  
      case HeadControlMode::lookAroundWhenBallCaught:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        lookAroundWhenBallCaught();
        break;  
      case HeadControlMode::lookAroundWhenBallJustCaught:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        lookAroundWhenBallJustCaught();
        break;  
      case HeadControlMode::lookStraightAhead:
        setJoints(0,0,0,200);
        break;  
      case HeadControlMode::catchBall:
        if (getRobotConfiguration().getRobotDesign()==RobotDesign::ERS7)
        {
          setJoints(-1064000, 0, 439000,400,0);
        }
        else //ERS210
        {
          if(motionRequest.motionType == MotionRequest::walk
            && (motionRequest.walkParams.translation.y > 50.0 ||
            motionRequest.walkParams.rotation > 0.78))
          {
            setJoints(-975000,-175000,0,400,-700000);
            lastScanWasLeft = false;
          }
          else if(motionRequest.motionType == MotionRequest::walk
            && (motionRequest.walkParams.translation.y < -50.0 ||
            motionRequest.walkParams.rotation < -0.78))
          {
            setJoints(-975000,175000,0,400,-700000);
            lastScanWasLeft = true;
          }
          else
          {
            setJoints(-975000,0,0,400,-700000);
          }
        }
        break;
      case HeadControlMode::stayAsForced:
        stayAsForced();
        break;  
      case HeadControlMode::followTail:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        followTail();
        break;  
      case HeadControlMode::lookToStars:
        
        if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
        {
          setJoints(toMicroRad(jointLimitHeadTiltP),0,toMicroRad(jointLimitHeadRollP),200, -700000);
        }
        
        else
        {        
          setJoints(toMicroRad(jointLimitHeadTiltP),0,0,200, -700000);
        }
        break;
      case HeadControlMode::lookAtPoint:
        {
          double tilt,pan,roll;
          Vector3<double> point(headControlMode.point.x,headControlMode.point.y,headControlMode.point.z);
          lookAtPoint(point,headControlMode.offset,tilt,pan,roll);
          setJoints(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll), 200,0);
          break;
        }
      case HeadControlMode::watchOrigin:
        {
          double tilt,pan,roll;
          Vector3<double> point(0,0,180);
          //its OK to start immediately here, because we scan smooth and in a way that we can always see it again:
          if (robotPose.getValidity()<0.5)
          {
            //if we dont know where to look, we look there smooth:
            if(headPathPlanner.isLastPathFinished())
            {
              unsigned long blindTime=SystemCall::getTimeSince(lastTimeOfGoodSL);
              double odoRot=currentOdometryData.rotation-lastOdometryData.rotation;
              if ((fabs(robotPose.rotation)<1.3)&&
                ((blindTime<500)||
                ((fabs(odoRot)<0.004)&&(blindTime<1300))
                )
                )
              {
                //scan for origin
                if(lastScanWasLeft)
                {
                  lookAtPoint(point,Vector2<int>(-65,0),tilt,pan,roll);
                  Vector3<long> right(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll));
                  Vector3<long> points[1]={right};
                  headPathPlanner.init(points,1, 550);
                }
                else
                {
                  lookAtPoint(point,Vector2<int>(65,0),tilt,pan,roll);
                  Vector3<long> left(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll));
                  Vector3<long> points[1]={left};
                  headPathPlanner.init(points,1, 550);
                }
                lastScanWasLeft=!lastScanWasLeft;
              }
              else if (odoRot==0)
              {
                //2do: somethimg usefull
              }
              else if (odoRot>0)
              {
                //if we can/will see the origin, then left
                lastScanWasLeft=true;
                static Vector3<long> point210(300000,1500000 ,0);
                static Vector3<long> point7(50000,1500000 ,50000);
                if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
                {
                  Vector3<long> points[1]={point7};
                  headPathPlanner.init(points,1, 450);
                }
                else
                {
                  Vector3<long> points[1]={point210};
                  headPathPlanner.init(points,1, 450);
                }
              }
              else
              {
                //if we can/will see the origin, then right
                lastScanWasLeft=false;
                static Vector3<long> point210(300000,-1500000 ,0);
                static Vector3<long> point7(50000,-1500000 ,50000);
                if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
                {
                  Vector3<long> points[1]={point7};
                  headPathPlanner.init(points,1, 450);
                }
                else
                {
                  Vector3<long> points[1]={point210};
                  headPathPlanner.init(points,1, 450);
                }
              }
            }
            headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
          }
          else
          {
            //if we know where to look, we look there immediately:
            headPathPlanner.init();
            if (robotPose.rotation<-1.45)
            {
              if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
              {
                setJoints(0,1500000 ,0, 1,0);
              }
              else
              {
                setJoints(300000,1500000 ,0, 1,0);
              }
            }
            else if (robotPose.rotation>1.45)
            {
              if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
              {
                setJoints(0,-1500000 ,0, 1,0);
              }
              else
              {
                setJoints(300000,-1500000 ,0, 1,0);
              }
            }
            else
            {
              double rota=-robotPose.speedbyDistanceToGoal*30;
              if(getRobotConfiguration().getRobotDesign()!= RobotConfiguration::ERS7)
              {
                //old head is much to fast with *30, new would be too slow with *0/*15
                rota /= 2;
              }
              lookAtPoint(point,Vector2<int>((int)rota,0),tilt,pan,roll);
              setJoints(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll), 200,0);
            }
          }
          break;
        }
      case HeadControlMode::lookParallelToGround:
        if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
        {
          if (headState.bodyTilt>0)
          {
            setJoints(0,0,toMicroRad(headState.bodyTilt));
          }
          else
          {
            setJoints(toMicroRad(headState.bodyTilt),0,0);
          }
        }
        else
        {
          setJoints(toMicroRad(headState.bodyTilt),0,-toMicroRad(headState.bodyRoll));
        }
        break;
      case HeadControlMode::scanMediumWidth:
        setJoints(
          toMicroRad(headState.bodyTilt - .3), 
          toMicroRad(cameraInfo.openingAngleWidth/4*sin((double )SystemCall::getCurrentSystemTime()/200.0)), 
          0);
        break;
      case HeadControlMode::lookJustBelowHorizon:
        setJoints(
          toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight*.5), 
          toMicroRad(cameraInfo.openingAngleWidth/4*sin((double )SystemCall::getCurrentSystemTime()/200.0)), 
          0);
        break;
      case HeadControlMode::direct:
        moveHead(headControlMode.tilt, headControlMode.pan, headControlMode.roll, headControlMode.speed, headControlMode.mouth);
      case HeadControlMode::optimizerMode:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        optimizerMode();
        break;
      default: 
        setJoints(
         0,
         toMicroRad(-robotPose.rotation),
         //toMicroRad(-currentOdometryData.rotation),
         0);
        break;
      }
    }
  }
  
  headControlModeExecutedLastTime.headControlMode = headControlMode.headControlMode;
  lastHeadControlState = headControlState;
  lastOdometryData = currentOdometryData;
  headPathPlanner.lastTilt=headMotionRequest.tilt;
  headPathPlanner.lastPan=headMotionRequest.pan;
  headPathPlanner.lastRoll=headMotionRequest.roll;
}


void GT2003HeadControl::getLookAtBallAngles(double& tilt,double& pan,double& roll)
{
  double distanceToBall = Geometry::distanceTo(robotPose.getPose(), ballPosition.seen);
  
  // calculate where the ball should be in the image:
  // far away balls are small and close to the horizon,
  // close balls appear under horizon...   
  if (distanceToBall > 500) distanceToBall = 500;
  int yOffset = (int)(35 - 60.0*(500 - distanceToBall)/500.0);
  yOffset = 0;

  lookAtPoint(
    Vector3<double>(ballPosition.seen.x, ballPosition.seen.y, ballRadius), 
    Vector2<int>(0, yOffset), 
    tilt, pan, roll);

}

void GT2003HeadControl::executeLookAtBallGoalie()
{
  double tilt, pan, roll,stepWidth= 2 * pi / 180 ,scanBorder = pi/7;
  
  getLookAtBallAngles(tilt,pan,roll);
  
  // clipped to a minimum angle to prevent from touching the ball
  if (tilt < -0.86) tilt=-0.86;
  
  if (leftWatch)
  {
    if (fabs(scanPan) < scanBorder)
      scanPan += stepWidth;
    else
    {
      scanPan -= stepWidth;
      leftWatch = false;
    }
  }
  else
  {
    if (fabs(scanPan) < scanBorder)
      scanPan -= stepWidth;
    else
    {
      scanPan += stepWidth;
      leftWatch = true;
    }
    
  }
  
  setJoints(toMicroRad(tilt), toMicroRad(pan+scanPan), toMicroRad(roll));
}

void GT2003HeadControl::executeLookAtBall()
{
  double tilt, pan, roll;
  
  getLookAtBallAngles(tilt,pan,roll);
  // clipped to a minimum angle to prevent from touching the ball
  if (tilt < -0.86) tilt=-0.86;
  setJoints(toMicroRad(tilt), toMicroRad(pan+scanPan), toMicroRad(roll));
}

void GT2003HeadControl::executeBallJustLost()
{
  if (lastHeadControlState!=headControlState)
  {
    // scan around propagated ball position can look pretty ugly in easy cases, therefore we scan around seen ball
    //Vector2<double> propagatedBallPos = ballPosition.propagated.getPropagatedPosition(SystemCall::getCurrentSystemTime());
    double tilt,pan,roll;
    getLookAtBallAngles(tilt,pan,roll);
    Vector3<long> littleLeft( toMicroRad(tilt),        toMicroRad(pan)+300000,toMicroRad(roll));
    Vector3<long> littleRight(toMicroRad(tilt),        toMicroRad(pan)-300000,toMicroRad(roll));
    Vector3<long> littleUp(   toMicroRad(tilt)+120000, toMicroRad(pan),toMicroRad(roll));
    Vector3<long> littleDown( toMicroRad(tilt)-280000, toMicroRad(pan),toMicroRad(roll));
    Vector3<long> littlePoints[4]={littleDown,littleLeft,littleUp,littleRight};
    headPathPlanner.init(littlePoints,4, 500);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void GT2003HeadControl::executeBallLost()
{ 
  Vector3<long> left;
  Vector3<long> right;
  Vector3<long> halfLeftDown;
  Vector3<long> halfRightDown;
  Vector3<long> up;
  Vector3<long> down;
  
  
  if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
  {
    left.x =  toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 4.0); 
    left.y = 1500000; 
    left.z = toMicroRad(fromDegrees(-10));
    
    right.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 4.0); 
    right.y = -1500000;
    right.z = toMicroRad(fromDegrees(-10));
    
    halfLeftDown.x = -600000;
    halfLeftDown.y =  800000;
    halfLeftDown.z = toMicroRad(fromDegrees(15));
    
    halfRightDown.x = -600000;
    halfRightDown.y = -800000;
    halfRightDown.z = toMicroRad(fromDegrees(15));
    
    up.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0);
    up.y = 0;
    up.z = toMicroRad(fromDegrees(25));
    
    down.x = -960000;
    down.y = 0;
    down.z = 0;
  }
  
  else 
  {
    left.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 4.0);
    left.y = 1500000;
    left.z = 0;
    
    right.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 4.0); 
    right.y = -1500000;
    right.z = 0;
    
    halfLeftDown.x = -600000;
    halfLeftDown.y =  800000;
    halfLeftDown.z = 0;
    
    halfRightDown.x = -600000;
    halfRightDown.y = -800000;
    halfRightDown.z = 0;
    
    up.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0);
    up.y = 0;
    up.z = 0;
    
    down.x = -960000;
    down.y = 0;
    down.z = 0;
  }
  if ((lastHeadControlState!=headControlState)||headPathPlanner.isLastPathFinished())
  {
    if (lastScanWasLeft)
    {
      Vector3<long> points[4]={down,halfRightDown,right,up};
      headPathPlanner.init(points,4, 750);
    }
    else
    {
      Vector3<long> points[4]={up,left,halfLeftDown,down};
      headPathPlanner.init(points,4, 750);
    }
    lastScanWasLeft = !lastScanWasLeft;
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void GT2003HeadControl::executeReturnToBall()
{
  if (lastHeadControlState!=headControlState)
  {
    double tilt, pan, roll;
    getLookAtBallAngles(tilt,pan,roll);
    if (tilt < -0.86) tilt=-0.86;
    
    Vector3<long> ball(toMicroRad(tilt), toMicroRad(pan),toMicroRad(roll));
    //calculated time proportional to effort to reach ball angles: 400ms for rotation of pi:
    long time = (long)(sqrt(
      (double)(headPathPlanner.lastTilt-tilt)*(double)(headPathPlanner.lastTilt-tilt)+
      (double)(headPathPlanner.lastPan-pan)*(double)(headPathPlanner.lastPan-pan)+
      (double)(headPathPlanner.lastRoll)*(double)(headPathPlanner.lastRoll))/pi/1000000*400);
    headPathPlanner.init(&ball,1, time);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void GT2003HeadControl::executeSearchAutoUpperScan()
{
  if (lastHeadControlState!=headControlState)
  {
    if (lastScanWasLeft)
    {
      //initialize upper right scan, maximum 90
      Vector3<long> centerTop(    160000, headPathPlanner.lastPan-70000,0);
      Vector3<long> rightTop(     160000, max(-1550000,headPathPlanner.lastPan-1550000),0);
      Vector3<long> rightBottom( -400000, max(-1550000,headPathPlanner.lastPan-1550000),0);
      Vector3<long> points[3]={centerTop,rightTop,rightBottom};
      headPathPlanner.init(points,3, 480);
    }
    else
    {
      //initialize upper left scan, maximum 90
      Vector3<long> centerTop(    160000, headPathPlanner.lastPan+70000,0);
      Vector3<long> leftTop(      160000, min(1550000,headPathPlanner.lastPan+1550000),0);
      Vector3<long> leftBottom(  -400000, min(1550000,headPathPlanner.lastPan+1550000),0);
      Vector3<long> points[3]={centerTop,leftTop,leftBottom};
      headPathPlanner.init(points,3, 480);
    }
    lastScanWasLeft = !lastScanWasLeft;
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void GT2003HeadControl::executeSearchAutoLowerScan()
{
  if (lastHeadControlState!=headControlState)
  {
    double tilt,pan,roll;
    getLookAtBallAngles(tilt,pan,roll);
    if(lastScanWasLeft) // == (headPathPlanner.lastPan > 0)
    {
      //initialize lower left scan
      Vector3<long> centerBottom(-400000, toMicroRad(pan)+70000,toMicroRad(roll));
      Vector3<long> ball(toMicroRad(tilt),toMicroRad(pan)-50000,toMicroRad(roll));
      Vector3<long> points[2]={centerBottom,ball};
      headPathPlanner.init(points,2, 400);
    }
    else
    {
      //initialize lower right scan
      Vector3<long> centerBottom(-400000, toMicroRad(pan)-70000,toMicroRad(roll));
      Vector3<long> ball(toMicroRad(tilt),toMicroRad(pan)+50000,toMicroRad(roll));
      Vector3<long> points[2]={centerBottom,ball};
      headPathPlanner.init(points,2, 400);
    }
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}



void GT2003HeadControl::lookLeft()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> down(-1200000,headPathPlanner.lastPan,0);
    Vector3<long> left(-1300000,1300000,0);
    Vector3<long> points[2]={down,left};
    headPathPlanner.init(points,2, 400);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void GT2003HeadControl::lookRight()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> down(-1200000,headPathPlanner.lastPan,0);
    Vector3<long> right(-1300000,-1300000,0);
    Vector3<long> points[2]={down,right};
    headPathPlanner.init(points,2, 400);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}


void GT2003HeadControl::searchForLandmarks()
{
  //just like last year, but
  /** @todo may be we would like to see lines too */
  
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0),  1500000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[2]={right,left};
      headPathPlanner.init(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, 800);
    }
    else
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, 800);
    }
  }
}

void GT2003HeadControl::searchForLines()
{
  Vector3<long> left(0, 1500000, 0);
  Vector3<long> right(0, -1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[2]={right,left};
      headPathPlanner.init(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, 800);
    }
    else
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, 800);
    }
  }
}

void GT2003HeadControl::searchForSpecialBall()
{
  
  //Vector3<long> centerTop(    160000, headPathPlanner.lastPan-70000,0);
  Vector3<long> rightTop(     160000, -1500000,0);
  Vector3<long> rightCenter( -120000, -1500000,0);
  Vector3<long> down(        -900000,        0,0);
  Vector3<long> leftCenter(  -120000,  1500000,0);
  Vector3<long> leftTop(      160000,  1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[3]={rightTop,down,leftTop};
      headPathPlanner.init(points,3, 8000);
    }
    else
    {
      Vector3<long> points[3]={leftTop,down,rightTop};
      headPathPlanner.init(points,3, 8000);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[3]={leftTop,down,rightTop};
      headPathPlanner.init(points,3, 8000);
    }
    else
    {
      Vector3<long> points[3]={rightTop,down,leftTop};
      headPathPlanner.init(points,3, 8000);
    }
  }
}

void GT2003HeadControl::searchForObstacles()
{
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0),  1500000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1500000,0);
  Vector3<long> center(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 2.0), 0,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points, 2, 800);
    }
    else
    {
      Vector3<long> points[3]={right, center, left};
      headPathPlanner.init(points, 3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points, 1, 800);
    }
    else
    {
      Vector3<long> points[2]={center, left};
      headPathPlanner.init(points, 2, 800);
    }
  }
}

void GT2003HeadControl::searchForObstaclesInFront()
{
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), 1400000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1400000,0);
  Vector3<long> center(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 1.5), 0,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points, 2, 800);
    }
    else
    {
      Vector3<long> points[3]={right, center, left};
      headPathPlanner.init(points, 3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points, 1, 800);
    }
    else
    {
      Vector3<long> points[2]={center, left};
      headPathPlanner.init(points, 2, 800);
    }
  }
}


void GT2003HeadControl::lookAroundWhenBallCaught()
{
  Vector3<long> left(  300000, 1200000,0);
  Vector3<long> middle(     0,       0,0);
  Vector3<long> right( 300000,-1200000,0);
  //The mode is active the first time after another one
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    //If current pan is left:
    if(headPathPlanner.lastPan > 0)
    {
      //from current position to left, center and right.
      Vector3<long> points[3]={left,middle,right};
      headPathPlanner.init(points,3, 800);
    }
    else
    {
      //from current position to right, center and left.
      Vector3<long> points[3]={right,middle,left};
      headPathPlanner.init(points,3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    //init the next loop
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={middle,right};
      headPathPlanner.init(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={middle,left};
      headPathPlanner.init(points,2, 800);
    }
  }
}

void GT2003HeadControl::lookAroundWhenBallJustCaught()
{
  Vector3<long> left(  100000, 500000,0);
  Vector3<long> right( 100000,-500000,0);
  
  //The mode is active the first time after another one
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    //If current pan is left:
    if(headPathPlanner.lastPan > 0)
    {
      //from current position to left, center and right.
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, 350);
    }
    else
    {
      //from current position to right, center and left.
      Vector3<long> points[3]={right,left};
      headPathPlanner.init(points,2, 350);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    //init the next loop
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, 350);
    }
    else
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, 350);
    }
  }
}

void GT2003HeadControl::stayAsForced()
{
  const int tiltTolerance = 60000;
  const int panTolerance  = 60000;
  const int rollTolerance = 50000;
  
  long tiltDiff = sensorDataBuffer.lastFrame().data[SensorData::headTilt] - headMotionRequest.tilt;
  long panDiff  = sensorDataBuffer.lastFrame().data[SensorData::headPan]  - headMotionRequest.pan;
  long rollDiff = sensorDataBuffer.lastFrame().data[SensorData::headRoll] - headMotionRequest.roll;
  
  if (tiltDiff > tiltTolerance)
  {
    headMotionRequest.tilt += tiltDiff-tiltTolerance;
  }
  else if (tiltDiff < -tiltTolerance)
  {
    headMotionRequest.tilt += tiltDiff+tiltTolerance;
  }
  
  if (panDiff > panTolerance)
  {
    headMotionRequest.pan += panDiff-panTolerance;
  }
  else if (panDiff < -panTolerance)
  {
    headMotionRequest.pan += panDiff+panTolerance;
  }
  
  if (rollDiff > rollTolerance)
  {
    headMotionRequest.roll += rollDiff-rollTolerance;
  }
  else if (rollDiff < -rollTolerance)
  {
    headMotionRequest.roll += rollDiff+rollTolerance;
  }
}

void GT2003HeadControl::followTail()
{
  long tilt, pan, tilt2;
  if(getRobotConfiguration().getRobotDesign() == RobotConfiguration::ERS7)
  {
    tilt2 = sensorDataBuffer.lastFrame().data[SensorData::headTilt2];
    if(sensorDataBuffer.lastFrame().data[SensorData::backF] > 15)
    {
      if(sensorDataBuffer.lastFrame().data[SensorData::backR] > 15) tilt2 = 0;
      else tilt2 -= toMicroRad(fromDegrees(20));
    }
    else if(sensorDataBuffer.lastFrame().data[SensorData::backR] > 15) tilt2 += toMicroRad(fromDegrees(20));
  }
  else // not ERS7
  {
    tilt2 = 0;
  }
  
  pan = toMicroRad(
    fromMicroRad(sensorDataBuffer.lastFrame().data[SensorData::tailPan]) / jointLimitTailPanN * jointLimitHeadPanN);
  
  tilt = toMicroRad(
    (fromMicroRad(sensorDataBuffer.lastFrame().data[SensorData::tailTilt]) - jointLimitTailTiltN) /
    (jointLimitTailTiltP - jointLimitTailTiltN) * (jointLimitHeadTiltP - jointLimitHeadTiltN) 
    +jointLimitHeadTiltN
    );
  
  setJoints(tilt, pan, tilt2);
}

void GT2003HeadControl::buildCameraMatrix(const SensorData& data, const HeadState& headState)
{
  if(lastFrame != data.frameNumber)
  {
    lastFrame = data.frameNumber;
    currentTilt = fromMicroRad(data.data[SensorData::headTilt]);
    currentPan  = fromMicroRad(data.data[SensorData::headPan]);
    currentRoll = fromMicroRad(data.data[SensorData::headRoll]);
    
    Geometry::calculateCameraMatrix(currentTilt, currentPan, currentRoll, headState, cameraMatrix2);
  }
}

void GT2003HeadControl::fromWorld2CameraCoordinates
(
 const Vector3<double> pointIn3D,  
 Vector3<double> &pointInCameraCoordinates
 )
{
  RotationMatrix bodyRotation;
  Vector3<double> bodyTranslation;
  //  double phi = -this->robotPose.getPose().getAngle();
  bodyRotation.fromKardanRPY(0, 0, -robotPose.rotation);
  bodyTranslation.x = robotPose.translation.x; 
  bodyTranslation.y = robotPose.translation.y;
  bodyTranslation.z = 0;
  
  pointInCameraCoordinates = 
    cameraMatrix2.rotation.invert()*((bodyRotation.invert()*(pointIn3D - bodyTranslation) - cameraMatrix2.translation));
}

void GT2003HeadControl::fromWorld2RobotCoordinates
(
 const Vector3<double> pointIn3D,  
 Vector3<double> &pointInRobotCoordinates
 )
{
  RotationMatrix bodyRotation;
  Vector3<double> bodyTranslation;
  
  bodyRotation.fromKardanRPY(0, 0, -robotPose.rotation);
  
  bodyTranslation.x = robotPose.translation.x; 
  bodyTranslation.y = robotPose.translation.y;
  bodyTranslation.z = 0;
  
  pointInRobotCoordinates = bodyRotation.invert()*(pointIn3D - bodyTranslation);
}

void GT2003HeadControl::optimizerMode()
{  
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0),  1500000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1500000,0);
  
  int duration=4000;

  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[2]={right,left};
      headPathPlanner.init(points,2, duration);
    }
    else
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, duration);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, duration);
    }
    else
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, duration);
    }
  }
}



/*
* Change log :
* 
* $Log: GT2003HeadControl.cpp,v $
* Revision 1.35  2004/05/17 19:21:51  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.34  2004/05/11 16:37:04  dueffert
* ERS7 psd and behindMe stuff in watchOrigin improved
*
* Revision 1.33  2004/05/04 09:05:35  dueffert
* headPathPlanner bug fixed in all copies
*
* Revision 1.32  2004/05/03 18:46:08  dueffert
* headPathPlanner bug fixed; HCM::none reimplemented; watchOrigin improved
*
* Revision 1.31  2004/03/17 16:18:49  thomas
* added preversion of motion optimisation with behaviour, selflocator, headcontrol and robotcontrol dialog
*
* Revision 1.30  2004/03/17 09:33:59  jhoffman
* changes to state machine, look at ball improved, look just below horizon recovered from revision 1.23
*
* Revision 1.29  2004/03/10 10:00:10  dueffert
* copy'n'paste bug fixed; missing normalization added
*
* Revision 1.28  2004/03/08 01:38:54  roefer
* Interfaces should be const
*
* Revision 1.27  2004/03/05 15:47:07  dueffert
* several bugs in watchOrigin fixed
*
* Revision 1.26  2004/03/04 18:17:15  juengel
* Added ERS7 version of catch ball.
*
* Revision 1.25  2004/03/04 12:38:56  dueffert
* lookAtPoint completely rewritten
*
* Revision 1.24  2004/03/04 10:05:17  jhoffman
* - motion process now uses odometry to propagate the robot pose while no new robot pose is being sent (this makes headcontrol a little better)
* - removed headcontrol member variable "propagatedPose" from headcontrol and cognition->motion-sender
*
* Revision 1.23  2004/02/29 17:28:53  dueffert
* watchOrigin work on ERS210 *and* ERS7 now
*
* Revision 1.22  2004/02/23 16:46:59  dueffert
* watchOrigin improved
*
* Revision 1.21  2004/02/18 14:41:47  dueffert
* ERS7 support for lookParallelToGround and watchOrigin added
*
* Revision 1.20  2004/02/16 17:51:06  dueffert
* ERS7 support for lookAtPoint added
* copy'n'paste bug for variable image size fixed
*
* Revision 1.19  2004/02/12 14:41:36  juengel
* added visualization to lookAtPoint
*
* Revision 1.18  2004/01/20 16:15:52  dueffert
* watchOrigin improved; correct indent
*
* Revision 1.17  2004/01/16 15:48:44  dueffert
* headControlMode added
*
* Revision 1.16  2004/01/14 14:40:35  juengel
* Bug fixed.
*
* Revision 1.15  2004/01/13 02:17:01  richert
* finetuning SimpleMotionDetector
*
* Revision 1.14  2004/01/12 18:07:35  goetzke
* adapted modes "lookToStars" and "searchForBalls" for ERS7
*
* Revision 1.13  2004/01/09 20:05:46  richert
* going on with function headMove (still not works!)
*
* Revision 1.12  2004/01/08 18:19:28  mkunz
* distancePanCenterToCamera -> distancePanCenterToCameraX
* new: distancePanCenterToCameraZ
*
* Revision 1.11  2004/01/08 12:35:28  juengel
* tilt2 changes faster in mode followTail
*
* Revision 1.10  2004/01/07 17:38:48  richert
* started new method headMove
*
* Revision 1.9  2004/01/07 14:26:38  richert
* no message
*
* Revision 1.8  2004/01/07 04:09:43  richert
* added HeadControlMode "direct"
*
* Revision 1.7  2004/01/05 17:57:58  juengel
* HeadControlMode followTail uses back switches to adjust tilt2.
*
* Revision 1.6  2004/01/01 10:58:50  roefer
* RobotDimensions are in a class now
*
* Revision 1.5  2003/12/15 11:47:05  juengel
* Introduced CameraInfo
*
* Revision 1.4  2003/12/10 14:53:12  dueffert
* warning removed
*
* Revision 1.3  2003/12/09 16:31:19  jhoffman
* added legacy head control modes from gt2003 (lookjustbelowhorizon)
*
* Revision 1.2  2003/12/05 14:20:06  jhoffman
* removed unused headcontrolmodes
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.16  2003/08/08 11:59:59  dueffert
* doxygen warning fixed
*
* Revision 1.15  2003/07/30 14:49:08  dueffert
* lookParallelToGround implemented for initial looking at checkerboard
*
* Revision 1.14  2003/07/09 11:42:19  jhoffman
* headcontrol mode as used in challenge 3
*
* Revision 1.13  2003/07/06 23:12:33  dassler
* ups.. State bersehen
*
* Revision 1.12  2003/07/06 22:18:35  dassler
* Neuen Goalie HeadControl eingebaut.
* StateMachine gendert.
*
* Goalie hat nun einen eigenen LookForBallGoalie
*
* Revision 1.11  2003/07/06 12:17:12  roefer
* Challenge 2 finished, some parts not checked in
*
* Revision 1.10  2003/07/06 02:13:43  loetzsch
* search-auto-for-goalie looks only 3000 ms consec. at the ball
*
* Revision 1.9  2003/07/06 01:41:51  dueffert
* scanAutoForGoalie look at ball time decreased
*
* Revision 1.8  2003/07/05 21:36:19  dassler
* Goalie HeadControlMode
* Alle 6 Sek. sich umschauen, wenn Ball weit weg (ber der Mittellinie) alle 3 Sek
*
* Revision 1.7  2003/07/05 19:34:27  dueffert
* ball speed is only considered in comments now :-)
*
* Revision 1.6  2003/07/03 16:07:12  loetzsch
* no message
*
* Revision 1.5  2003/07/03 10:35:59  roefer
* New head control  mode search-for-lines
*
* Revision 1.4  2003/07/03 10:09:42  dueffert
* executeBallLost improved
*
* Revision 1.3  2003/07/02 20:42:49  dueffert
* lookAroundWhenBallJustCaught changed
*
* Revision 1.2  2003/07/02 16:47:26  dueffert
* lookAroundWhenBallJustCaught added
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.74  2003/06/30 11:49:44  schmidt
* Corrected search-for-special-ball
*
* Revision 1.73  2003/06/29 08:40:17  roefer
* Warnings removed
*
* Revision 1.72  2003/06/28 14:06:27  loetzsch
* made compilable again
*
* Revision 1.71  2003/06/27 17:56:18  kudlacik
* Insert new HeadControlMode searchForSpecialBall
*
* Revision 1.70  2003/06/27 13:17:57  jhoffman
* work on obstacle avoider challenge,
* added headcontrolmode,
* added method needed to determine empty space,
* updatet drawing method to reflect actual corridor size
*
* Revision 1.69  2003/06/27 11:21:00  dueffert
* lookAtPoint perfected, simpleLookAtPoint removed
*
* Revision 1.68  2003/06/26 16:02:17  dueffert
* smarter looking at ball from different states, searchAuto() scans ball+-90 maximum
*
* Revision 1.67  2003/06/26 14:21:53  dueffert
* returnToBall() speed corrected
*
* Revision 1.66  2003/06/26 12:05:26  dueffert
* sign bug fixed
*
* Revision 1.65  2003/06/24 14:35:22  dueffert
* isLastPathFinished() and first exact lookAtPoint() added
*
* Revision 1.64  2003/06/23 14:30:08  dueffert
* executeReturnToBall() implemented
*
* Revision 1.63  2003/06/22 22:16:20  cesarz
* small bugfix (missing brackets)
*
* Revision 1.62  2003/06/22 15:58:41  dueffert
* setting lastScanWasLeft in other modes too
*
* Revision 1.61  2003/06/22 15:16:25  dueffert
* global state machine added
*
* Revision 1.60  2003/06/22 11:59:27  loetzsch
* lookAtBall looks deeper now
*
* Revision 1.59  2003/06/22 10:10:13  dueffert
* searchForBall improved
*
* Revision 1.58  2003/06/21 19:39:32  loetzsch
* lookAtBall looks some deeper now
*
* Revision 1.57  2003/06/21 16:36:07  loetzsch
* clipped the tilt angle in lookAtBall()
*
* Revision 1.56  2003/06/21 15:27:12  juengel
* Reanimated searchForObstacles.
*
* Revision 1.55  2003/06/21 12:33:31  dueffert
* left/right bug fixed
*
* Revision 1.54  2003/06/21 11:56:59  dueffert
* some tuning
*
* Revision 1.53  2003/06/21 11:01:42  dueffert
* copy'n'paste bug fixed
*
* Revision 1.52  2003/06/21 10:49:16  dueffert
* number conversion bug fixed
*
* Revision 1.51  2003/06/21 09:44:19  dueffert
* a bunch of little improvements
*
* Revision 1.50  2003/06/20 15:30:34  dueffert
* shut mouth by default
*
* Revision 1.49  2003/06/19 18:33:07  juengel
* lookAtStarsAdded
*
* Revision 1.48  2003/06/18 17:03:30  dueffert
* lookLeft/Right bug fixed
*
* Revision 1.47  2003/06/18 16:12:41  dueffert
* searchAuto improved
*
* Revision 1.46  2003/06/18 10:56:55  dueffert
* minor improvements: lookLeft, lookRight, common lookAtBall, shorter landmark scan in searchAuto
*
* Revision 1.45  2003/06/18 08:52:12  thomas
* updated: gt2003-behavior changed to current head-control
*
* Revision 1.44  2003/06/17 20:00:06  dueffert
* logic added
*
* Revision 1.43  2003/06/17 16:55:38  dueffert
* bug fixes and additions
*
* Revision 1.42  2003/06/17 13:41:47  dueffert
* some stuff added
*
* Revision 1.41  2003/06/16 20:09:01  jhoffman
* interpolate odometry and propagate ball position for look at ball
*
* Revision 1.40  2003/06/16 09:43:10  dueffert
* scope bug fixed
*
* Revision 1.39  2003/06/15 22:42:22  loetzsch
* no message
*
* Revision 1.38  2003/06/15 18:13:12  jhoffman
* added simple states for "search fo ball"
*
* Revision 1.37  2003/06/15 16:43:56  jhoffman
* propagated position is calculated by a function rather then in iteratively in cognition
*
* Revision 1.36  2003/06/13 17:59:27  juengel
* Added searchForObstacles.
* Removed "unused phi" - warnings.
*
* Revision 1.35  2003/06/13 17:05:28  jhoffman
* added some code necessary for looking at the ball
*
* Revision 1.34  2003/06/12 12:19:57  dueffert
* lookAtPoint added
*
* Revision 1.33  2003/06/06 13:54:06  dueffert
* functionality added
*
* Revision 1.32  2003/06/05 15:50:28  dueffert
* some ideas added
*
* Revision 1.31  2003/06/02 14:46:24  dueffert
* bug fixed
*
* Revision 1.30  2003/06/02 14:44:01  dueffert
* some modes implemented
*
* Revision 1.29  2003/06/02 13:17:07  dueffert
* ball holding removed from lookLeft/lookRight
*
* Revision 1.28  2003/06/02 12:56:23  dueffert
* bugs fixed
*
* Revision 1.27  2003/05/28 15:34:27  dueffert
* new GT2003HeadControl
*
* Revision 1.26  2003/05/28 08:43:06  juengel
* Added HeadPathPlanner.
*
* Revision 1.25  2003/05/27 16:17:40  juengel
* Renamed GT2003HeadControl to HUGO2003HeadControl.
* Added GT2003HeadControl.
*
*
*/
