/**
* @file GT2003HeadControl.h
* 
* This file contains a class for head-behavior.
*
* @author Jan Hoffmann
*/

#ifndef __GT2003HeadControl_h__
#define __GT2003HeadControl_h__

#include "HeadControl.h"
#include "Tools/Math/PIDsmoothedValue.h"


//in urad per frame: 4000 = 28.6/s
#define minHeadSpeed 4000

/**
* @class GT2003HeadControl
*
* A solution of the HeadControl module.
*/
class GT2003HeadControl : public HeadControl
{
public:
/** 
* Constructor.
* @param interfaces The paramters of the HeadControl module.
  */
  GT2003HeadControl(const HeadControlInterfaces& interfaces);
  
  /** Destructor. */
  ~GT2003HeadControl() {}
  
  /** Executes the module */
  virtual void execute();
  
  enum headControlStates
  {
    otherModes,
      lookAtBall,
      lookAtBallGoalie,
      ballJustLost,
      ballLost,
      returnToBall,
      searchAutoUpperScan,
      searchAutoLowerScan,
      numOfStates
  } headControlState, lastHeadControlState;
  
  bool lastScanWasLeft,leftWatch;
  double scanPan;
  
private:
  CameraInfo cameraInfo;
  
  void executeLookAtBall();
  void executeLookAtBallGoalie();
  void executeBallJustLost();
  void executeBallLost();
  void executeReturnToBall();
  void executeSearchAutoUpperScan();
  void executeSearchAutoLowerScan();
  
  void searchForLandmarks();
  void searchForLines();
  void searchForSpecialBall();
  void searchForObstacles();
  void searchForObstaclesInFront();
  void lookAroundWhenBallJustCaught();
  void lookAroundWhenBallCaught();
  void stayAsForced();
  void followTail();
  void lookLeft();
  void lookRight();
  void lookJustBelowHorizon();
  void optimizerMode();
  
  /** should be obsolete
  2do: CameraMatrix woanders herbekommen!
  */
  void buildCameraMatrix(const SensorData& data, const HeadState& headState);
  unsigned long lastFrame;
  
  /** 3d transformation from world coordinates to camera coordinates using the camera matrix */
  void fromWorld2CameraCoordinates(const Vector3<double> pointIn3D, Vector3<double> &pointInCameraCoordinates);
  
  /** 3d transformation from world coordinates to robot coordinates */
  void fromWorld2RobotCoordinates(const Vector3<double> pointIn3D, Vector3<double> &pointInRobotCoordinates);
  
  /** look at 3d-point on field with offset point in camera image */
  void lookAtPoint(const Vector3<double> &pos,const Vector2<int> &offset,double& tilt,double& pan,double& roll);
  /** look at 3d-point subroutine trying to find tilt1 solution for given tilt2/roll. this is useful on ERS210 and ERS7
  *@return true when matching angles were found */
  bool lookAtPointFixTilt2(const Vector3<double> &aim, const double& xtan, const double& ytan, double& tilt, double& pan, const double& tilt2);
  /** look at 3d-point subroutine trying to find tilt2 solution for given tilt1. this is only useful on ERS7
  *@return true when matching angles were found */
  bool lookAtPointFixTilt1(const Vector3<double> &aim, const double& xtan, const double& ytan, const double& tilt, double& pan, double& tilt2);
  
  /** calculate the angles for looking at the seen ball with a certain camera pixel depending on ball distance */
  void getLookAtBallAngles(double& tilt,double& pan,double& roll);
  
  /** just a fake for compatibility, no smoothing yet (speed is ignored) */
  void setJoints(long tilt, long pan, long roll, long speed=100, long mouth=0);
  
  /* direct HeadControl with speed parameter (smooting) */
  void moveHead(long tilt=0, long pan=0, long roll=0, long speed=400, long mouth=0);
  
  /** Indicates if this head control mode was active the last time execute was called */
  HeadControlMode headControlModeExecutedLastTime;

  /** current odometry data of last call to head control */
  OdometryData lastOdometryData;
  /** time of last good self localization */
  unsigned long lastTimeOfGoodSL;
  
  /** This struct can calculate a smooth series of head joint angles from a gives set of way points and an overall duration */
  struct HeadPathPlanner
  {
  public:
  /**
  * Resets/Initializes a set of points to visit in a certain time
  * @param vectors set of arcs to visit
  * @param numberOfVectors number of Vector3s in param vectors
  * @param duration time in ms to reach the last position in param vecors
    */
    void init(const Vector3<long>* vectors=0, int numberOfVectors=0, unsigned long duration=8);
    
    /**
    * default constructor
    */
    HeadPathPlanner():lastTilt(0),lastPan(0),lastRoll(0),currentPoint(0),currentFrame(0),numberOfFrames(0),numberOfPoints(0) {}
    
    /**
    * Initializes a set of points to visit in a certain time. These points are
    * assumed to be cyclic. The first and last point to visit is the one closest
    * to the current head position.
    * @param vectors set of arcs to visit
    * @param numberOfVectors number of Vector3s in param vectors
    * @param duration time in ms to reach the last position in param vecors
    */
    void initLoop(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration);
    
    /**
    * Calculates the angles for tilt pan and roll
    * @return true if the last point is reached.
    */
    bool getAngles(long& tilt, long& pan, long& roll);
    
    /**
    * Return whether the last initialized path is already finished
    * @return true if last path is finished.
    */
    inline bool isLastPathFinished() { return (currentFrame>=numberOfFrames); }
    
    long lastTilt;
    long lastPan;
    long lastRoll;
    
  private:
    long currentPoint;
    long currentFrame;
    long numberOfFrames;
    
    enum {maxNumberOfPoints = 20};
    long numberOfPoints;
    Vector3<long> points[maxNumberOfPoints];
    double firstFrame[maxNumberOfPoints];
  };
  
  HeadPathPlanner headPathPlanner;
  
  // needed for transformations and for simple look at point
  CameraMatrix cameraMatrix2;
  double currentTilt, currentPan, currentRoll;
  
};

#endif //__GT2003HeadControl_h__

/*
* Change log :
* 
* $Log: GT2003HeadControl.h,v $
* Revision 1.15  2004/05/17 19:21:51  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.14  2004/05/03 18:46:17  dueffert
* watchOrigin improved
*
* Revision 1.13  2004/03/17 16:18:49  thomas
* added preversion of motion optimisation with behaviour, selflocator, headcontrol and robotcontrol dialog
*
* Revision 1.12  2004/03/08 01:38:54  roefer
* Interfaces should be const
*
* Revision 1.11  2004/03/05 15:51:48  dueffert
* small initialization bug fixed
*
* Revision 1.10  2004/03/04 12:38:55  dueffert
* lookAtPoint completely rewritten
*
* Revision 1.9  2004/03/04 10:05:19  jhoffman
* - motion process now uses odometry to propagate the robot pose while no new robot pose is being sent (this makes headcontrol a little better)
* - removed headcontrol member variable "propagatedPose" from headcontrol and cognition->motion-sender
*
* Revision 1.8  2004/02/23 16:54:19  dueffert
* beautified
*
* Revision 1.7  2004/02/16 17:50:53  dueffert
* ERS7 support for lookAtPoint added
*
* Revision 1.6  2004/01/30 15:42:59  dueffert
* missing initialization added
*
* Revision 1.5  2004/01/09 20:05:46  richert
* going on with function headMove (still not works!)
*
* Revision 1.4  2004/01/07 17:38:48  richert
* started new method headMove
*
* Revision 1.3  2003/12/15 11:47:05  juengel
* Introduced CameraInfo
*
* Revision 1.2  2003/12/09 16:31:20  jhoffman
* added legacy head control modes from gt2003 (lookjustbelowhorizon)
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.5  2003/08/09 14:53:10  dueffert
* files and docu beautified
*
* Revision 1.4  2003/07/06 22:18:35  dassler
* Neuen Goalie HeadControl eingebaut.
* StateMachine gendert.
*
* Goalie hat nun einen eigenen LookForBallGoalie
*
* Revision 1.3  2003/07/03 10:35:59  roefer
* New head control  mode search-for-lines
*
* Revision 1.2  2003/07/02 16:47:26  dueffert
* lookAroundWhenBallJustCaught added
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.37  2003/06/27 17:56:19  kudlacik
* Insert new HeadControlMode searchForSpecialBall
*
* Revision 1.36  2003/06/27 13:17:57  jhoffman
* work on obstacle avoider challenge,
* added headcontrolmode,
* added method needed to determine empty space,
* updatet drawing method to reflect actual corridor size
*
* Revision 1.35  2003/06/27 11:21:00  dueffert
* lookAtPoint perfected, simpleLookAtPoint removed
*
* Revision 1.34  2003/06/26 16:02:18  dueffert
* smarter looking at ball from different states, searchAuto() scans ball+-90 maximum
*
* Revision 1.33  2003/06/24 14:35:22  dueffert
* isLastPathFinished() and first exact lookAtPoint() added
*
* Revision 1.32  2003/06/22 15:16:25  dueffert
* global state machine added
*
* Revision 1.31  2003/06/21 10:49:16  dueffert
* number conversion bug fixed
*
* Revision 1.30  2003/06/21 09:43:49  dueffert
* beautified
*
* Revision 1.29  2003/06/18 16:12:42  dueffert
* searchAuto improved
*
* Revision 1.28  2003/06/18 10:56:55  dueffert
* minor improvements: lookLeft, lookRight, common lookAtBall, shorter landmark scan in searchAuto
*
* Revision 1.27  2003/06/17 20:00:06  dueffert
* logic added
*
* Revision 1.26  2003/06/17 16:55:38  dueffert
* bug fixes and additions
*
* Revision 1.25  2003/06/16 20:09:01  jhoffman
* interpolate odometry and propagate ball position for look at ball
*
* Revision 1.24  2003/06/15 18:13:12  jhoffman
* added simple states for "search fo ball"
*
* Revision 1.23  2003/06/15 16:43:56  jhoffman
* propagated position is calculated by a function rather then in iteratively in cognition
*
* Revision 1.22  2003/06/13 17:58:36  juengel
* Added searchForObstacles.
*
* Revision 1.21  2003/06/13 17:05:28  jhoffman
* added some code necessary for looking at the ball
*
* Revision 1.20  2003/06/12 12:19:57  dueffert
* lookAtPoint added
*
* Revision 1.19  2003/06/06 13:54:06  dueffert
* functionality added
*
* Revision 1.18  2003/05/28 15:34:27  dueffert
* new GT2003HeadControl
*
* Revision 1.17  2003/05/28 08:43:06  juengel
* Added HeadPathPlanner.
*
* Revision 1.16  2003/05/27 16:17:40  juengel
* Renamed GT2003HeadControl to HUGO2003HeadControl.
* Added GT2003HeadControl.
*
*/
