/**
* @file DDD2004BallSpecialist.h
* 
* This file contains a class for Image Processing.
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
* @author Max Risler
* @author Ronnie Brunn
* @author Michael Kunz
*/

#ifndef __DDD2004BallSpecialist_h_
#define __DDD2004BallSpecialist_h_

#include "Tools/Math/Geometry.h"
#include "Tools/Debugging/DebugImages.h"
#include "Platform/GTAssert.h"

#include "Representations/Perception/ColorTable.h"
#include "Representations/Perception/Image.h"
#include "Representations/Perception/CameraMatrix.h"
#include "Representations/Perception/BallPercept.h"

#include "DDD2004ColorCorrector.h"
#include "Tools/Math/Common.h"

/**
The BallSpecialist finds a single ball in an image.
*/
class DDD2004BallSpecialist  
{
public:
  DDD2004BallSpecialist(const DDD2004ColorCorrector& colorCorrector);
  
  /** 
  * Searches for the ball in the image, starting from the secified point. 
  */
  void searchBall
  (
    const Image& image,
    const ColorTable& colorTable,
    const CameraMatrix& cameraMatrix,
    int x, int y,
    BallPercept& ballPercept
  );
  
  /**
  * Returns the Similarity of the given color to orange.
  */
  unsigned char getSimilarityToOrange (unsigned char y, unsigned char u, unsigned char v)
  {
/*      double r = y + 1.140 * (u - 128);
      double g = y - 0.394 * (v - 128) - 0.581 * (u - 128);
      double b = y + 2.028 * (v - 128);
      if(r < 0) r = 0; if(r > 255) r = 255;
      if(g < 0) g = 0; if(g > 255) g = 255;
      if(b < 0) b = 0; if(b > 255) b = 255;*/
    unsigned char r, g, b;
    Image::convertFromYCbCrToRGB(y, u, v, r, g, b);

	  return (unsigned char) min(max(r-b-abs(2*g-r),0),255);
  }

private:

  /**
  * The struct represents a ball point.
  */
  class BallPoint : public Vector2<int>
  {
  public:
    BallPoint() : atBorder(false), greenIsClose(true), yellowIsClose(false), hardEdge(true){};
    bool greenIsClose;
    bool yellowIsClose;
    bool hardEdge;
    bool atBorder;

    BallPoint& operator=(const Vector2<int>& other)
    {
      x = other.x;
      y = other.y;

      return *this;
    }
  };

  class BallPointList
  {
  public:
    BallPointList() : number(0) {};

    enum {maxNumberOfPoints = 400}; /**< The maximum number of ball points. */
    BallPoint ballPoints[maxNumberOfPoints]; /**< The ball points. */
    int number; /**< The actual number of ball points. */

    void add(const BallPoint& ballPoint)
    {
      ASSERT(number < maxNumberOfPoints);
      ballPoints[number++] = ballPoint;
      DOT(imageProcessor_ball2, ballPoint.x / 2, ballPoint.y / 2,
        (ballPoint.hardEdge) ? Drawings::blue : Drawings::orange, 
        (ballPoint.atBorder) ? Drawings::black :
        (ballPoint.greenIsClose) ? Drawings::green :
        (ballPoint.yellowIsClose) ? Drawings::yellow :
        Drawings::white
      );
    }

    BallPoint& operator[] (int i) { return ballPoints[i]; }
    const BallPoint& operator[] (int i) const { return ballPoints[i]; }
  };

  /** Scan for the ball starting at a given trigger point */
  void scanForBallPoints
  (
   const Image& image,
   const CameraInfo& bwCameraInfo,
   const ColorTable& colorTable,
   int x, int y,
   BallPointList& ballPoints,
   int& countOrange,
   int& maxOrangePerLine,
   int& countPixel
  );

  /** Finds the end of the ball */
  bool findEndOfBall(
    const Image& image,
    const CameraInfo& bwCameraInfo,
    const ColorTable& colorTable,
    const BallPoint& start,
    const Vector2<int>& step,
    BallPoint& destination,
    int& countOrange,
    int& maxOrangePerLine,
    int& countPixel
    );

  /**
  * The function tries to calculate the ball percept by using the Levenberg-Marquardt
  * algorithm. The function fails if less than 3 points are available.
  * @return true if ball percept was created
  */
  bool createBallPerceptLevenbergMarquardt
  (
    const BallPointList& ballPoints,
    Vector2<int>& center,
    double& radius
  );


  /*
  * Check if a list of points is inside a given circle
  */
  bool checkIfPointsAreInsideBall(const BallPointList& ballPoints, Vector2<int>& center, double radius);

  /**
  * The function checks whether a ball percept is plausible and will add it if so.
  * @param centerX The x-coordinate of the center of the ball.
  * @param centerY The y-coordinate of the center of the ball.
  * @param radius The radius of the ball.
  */
  void addBallPercept
  (
    const Image& image,
    const CameraInfo& bwCameraInfo,
    const CameraMatrix& cameraMatrix,
    const Vector2<int>& center,
    double radius,
    BallPercept& ballPercept
  );  

  const DDD2004ColorCorrector& colorCorrector;
};

#endif// __BallSpecialist_h_

/*
* Change log :
* 
* $Log: DDD2004BallSpecialist.h,v $
* Revision 1.13  2004/05/03 16:22:45  mkunz
* use global YCbCr-->RGB method
*
* Revision 1.12  2004/04/07 12:28:56  risler
* ddd checkin after go04 - first part
*
* Revision 1.8  2004/04/02 15:16:31  mkunz
* hard edge detection basics
*
* Revision 1.7  2004/04/02 01:51:03  mkunz
* better handling og highlights and ghost balls in yellow goal
*
* Revision 1.6  2004/03/30 11:10:39  mkunz
* bwCameraInfo intrduced
*
* Revision 1.5  2004/03/29 20:46:02  risler
* use high resolution
*
* Revision 1.4  2004/03/29 12:41:34  park
* no message
*
* Revision 1.3  2004/03/29 11:28:04  park
* no message
*
* Revision 1.2  2004/03/29 11:14:29  park
* added refiner for Ball-Percept
*
* Revision 1.1.1.1  2004/03/29 08:28:47  Administrator
* initial transfer from tamara
*
* Revision 1.11  2004/03/28 19:36:04  risler
* removed the now obsolete createBallPercept function
*
* Revision 1.10  2004/03/28 18:52:12  uhrig
* Removed function createBallPerceptFrom3Points (and all functions called exclusively by that function) because the ball percept can be created by the Levenberg-Marquardt algorithm, too.
*
* Revision 1.9  2004/03/28 17:08:14  risler
* added counting all scanned pixels
* added test for number of orange pixels
*
* Revision 1.8  2004/03/27 18:43:17  risler
* bugs fixed
*
* Revision 1.7  2004/03/27 17:17:00  risler
* ball perception revisited
*
* Revision 1.6  2004/03/27 14:39:37  risler
* moved scan to scanForBallPoints
*
* Revision 1.5  2004/03/26 19:43:33  mkunz
* atBorder and greenIsNear added to findEndOfBall
*
* Revision 1.4  2004/03/26 18:45:48  risler
* removed old ball recognition stuff
*
* Revision 1.3  2004/03/26 14:25:17  risler
* ball specialist drawing fixed
*
* Revision 1.2  2004/03/26 14:02:39  brunn
* added getSimilarityToOrange
*
* Revision 1.1  2004/03/26 09:12:42  risler
* added DDD2004BallSpecialist
* moved ball related stuff from ImageProcessor to BallSpecialist
*
* Revision 1.3  2003/05/20 15:30:37  risler
* removed remains of color table hack
*
* Revision 1.2  2003/05/12 00:03:29  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/04/15 17:19:43  risler
* moved DDDGO2003 ImageProcessor to own module
* removed ContinuousBasicBehaviorTester
*
* Revision 1.9  2003/04/15 15:52:09  risler
* DDD GO 2003 code integrated
*
* Revision 1.10  2003/04/10 21:45:21  max
* hack: colortable access via non virtual function
*
* Revision 1.9  2003/04/08 17:02:50  dthomas
* modified: "ghost"-balls should no more be detected as ball-percept
*
* Revision 1.8  2003/04/08 00:59:41  max
* tamara update
*
* Revision 1.8  2003/04/06 21:37:03  juengel
* Ball distance is calculated by size and not by angle, if cameraMatrix is not valid.
*
* Revision 1.7  2003/02/19 14:59:55  roefer
* pColorTable -> colorTable
*
* Revision 1.6  2003/02/18 21:29:16  osterhues
* Changed all instances of ColorTable64 to new base class ColorTable
*
* Revision 1.5  2002/09/22 18:40:51  risler
* added new math functions, removed GTMath library
*
* Revision 1.4  2002/09/22 09:15:33  risler
* Geometry.h moved to directory Math
*
* Revision 1.3  2002/09/19 23:38:53  juengel
* Changed debug image mechanisms.
*
* Revision 1.2  2002/09/17 23:55:21  loetzsch
* - unraveled several datatypes
* - changed the WATCH macro
* - completed the process restructuring
*
* Revision 1.1  2002/09/10 15:36:15  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1  2002/09/07 13:36:57  loetzsch
* unified the vision modules into one module "ImageProcessor"
* - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
*   and PlayersPerceptor were are combined to the new solution
*   "BlobImageProcessor"
* - The GridImageProcessor and the SubPixelGradientCalculator became
*   a solution of "ImageProcessor"
*
* Revision 1.5  2002/07/23 13:33:39  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.4  2002/06/08 17:01:35  juengel
* Removed close ball bug.
*
* Revision 1.3  2002/05/25 17:51:27  juengel
* Preparations for Ball Collection Challenge.
*
* Revision 1.2  2002/05/13 13:19:20  juengel
* findEndOfBall modified.
*
* Revision 1.1.1.1  2002/05/10 12:40:14  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.5  2002/05/06 17:52:35  juengel
* Changed searchEndOfFlags
*
* Revision 1.4  2002/05/03 11:52:08  juengel
* ...
*
* Revision 1.3  2002/04/03 12:50:20  juengel
* Flags and ball.
*
* Revision 1.2  2002/04/02 10:30:34  juengel
* GridImageProcessor enhanced.
*
* Revision 1.1  2002/03/16 13:43:37  juengel
* GridImageProcessor created.
*
* Revision 1.3  2002/02/03 20:42:26  juengel
* Ball perception in improved.
*
* Revision 1.2  2002/02/03 14:37:58  juengel
* Drawing of the world state removed from Berlin2001BehaviorControl.
* Drawing method for world states added to PaintMethods.
* Drawing of the world state added to the Processes with BehaviorControl.
*
* Revision 1.1  2002/01/22 14:53:30  juengel
* ImageToPerceptCollection eingefhrt
*
*
*/
