/**
* @file DDD2004ColorCorrector.cpp
*
* Implementation of class DDD2004ColorCorrector
*
* @author Max Risler
*/
#include "DDD2004ColorCorrector.h"
#include "Tools/Location.h"

DDD2004ColorCorrector::DDD2004ColorCorrector()
{
  if(getRobotConfiguration().getRobotDesign() == RobotDesign::ERS7)
    init();
  else
    disable();
}

void DDD2004ColorCorrector::init()
{
  //setup radius table
  for (int x = 0; x < cameraResolutionWidth_ERS7; x++)
    for (int y = 0; y < cameraResolutionHeight_ERS7; y++)
      radiusTable[x][y] = getRadius(x,y);

  int maxRadius = radiusTable[0][0];
  
  InBinaryFile file(getLocation().getFilename("white.img"));
  if (file.exists())
  {
    int i,c,j,x,y;
    long pixelSum[cameraResolutionWidth_ERS7][3];
    int pixelNum[cameraResolutionWidth_ERS7];

    for (i = 0; i <= maxRadius; i++)
    {
      for (j = 0; j < 3; j++)
        pixelSum[i][j] = 0;
      pixelNum[i] = 0;
    }

    do 
    {
      Image image;
      //file >> image;
      file >> image.cameraInfo.resolutionWidth 
           >> image.cameraInfo.resolutionHeight 
           >> image.frameNumber;
 	    for(y = 0; y < image.cameraInfo.resolutionHeight; ++y)
        for(c = 0; c < 3; ++c)
          file.read(&image.image[y][c][0], image.cameraInfo.resolutionWidth);

      for (x = 0; x < image.cameraInfo.resolutionWidth; x++)
        for (y = 0; y < image.cameraInfo.resolutionHeight; y++)
        {
          int r = radiusTable[x][y];
          pixelSum[r][0] += image.image[y][0][x];
          pixelSum[r][1] += image.image[y][1][x];
          pixelSum[r][2] += image.image[y][2][x];
          pixelNum[r]++;
        }
    }
    while(!file.eof());

    if (pixelNum[0] == 0)
    {
      OUTPUT(idText, text, "DDD2004ColorCorrector: white.img contains no images.");
      disable();
      return;
    }

    for (i = 0; i <= maxRadius; i++)
      for (j = 0; j < 3; j++)
      {
        pixelSum[i][j] /= pixelNum[i];
        for (c = 0; c < 256; c++)
        {
          long x = c * pixelSum[0][j] / pixelSum[i][j];
          if (x > 255) x = 255;
          correctionTable[i][c][j] = (unsigned char) x;
        }
      }

  } else {
    OUTPUT(idText, text, "DDD2004ColorCorrector: white.img not found.");
    disable();
  }
}

void DDD2004ColorCorrector::disable()
{
  for (int x = 0; x < cameraResolutionWidth_ERS7; x++)
    for (int y = 0; y < cameraResolutionHeight_ERS7; y++)
      radiusTable[x][y] = 0;
  for (int c=0; c<256; c++)
  {
    correctionTable[0][c][0] = c;
    correctionTable[0][c][1] = c;
    correctionTable[0][c][2] = c;
  }
}

/*
* Change log :
* 
* $Log: DDD2004ColorCorrector.cpp,v $
* Revision 1.6  2004/03/24 12:56:18  risler
* no message
*
* Revision 1.5  2004/03/22 19:04:47  risler
* fixed DDD ColorCorrector image reading
*
* Revision 1.4  2004/03/18 16:59:13  risler
* whiteImage.log renamed to whiteImg.log
*
* Revision 1.3  2004/03/18 16:25:47  risler
* whiteImage.log renamed to whiteImg.log
*
* Revision 1.2  2004/03/17 17:16:08  risler
* whiteImage.log moved to Location directory
*
* Revision 1.1  2004/03/17 15:31:16  risler
* added DDD2004ColorCorrector
*
*
*/
