/**
* @file GT2003ImageProcessor.h
*
* Definition of class GT2003ImageProcessor
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/

#ifndef __GT2003ImageProcessor_h_
#define __GT2003ImageProcessor_h_


#include "Modules/ImageProcessor/ImageProcessor.h"
#include "GT2003FlagSpecialist.h"
#include "../ImageProcessorTools/GoalRecognizer.h"
#include "../ImageProcessorTools/ManualCalibration.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Debugging/DebugImages.h"

/**
* @class GT2003ImageProcessor
*
* The lines image processor recognizes characteristic lines in the image.
* Four types of lines are distinguished:
* edges between the skyblue goal and the field, edges between the yellow goal 
* and the field, edges between the border and the field, and edges between the
* field lines and the field.
*
* The module scans vertical and horizontal lines in the image from top to bottom
* and from left to right. As the green of the field is very dark, all edges are
* characterized by a big difference of the y-channel of adjacent pixels. An
* increase in the y-channel followed by a decrease is an indication for an edge.
*
* The projection of the pixels on the field plane is used to determine their
* relative position to the robot.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
*/ 

class GT2003ImageProcessor : public ImageProcessor
{
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the GT2003ImageProcessor module.
  */
  GT2003ImageProcessor(const ImageProcessorInterfaces& interfaces);

  /** Executes the module */
  virtual void execute();
  virtual bool handleMessage(InMessage& message);

private:
  ManualCalibration manualCalibration;

  /**
  * The struct represents a ball point.
  */
  struct BallPoint : Vector2<int>
  {
    bool isBottom, /**< Is the point at the lower side of the ball? */
         greenIsClose;
  };

  double xFactor, /**< Factor to convert the pixel coordinate space to the anglular coordinate space. */
         yFactor; /**< Factor to convert the pixel coordinate space to the anglular coordinate space. */
  int yThreshold; /**< Brightness increase threshold. */
  int vThreshold; /**< Brightness decrease threshold. */
  Geometry::Line horizon; /**< The horizon. */
  int orangeCount,  /**< Number of columns with ball points. */
      noOrangeCount, /**< Number of columns without a ball point. */
      noRedCount, /**< Number of columns without a red robot point. */
      noBlueCount, /**< Number of columns without a blue robot point. */
      noGoalCount, /**< Number of columns without a opponent goal seen. */
      closestBottom; /**< Closest bottom point on the grid. */
  Vector2<int> firstRed, /**< First red robot point in a cluster. */
               closestRed, /**< Closest red robot point in a cluster. */
               lastRed, /**< Last red robot point in a cluster. */
               firstBlue, /**< First blue robot point in a cluster. */
               closestBlue, /**< Closest blue robot point in a cluster. */
               lastBlue, /**< Last blue robot point in a cluster. */
               firstBall, /**< First center between top and bottom of a ball. */
               lastBall, /**< Last center between top and bottom of a ball. */
               firstFlag, /**< First flag point in a cluster. */
               lastFlag; /**< Last flag point in a cluster. */
  bool goalAtBorder; /**< Is the first goal point at the image border? */
  enum {maxNumberOfPoints = 400}; /**< The maximum number of ball points. */
  BallPoint ballPoints[maxNumberOfPoints]; /**< The ball points. */
  int numberOfBallPoints; /**< The actual number of ball points. */
  int flagCount[6]; /**< Histogram for flag points. */

  CameraMatrix cmBall, /**< Camera matrix without ball radius. */
               cmTricot; /**< Camera matrix without tricot height. */

  GT2003FlagSpecialist flagSpecialist; /**< The flag specialist. */
//  GT2003GoalSpecialist goalSpecialist; /**< The goal specialist. */

  /**
  * The function scans columns for line points.
  */
  void scanColumns();

  /** 
  * The function scans rows for line points. 
  */
  void scanRows();

  /** 
  * The function scans a line for line points. 
  * @param start The start point of the line.
  * @param end The end point of the line.
  * @param vertical Vertical lines are scanned for more information.
  * @param noLines Should the line not be scanned for points on field lines or borders?
  */
  void scan(const Vector2<int>& start, const Vector2<int>& end,
            bool vertical, bool noLines);

  /** 
  * The function clusters points of balls.
  * @param topPoint The top point of the current scan column.
  * @param bottomPoint The bottom point of the current scan column.
  * @param first The upper ball point. If 0, no ball has been found.
  * @param last The lower ball point. If 0, no ball has been found.
  * @param greenAboveOrange Was there green above the top point?
  * @param greenBelowOrange Was there green below the bottom point?
  */
  void clusterBalls(const unsigned char* topPoint,
                    const unsigned char* bottomPoint,
                    const unsigned char* first,
                    const unsigned char* last,
                    bool greenAboveOrange,
                    bool greenBelowOrange);

  /**
  * The function creates a ball percept from the ball points stored in "ballPoints".
  * It replaces the previous percept if the new ball is larger than the larger one 
  * found so far.
  */
  void createBallPercept();

  /**
  * The function filters outlier points.
  */
  void filterBallPoints();

  /**
  * The functions tries to calculate the ball percept by calculating the center of the ball.
  * @param onlyIfGreenIsClose Use only points that are close to green.
  * @return Has a percept been generated?
  */
  bool createBallPerceptFrom3Points(bool onlyIfGreenIsClose);

  /**
  * The function selects three points from the array "ballPoints".
  * The points will be used to determine the center of the ball.
  * @param point1 The index of the first point is returned to this variable.
  * @param point2 The index of the second point is returned to this variable.
  * @param point3 The index of the third point is returned to this variable.
  * @param onlyIfGreenIsClose Use only points that are close to green.
  * @return Were there enough adequate points?
  */
  bool select3Points(int& point1, int& point2, int& point3, bool onlyIfGreenIsClose) const;

  /**
  * The function determines the center of the ball by cutting the middle perpendiculars.
  * @param v1 The first point on the border of the ball.
  * @param v2 The second point on the border of the ball.
  * @param v3 The third point on the border of the ball.
  * @return The center of the ball.
  */
  Vector2<int> cutMiddlePerpendiculars(Vector2<int>& v1,
                                       Vector2<int>& v2,
                                       Vector2<int>& v3) const;

  /**
  * The function checks whether a ball percept is plausible and will add it if so.
  * @param centerX The x-coordinate of the center of the ball.
  * @param centerY The y-coordinate of the center of the ball.
  * @param radius The radius of the ball.
  */
  void addBallPercept(
    int centerX,
    int centerY,
    double radius
    );  


  /**
  * The function checks whether a ball percept is plausible.
  * @param offset The offset of the ball.
  * @return Is enough orange inside the ball? 
  */
  bool checkOrangeInsideBall(const Vector2<double>& offset);

  /**
  * The function checks whether a ball percept is plausible.
  * @param circle A circle in the image.
  * @return Is enough orange inside the ball? 
  */
  bool checkOrangeInsideBall(const Geometry::Circle& circle);

  
  /** 
  * The function clusters points of red and blue robots.
  * @param bottomPoint The bottom point of the current scan column.
  * @param redFound Has a red robot point been found? In that case, the last
  *                 entry in the lines percept is that point.
  * @param blueFound Has a blue robot point been found? In that case, the last
  *                  entry in the lines percept is that point.
  */
  void clusterRobots(const unsigned char* bottomPoint, bool redFound, bool blueFound);

  /** 
  * The function clusters points of the opponent's goal.
  */
  void clusterGoal();

  /** 
  * The function adds or replaces the direction to the free part of the opponent's goal.
  */
  void addGoal(const Pose2D& camera, 
               const Vector2<double>& first, const Vector2<double>& last);

  /** 
  * The function clusters flag points.
  * @param flagType The type of the flag.
  * @param point The center of the pink area.
  */
  void clusterFlags(Flag::FlagType flagType, const Vector2<int>& point);

  /**
  * The function filters the percepts, i.e. it removes potential misreadings.
  */
  void filterPercepts();

  /** 
  * The function converts an address to pixel coordinates.
  * @param p An address in image.image.
  * @return The x- and y-coordinates of the corresponding pixel.
  */
  Vector2<int> getCoords(const unsigned char* p) const;

  //!@name Helpers for grid drawing
  //!@{
  const unsigned char* last;
  Drawings::Color lineColor;
  void plot(const unsigned char* p,Drawings::Color color);
  //!@}

  //obstacle buffers
/*  Vector2<int> obstaclesNearPointOnField2;
  bool obstaclesFarPointOnField2IsOnImageBorder;
  
  Vector2<int> obstaclesFarPointOnField1;
  Vector2<int> obstaclesFarPointOnField2;
*/
  double angleBetweenDirectionOfViewAndGround;

  int numberOfScannedPixels;

  DECLARE_DEBUG_IMAGE(imageProcessorPlayers);
};
#endif// __GT2003ImageProcessor_h_

/*
* Change log :
* 
* $Log: GT2003ImageProcessor.h,v $
* Revision 1.13  2004/04/18 11:58:24  nistico
* Removed CT32K_LAYOUT
*
* Revision 1.12  2004/04/09 18:51:41  roefer
* No ColorTable32k
*
* Revision 1.11  2004/04/08 15:33:06  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.10  2004/03/12 17:22:26  nistico
* - Fixed a problem in CT32K_LAYOUT with localization
*
* Revision 1.9  2004/03/08 01:38:59  roefer
* Interfaces should be const
*
* Revision 1.8  2004/03/04 17:51:18  nistico
* -Bug with MSH2004ColorCorrection fixed
*
* Revision 1.7  2004/03/03 17:18:43  nistico
* -Introduced fast ColorTable32K access through conditional compilation
* For Dortmund: uncomment (define) CT32K_LAYOUT in GT2003ImageProcessor.h
* For other universities: leave undefined :-)
*
* Revision 1.6  2004/03/01 11:56:44  juengel
* Recognition of obstacle types added.
*
* Revision 1.5  2003/12/04 17:36:59  juengel
* comment corrected
*
* Revision 1.4  2003/12/01 16:26:13  juengel
* Added calculation of "pixel usage" goal pixels are not counted yet.
*
* Revision 1.3  2003/11/15 17:09:02  juengel
* changed BallPercept
*
* Revision 1.2  2003/10/12 20:20:15  juengel
* ColorTable is not const anymore.
* Added manual calibration.
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.6  2003/09/01 10:05:08  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.5  2003/08/18 12:07:49  juengel
* Moved method calculateLineSize to class Geometry.
*
* Revision 1.4  2003/08/08 11:25:59  dueffert
* doxygen docu corrected
*
* Revision 1.3  2003/07/29 12:48:35  juengel
* Moved calculateHorizon, getDistanceBySize and getSizeByDistance to Geometry.
*
* Revision 1.2  2003/07/21 13:42:21  juengel
* Moved ColorTableAuto and GoalRecognizer to ImageProcessorTools.
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.10  2003/06/27 11:18:12  juengel
* Distance to a ball above the horizon is calculated by size.
*
* Revision 1.9  2003/06/25 18:43:01  juengel
* Added nearPoints, farPoints[maxNumberOfPoints] and farPointIsOnImageBorder to ObstaclesPercept.
*
* Revision 1.8  2003/06/22 17:09:09  juengel
* Added filter for ObstaclesPercept.
*
* Revision 1.7  2003/06/20 21:12:45  roefer
* If camera matrix is invalid, ball distance is calculated from size
*
* Revision 1.6  2003/06/12 16:57:59  juengel
* Added GoalRecognizer, removed GoalSpecialist.
*
* Revision 1.5  2003/05/07 16:07:50  roefer
* Major ball recognition bug fixed
*
* Revision 1.4  2003/05/04 21:48:46  roefer
* Filtering ball point outliers
*
* Revision 1.3  2003/05/04 17:31:52  roefer
* Flag and goal specialists added to GT2003IP
*
* Revision 1.2  2003/05/03 18:09:55  roefer
* Ball sampling only for small balls
*
* Revision 1.1  2003/05/01 13:24:24  roefer
* Copied LinesImageProcessor2 to GT2003ImageProcessor
*
*
*/
