/**
* @file GridImageProcessor2.cpp
* 
* This file contains a class for Image Processing.
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
*/

#include "GridImageProcessor2.h"
#include "../ImageProcessorTools/GoalRecognizer.h"

#include "Tools/FieldDimensions.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"
#include "Tools/Math/Histogram.h"
#include "Tools/RingBuffer.h"
#include "Tools/Debugging/GenericDebugData.h"

#include "Representations/Perception/ColorTable64.h"

#include "Platform/SystemCall.h"
#include "Platform/GTAssert.h"

GridImageProcessor2::GridImageProcessor2(const ImageProcessorInterfaces& interfaces)
: 
ImageProcessor(interfaces), 
groundDetector(
               cameraMatrix, 
               image.cameraInfo, 
               lastColorSpaceUsageCounter, 
               averageIntensityOverLastImages, 
               colorTable64, colorTableCuboids, colorTableReferenceColor, 
               horizonLine, 
               goalIndicationAboveHorizon, goalIndicationBelowHorizon,
               foundGoal,
               obstaclesPercept, linesPercept),
               
               ballDetector(image, cameraMatrix, image.cameraInfo, colorTable64, colorTableCuboids, colorTableReferenceColor, ballPercept),
               
               orangeCalibration(cameraMatrix, image.cameraInfo, colorTable64, colorTableCuboids, colorTableReferenceColor),
               
//               manualCalibration(image, cameraMatrix, colorTable64, calibrationRequest),
               
               lastColorSpaceUsageCounter(colorSpaceUsageCounter1),
               
               currentColorSpaceUsageCounter(colorSpaceUsageCounter2)

//               lineSegment(image.cameraInfo)
{
  //  PRINT("GIP2 Construction start");

  //debug parameters
  returnCondition = 5;

  InTextFile parametersFile("gi2param.cfg");
  if (parametersFile.exists())
  {
    parametersFile >> parameters;
  }
  else { OUTPUT(idText, text, "GridImageProcessor2DlgBar could not load gi2param.cfg"); }

  //  colorTable64 = (ColorTable64&)colorTable;
  //  PRINT("GIP2 Construction end");

  averageIntensity[0].init();
}

GridImageProcessor2::~GridImageProcessor2()
{
}

void GridImageProcessor2::execute()
{
  INIT_DEBUG_IMAGE(imageProcessorScanLines, image);
  INIT_DEBUG_IMAGE(imageProcessorBall, image);
  Image emptyImage;
  /*
  for(int x = 0; x < cameraResolutionWidth; x++)
  {
  for(int y = 0; y < cameraResolutionHeight; y++)
  {
  emptyImage.image[y][0][x] = 255;
  emptyImage.image[y][1][x] = 127;
  emptyImage.image[y][2][x] = 127;
  }
  }
  */

  INIT_DEBUG_IMAGE(imageProcessorGeneral, emptyImage);

  INIT_DEBUG_IMAGE(classificationY, emptyImage);
  INIT_DEBUG_IMAGE(classificationU, emptyImage);
  INIT_DEBUG_IMAGE(classificationV, emptyImage);

  if(parameters.resetAutoCalibration)
  {
    colorTable64.clear();
    colorTableCuboids.clear();
    colorTableReferenceColor.clear();
  }
  if(parameters.setColorTableToDefault)
  {
    colorTable64 = (ColorTable64&)colorTable;
  }
  if(parameters.useLongScanLinesOnly)
  {
    parameters.desiredNumberOfScanLines = image.cameraInfo.resolutionWidth;
  }
  else parameters.desiredNumberOfScanLines = image.cameraInfo.resolutionWidth;


  // initialisation

  ballPercept.reset(image.frameNumber);
  linesPercept.reset(image.frameNumber);
  landmarksPercept.reset(image.frameNumber);
  playersPercept.reset(image.frameNumber);
  obstaclesPercept.reset(image.frameNumber);


  groundDetector.init();
  ballDetector.init();
  orangeCalibration.init();

  ColorSpaceUsageCounter& buf = lastColorSpaceUsageCounter;
  lastColorSpaceUsageCounter = currentColorSpaceUsageCounter;
  currentColorSpaceUsageCounter = buf;

  currentColorSpaceUsageCounter.reset();

  accumulatedIntensity[0] = 0;
  accumulatedIntensity[1] = 0;
  accumulatedIntensity[2] = 0;
  numberOfScannedPixelsOnLongLines = 0;
  numberOfScannedPixels = 0;

  accumulatedEdgeness[0] = 0;
  accumulatedEdgeness[1] = 0;
  accumulatedEdgeness[2] = 0;


  // manual color calibration
//  manualCalibration.execute();

  // calculate the horizon
  horizonLine = Geometry::calculateHorizon(cameraMatrix, image.cameraInfo);

  LINE(imageProcessor_horizon,
    (int)(horizonLine.base.x - 500 * horizonLine.direction.x), (int)(horizonLine.base.y - 500 * horizonLine.direction.y), 
    (int)(horizonLine.base.x + 500 * horizonLine.direction.x), (int)(horizonLine.base.y + 500 * horizonLine.direction.y), 
    1, Drawings::ps_solid, Drawings::white );

  // The use of a fixed horizon is useful for visualzation
  if(parameters.useFixedHorizon)
  {
    horizonLine.base.x = image.cameraInfo.resolutionWidth / 2;
    horizonLine.base.y = 20;
    horizonLine.direction.x = 1;
    horizonLine.direction.y = 0;
  }

  // calculate the scan lines
  verticalLine.base = horizonLine.base;
  verticalLine.direction.x = - horizonLine.direction.y;
  verticalLine.direction.y = horizonLine.direction.x;

  numberOfScanLines = calculateVerticalScanLines();

  // scan the image
  //scanVerticalLines();


  INFO(printPixelUsage, idText, text, "used pixels: " << 100 * numberOfScannedPixels / (image.cameraInfo.resolutionWidth * image.cameraInfo.resolutionHeight) << " %");

  if(numberOfScannedPixelsOnLongLines != 0)
  {
    averageIntensity[0].add(accumulatedIntensity[0] / numberOfScannedPixelsOnLongLines);
    averageIntensity[1].add(accumulatedIntensity[1] / numberOfScannedPixelsOnLongLines);
    averageIntensity[2].add(accumulatedIntensity[2] / numberOfScannedPixelsOnLongLines);

    averageIntensityOverLastImages.x = averageIntensity[0].getSum() / averageIntensity[0].getNumberOfEntries();
    averageIntensityOverLastImages.y = averageIntensity[1].getSum() / averageIntensity[1].getNumberOfEntries();
    averageIntensityOverLastImages.z = averageIntensity[2].getSum() / averageIntensity[2].getNumberOfEntries();

    colorTableCuboids.setThresholds(white, 
      min(
      (averageIntensityOverLastImages.x + 255 + 255 ) / 3, 
      averageIntensityOverLastImages.x * 2
      ),
      0, 0, 255, 255, 255);

    colorTableCuboids.setThresholds(green, 
      (int)(averageIntensityOverLastImages.x * 0.8), 
      (int)(averageIntensityOverLastImages.y * 0.9), 
      (int)(averageIntensityOverLastImages.z * 0.9), 
      (int)(averageIntensityOverLastImages.x * 1.2), 
      (int)(averageIntensityOverLastImages.y * 1.1), 
      (int)(averageIntensityOverLastImages.z * 1.1)
      );

    averageEdgeness[0].add(accumulatedEdgeness[0] / numberOfScannedPixelsOnLongLines);
    averageEdgeness[1].add(accumulatedEdgeness[1] / numberOfScannedPixelsOnLongLines);
    averageEdgeness[2].add(accumulatedEdgeness[2] / numberOfScannedPixelsOnLongLines);

    averageEdgenessOverLastImages.x = averageEdgeness[0].getSum() / averageEdgeness[0].getNumberOfEntries();
    averageEdgenessOverLastImages.y = averageEdgeness[1].getSum() / averageEdgeness[1].getNumberOfEntries();
    averageEdgenessOverLastImages.z = averageEdgeness[2].getSum() / averageEdgeness[2].getNumberOfEntries();

    if(parameters.enableThresholdCalibration)
    {
      parameters.threshold[0] = averageIntensityOverLastImages.x / 5;
    }
  }
  // analyse the results of the scan
  ballDetector.createBallPercept();
  orangeCalibration.calibrateOrange();
  groundDetector.createObstaclesPercept();
  groundDetector.createGoalIndication();

  if(groundDetector.lightingHasChanged())
  {
    colorTable64.clear();
    colorTableCuboids.clear();
    colorTableReferenceColor.clear();
  }

  if(foundGoal)
  {
    GoalRecognizer goalRecognizer(
      image, cameraMatrix, colorTable64, 
      goalIndicationAboveHorizon, goalIndicationBelowHorizon, 
      obstaclesPercept, landmarksPercept);
    goalRecognizer.getGoalPercept(landmarksPercept);
  }
  INFO(sendColorTable64, idColorTable64, bin, colorTable64);
  INFO(sendColorTableCuboids, idColorTableCuboids, bin, colorTableCuboids);
  INFO(sendColorTableReferenceColor, idColorTableReferenceColor, bin, colorTableReferenceColor);

  GENERATE_DEBUG_IMAGE(segmentedImage1,
    colorTableCuboids.generateColorClassImage(image, segmentedImage1ColorClassImage);  );
  
  GENERATE_DEBUG_IMAGE(segmentedImage2,
    colorTableReferenceColor.generateColorClassImage(image, segmentedImage2ColorClassImage);  );

  GENERATE_DEBUG_IMAGE(segmentedImage3,
    colorTable64.generateColorClassImage(image, segmentedImage3ColorClassImage);  );

#if defined(_WIN32) && !defined(SIMROBOT)
  visualize();
#endif

  SEND_DEBUG_COLOR_CLASS_IMAGE(segmentedImage1);
  SEND_DEBUG_COLOR_CLASS_IMAGE(segmentedImage2);
  SEND_DEBUG_COLOR_CLASS_IMAGE(segmentedImage3);
  SEND_DEBUG_IMAGE(imageProcessorGeneral);
  SEND_DEBUG_IMAGE(imageProcessorScanLines);
  SEND_DEBUG_IMAGE(imageProcessorGradients);
  SEND_DEBUG_IMAGE(imageProcessorBall);
  SEND_DEBUG_IMAGE(classificationY);
  SEND_DEBUG_IMAGE(classificationU);
  SEND_DEBUG_IMAGE(classificationV);
  SEND_DEBUG_IMAGE(colorFrequency);
  DEBUG_DRAWING_FINISHED(imageProcessor_general);
  DEBUG_DRAWING_FINISHED(imageProcessor_horizon);
  DEBUG_DRAWING_FINISHED(imageProcessor_scanLines);
  DEBUG_DRAWING_FINISHED(imageProcessor_coloredSegments1);
  DEBUG_DRAWING_FINISHED(imageProcessor_coloredSegments2);
  DEBUG_DRAWING_FINISHED(imageProcessor_coloredSegments3);
  DEBUG_DRAWING_FINISHED(imageProcessor_calibration1);
  DEBUG_DRAWING_FINISHED(imageProcessor_calibration2);
  DEBUG_DRAWING_FINISHED(imageProcessor_flagsAndGoals);
  DEBUG_DRAWING_FINISHED(imageProcessor_obstacles);
  DEBUG_DRAWING_FINISHED(imageProcessor_ground);
  DEBUG_DRAWING_FINISHED(imageProcessor_gradients);
  DEBUG_DRAWING_FINISHED(sketch);
}

int GridImageProcessor2::calculateVerticalScanLines()
{
  int currentIndex = 0;
  int currentLongLineIndex = 0;

  Vector2<double> intersection;

  Vector2<int> topPoint, bottomPoint;

  Geometry::Line lineAboveHorizon[2], lineBelowHorizon;
  Geometry::Line scanLine;

  scanLine.direction = verticalLine.direction;

  // upper boundary for scan lines
  lineAboveHorizon[0].direction = horizonLine.direction;
  lineAboveHorizon[0].base = horizonLine.base - verticalLine.direction * 20;

  lineAboveHorizon[1].direction = horizonLine.direction;
  lineAboveHorizon[1].base = horizonLine.base - verticalLine.direction * 5;

  // lower boundaries for scan lines
  lineBelowHorizon.direction = horizonLine.direction;

  // calculate and scan lines
  int i;
  for(i = 0; i < parameters.desiredNumberOfScanLines; i++)
  {
    lineAboveHorizon[1].base = 
      horizonLine.base + 
      scanLine.direction * 
      (- groundDetector.getDistanceFromFieldBorderToHorizon() - 15);
    lineBelowHorizon.base = 
      horizonLine.base + 
      scanLine.direction * 
      (- groundDetector.getDistanceFromFieldBorderToHorizon() + 10 + 10 * (i % 2));

    scanLine.base.x = image.cameraInfo.resolutionWidth / 2 + (i - parameters.desiredNumberOfScanLines / 2) * horizonLine.direction.x * 1;
    scanLine.base.y = image.cameraInfo.resolutionHeight / 2 + (i - parameters.desiredNumberOfScanLines / 2) * horizonLine.direction.y * 1;

    //Does the scan line intersect with the image?
    if(Geometry::getIntersectionPointsOfLineAndRectangle(
      Vector2<int>(0,0),
      Vector2<int>(image.cameraInfo.resolutionWidth - 1,image.cameraInfo.resolutionHeight - 1),
      scanLine, topPoint, bottomPoint) )
    {
      //      int lineType = i & 3; // different lengths of scan lines
      if(i%parameters.scanLineSpacing == 0 || parameters.useLongScanLinesOnly)
      {
        Geometry::getIntersectionOfLines(lineAboveHorizon[0], scanLine, intersection);
      }
      else
      {
        Geometry::getIntersectionOfLines(lineAboveHorizon[1], scanLine, intersection);
      }

      //Is the bottomPoint below the horizon?
      if(scanLine.direction.y / (intersection.y - bottomPoint.y) < 0)
      {
        //Is the intersection with the horizon point inside the image?
        if(intersection.x < image.cameraInfo.resolutionWidth && intersection.x >= 0 &&
          intersection.y < image.cameraInfo.resolutionHeight && intersection.y >= 0) 
        {
          topPoint.x = (int)intersection.x;
          topPoint.y = (int)intersection.y;
        }
        if(i%parameters.scanLineSpacing == 0 || parameters.useLongScanLinesOnly)         //        int lineType = 0;
        {
          //scanVertical(currentLongLineIndex++, true, topPoint, bottomPoint);
          scanVertical2(currentLongLineIndex++, topPoint, bottomPoint);
          scanVertical3(topPoint, bottomPoint);
          DOT(imageProcessor_general, bottomPoint.x, bottomPoint.y, Drawings::red, Drawings::red);
          DOT(imageProcessor_general, topPoint.x, topPoint.y, Drawings::blue, Drawings::white);
        }
        else
        {
          Geometry::getIntersectionOfLines(lineBelowHorizon, scanLine, intersection);
          //Is the intersection with the horizon point inside the image?
          if(intersection.x < image.cameraInfo.resolutionWidth && intersection.x >= 0 &&
            intersection.y < image.cameraInfo.resolutionHeight && intersection.y >= 0) 
          {
            bottomPoint.x = (int)intersection.x;
            bottomPoint.y = (int)intersection.y;
            //scanVertical(currentIndex, false, topPoint, bottomPoint);
            scanVertical3(topPoint, bottomPoint);
            DOT(imageProcessor_general, bottomPoint.x, bottomPoint.y, Drawings::red, Drawings::red);
            DOT(imageProcessor_general, topPoint.x, topPoint.y, Drawings::blue, Drawings::blue);
          }
        }
        currentIndex++;
      }//Is the bottomPoint below the horizon?
    }//Does the scan line intersect with the image?
  }//for(int i = 0; i < parameters.desiredNumberOfScanLines; i++)
  return currentIndex;
}// calculateVerticalScanLines()

void GridImageProcessor2::scanVertical2
(
 int longLineIndex,
 const Vector2<int>& start, const Vector2<int>& end
 )
{
  LINE(imageProcessor_scanLines, start.x, start.y, end.x, end.y,
  0, Drawings::ps_solid, Drawings::white);

  LineSegmentation lineSegmentation(start, end, image, 
    parameters.threshold[0], parameters.threshold[1], parameters.threshold[2], currentColorSpaceUsageCounter);

  numberOfScannedPixelsOnLongLines += lineSegmentation.getNumberOfScannedPixels();
  accumulatedIntensity[0] += lineSegmentation.getAccumulatedIntensity(0);
  accumulatedIntensity[1] += lineSegmentation.getAccumulatedIntensity(1);
  accumulatedIntensity[2] += lineSegmentation.getAccumulatedIntensity(2);

  for(int i = 0; i < lineSegmentation.getNumberOfSegments(); i++)
  {
    LineSegment& currentSegment = lineSegmentation.getSegment(i);
    currentSegment.beginIsBelowHorizon = 
      Geometry::getDistanceToLine(horizonLine, Vector2<double>(currentSegment.begin.x, currentSegment.begin.y)) < 0;
    currentSegment.endIsBelowHorizon = 
      Geometry::getDistanceToLine(horizonLine, Vector2<double>(currentSegment.end.x, currentSegment.end.y)) < 0;

    groundDetector.addSegment(currentSegment, longLineIndex);
    ballDetector.addSegment(currentSegment, longLineIndex);
    orangeCalibration.addSegment(currentSegment, longLineIndex);

    drawSegment(currentSegment);
  }
}

void GridImageProcessor2::scanVertical3
(
 const Vector2<int>& start, const Vector2<int>& end
 )
{
  LINE(imageProcessor_scanLines, start.x, start.y, end.x, end.y,
    0, Drawings::ps_solid, Drawings::white);

  int x = 0, y = 0; //the current position in the image
  int lastX, lastY;

  colorClass color;
  RingBuffer<colorClass, 5> colorOfRun;
  RingBuffer<int, 5> lengthOfRun;
  RingBuffer<Vector2<int>, 5> beginOfRun;
  RingBuffer<Vector2<int>, 5> endOfRun;

  int lengthOfCurrentRun = 0;

  for(int i = 0; i < 5; i++)
  {
    colorOfRun.add(noColor);
    lengthOfRun.add(0);
    beginOfRun.add(Vector2<int>(0,0));
    endOfRun.add(Vector2<int>(0,0));
  }
  int pixelsSinceLastOrange = 0;
  Vector2<int> lastOrange;

  Geometry::PixeledLine line(start, end);
  if(line.getNumberOfPixels() > 3)
  {
    //for all pixels of the current line
    for(int z = 2; z < line.getNumberOfPixels(); z++) 
    {
      numberOfScannedPixels++;

      x = line.getPixelX(z); y = line.getPixelY(z);
      lastX = x; lastY = y;
      DEBUG_IMAGE_SET_PIXEL_Y(imageProcessorScanLines, x, y, 127);

      // color class based scan for ball
      color = colorTable64.getColorClass(image.image[y][0][x], image.image[y][1][x], image.image[y][2][x]);

      // a new run starts if the color changes ...
      lengthOfCurrentRun++;
      if(color != colorOfRun[0] || z == line.getNumberOfPixels() - 1) 
      {
        // ... except the current run is orange and the new pixel is black
        if(color == orange) pixelsSinceLastOrange = 0;
        if(colorOfRun[0] == orange && pixelsSinceLastOrange == 0) lastOrange = Vector2<int>(lastX, lastY);
        if(color == noColor && colorOfRun[0] == orange) pixelsSinceLastOrange++;
        else
        {
          colorOfRun.add(color);
          beginOfRun.add(Vector2<int>(x,y));
          endOfRun.add(Vector2<int>(x,y));
          lengthOfRun.add(0);
          endOfRun[1].x = lastX;
          endOfRun[1].y = lastY;
          lengthOfRun[1] = lengthOfCurrentRun;
          lengthOfCurrentRun = 0;

          int lengthOfOrange = -2;
          if(colorOfRun[2] == orange) 
            lengthOfOrange = (int)Geometry::distance(beginOfRun[2], endOfRun[2]);

          if(
            pixelsSinceLastOrange <= lengthOfOrange - pixelsSinceLastOrange &&
            ( 
            (colorOfRun[1] == green && lengthOfRun[1] >= 1
            || 
            colorOfRun[1] == white && lengthOfRun[1] <= lengthOfRun[2]
            )
              )
              )
            {
              Vector2<int>ball1, ball2;
              if(
                colorOfRun[3] == noColor && colorOfRun[4] == white && lengthOfRun[3] - 1 <= lengthOfRun[4] || 
                colorOfRun[3] == noColor && colorOfRun[4] == green && lengthOfRun[3] - 1 <= lengthOfRun[4] ||
                colorOfRun[3] == white
                )
              {
                ball1 = beginOfRun[2]; //begin of orange
                ball2 = lastOrange; //endOfRun[1]; //end of orange
                ballDetector.addBallPoints(ball1, ball2);
                DEBUG_IMAGE_SET_PIXEL_RED(imageProcessorBall, ball1.x, ball1.y);
                DEBUG_IMAGE_SET_PIXEL_RED(imageProcessorBall, ball2.x, ball2.y);
              }
            }//colorOfRun[0] == green && colorOfRun[1] == orange)
        }//else
      }// new run

      GENERATE_DEBUG_IMAGE(imageProcessorGeneral, 
        if(colorOfRun[0] == orange) { DEBUG_IMAGE_SET_PIXEL_ORANGE(imageProcessorGeneral, x, y); }
        if(colorOfRun[0] == green) {DEBUG_IMAGE_SET_PIXEL_GREEN(imageProcessorGeneral, x, y); }
        if(colorOfRun[0] == white) {DEBUG_IMAGE_SET_PIXEL_WHITE(imageProcessorGeneral, x, y); } ); 
    }//for(int z = 2; z < line.getNumberOfPixels(); z++)
  }//if(line.getNumberOfPixels() > 3)
}

void GridImageProcessor2::scanVertical
(
 int longLineIndex, bool lineIsLong,
 const Vector2<int>& start, const Vector2<int>& end
 )
{
  COMPLEX_DRAWING(imageProcessor_scanLines,
    if(lineIsLong)
    {
      LINE(imageProcessor_scanLines, start.x, start.y, end.x, end.y,
        0, Drawings::ps_solid, Drawings::white);
    }
    else
    {
      LINE(imageProcessor_scanLines, start.x, start.y, end.x, end.y,
        0, Drawings::ps_solid, Drawings::gray);
    }
    );

    int x = 0, y = 0; //the current position in the image
    int lastX, lastY;

    colorClass color;

    RingBuffer<colorClass, 5> colorOfRun;
    RingBuffer<int, 5> lengthOfRun;
    RingBuffer<Vector2<int>, 5> beginOfRun;
    RingBuffer<Vector2<int>, 5> endOfRun;

    int lengthOfCurrentRun = 0;

    for(int i = 0; i < 5; i++)
    {
      colorOfRun.add(noColor);
      lengthOfRun.add(0);
      beginOfRun.add(Vector2<int>(0,0));
      endOfRun.add(Vector2<int>(0,0));
    }
    int pixelsSinceLastOrange = 0;
    Vector2<int> lastOrange;

    // for segmentation  
    RingBuffer<int,3> intensity[3]; //a buffer for the intensities of the three channels
    int changeY, changeU, changeV; //indicates the type of an edge:  1: up,   0: no edge,   -1: down

    enum {onEdge, betweenEdges} edgeState = betweenEdges; // states for edge detection
    int numberOfPixelsSinceLastEdge;

    enum {up, down, nothing} changeYState = nothing;
    Vector2<int> topOfRoof;

    Geometry::PixeledLine line(start, end);
    if(line.getNumberOfPixels() > 3)
    {
      // buffer the first two pixels
      x = line.getPixelX(0); y = line.getPixelY(0);
      intensity[0].add(image.image[y][0][x]);
      intensity[1].add(image.image[y][1][x]);
      intensity[2].add(image.image[y][2][x]);
      x = line.getPixelX(1); y = line.getPixelY(1);
      intensity[0].add(image.image[y][0][x]);
      intensity[1].add(image.image[y][1][x]);
      intensity[2].add(image.image[y][2][x]);

      // init local statistics
      numberOfPixelsSinceLastEdge = 1;
      Vector2<int> firstPixelAfterLastEdge(line.getPixelX(2), line.getPixelY(2));
      bool firstPixelAfterLastEdgeWasBelowHorizon =
        Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < -5;

      int accumulatedIntensitySinceLastChange[3] = {0,0,0};
      int minIntensitySinceLastChange[3] = {255, 255, 255};
      int maxIntensitySinceLastChange[3] = {0, 0, 0};

      //for all pixels of the current line
      for(int z = 2; z < line.getNumberOfPixels(); z++) 
      {
        numberOfScannedPixels++;

        x = line.getPixelX(z); y = line.getPixelY(z);
        lastX = x; lastY = y;
        DEBUG_IMAGE_SET_PIXEL_Y(imageProcessorScanLines, x, y, 127);

        // color class based scan for ball
        color = colorTable64.getColorClass(image.image[y][0][x], image.image[y][1][x], image.image[y][2][x]);

        // a new run starts if the color changes ...
        lengthOfCurrentRun++;
        if(color != colorOfRun[0] || z == line.getNumberOfPixels() - 1) 
        {
          // ... except the current run is orange and the new pixel is black
          if(color == orange) pixelsSinceLastOrange = 0;
          if(colorOfRun[0] == orange && pixelsSinceLastOrange == 0) lastOrange = Vector2<int>(lastX, lastY);
          if(color == noColor && colorOfRun[0] == orange) pixelsSinceLastOrange++;
          else
          {
            colorOfRun.add(color);
            beginOfRun.add(Vector2<int>(x,y));
            endOfRun.add(Vector2<int>(x,y));
            lengthOfRun.add(0);
            endOfRun[1].x = lastX;
            endOfRun[1].y = lastY;
            lengthOfRun[1] = lengthOfCurrentRun;
            lengthOfCurrentRun = 0;

            int lengthOfOrange = -2;
            if(colorOfRun[2] == orange) 
              lengthOfOrange = (int)Geometry::distance(beginOfRun[2], endOfRun[2]);

            if(
              pixelsSinceLastOrange <= lengthOfOrange - pixelsSinceLastOrange &&
              ( 
              (colorOfRun[1] == green && lengthOfRun[1] >= 1
              || 
              colorOfRun[1] == white && lengthOfRun[1] <= lengthOfRun[2]
              )
                )
                )
              {
                Vector2<int>ball1, ball2;
                if(
                  colorOfRun[3] == noColor && colorOfRun[4] == white && lengthOfRun[3] - 1 <= lengthOfRun[4] || 
                  colorOfRun[3] == noColor && colorOfRun[4] == green && lengthOfRun[3] - 1 <= lengthOfRun[4] ||
                  colorOfRun[3] == white
                  )
                {
                  ball1 = beginOfRun[2]; //begin of orange
                  ball2 = lastOrange; //endOfRun[1]; //end of orange
                  ballDetector.addBallPoints(ball1, ball2);
                  DEBUG_IMAGE_SET_PIXEL_RED(imageProcessorBall, ball1.x, ball1.y);
                  DEBUG_IMAGE_SET_PIXEL_RED(imageProcessorBall, ball2.x, ball2.y);
                }
              }//colorOfRun[0] == green && colorOfRun[1] == orange)
          }//else
        }// new run

        GENERATE_DEBUG_IMAGE(imageProcessorGeneral, 
          if(colorOfRun[0] == orange) { DEBUG_IMAGE_SET_PIXEL_ORANGE(imageProcessorGeneral, x, y); }
          if(colorOfRun[0] == green) {DEBUG_IMAGE_SET_PIXEL_GREEN(imageProcessorGeneral, x, y); }
          if(colorOfRun[0] == white) {DEBUG_IMAGE_SET_PIXEL_WHITE(imageProcessorGeneral, x, y); } ); 

          if(lineIsLong)
          {
            // segmentation of scan lines
            intensity[0].add(image.image[y][0][x]);
            intensity[1].add(image.image[y][1][x]);
            intensity[2].add(image.image[y][2][x]);

            // global statistics
            currentColorSpaceUsageCounter.addColor(image.image[y][0][x],image.image[y][1][x],image.image[y][2][x]);
            accumulatedIntensity[0] += intensity[0][0]; 
            accumulatedIntensity[1] += intensity[1][0]; 
            accumulatedIntensity[2] += intensity[2][0]; 
            numberOfScannedPixelsOnLongLines++;

            //detect edges
            int differenceY = intensity[0][0] - intensity[0][2];
            int differenceU = intensity[1][0] - intensity[1][2];
            int differenceV = intensity[2][0] - intensity[2][2];

            if(differenceY > parameters.threshold[0])       { changeY = 1; DOT(imageProcessor_gradients, x-1, y, Drawings::gray, Drawings::gray); }
            else if(differenceY < -parameters.threshold[0]) { changeY = -1;DOT(imageProcessor_gradients, x-1, y, Drawings::black, Drawings::black); }
            else                                              changeY = 0;

            if(differenceU > parameters.threshold[1])       { changeU = 1; DOT(imageProcessor_gradients, x, y, Drawings::pink, Drawings::pink); }
            else if(differenceU < -parameters.threshold[1]) { changeU = -1;DOT(imageProcessor_gradients, x, y, Drawings::red, Drawings::red);}
            else                                              changeU = 0;

            if(differenceV > parameters.threshold[2])       { changeV = 1; DOT(imageProcessor_gradients, x+1, y, Drawings::skyblue, Drawings::skyblue);}
            else if(differenceV < -parameters.threshold[2]) { changeV = -1;DOT(imageProcessor_gradients, x+1, y, Drawings::blue, Drawings::blue); }
            else                                              changeV = 0;

            accumulatedEdgeness[0] += abs(differenceY);
            accumulatedEdgeness[1] += abs(differenceU);
            accumulatedEdgeness[2] += abs(differenceV);

#if defined(_WIN32) && !defined(SIMROBOT)
            if(differenceY > parameters.threshold[0])
              classificationYImage.image[y][0][x] = 255;
            else classificationYImage.image[y][0][x] = 0;
            classificationYImage.image[y][1][x] = 127; 
            classificationYImage.image[y][2][x] = 127;

            if(differenceU > parameters.threshold[1])
              classificationUImage.image[y][0][x] = 255;
            else classificationUImage.image[y][0][x] = 0;
            classificationUImage.image[y][1][x] = 127; 
            classificationUImage.image[y][2][x] = 127;

            if(differenceV > parameters.threshold[2])
              classificationVImage.image[y][0][x] = 255;
            else classificationVImage.image[y][0][x] = 0;
            classificationVImage.image[y][1][x] = 127; 
            classificationVImage.image[y][2][x] = 127;
#endif      
            // edge state machine
            if(edgeState == betweenEdges)
            {
              if(changeY != 0 || changeU != 0 || changeV != 0)
              {
                //create and add the segment
                LineSegment newSegment(
                  firstPixelAfterLastEdge, x, y, numberOfPixelsSinceLastEdge,
                  accumulatedIntensitySinceLastChange[0] / numberOfPixelsSinceLastEdge,
                  accumulatedIntensitySinceLastChange[1] / numberOfPixelsSinceLastEdge,
                  accumulatedIntensitySinceLastChange[2] / numberOfPixelsSinceLastEdge,
                  minIntensitySinceLastChange, maxIntensitySinceLastChange,
                  firstPixelAfterLastEdgeWasBelowHorizon, 
                  Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < 0,
                  false,
                  image.cameraInfo
                  );

                ////currentColorSpaceUsageCounter.addColor(image.image[y][0][x],image.image[y][1][x],image.image[y][2][x]);

                drawSegment(newSegment);
                groundDetector.addSegment(newSegment, longLineIndex);
                ballDetector.addSegment(newSegment, longLineIndex);
                orangeCalibration.addSegment(newSegment, longLineIndex);

                // change state, reset counters
                DEBUG_IMAGE_SET_PIXEL_BLUE(imageProcessorScanLines, x + 1, y);
                edgeState = onEdge;
                if(changeY == 1) changeYState = up;
                else changeYState = nothing;

                firstPixelAfterLastEdge.x = x; firstPixelAfterLastEdge.y = y;
                accumulatedIntensitySinceLastChange[0] = intensity[0][0]; 
                accumulatedIntensitySinceLastChange[1] = intensity[1][0]; 
                accumulatedIntensitySinceLastChange[2] = intensity[2][0]; 

                minIntensitySinceLastChange[0] = intensity[0][0];
                minIntensitySinceLastChange[1] = intensity[1][0];
                minIntensitySinceLastChange[2] = intensity[2][0];

                maxIntensitySinceLastChange[0] = intensity[0][0];
                maxIntensitySinceLastChange[1] = intensity[1][0];
                maxIntensitySinceLastChange[2] = intensity[2][0];

                firstPixelAfterLastEdgeWasBelowHorizon =
                  Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < -1;

                numberOfPixelsSinceLastEdge = 1;
              }//if(changeY != 0 || changeU != 0 || changeV != 0)
              else
              {
                accumulatedIntensitySinceLastChange[0] += intensity[0][0]; 
                accumulatedIntensitySinceLastChange[1] += intensity[1][0]; 
                accumulatedIntensitySinceLastChange[2] += intensity[2][0]; 
                if(intensity[0][0] < minIntensitySinceLastChange[0]) minIntensitySinceLastChange[0] = intensity[0][0];
                if(intensity[1][0] < minIntensitySinceLastChange[1]) minIntensitySinceLastChange[1] = intensity[1][0];
                if(intensity[2][0] < minIntensitySinceLastChange[2]) minIntensitySinceLastChange[2] = intensity[2][0];

                if(intensity[0][0] > maxIntensitySinceLastChange[0]) maxIntensitySinceLastChange[0] = intensity[0][0];
                if(intensity[1][0] > maxIntensitySinceLastChange[1]) maxIntensitySinceLastChange[1] = intensity[1][0];
                if(intensity[2][0] > maxIntensitySinceLastChange[2]) maxIntensitySinceLastChange[2] = intensity[2][0];
                numberOfPixelsSinceLastEdge++;
              }
            }//if(edgeState == betweenEdges)
            else if(edgeState == onEdge)
            {
              if(changeY == 0 && changeU == 0 && changeV == 0) 
              {
                if(changeYState == down)
                {
                  //create and add a segment
                  LineSegment newSegment(
                    (firstPixelAfterLastEdge + topOfRoof) / 2, 
                    (x + topOfRoof.x) / 2, (y + topOfRoof.y) / 2, 
                    numberOfPixelsSinceLastEdge,
                    accumulatedIntensitySinceLastChange[0] / numberOfPixelsSinceLastEdge,
                    accumulatedIntensitySinceLastChange[1] / numberOfPixelsSinceLastEdge,
                    accumulatedIntensitySinceLastChange[2] / numberOfPixelsSinceLastEdge,
                    minIntensitySinceLastChange, maxIntensitySinceLastChange,
                    firstPixelAfterLastEdgeWasBelowHorizon,
                    Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < 0,
                    true,
                    image.cameraInfo);

                  LINE(imageProcessor_general, newSegment.begin.x, newSegment.begin.y, newSegment.end.x, newSegment.end.y, 2, Drawings::ps_solid, Drawings::red);

                  drawSegment(newSegment);
                  groundDetector.addSegment(newSegment, longLineIndex);
                }

                edgeState = betweenEdges;
                //reset counters
                firstPixelAfterLastEdge.x = x; firstPixelAfterLastEdge.y = y;
                accumulatedIntensitySinceLastChange[0] = intensity[0][0]; 
                accumulatedIntensitySinceLastChange[1] = intensity[1][0]; 
                accumulatedIntensitySinceLastChange[2] = intensity[2][0]; 

                minIntensitySinceLastChange[0] = intensity[0][0];
                minIntensitySinceLastChange[1] = intensity[1][0];
                minIntensitySinceLastChange[2] = intensity[2][0];

                maxIntensitySinceLastChange[0] = intensity[0][0];
                maxIntensitySinceLastChange[1] = intensity[1][0];
                maxIntensitySinceLastChange[2] = intensity[2][0];

                firstPixelAfterLastEdgeWasBelowHorizon =
                  Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < -1;

                numberOfPixelsSinceLastEdge = 1;
              }//if(changeY == 0 && changeU == 0 && changeV == 0)
              if(changeYState == up)
              {
                DEBUG_IMAGE_SET_PIXEL_RED(imageProcessorScanLines, x - 1, y);
                if(changeY == -1) 
                {
                  changeYState = down;
                  topOfRoof.x = x;
                  topOfRoof.y = y;
                }
              }
              if(changeYState == down)
              {
                DEBUG_IMAGE_SET_PIXEL_PINK(imageProcessorScanLines, x - 1, y);
                if(changeY == 1) changeYState = nothing;
              }
            }
          }//if(lineIsLong)
      }//for(int z = 2; z < line.getNumberOfPixels(); z++)

      if(lineIsLong && edgeState == betweenEdges)
      {
        LineSegment newSegment(
          firstPixelAfterLastEdge, x, y, numberOfPixelsSinceLastEdge,
          accumulatedIntensitySinceLastChange[0] / numberOfPixelsSinceLastEdge,
          accumulatedIntensitySinceLastChange[1] / numberOfPixelsSinceLastEdge,
          accumulatedIntensitySinceLastChange[2] / numberOfPixelsSinceLastEdge,
          minIntensitySinceLastChange, maxIntensitySinceLastChange,
          firstPixelAfterLastEdgeWasBelowHorizon,
          Geometry::getDistanceToLine(horizonLine, Vector2<double>(x,y)) < 0,
          false,
          image.cameraInfo);
        //end is in border = true

        drawSegment(newSegment);

        groundDetector.addSegment(newSegment, longLineIndex);
        ballDetector.addSegment(newSegment, longLineIndex);
        orangeCalibration.addSegment(newSegment, longLineIndex);
      }
    }//if(line.getNumberOfPixels() > 3)
}

bool GridImageProcessor2::handleMessage(InMessage& message)
{
  bool handled = false;

  switch(message.getMessageID())
  {
  case idParametersForGridImageProcessor2:
    {
      message.bin >> parameters;
      handled = true;
      break;
    }
  case idGridImageProcessor2DebugParameters:
    {
      GenericDebugData d;
      message.bin >> d;

      parameters.scanLineSpacing          = (int)d.data[0];
      parameters.desiredNumberOfScanLines = (int)d.data[1];
      returnCondition                     = (int)d.data[3];

      if(parameters.scanLineSpacing < 1) parameters.scanLineSpacing = 1;

      if(returnCondition == 11) 
      {
        colorTable64.clear();
        colorTableCuboids.clear();
        colorTableReferenceColor.clear();
      }
      //      if(returnCondition == 10) colorTable64 = (ColorTable64&)colorTable;

      handled = true;
      break;
    }
  }
  return handled;
}

void GridImageProcessor2::visualize()
{
#if defined(_WIN32) && !defined(SIMROBOT)
  int x,y;
  //color space usage
  currentColorSpaceUsageCounter.generateImage(image, colorFrequencyImage);

  Histogram histogramY(Histogram::imageIntensityY);
  Histogram histogramU(Histogram::imageIntensityU);
  Histogram histogramV(Histogram::imageIntensityV);

  for(x = 0; x < image.cameraInfo.resolutionWidth; x++)
  {
    for(y = 0; y < image.cameraInfo.resolutionHeight; y++)
    {
      histogramY.add(image.image[y][0][x]);
      histogramU.add(image.image[y][1][x]);
      histogramV.add(image.image[y][2][x]);
    }
  }
  INFO(sendHistogram_imageIntensityY, idHistogram, bin, histogramY);
  INFO(sendHistogram_imageIntensityU, idHistogram, bin, histogramU);
  INFO(sendHistogram_imageIntensityV, idHistogram, bin, histogramV);


  for(x = 0; x < image.cameraInfo.resolutionWidth; x++)
  {
    for(y = 0; y < image.cameraInfo.resolutionHeight; y++)
    {

      if(image.image[y][0][x] > averageIntensityOverLastImages.x * 1.3)
      {
        imageProcessorGradientsImage.image[y][0][x] = 200;
        imageProcessorGradientsImage.image[y][1][x] = image.image[y][1][x];
        imageProcessorGradientsImage.image[y][2][x] = image.image[y][2][x];
      }
      else if(image.image[y][0][x] > averageIntensityOverLastImages.x * 0.7)
      {
        imageProcessorGradientsImage.image[y][0][x] = averageIntensityOverLastImages.x;
        imageProcessorGradientsImage.image[y][1][x] = averageIntensityOverLastImages.y;
        imageProcessorGradientsImage.image[y][2][x] = averageIntensityOverLastImages.z;
      }
      else
      {
        imageProcessorGradientsImage.image[y][0][x] = 60;
        imageProcessorGradientsImage.image[y][1][x] = image.image[y][1][x];
        imageProcessorGradientsImage.image[y][2][x] = image.image[y][2][x];
      }
    }
  }
  /*
  histogramY.analyseClusters();
  histogramU.analyseClusters();
  histogramV.analyseClusters();
  for(x = 0; x < cameraResolutionWidth; x++)
  {
  for(y = 0; y < cameraResolutionHeight; y++)
  {
  classificationYImage.image[y][0][x] = 0;
  int clusterIndex;
  for(clusterIndex = 0; clusterIndex < histogramY.getNumberOfClusters(); clusterIndex++)
  {
  if(
  histogramY.getBeginOfCluster(clusterIndex) <= image.image[y][0][x] &&
  histogramY.getEndOfCluster(clusterIndex)   >= image.image[y][0][x] )
  classificationYImage.image[y][0][x] = (clusterIndex + 1) * 255 / histogramY.getNumberOfClusters();
  }
  classificationYImage.image[y][1][x] = 127; classificationYImage.image[y][2][x] = 127;

  classificationUImage.image[y][0][x] = 0;
  for(clusterIndex = 0; clusterIndex < histogramU.getNumberOfClusters(); clusterIndex++)
  {
  if(
  histogramU.getBeginOfCluster(clusterIndex) <= image.image[y][1][x] &&
  histogramU.getEndOfCluster(clusterIndex)   >= image.image[y][1][x] )
  classificationUImage.image[y][0][x] = (clusterIndex + 1) * 255 / histogramU.getNumberOfClusters();
  }
  classificationUImage.image[y][1][x] = 127; classificationUImage.image[y][2][x] = 127;

  classificationVImage.image[y][0][x] = 0;
  for(clusterIndex = 0; clusterIndex < histogramV.getNumberOfClusters(); clusterIndex++)
  {
  if(
  histogramV.getBeginOfCluster(clusterIndex) <= image.image[y][2][x] &&
  histogramV.getEndOfCluster(clusterIndex)   >= image.image[y][2][x] )
  classificationVImage.image[y][0][x] = (clusterIndex + 1) * 255 / histogramV.getNumberOfClusters();
  }
  classificationVImage.image[y][1][x] = 127; classificationVImage.image[y][2][x] = 127;
  }
  }
  */
#endif
}

void GridImageProcessor2::drawSegment(LineSegment lineSegment)
{
  //lines in color class for debug drawing
  COMPLEX_DRAWING(imageProcessor_coloredSegments3,

    colorClass colorOfLineSegment = colorTable64.getColorClass(lineSegment.averageIntensity[0], lineSegment.averageIntensity[1], lineSegment.averageIntensity[2]);

  Drawings::Color lineColor;
  switch(colorOfLineSegment)
  {
  case orange: lineColor = Drawings::orange; break;
  case yellowOrange: lineColor = Drawings::yellowOrange; break;
  case green: lineColor = Drawings::green; break;
  case white: lineColor = Drawings::white; break;
  case yellow: lineColor = Drawings::yellow; break;
  case skyblue: lineColor = Drawings::skyblue; break;
  case pink: lineColor = Drawings::pink; break;
  case red: lineColor = Drawings::red; break;
  case blue: lineColor = Drawings::blue; break;
  default: lineColor = Drawings::gray;
  }
  LINE(imageProcessor_coloredSegments3, 
    lineSegment.begin.x, lineSegment.begin.y,
    lineSegment.end.x, lineSegment.end.y,
    2, Drawings::ps_solid, lineColor);
  );

  //lines in color class for debug drawing
  COMPLEX_DRAWING(imageProcessor_coloredSegments2,

    colorClass colorOfLineSegment = colorTableReferenceColor.getColorClass(lineSegment.averageIntensity[0], lineSegment.averageIntensity[1], lineSegment.averageIntensity[2]);

  Drawings::Color lineColor;
  switch(colorOfLineSegment)
  {
  case orange: lineColor = Drawings::orange; break;
  case yellowOrange: lineColor = Drawings::yellowOrange; break;
  case green: lineColor = Drawings::green; break;
  case white: lineColor = Drawings::white; break;
  case yellow: lineColor = Drawings::yellow; break;
  case skyblue: lineColor = Drawings::skyblue; break;
  case pink: lineColor = Drawings::pink; break;
  default: lineColor = Drawings::gray;
  }
  LINE(imageProcessor_coloredSegments2,
    lineSegment.begin.x, lineSegment.begin.y,
    lineSegment.end.x, lineSegment.end.y,
    2, Drawings::ps_solid, lineColor);
  );

  //lines in color class for debug drawing
  COMPLEX_DRAWING(imageProcessor_coloredSegments1,

    colorClass colorOfLineSegment = colorTableCuboids.getColorClass(lineSegment.averageIntensity[0], lineSegment.averageIntensity[1], lineSegment.averageIntensity[2]);

  Drawings::Color lineColor;
  switch(colorOfLineSegment)
  {
  case green: lineColor = Drawings::green; break;
  case white: lineColor = Drawings::white; break;
  default: lineColor = Drawings::gray;
  }
  LINE(imageProcessor_coloredSegments1,
    lineSegment.begin.x, lineSegment.begin.y,
    lineSegment.end.x, lineSegment.end.y,
    4, Drawings::ps_solid, lineColor);
  );



  // lines in average color for debug image
#if defined(_WIN32) && !defined(SIMROBOT)
  //visualization
  /*  Geometry::PixeledLine linePixels(lineSegment.begin, lineSegment.end);
  for(int i = 0; i < linePixels.getNumberOfPixels(); i++)
  {
  imageProcessorGeneralImage.image[linePixels.getPixelY(i)][0][linePixels.getPixelX(i)] = lineSegment.averageIntensity[0];
  imageProcessorGeneralImage.image[linePixels.getPixelY(i)][1][linePixels.getPixelX(i)] = lineSegment.averageIntensity[1];
  imageProcessorGeneralImage.image[linePixels.getPixelY(i)][2][linePixels.getPixelX(i)] = lineSegment.averageIntensity[2];
  }
  */
#endif
}

/*
* Change log :
* 
* $Log: GridImageProcessor2.cpp,v $
* Revision 1.25  2004/03/08 01:39:00  roefer
* Interfaces should be const
*
* Revision 1.24  2004/03/01 11:49:40  juengel
* Added red and blue to visualization of segments.
*
* Revision 1.23  2004/02/12 14:16:20  juengel
* changed visualization of segmented image 1, 2, 3
*
* Revision 1.22  2004/02/05 14:36:22  juengel
* changed width of scan line drawings
*
* Revision 1.21  2003/12/31 23:50:36  roefer
* Removed inclusion of RobotDimensions.h because it is not used
*
* Revision 1.20  2003/12/15 11:47:02  juengel
* Introduced CameraInfo
*
* Revision 1.19  2003/12/04 17:39:00  juengel
* Moved LineSegmentation to own class.
*
* Revision 1.18  2003/12/01 16:26:12  juengel
* Added calculation of "pixel usage" goal pixels are not counted yet.
*
* Revision 1.17  2003/11/30 01:53:18  loetzsch
* prepared RobotControl port to Visual C++ .Net
*
* Revision 1.16  2003/11/26 11:43:31  juengel
* Goal recognition added.
*
* Revision 1.15  2003/11/17 11:39:28  juengel
* drawing scan lines changed
*
* Revision 1.14  2003/11/12 17:40:13  juengel
* color class based ball detection is done on all scan lines now
*
* Revision 1.13  2003/11/12 16:19:35  goehring
* frameNumber added to percepts
*
* Revision 1.12  2003/11/11 16:57:00  juengel
* "run based" ball detection
*
* Revision 1.11  2003/11/10 11:37:12  juengel
* Added extra ball scan lines.
*
* Revision 1.10  2003/11/07 11:21:19  juengel
* GridImageProcessor2 has more parameters now.
*
* Revision 1.9  2003/11/05 16:45:15  juengel
* Added drawing imageProcessor_ball2
*
* Revision 1.8  2003/11/03 20:16:09  juengel
* color class images can be sent from robot now
*
* Revision 1.7  2003/10/31 11:47:48  juengel
* Changed white for ColorTableCuboids.
*
* Revision 1.6  2003/10/30 18:31:06  juengel
* White and green of ColorTableCuboids are filled.
*
* Revision 1.5  2003/10/29 13:06:19  juengel
* New constructor for LineSegment.
*
* Revision 1.4  2003/10/23 07:24:20  juengel
* Renamed ColorTableAuto to ColorTableReferenceColor.
*
* Revision 1.3  2003/10/12 20:18:07  juengel
* ColorTable is not const anymore.
*
* Revision 1.2  2003/10/12 11:51:15  juengel
* Added ManualCalibration.
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.13  2003/09/26 15:27:49  juengel
* Renamed DataTypes to representations.
*
* Revision 1.12  2003/09/26 11:38:51  juengel
* - sorted tools
* - clean-up in DataTypes
*
* Revision 1.11  2003/09/01 16:20:47  juengel
* Removed DebugKey sendParametersFromGridImageProcessor2
*
* Revision 1.10  2003/09/01 15:08:36  juengel
* new drawings
*
* Revision 1.9  2003/09/01 10:16:17  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.8  2003/08/30 10:19:53  juengel
* DebugDrawings clean-up 1
*
* Revision 1.7  2003/08/29 13:13:51  juengel
* segments are generated for roof edges in y channel
*
* Revision 1.6  2003/08/25 17:21:49  juengel
* Added some debug images.
*
* Revision 1.5  2003/08/18 12:09:23  juengel
* Added segment based detection.
*
* Revision 1.4  2003/07/29 12:47:08  juengel
* Moved calculateHorizon to Geometry
*
* Revision 1.3  2003/07/21 13:44:07  juengel
* clean up
*
* Revision 1.2  2003/07/19 19:31:04  juengel
* Improved AutoCalibration
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.53  2003/06/25 18:42:34  juengel
* Added nearPoints, farPoints[maxNumberOfPoints] and farPointIsOnImageBorder to ObstaclesPercept.
*
* Revision 1.52  2003/06/13 14:53:31  juengel
* Added scanLinesForObstacles.
*
* Revision 1.51  2003/06/12 16:55:22  juengel
* Moved RangeArray to its own file.
*
* Revision 1.50  2003/05/27 16:18:36  juengel
* Renamed some variables.
*
* Revision 1.49  2003/05/26 13:07:37  juengel
* Improved recognition of goals.
*
* Revision 1.48  2003/05/26 08:21:40  juengel
* Improved recognition of goals.
*
* Revision 1.47  2003/05/12 10:29:24  dueffert
* unused variable warnings removed
*
* Revision 1.46  2003/05/06 16:52:07  juengel
* Warning removed.
*
* Revision 1.45  2003/05/05 12:17:15  juengel
* Moved some parameters to the parameters struct.
*
* Revision 1.44  2003/05/04 17:34:33  roefer
* Support for debug images only in RobotControl
*
* Revision 1.43  2003/05/01 08:15:21  roefer
* specialPercept.reset() added
*
* Revision 1.42  2003/04/26 11:35:26  juengel
* Three scan lines are used to check vertical for goal indications.
*
* Revision 1.41  2003/04/23 16:14:00  juengel
* Changed calculation and scan of horizontal lines, added calculation of best angle two own and opponent goal.
*
* Revision 1.40  2003/04/05 16:50:22  juengel
* Calculation of ballPercept changed.
*
* Revision 1.39  2003/04/04 17:18:32  juengel
* Warnings removed.
*
* Revision 1.38  2003/04/03 18:50:08  juengel
* Improved AutoCalibration.
*
* Revision 1.37  2003/04/02 13:23:06  juengel
* New flag recognition
*
* Revision 1.36  2003/04/01 20:22:05  juengel
* colorTableReferenceColor.getColorClassForFlagsAndGoals is used.
*
* Revision 1.35  2003/04/01 13:59:59  juengel
* Added ColorTableReferenceColor.
*
* Revision 1.34  2003/03/31 20:55:34  juengel
* calculateSingleBallPercept
*
* Revision 1.33  2003/03/29 21:15:53  juengel
* Improved ball recognition, flag recognition and auto calibration.
*
* Revision 1.32  2003/03/28 14:34:55  juengel
* Added new circle calculation methods.
*
* Revision 1.31  2003/03/25 11:03:09  juengel
* New calculation of ball circle.
*
* Revision 1.30  2003/03/22 16:55:59  juengel
* Improved flag perception, goal percept is only filled when a goal is seen.
*
* Revision 1.29  2003/03/15 22:31:18  juengel
* Some debug parameters are initialized now.
*
* Revision 1.28  2003/03/15 13:32:04  juengel
* Added recognition of goals.
*
* Revision 1.27  2003/03/12 12:36:49  juengel
* clean up
*
* Revision 1.26  2003/03/11 11:33:30  juengel
* Constructor reads parameters from gi2params.cfg,
* obstaclesPercept is filled.
*
* Revision 1.25  2003/03/10 13:56:57  juengel
* Worked at goal recognition
*
* Revision 1.24  2003/03/08 11:58:32  juengel
* improved recognition of ball
*
* Revision 1.23  2003/03/07 17:08:53  juengel
* warnings removed
*
* Revision 1.22  2003/03/07 11:36:56  juengel
* Added calculation of flag distances.
*
* Revision 1.21  2003/03/06 18:16:52  dueffert
* fixed max problem (Math/Common.h has to be included after iostream.h!?!)
*
* Revision 1.20  2003/03/06 14:29:07  juengel
* Added several returns for time measurement.
*
* Revision 1.19  2003/03/05 17:18:34  juengel
* Improved flag perception.
*
* Revision 1.18  2003/03/04 14:54:23  juengel
* Added player perception.
*
* Revision 1.17  2003/02/27 12:55:41  juengel
* Clean up.
*
* Revision 1.16  2003/02/25 14:00:58  juengel
* dies und das
*
* Revision 1.15  2003/02/21 16:34:16  dueffert
* common pi in own code, warnings removed, platform compatibility restored
*
* Revision 1.14  2003/02/17 11:00:13  dueffert
* greenshills backport
*
* Revision 1.13  2003/02/13 09:49:33  juengel
* GT2003 compiles now
*
* Revision 1.12  2003/02/13 09:40:01  juengel
* added auto calibration of green tolerance
* improved recognition of flags
*
* Revision 1.11  2003/02/11 17:08:59  juengel
* recognition of flags added
*
* Revision 1.10  2003/02/09 16:47:16  juengel
* enabled auto calibration
*
* Revision 1.9  2003/02/09 15:12:59  juengel
* Added scanLinesForLinesGoalsBallAndPlayers.
*
* Revision 1.8  2003/02/09 13:02:10  juengel
* Added scanLinesForLines.
*
* Revision 1.7  2003/02/09 10:09:15  juengel
* clean up
*
* Revision 1.6  2003/02/08 18:41:47  juengel
* big change
*
* Revision 1.5  2003/01/30 22:32:35  juengel
* Added LinesPerceptor functionality.
*
* Revision 1.4  2003/01/13 10:33:16  juengel
* pattern matching for flags
*
* Revision 1.3  2002/12/02 12:02:58  juengel
* Line clipping for flag predictions added.
*
* Revision 1.2  2002/12/01 17:39:37  juengel
* Added calculation of the horizon and prediction of flags.
*
* Revision 1.1  2002/11/26 14:19:25  juengel
* GridImageProcessor2 added.
*
*/
