/**
* @file GridImageProcessorTSL.cpp
* 
* This file contains a class for Image Processing.
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
* @author <a href="mailto:sebastian.deutsch@gmx.de">Sebastian Deutsch</a>
* @author <a href=mailto:robocup@andreosterhues.de>Andr Osterhues</a>
*
*/

#include "GridImageProcessorTSL.h"
#include "Tools/Debugging/Debugging.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Debugging/DebugImages.h"
#include "Tools/FieldDimensions.h"
#include <float.h>

#define HORIZ_TOP       0
#define HORIZ_BOTTOM    2
#define VERTICAL_LEFT   3
#define VERTICAL_RIGHT  1

#define SCAN_PINK       0
#define SCAN_YELLOW     1
#define SCAN_SKYBLUE    2
#define SCAN_ORANGE     3
#define SCAN_BLUE       4
#define SCAN_RED        5

#define SCAN_STARTCOLOR SCAN_PINK
#define SCAN_ENDCOLOR   SCAN_RED+1

GridImageProcessorTSL::GridImageProcessorTSL(const ImageProcessorInterfaces& interfaces)
: ImageProcessor(interfaces)
{
}

GridImageProcessorTSL::~GridImageProcessorTSL()
{
}

void GridImageProcessorTSL::execute()
{
  register int x, y, z;

  INIT_DEBUG_IMAGE(imageProcessorGeneral, image);

  #ifdef _WIN32
  // Create segmented image
  for (y = 0; y < image.cameraInfo.resolutionHeight; y++)
  {
    for (x = 0; x < image.cameraInfo.resolutionWidth; x++)
    {
      switch (colorTable.getColorClass(image.image[y][0][x],  image.image[y][1][x], image.image[y][2][x]))
      {
        case noColor:
        {
          DEBUG_IMAGE_SET_PIXEL_BLACK(segmentedImage1, x, y);
          break;
        }
        case orange:
        {
          DEBUG_IMAGE_SET_PIXEL_ORANGE(segmentedImage1, x, y);
          break;
        }
        case yellow:
        {
          DEBUG_IMAGE_SET_PIXEL_YELLOW(segmentedImage1, x, y);
          break;
        }
        case skyblue:
        {
          DEBUG_IMAGE_SET_PIXEL_BLUE(segmentedImage1, x, y);
          break;
        }
        case pink:
        {
          DEBUG_IMAGE_SET_PIXEL_PINK(segmentedImage1, x, y);
          break;
        }
        case red:
        {
          DEBUG_IMAGE_SET_PIXEL_RED(segmentedImage1, x, y);
          break;
        }
        case blue:
        {
          DEBUG_IMAGE_SET_PIXEL_DARK_BLUE(segmentedImage1, x, y);
          break;
        }
        case green:
        {
          DEBUG_IMAGE_SET_PIXEL_GREEN(segmentedImage1, x, y);
          break;
        }
        case gray:
        {
          DEBUG_IMAGE_SET_PIXEL_GRAY(segmentedImage1, x, y);
          break;
        }
        case white:
        {
          DEBUG_IMAGE_SET_PIXEL_WHITE(segmentedImage1, x, y);
          break;
        }
      }
    }
  }
  #endif //_WIN32
  
  int i,j;
  colorClass color;

  Geometry::Line horizonLine;
  double horizonAngle;
  horizonLine = calculateHorizon(horizonAngle);
  
  colorClass searchCol[SCAN_ENDCOLOR];
  int drawingsCol[SCAN_ENDCOLOR];
  Vector2<int> colorMostTop[5][SCAN_ENDCOLOR];
  Vector2<int> colorMostLeft[5][SCAN_ENDCOLOR];
  Vector2<int> colorMostRight[5][SCAN_ENDCOLOR];
  Vector2<int> colorMostBottom[5][SCAN_ENDCOLOR];
  int colorCount[SCAN_ENDCOLOR];
  int colorObjectCount[SCAN_ENDCOLOR];
  bool colorBlankLine[SCAN_ENDCOLOR];
  bool colorFound[SCAN_ENDCOLOR];

  searchCol[0] = pink;
  searchCol[1] = yellow;
  searchCol[2] = skyblue;
  searchCol[3] = orange;
  searchCol[4] = blue;
  searchCol[5] = red;

  drawingsCol[0] = Drawings::pink;
  drawingsCol[1] = Drawings::yellow;
  drawingsCol[2] = Drawings::skyblue;
  drawingsCol[3] = Drawings::orange;
  drawingsCol[4] = Drawings::blue;
  drawingsCol[5] = Drawings::red;

  for(j=SCAN_STARTCOLOR; j<SCAN_ENDCOLOR; j++) {
    
    // initialise scope variables
    colorCount[j]              = 0;
    colorObjectCount[j]        = 0;
    colorFound[j]              = false;
    
    // max. 5 object per scan
    for(i=0; i<5; i++)
    {
      colorMostLeft[i][j].x    = image.cameraInfo.resolutionWidth-1;
      colorMostLeft[i][j].y    = image.cameraInfo.resolutionHeight / 2;
      colorMostTop[i][j].x     = image.cameraInfo.resolutionWidth / 2;
      colorMostTop[i][j].y     = image.cameraInfo.resolutionHeight-1;
      colorMostRight[i][j].x   = 0;
      colorMostRight[i][j].y   = image.cameraInfo.resolutionHeight / 2;
      colorMostBottom[i][j].x  = 0;
      colorMostBottom[i][j].y  = 0;
    }
  }

  // do the scan
  for(x=0; x<image.cameraInfo.resolutionWidth; x++)
  {
    for(z=SCAN_STARTCOLOR; z<SCAN_ENDCOLOR; z++) {
      colorBlankLine[z] = true;
      colorCount[z] = 0;
    }

    for(y=image.cameraInfo.resolutionHeight-1; y>=0; y--)
    {
      
      color = colorTable.getColorClass(image.image[y][0][x],  image.image[y][1][x], image.image[y][2][x]);
            
      for(z=SCAN_STARTCOLOR; z<SCAN_ENDCOLOR; z++) {

        if(color == searchCol[z]) {
          colorCount[z]++;

          if(colorCount[z]>3)
          {
            colorFound[z] = true;
            colorBlankLine[z] = false;

            if(x < colorMostLeft[colorObjectCount[z]][z].x)
            {
              colorMostLeft[colorObjectCount[z]][z].x = x;
              colorMostLeft[colorObjectCount[z]][z].y = y;
            }
            if(y < colorMostTop[colorObjectCount[z]][z].y) 
            {
              colorMostTop[colorObjectCount[z]][z].x = x;
              colorMostTop[colorObjectCount[z]][z].y = y;
            }
            if(x > colorMostRight[colorObjectCount[z]][z].x)
            {
              colorMostRight[colorObjectCount[z]][z].x = x;
              colorMostRight[colorObjectCount[z]][z].y = y;
            }
            if(y > colorMostBottom[colorObjectCount[z]][z].y)
            {
              colorMostBottom[colorObjectCount[z]][z].x = x;
              colorMostBottom[colorObjectCount[z]][z].y = y;
            }
          }
        }
        else 
        {
          colorCount[z] = 0;
        }
      } /* z */
    } /* y */

    for(z=SCAN_STARTCOLOR; z<SCAN_ENDCOLOR; z++) {
      if(colorFound[z] && colorBlankLine[z]) {
        colorObjectCount[z]++;
        if(colorObjectCount[z]>4) colorObjectCount[z] = 4;
        colorFound[z] = false;
      }
    }
  } /* x */

  // correct colorObject count in case the box isnt seen
  for(z=SCAN_STARTCOLOR; z<SCAN_ENDCOLOR; z++) {
    if(colorFound[z] && colorObjectCount[z]==0) colorObjectCount[z]++;
  }

  // draw debug outputs
  for(z=SCAN_STARTCOLOR; z<SCAN_ENDCOLOR; z++) {
    for(i = 0; i<colorObjectCount[z]; i++) {
      LINE(sketch, colorMostLeft[i][z].x , colorMostTop[i][z].y,    colorMostRight[i][z].x, colorMostTop[i][z].y,    1, Drawings::ps_solid, drawingsCol[z]);
      LINE(sketch, colorMostRight[i][z].x, colorMostTop[i][z].y,    colorMostRight[i][z].x, colorMostBottom[i][z].y, 1, Drawings::ps_solid, drawingsCol[z]);
      LINE(sketch, colorMostRight[i][z].x, colorMostBottom[i][z].y, colorMostLeft[i][z].x , colorMostBottom[i][z].y, 1, Drawings::ps_solid, drawingsCol[z]);
      LINE(sketch, colorMostLeft[i][z].x , colorMostBottom[i][z].y, colorMostLeft[i][z].x , colorMostTop[i][z].y,    1, Drawings::ps_solid, drawingsCol[z]);
    }
  }
  
  
  
  
  
  DEBUG_DRAWING_FINISHED(sketch);
  SEND_DEBUG_IMAGE(imageProcessorGeneral);
  SEND_DEBUG_IMAGE(segmentedImage1);
}

Geometry::Line GridImageProcessorTSL::calculateHorizon(double& horizonAngle)
{
  double x1, y1, x2, y2;
  double r31 = cameraMatrix.rotation.c[0].z;
  double r32 = cameraMatrix.rotation.c[1].z;
  double r33 = cameraMatrix.rotation.c[2].z;

  if (r33 == 0) 
  {
    r33 = 0.00001;
  }
  
  double v1 = (174.0 * 0.5) / tan(image.cameraInfo.openingAngleWidth * 0.5);
  double v2 = 174.0 * 0.5;
  y1 = -(-(image.cameraInfo.resolutionHeight / 2 * r33) - (r31 * v1) - (r32 * v2) ) / r33;
  
  v2 = -174.0 * 0.5;
  
  y2 = -( -(image.cameraInfo.resolutionHeight / 2 * r33) - (r31 * v1) - (r32 * v2) ) / r33; 
  
  x1 = 1;
  x2 = image.cameraInfo.resolutionWidth;
  
  Vector3<double> toTopOfHead;
  Vector3<double> toTopOfHeadWorld;
  
  toTopOfHead.x = 0;
  toTopOfHead.y = 0;
  toTopOfHead.z = 1;
  
  toTopOfHeadWorld = cameraMatrix.rotation * toTopOfHead;
  if (toTopOfHeadWorld.z < 0)
  {
    double xBuffer, yBuffer;
    xBuffer = x1;
    yBuffer = y1;
    x1 = x2;
    y1 = y2;
    x2 = xBuffer;
    y2 = yBuffer;
  }
  LINE(imageProcessor_horizon, x1, y1, x2, y2, 1, Drawings::ps_solid, Drawings::white);

  Geometry::Line horizon;

  horizon.base.x = (x1 + x2) * 0.5;
  horizon.base.y = (y1 + y2) * 0.5;
  horizon.direction.x = x2 - x1;
  horizon.direction.y = y2 - y1;
  horizon.normalizeDirection();

  horizonAngle = horizon.direction.angle();

  if (x1 == image.cameraInfo.resolutionWidth)
  {
    if (y1 < image.cameraInfo.resolutionHeight / 2)
    {
      horizonAngle += pi;
    }
    else if ( y1 > image.cameraInfo.resolutionHeight / 2)
    {
      horizonAngle -= pi;
    }
  }
  return horizon;
}

/*
* Change log :
* 
* $Log: GridImageProcessorTSL.cpp,v $
* Revision 1.3  2004/03/08 01:39:01  roefer
* Interfaces should be const
*
* Revision 1.2  2003/12/15 11:47:03  juengel
* Introduced CameraInfo
*
* Revision 1.1  2003/10/06 14:10:15  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/01 10:16:17  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.2  2003/08/30 10:16:42  juengel
* DebugDrawings clean-up 1
*
* Revision 1.1.1.1  2003/07/02 09:40:24  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.2  2003/05/20 21:57:19  roefer
* #include bug fixed, warnings removed
*
* Revision 1.1  2003/05/20 15:52:14  deutsch
* A TSL optimized grid image processor.
* not working yet.
*
* Revision 1.15  2003/04/09 11:55:11  osterhues
* Workaround for _finite to catch NaN
*
* Revision 1.14  2003/04/09 11:44:16  deutsch
* added approximation function for distance.
*
* Revision 1.13  2003/04/09 10:17:53  hebbel
* fixed compile error
*
* Revision 1.12  2003/04/09 04:40:53  deutsch
* now just one enemy is percept per frame to avoid very long enemys (scan 2 for 1).
*
* Revision 1.11  2003/04/09 00:02:54  deutsch
* Desperade try to improve *everything* in the last minutes.
*
* Revision 1.10  2003/04/07 15:24:24  deutsch
* no message
*
* Revision 1.9  2003/04/02 13:07:56  deutsch
* Optimized angle calculation for ED.
*
* Revision 1.8  2003/04/02 00:06:52  deutsch
* Added X distance calculation for enemy detection.
*
* Revision 1.7  2003/04/01 04:27:09  deutsch
* Added enemy cognition.
*
* Revision 1.6  2003/03/31 16:30:31  osterhues
* Copied Ball-, Flag- and GoalSpecialist from GridImageProcessor
*
* Revision 1.5  2003/03/31 04:16:32  osterhues
* Added support for segmented image
*
* Revision 1.4  2003/03/27 03:21:53  deutsch
* no message
*
* Revision 1.3  2003/03/25 18:42:02  osterhues
* Imported code from GridImageProcessor
* Small optimizations
*
* Revision 1.2  2003/03/19 14:23:53  osterhues
* no message
*
* Revision 1.1  2003/03/04 15:41:55  osterhues
* no message
*
* Revision 1.3  2003/03/04 15:38:15  osterhues
* no message
*
* Revision 1.1  2003/03/04 15:02:21  osterhues
* Changed GridImageProcessorDo to GridImageProcessorTSL
*
* Revision 1.6  2003/02/18 18:59:32  roefer
* Warning removed
*
* Revision 1.5  2003/02/17 16:36:02  deutsch
* no message
*
* Revision 1.4  2003/02/06 20:33:15  deutsch
* TSLTools converted for RobotControl
*
* Revision 1.3  2003/01/08 11:06:26  loetzsch
* did not compile for the OpenR platform
*
* Revision 1.2  2003/01/06 21:19:08  deutsch
* Horizon calculation added.
*
* Revision 1.1  2003/01/03 17:22:42  deutsch
* added class GridmageProcesserDo
*
*
*/
