/** 
* @file ColorTableCuboids.h
* Declaration of class ColorTableCuboids.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

#ifndef _ColorTableCuboids_h_
#define _ColorTableCuboids_h_

#include "Representations/Perception/Image.h"
#include "Tools/ColorClasses.h"

/**
* @class ColorTableCuboids
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/
class ColorTableCuboids : public ColorTable
{
public:
  /** Constructor */
  ColorTableCuboids();

  /** Destructor */
  ~ColorTableCuboids();

  /** 
  * Calculates the color classes of a point in the yuv color space.
  * @param y the y value of the pixel
  * @param u the u value of the pixel
  * @param v the v value of the pixel
  */
  virtual colorClass getColorClass (
    const unsigned char y, 
    const unsigned char u, 
    const unsigned char v
    ) const;


  /**
  * Segments an image to a color class image.
  * 
  * This doesn't need to used in the image processor, but is needed for visualisation
  * of color tables.
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  * @return A reference to the created color class image.
  */
  virtual void generateColorClassImage(
    const Image& image, 
    ColorClassImage& colorClassImage) const;

  void clear();

  void setThresholds(
    colorClass color,
    int minY, int minU, int minV, int maxY, int maxU, int maxV
    );

  int getThresholdMin(colorClass color, int channel) const;
  int getThresholdMax(colorClass color, int channel) const;


private:
  int thresholdMin[numOfColors][3];
  int thresholdMax[numOfColors][3];
  unsigned long int lastTimeWhenThresholdWasSet[numOfColors];
};

/**
 * Streaming operator that reads ColorTableCuboids from a stream.
 * @param stream The stream from which is read.
 * @param colorTableCuboids The ColorTableCuboids object.
 * @return The stream.
 */ 
In& operator>>(In& stream, ColorTableCuboids& colorTableCuboids);
 
/**
 * Streaming operator that writes GridImageProcessor2Parameters to a stream.
 * @param stream The stream to write on.
 * @param colorTableCuboids The ColorTableCuboids object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const ColorTableCuboids& colorTableCuboids);



#endif   //  _ColorTableCuboids_h_

/*
 * Change log :
 * 
 * $Log: ColorTableCuboids.h,v $
 * Revision 1.2  2003/10/29 13:13:43  juengel
 * added method "clear"
 *
 * Revision 1.1  2003/10/23 07:15:17  juengel
 * Renamed ColorTableAuto to ColorTableReferenceColor,
 * added ColorTableCuboids.
 *
 */
