/** 
* @file ColorTableReferenceColor.h
* Declaration of class ColorTableReferenceColor.
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/

#ifndef _ColorTableReferenceColor_h_
#define _ColorTableReferenceColor_h_

#include "Representations/Perception/Image.h"
#include "Tools/ColorClasses.h"

/**
* @class ColorTableReferenceColor
*
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Jngel</A>
*/
class ColorTableReferenceColor : public ColorTable
{
public:
  /** Constructor */
  ColorTableReferenceColor();

  /** Destructor */
  ~ColorTableReferenceColor();

  /** 
  * Calculates the color classes of a point in the yuv color space.
  * @param y the y value of the pixel
  * @param u the u value of the pixel
  * @param v the v value of the pixel
  */
  virtual colorClass getColorClass (
    const unsigned char y, 
    const unsigned char u, 
    const unsigned char v
    ) const;


  /**
  * Segments an image to an color class image.
  * 
  * This doesn't need to used in the image processor, but is needed for visualisation
  * of color tables.
  * @param image A reference to the image to be segmented
  * @param colorClassImage A reference to the color class image to be created
  * @return A reference to the created color class image.
  */
  virtual void generateColorClassImage(
    const Image& image, 
    ColorClassImage& colorClassImage) const;


  int getAverageGreen(int channel) const;
  int getGreenMin(int channel) const;
  int getGreenMax(int channel) const;
  
  void setGreenMin(int channel, int value);
  void setGreenMax(int channel, int value);

  void expandGreen(int greenMin[3], int greenMax[3]);

  void clear();

private:
  void setGreen();

  int averageGreen[3];
  int greenMin[3];
  int greenMax[3];

  bool greenIsValid;
  int setGreenCounter;

  int skyblueAutoMin[3];
  int skyblueAutoMax[3];

  int yellowAutoMin[3];
  int yellowAutoMax[3];

  int orangeAutoMin[3];
  int orangeAutoMax[3];

  int pinkAutoMin[3];
  int pinkAutoMax[3];

  int redAutoMin[3];
  int redAutoMax[3];

  int whiteAutoMin[3];
  int whiteAutoMax[3];
  
  int whiteFixedMin[3];
  int whiteFixedMax[3];
};

/**
 * Streaming operator that reads ColorTableReferenceColor from a stream.
 * @param stream The stream from which is read.
 * @param colorTableReferenceColor The ColorTableReferenceColor object.
 * @return The stream.
 */ 
In& operator>>(In& stream, ColorTableReferenceColor& colorTableReferenceColor);
 
/**
 * Streaming operator that writes GridImageProcessor2Parameters to a stream.
 * @param stream The stream to write on.
 * @param colorTableReferenceColor The ColorTableReferenceColor object.
 * @return The stream.
 */ 
Out& operator<<(Out& stream, const ColorTableReferenceColor& colorTableReferenceColor);



#endif   //  _ColorTableReferenceColor_h_

/*
 * Change log :
 * 
 * $Log: ColorTableReferenceColor.h,v $
 * Revision 1.4  2003/11/10 11:35:59  juengel
 * Added pink.
 *
 * Revision 1.3  2003/11/03 20:19:22  juengel
 * Added color class yellowOrange
 *
 * Revision 1.2  2003/10/29 13:13:14  juengel
 * added method "expandGreen"
 *
 * Revision 1.1  2003/10/23 07:15:17  juengel
 * Renamed ColorTableAuto to ColorTableReferenceColor,
 * added ColorTableCuboids.
 *
 * Revision 1.1  2003/10/06 14:10:14  cvsadm
 * Created GT2004 (M.J.)
 *
 * Revision 1.5  2003/09/26 15:27:49  juengel
 * Renamed DataTypes to representations.
 *
 * Revision 1.4  2003/08/29 11:59:38  roefer
 * Comment corrected
 *
 * Revision 1.3  2003/08/18 12:06:01  juengel
 * Added streaming operators.
 *
 * Revision 1.2  2003/07/29 12:44:26  juengel
 * Optimization
 *
 * Revision 1.1  2003/07/21 13:40:24  juengel
 * Moved ColorTableReferenceColor and GoalRecognizer to ImageProcessorTools.
 *
 * Revision 1.2  2003/07/19 19:31:04  juengel
 * Improved AutoCalibration
 *
 * Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
 * created new repository for the competitions in Padova from the 
 * tamara CVS (Tuesday 2:00 pm)
 *
 * removed unused solutions
 *
 * Revision 1.4  2003/05/05 12:27:27  juengel
 * Added method setGreen.
 *
 * Revision 1.3  2003/04/23 16:17:55  juengel
 * ColorTableReferenceColor::getColorClassForFlagsAndGoals returns a color class now.
 *
 * Revision 1.2  2003/04/02 13:21:11  juengel
 * Improved flag colors.
 *
 * Revision 1.1  2003/04/01 13:59:34  juengel
 * Added ColorTableReferenceColor
 *
 *
 */
